# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re, sys, socket
from Arnet import Subnet, Mask, NsLib
from IpLibConsts import DEFAULT_VRF
from collections import namedtuple, Counter
import Tac
from Bunch import Bunch
from RoutingUtils import RouteFec, RouteVia
from DeviceNameLib import kernelIntfToEosIntf

# pylint: disable-msg=C0321
# pylint: disable-msg=W1401

class RibdTask( Bunch ):
   numkeys = ( 'priority', 'n_newpolicy', 'n_reinit', 'n_flash',
               'n_reinit_async', 'n_rth_flash', 'n_rth_filtered', 'port',
               'n_rth_filtered_last', 'peeras', 'localas' )
   def __init__(self, **kwargs):
      d2 = self._parseTaskName( kwargs['name'] )
      if d2:
         kwargs.update(d2)
      for c in RibdTask.numkeys:
         if c in kwargs:
            if kwargs[c] is not None:
               kwargs[c] = int(kwargs[c])
      Bunch.__init__(self, **kwargs)

   def _parseTaskName( self, name ):
      '''
      task_name_r in gated/task.c prints task name differently based on whether
      task_addr and task_pid is set. For BGP tasks, the name begins with
      BGP_<peer_as>_<local_as> or BGP_<peer_as>
      '''
      m = re.match( r'BGP_(?P<peeras>\d+)(_(?P<localas>\d+))?' + \
            r'(\.(?P<peeraddr>[^+]+))?(\+(?P<port>\d+))?', name, re.S )
      if m:
         return m.groupdict()
      return None

   def isBgpPeer( self ):
      return self.name.startswith('BGP_') and not self.name.startswith('BGP_Proto')

   def isRibOut( self ):
      return self.name.startswith('RIB-Out')

   def __eq__( self, other ):
      if not isinstance(other, RibdTask):
         return NotImplemented
      return self.name == other.name and \
         self.n_newpolicy == other.n_newpolicy and \
         self.n_reinit == other.n_reinit and \
         self.n_flash == other.n_flash and \
         self.n_reinit_async == other.n_reinit_async

   def __ne__( self, other ):
      if not isinstance(other, RibdTask):
         return NotImplemented
      return not self.__eq__(other)

ipv4Re = r"[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}\.[12]?\d{1,2}"
ipv6Re = r"(" + \
         r"::([0-9a-f]{1,4}:)*" + ipv4Re + \
         r"|::([0-9a-f]{1,4}:)*[0-9a-f]{1,4}" + \
         r"|::" + \
         r"|([0-9a-f]{1,4}:)+:([0-9a-f]{1,4}:)*" + ipv4Re + \
         r"|([0-9a-f]{1,4}:)+:([0-9a-f]{1,4}:)*[0-9a-f]{1,4}" + \
         r"|([0-9a-f]{1,4}:)+:" + \
         r"|([0-9a-f]{1,4}:)+[0-9a-f]{1,4}" + \
         r")(%\w+)?"
srteipv4Re = r"\d+\|\d+\|" + ipv4Re
srteipv6Re = r"\d+\|\d+\|" + ipv6Re
maskLen4Re = r"\d{1,2}"
maskLen6Re = r"\d{1,3}"
rtStatesIgnorable = set( [ 'delete', 'release', 'hidden', 'suppressed',
                           'negpref', 'policyreject' ] )

class RibdDumpAttributeParser:
   def __init__( self, attributesText ):
      self.attributesText_ = attributesText

   def raw( self ):
      return self.attributesText_

   # XXX: the default delimeter between attribute name and value is an optional
   # ':' followed by one or more whitespace chars. Allowing the caller the
   # option to be more stringent allows us to avoid cases where the attribute's
   # name can appear elsewhere as a single word in the input, but as an
   # attribute name. cf. AutoLocalAddr ibid.

   def prp( self, propName, _type, findAll=False, delim=None):
      if delim is None:
         delim = r":*\s+"
      genericPattern = propName + delim + r"(?P<propValue>"+ _type + ")"
      if findAll:
         match = re.finditer( r""+ genericPattern,
                              self.attributesText_, re.S | re.M )
      else:
         match = re.search( r""+ genericPattern,
                            self.attributesText_, re.S | re.M )
      if match:
         if findAll:
            result = []
            for m in match:
               result.append( m.group( 'propValue' ) )
            return result
         else:
            return match.group( 'propValue' )
      else:
         if findAll:
            return []
         return None

   def getIPPrp( self, propName, findAll=False, delim=None ):
      return self.prp( propName, ipv4Re, findAll=findAll, delim=delim )

   def getIP6Prp( self, propName, findAll=False, delim=None ):
      return self.prp( propName, ipv6Re, findAll=findAll, delim=delim )

   def getIPprefixPrp( self, propName, findAll=False ):
      return self.prp( propName, ipv4Re + "/%s" % maskLen4Re, findAll=findAll )

   def getIP6prefixPrp( self, propName, findAll=False ):
      return self.prp( propName, ipv6Re + "/%s" % maskLen6Re, findAll=findAll )

   def getStringPrp( self, propName, findAll=False ):
      return self.prp( propName, r"\w+", findAll=findAll )

   def getNumPrp( self, propName, findAll=False ):
      return self.prp( propName, r"\d+", findAll=findAll )

   def getIntPrp( self, propName, findAll=False ):
      return self.prp( propName, r"-?\d+", findAll=findAll )

   def getFloatPrp( self, propName, findAll=False ):
      ret = self.prp( propName, r'[\d]+\.[\d]+' )
      return 0.0 if not ret else float( ret )

   def getHexProp( self, propName, findAll=False ):
      return self.prp( propName, r'[\da-fA-F]+', findAll=findAll )

class MioDumpParser( RibdDumpAttributeParser ):
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__( self,
         re.sub( r'^1\d0\s+', '', rawstr, flags=re.M ) )

   def getMioValue( self, path, tag ):
      values = self.prp( '/%s' % re.escape( path ), '((?!/).)*' )
      if values:
         # parse the the pattern:  '<sid> <tag> <type>: <value>'
         m = re.search(r'\n.*\s+%s.*:\s+(?P<value>\w+)' % tag, values )
         if m:
            return m.group( 'value' )

   def relayHashFirstAvailable( self ):
      return int( self.getIntPrp( 'Relay hash first available' ) )

   def relayNoReplyHandle( self ):
      return int( self.getIntPrp( 'Relay no reply handle' ) )

class RouteEntry( RibdDumpAttributeParser ):
   def __init__( self, address, maskLen, attributesText, addrFamily=socket.AF_INET ):
      RibdDumpAttributeParser.__init__(self, attributesText)
      self.address_ = address
      self.maskLen_ = maskLen
      self.af_ = addrFamily

   def address( self ):
      return self.address_

   def maskLen( self ):
      return self.maskLen_

   def mask( self ):
      return str( Mask( self.maskLen_, addrFamily=self.af_ ) )

   def _getProtocol( self ):
      match = re.match( r".*\t\t(?P<active>[\*\+\-/]*)(?P<protocol>[\w-]+)\s+"
                        r"(?P<memAddr>[0-9a-fA-Fx]+)\s+Preference:",
                        self.attributesText_, re.S | re.M )
      return match

   def protocol( self ):
      match = self._getProtocol()
      return match.group( "protocol" )

   def memAddr( self ):
      match = self._getProtocol()
      return match.group( "memAddr" )

   def isBestPath( self ):
      # Here we do not differentiate between RIB (unicast/multicast)
      match = self._getProtocol()
      return bool( match.group( "active" ) and match.group( "active" )[0] in "*+" )

   def isHolddown( self ):
      # Here we do not differentiate between RIBs (unicast/multicast)
      match = self._getProtocol()
      return bool( match.group( "active" ) and "/" in match.group( "active" ) )

   def getNhes( self ):
      # there can be multiple "Nhe:" lines (e.g. for a Static ECMP route)
      matches = re.finditer( r"\s+Nhe: (?P<nhIpAddr>%s|%s) \(\w+\)"
                             r"\sInstance: (?P<rtInst>[\d]+)/(?P<nheInst>[\d]+)" \
                             r"\sResolution: (?P<rrStatus>[\w]+)\s*\n" \
                             % ( ipv4Re, ipv6Re ),
                             self.attributesText_, re.S | re.M )
      nheDicts = {}
      if matches:
         for match in matches:
            nheDict = match.groupdict()
            nheDict[ 'rtInst' ] = int( nheDict[ 'rtInst' ] )
            nheDict[ 'nheInst' ] = int( nheDict[ 'nheInst' ] )
            nheDictKey = nheDict[ 'nhIpAddr' ]
            nheDicts[ nheDictKey ] = nheDict
      return nheDicts

   def isRecursive( self ):
      # Returns True if the routes nexthop is being recursively resolved
      return ( len( self.getNhes() ) > 0 )

   def rrStatus( self, nhIp ):
      nheDicts = self.getNhes()
      if nhIp in nheDicts:
         return nheDicts[ nhIp ][ 'rrStatus' ]
      return None

   def isInSyncWithNhe( self, nhIp=None ):
      nheDicts = self.getNhes()
      if nhIp:
         nheDict = nheDicts[ nhIp ]
         return ( nheDict[ 'rtInst' ] == nheDict[ 'nheInst' ] )
      else:
         # check that the route is in sync with each NHE
         for nheDict in nheDicts.itervalues():
            if ( nheDict[ 'rtInst' ] != nheDict[ 'nheInst' ] ):
               return False
      return True

   def nheInstance( self, nhIp ):
      nheDicts = self.getNhes()
      if nhIp in nheDicts:
         return nheDicts[ nhIp ][ 'nheInst' ]
      return None

   def rtInstance( self, nhIp ):
      nheDicts = self.getNhes()
      if nhIp in nheDicts:
         return nheDicts[ nhIp ][ 'rtInst' ]
      return None

   def nhe( self, af=None ):
      return self.getIPPrp( 'Nhe', af=af )

   def source( self ):
      if self.af_ == socket.AF_INET6:
         return self.getIP6Prp( 'Source' )
      else:
         return self.getIPPrp( 'Source' )

   def resolution( self ):
      return self.prp( 'Resolution', '\w+' )
   
   def isLabel( self ):
      states = self.state()
      return ( states and 'Label' in states )

   def getIPPrp( self, propName, findAll=False, delim=None, af=None ):
      # pylint: disable-msg=arguments-differ
      af = af or self.af_
      if af == socket.AF_INET:
         return self.prp( propName, ipv4Re, delim=delim )
      else:
         return self.prp( propName, ipv6Re, delim=delim )

   def getStringPrp( self, propName, findAll=False ):
      return self.prp( propName, r"\w+" )

   def getNumPrp( self, propName, findAll=False ):
      return self.prp( propName, r"\d+" )

   def getIntPrp( self, propName, findAll=False ):
      return self.prp( propName, r"-?\d+" )

   def preference( self ):
      return self.getIntPrp( 'Preference' )

   def attrId( self ):
      return int(self.prp( 'AttrId', '[0-9a-f]+' ), 16)

   def protoInstanceId( self ):
      return self.getNumPrp( 'InstanceId' )

   def announcementBits( self ):
      return self.prp( r'Announcement bits\(\d+\)', r'[^\n]+' )

   def localAS( self ):
      return self.getNumPrp( 'Local AS' )

   def age( self ):
      return self.getNumPrp( 'Age' )

   def metric( self ):
      return self.getNumPrp( 'Metric' )

   def localPref( self ):
      return self.getNumPrp( 'Metric2' )

   def peerAS( self ):
      return self.getNumPrp( 'Peer AS' )

   def tag( self ):
      return self.getNumPrp( 'Tag' )

   def rxPathId( self ):
      return self.getNumPrp( 'rx_pathid' )

   def task( self ):
      return self.getStringPrp( 'Task' )

   def state( self ):
      return self.prp( 'State', r'<[\w ]+>' )
   
   def isPolRejected( self ):
      return 'PolicyReject' in self.state()
   
   def isNegPref( self ):
      return 'NegPref' in self.state()

   def states( self ):
      return [ s.lower() for s in self.state()[1:-1].split() ]

   def ignorable( self ):
      return True if set( self.states() ).intersection( rtStatesIgnorable ) \
            else False

   def isBgpEcmp( self ):
      return 'bgpecmp' in self.states()

   def listenTaskTsi( self ):
      return self.prp( 'BGP route modification:', '.+' )

   def aptxAnyListentTaskTsi( self ):
      return self.prp( 'BGP \(aptx any\) route modification:', '.+' )

   def communities( self ):
      return self.prp( 'Communities', r'[\d: ]+' )

   def adjacency( self ):
      return long(self.prp( 'Adjacency Id', '[0-9a-f]+' ), 16)

   def ngw( self ):
      return self.getNumPrp( 'N_gw' )

   def origin( self ):
      genericPattern = r".*AS Path:\s+([0-9{}()\[\],]+\s)*(?P<propValue>i|e|\?)"
      match = re.match( r""+ genericPattern,
                  self.attributesText_, re.S | re.M )
      if match:
         return match.group( 'propValue' ).strip()
      else:
         return None

   def asPath( self ):
      match = re.match( r".*AS Path:\s+(?P<aspath>([0-9{}()\[\],]+\s)*)",
                        self.attributesText_, re.S | re.M )
      if match:
         return match.group( 'aspath' ).strip()
      else:
         return None

   def asPathHashId( self ):
      match = re.match( r'.*AS Path:.*HashID\s+(?P<hashId>\d+)',
                        self.attributesText_, re.S | re.M )
      if match:
         return int( match.groups( 'hashId' )[ 0 ] )

   def atomicagg( self ):
      match = re.search( r""+ 'Atomic',
                  self.attributesText_, re.S | re.M )
      return match is not None

   # Return AS#:a.b.c.d
   def aggregator( self ):
      return self.prp( 'AG', r'[\d:.]+' )

   def extcommunities( self ):
      lst = []
      propName = 'Route-Origin-AS'
      _type = r'[\d:]+'
      genericPattern = propName + ":(?P<propValue>"+ _type + ")"
      offset = 0
      while True:
         match = re.search( r""+ genericPattern,
                     self.attributesText_[ offset: ] )
         if match:
            lst.append( match.group( 'propValue' ).strip() )
            offset += match.end()
         else:
            return lst

   def _contributorAttributes( self, attrMatchText ):
      attributes = re.search( r"proto\s+(?P<proto>\S*)\s+metric\s+(?P<metric>\S*)"
                              r"\s+preference\s+(?P<preference>\d*)\s+"
                              r"RIB:\s+(?P<rib>\S*)", attrMatchText )
      return attributes.groupdict()

   def successfulContributors( self ):
      match = re.search( "(Contributing Routes:\n)(.*?)(\n\n)",
                         self.attributesText_ ,re.MULTILINE | re.DOTALL )
      if match:
         matchText = match.group()
         routes = re.finditer( r"\s+(?P<address>.*?)\s+mask\s+(?P<mask>.*?)\s+"
                               "(?P<attributes>.*)\n", matchText )
         return [ dict( r.groupdict().items() + \
                        self._contributorAttributes(
                                       r.group( "attributes" ) ).items() ) \
                   for r in routes ]
      return []

   def srTeBsid( self ):
      bsid = self.prp( 'SRTE-bsid', r'[\d\-a-fA-F0x]+' )
      if bsid is not None:
         return int( bsid, 16 ) if bsid != '-' else - 1

   def srTeEnlp( self ):
      enlp = self.prp( 'SRTE-enlp', r'[\d]+' )
      return int( enlp ) if enlp is not None else None

   def srTePreference( self ):
      pref = self.prp( 'SRTE-pref', r'\d+' )
      return int( pref )

   def srTeSlWt( self ):
      slwts = self.prp( r'SRTE-SL\d+', r'[\[\]0x\da-fA-F\-Wt: ]+', findAll=True,
                        delim=r': ' )
      slws = []
      for slw in slwts:
         sl = re.findall( '0x[0-9a-f]+', slw )
         sl = [ int( s, 16 ) for s in sl ]
         if 'hasNonType1' in slw:
            wt = slw.split( 'Wt:' )[ 1 ].split( '' )[ 0 ].strip()
         else:
            wt = slw.split( 'Wt:' )[ 1 ].strip()
         wt = int( wt ) if wt != '-' else None
         slws.append( ( sl, wt ) )
      return slws
   
   def getDynPolicyMatches( self ):
      return int( self.prp( 'Num matching dyn policies', '[0-9a-f]+' ) )

communityRe = \
      r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
      r"(?P<hash>\d+)\s+Count\s+(?P<count>\d+)\s+" + \
      r"Flags\s+(?P<flags>(\w+|<\w+ >))\s+\((?P<flagvalue>0x\d+)\)\s+" + \
      r"Communities:\s(?P<comm>(((\d+:\d+\s)+)|(regex.*?, state.*?\s)))"

localAsInfoEntriesRe = \
      r"\s+AS: (\d+) count: (\d+) loop: (\d+)"

bgJobInfoEntriesRe = \
      r"\s+(\S+) (\S+) (.+) priority (\d+) runs (\d+) <(\S+)>"

def cacheEntries( cache ):
   '''
   Each entry in this section is as follows
   150             Cache 0xdeadbeef size 10 cleared 4
   150             AsPath 3 useCount 0 match no
   150             AsPath 4 useCount 0 match no
   150             AsPath 5 useCount 0 match no
   150             AsPath 6 useCount 0 match no
   150             Cache End
   '''
   if not cache:
      return []
   entries = []
   pat = r'AsPath (?P<aspathid>\d+) useCount (?P<useCount>\d+) ' + \
         r'match (?P<match>\S+)'
   lst = re.finditer( pat, cache, re.S | re.M )
   for m in lst:
      entries.append( Bunch( **m.groupdict() ) )
   return entries

class CommunityList( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.address = int( self.address, 16 )
      self.commListAddress = int( self.commListAddress, 16 )
      self.refcount = int( self.refcount )
      self.entry = None
      self.exact = self.name.startswith('.exact')
      self.cacheSize = int( self.cacheSize ) if self.cacheSize else 0
      self.cacheClearCount = int( self.cacheClearCount ) \
                              if self.cacheClearCount else 0
      self.cacheId = int( self.cacheId ) if self.cacheId else 0

   @Tac.memoize
   def cacheEntries( self ):
      return cacheEntries( self.cache )

   @staticmethod
   def parse( content ):
      """Parses ADVFT_COMM_LIST entries raw content dumped by show tech ribd or
      show policy GII command returns a dictionary of @CommunityList objects
      with name as key. Object for corresponding as_comm_list
      (CommunityListEntry) object is also parsed and stored"""
      cdict = {}
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      cles = CommunityListEntry.parse( content )
      m = re.match( r".*(?P<commlists>Community Lists:.*?)" \
                     r"^\s*Community Lists", content, re.S | re.M )
      if m:
         clpat = r'\s+(?P<name>\S+) Id (?P<id>\S+) ' \
               r'refcount (?P<refcount>\d+) ' \
               r'(?P<address>\w+) list (?P<commListAddress>\w+) ' \
               r'as_comm_list <(\S+\s)*syment(\s\S+)*> (?P<commtype>\S+)' \
               r'(?P<cache>\s*Cache (?P<cacheId>\d+) size (?P<cacheSize>\d+) ' \
               r'cleared (?P<cacheClearCount>\d+)\n.*?Cache End)?'
         mlst = re.finditer( clpat, content, re.S | re.M )
         for lm in mlst:
            clist = CommunityList( **lm.groupdict() )
            cdict[ clist.name ] = clist
            if clist.id in cles:
               clist.entry = cles[ clist.id ]
      return cdict

class CommunityListEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.address = int( self.address, 16 )
      self.refcount = int( self.refcount )
      self.count = int( self.count )
      self.communities = {}

   @staticmethod
   def parse( content ):
      """Parses standard community list entries raw content dumped by show tech
      ribd or show policy GII command returns a dictionary of
      @CommunityListEntry objects with ID as key"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      commEntries = CommunityEntry.parse( content )
      extCommEntries = ExtCommunityEntry.parse( content )
      cdict = {}
      m = re.match( r".*(?P<commlists>Community List Table.*?)" \
                     r"^\s*Community List", content, re.S | re.M )
      if m:
         clpat = r'Id (?P<id>\S+) (?P<address>\w+)' \
               r' refcount (?P<refcount>\d+)' \
               r' flags (?P<flags>\w+) count (?P<count>\d+)' \
               r' (?P<commtype>\S+)' \
               r'\s*Comm Entries:\s?(?P<entries>[^\n]+)'
         content = m.group( 'commlists' )
         mlst = re.finditer( clpat, content, re.S | re.M )
         for lm in mlst:
            clist = CommunityListEntry( **lm.groupdict() )
            if clist.entries.strip() != "":
               entries = map( int, clist.entries.strip().split() )
               if clist.commtype == 'standard':
                  for e in entries:
                     clist.communities[ e ] = commEntries[ e ]
               else:
                  for e in entries:
                     clist.communities[ e ] = extCommEntries[ e ]
            cdict[ clist.id ] = clist
      return cdict

class IsisIntfInfo( Bunch ):
   def __init__( self, **kwds ):
      Bunch.__init__(self, **kwds )

   @staticmethod
   def parse( content ):
      """Parses Interface information from raw content dumped by show tech ribd"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      m = re.match( r".*(?P<isisIntfInfo>circuits:.*?)\s*level \d lsp database:",
                    content, re.S | re.M )
      if m:
         content = m.group( 'isisIntfInfo' )

      ret = {}
      intfNamePattern = r'\s*name'
      intfPattern = r"\s*(?P<intfName>([\w/\.]+)) index (?P<ifIndex>(\d)+) " \
         r"level (?P<levelNumber>(\d)+)\s*(?P<data>(.*\n)*)" \
         r"\s*BFD state is (?P<bfdState>\w+)\s*BFD req sent is (?P<bfdReqSent>(\d)+)"

      t = re.split(intfNamePattern, content, re.M )

      for intfIter in range( 1, len( t ) ):
         intf = re.match( intfPattern, t[ intfIter ] )
         ret[ intf.group( 'intfName' ) ] = IsisIntfInfo( **intf.groupdict() )

      return ret

class CommunityEntry( Bunch ):
   def __init__( self, **kwds ):
      Bunch.__init__(self, **kwds )
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )
      self.flagvalue = int( self.flagvalue, 16 )

   @staticmethod
   def parse( content ):
      """Parses standard community entries raw content dumped by show tech ribd
      or show policy GII command returns a dictionary of @CommunityEntry objects
      with ID of the entry as key"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      lst = re.finditer( r"(" + communityRe + ")", content, re.S | re.M )
      clist = {}
      for m in lst:
         c = CommunityEntry( **m.groupdict() )
         clist[ c.id ] = c
      return clist

   @staticmethod
   def parseAll( content ):
      """Parses standard community entries raw content dumped by show tech ribd
      or show policy GII command returns a dictionary of @CommunityEntry objects
      with ID of the entry as key"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      if re.search( r"No communities in the table", content, re.S | re.M ):
         return {}
      m = re.search( r"Communities Table\s+(?P<CommunitiesTable>.*?)" + \
                     r"\sExtended Communities Table.*", content, re.S | re.M )
      content = m.group( 'CommunitiesTable' )
      communityReAll = \
            r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
            r"(?P<hash>\d+)\s+Count\s+(?P<count>\d+)\s+" + \
            r"Flags\s+(?P<flags>[<\w\s>]+)\s+\((?P<flagvalue>0x\d+)\)\s+" + \
            r"Communities:\s+(?P<comm>.*?)\n"
      lst = re.finditer( r"(" + communityReAll + ")", content, re.S | re.M )
      clist = {}
      for m in lst:
         c = CommunityEntry( **m.groupdict() )
         clist[ c.id ] = c
      return clist

class ExtCommunityEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )
      self.flagvalue = int( self.flagvalue, 16 )

   def __repr__( self ):
      return repr( self.__dict__ )

   @staticmethod
   def parse( content ):
      """Parses standard extended community entries raw content dumped by show
      tech ribd or show policy GII command returns a dictionary of
      @ExtCommunityEntry objects with ID of the entry as key"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      if re.search( r"No extended communities in the table", content, re.S | re.M ):
         return {}
      m = re.search( r"Extended Communities Table\s+(?P<CommunitiesTable>.*?)" +
                     r"\sLarge Communities Table.*", content, re.S | re.M )
      content = m.group( 'CommunitiesTable' )
      pat = \
         r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
         r"(?P<hash>\d+)\s+Count\s+(?P<count>\d+)\s+" + \
         r"Flags\s+(?P<flags>\S+)\s+\((?P<flagvalue>0x\d+)\)\s+" + \
         r"Communities: (?P<comm>(\S+\s)*)"
      clist = {}
      lst = re.finditer( pat, content, re.S | re.M )
      for m in lst:
         c = ExtCommunityEntry( **m.groupdict() )
         clist[ c.id ] = c
      return clist

class LargeCommunityEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )
      self.flagvalue = int( self.flagvalue, 16 )

   def __repr__( self ):
      return repr( self.__dict__ )

   @staticmethod
   def parse( content ):
      """Parses large community entries raw content dumped by show
      tech ribd or show policy GII command returns a dictionary of
      @LargeCommunityEntry objects with ID of the entry as key"""
      content = re.sub( r'^1\d0\s', '', content, flags=re.M )
      if re.search( r"No large communities in the table", content, re.S | re.M ):
         return {}
      m = re.search( r"Large Communities Table\s+(?P<CommunitiesTable>.*?)" +
                     r"\sAS Path Info Table.*", content, re.S | re.M )
      content = m.group( 'CommunitiesTable' )
      pat = \
         r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
         r"(?P<hash>\d+)\s+Count\s+(?P<count>\d+)\s+" + \
         r"Flags\s+(?P<flags>\S+)\s+\((?P<flagvalue>0x\d+)\)\s+" + \
         r"Communities: (?P<comm>(\S+\s)*)"
      clist = {}
      lst = re.finditer( pat, content, re.S | re.M )
      for m in lst:
         c = LargeCommunityEntry( **m.groupdict() )
         clist[ c.id ] = c
      return clist

class RibOutPeer( dict ):
   def __init__(self, **kwargs):
      dict.__init__(self, kwargs)
      self.__dict__.update(kwargs)
   def __repr__( self ):
      keys = self.__dict__.keys()
      keys.sort()
      args = ', '.join(['%s=%r' % (key, self[key]) for key in keys])
      return '%s(%s)' % (self.__class__.__name__, args)

class RibOutBgpAnnounceQueue( RibOutPeer ):
   pass

ospf2LsaRe = ( r"type (?P<type>\S+) id (?P<lsid>\d+.\d+.\d+.\d+) advrt "
               "(?P<advrt>\d+.\d+.\d+.\d+) seq (?P<seq>\S+) \[mem: \S+\]"
               "\s+left \S+ right \S+ bitindex \S+\s+age (?P<age>\d+) "
               "cksum \S+ len \d+ flags <(?P<flags>[^>]*)>" )

class Ospf2Lsas( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)

class BgpInstance( RibdDumpAttributeParser ):
   '''
   Parses Bgp Instance dump (output of show bgp dump instance)
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self, 
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   def maxEcmp( self ):
      return int( self.getNumPrp( 'Max ecmp' ) )

   def maxPaths( self ):
      return int( self.getNumPrp( 'Max paths' ) )

   def isIpv4Unicast( self ):
      return self.getStringPrp( 'Default IPv4 Unicast' ) == 'yes'

   def isIpv6Unicast( self ):
      return self.getStringPrp( 'Default IPv6 Unicast' ) == 'yes'

class BgpPeer( RibdDumpAttributeParser ):
   '''
   Parses BGP peer dump (bgp_peer_dump), section of 'rdc dump' or one produced
   by GII command 'show bgp dump peer <addr>'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   def state( self ):
      return self.getStringPrp( 'State' )

   def flags( self ):
      return self.prp( 'Flags', r'<[\w ]+>' )

   def peeringIntf( self ):
      ifap = self.getStringPrp( 'ifap' )
      return None if ifap == 'NULL' else ifap

   def peerType( self ):
      return self.getStringPrp( 'Type' )

   def hisCapabilities( self ):
      return self.prp( 'His Capabilities', r'<[\w ]+>' )

   def myCapabilities( self ):
      return self.prp( 'My Capabilities', r'<[\w ]+>' )

   def options( self ):
      return self.prp( 'Options', r'<[\w ]+>' )

   def maxRoutes( self ):
      return int( self.getNumPrp( 'Max Routes' ) )

   def maxAcceptedRoutes( self ):
      return int( self.getNumPrp( 'Max Accepted Routes' ) )

   def outDelay( self ):
      od = self.getNumPrp( 'Out-delay' )
      return int( od ) if od is not None else 0

   def linkbandwidthDelay( self ):
      ld = self.getNumPrp( 'Linkbandwidth-delay' )
      return int( ld ) if ld is not None else 0

   def allowAsLoop( self ):
      return int( self.getNumPrp( 'Allowas loop' ) )

   def totalRoutes( self ):
      return int( self.getNumPrp( 'Total Routes' ) )

   def metricOut( self ):
      return int( self.getNumPrp( 'Metric Out' ) )

   def weight( self ):
      return int( self.getNumPrp( 'Weight' ) )

   def weightV4( self ):
      return int( self.getNumPrp( 'IPV4 Weight' ) )

   def weightV6( self ):
      return int( self.getNumPrp( 'IPV6 Weight' ) )

   def preference( self ):
      return int( self.getNumPrp( 'Preference' ) )

   def peerID( self ):
      return self.getStringPrp( 'Peer ID' )

   def localID( self ):
      return self.getIPPrp( 'Local ID' )

   def activeHoldTime( self ):
      return int( self.getNumPrp( 'Active Holdtime') )

   def activeKeepAliveTime( self ):
      return int(self.getNumPrp( 'Active KeepAliveTime') )

   def sendEor( self ):
      return self.getStringPrp( 'Send-Eor' )

   def idleHoldTimeState( self ):
      return self.getStringPrp( 'IdleHold timer is' )

   def idleHoldTimeValue( self ):
      return self.getIntPrp( 'Current IdleHold Timer Value' )

   def idleHoldTimeLast( self ):
      return self.getIntPrp( 'Last IdleHold Timer expire Time' )

   def defaultUnsupIdleHoldTime( self ):
      return self.getIntPrp( 'Default IdleHold timer for UnsupportedCap is' )

   def ribOut( self ):
      return self.getStringPrp( 'RIB-Out')

   # XXX: The literal string "AutoLocalAddr" can also appear as a token in the
   # "Options" flags. We must be stricter with the delimiter, AutoLocalAddr can
   # return a bogus match.

   def autoLocalAddr( self ):
      ala = self.getIPPrp( 'AutoLocalAddr', delim=r':\s*' )
      if ala:
         return ala
      return self.getIP6Prp( 'AutoLocalAddr', delim=r':\s*' )

   def localAddr( self ):
      la = self.getIPPrp( 'Local' )
      if la:
         return la
      return self.getIP6Prp( 'Local' )

   @Tac.memoize
   def ribOutBit( self ):
      m = re.search(r'RIB-Out \S+\s+Bit: (?P<brobit>\d+)', self.attributesText_,
                  re.M | re.S )
      if m:
         return int( m.group( 'brobit' ) )
      return -1

   def rtInitTime( self ):
      return self.getFloatPrp( 'Rt Init Time' )

   def importPolicy( self ):
      return self.prp( 'Peer Import Policy', r'\S+')

   def exportPolicy( self ):
      return self.prp( 'Peer Export Policy', r'\S+')

   def importPolicyIPv4Unicast( self ):
      return self.prp( 'Peer Import Policy for IPv4 Unicast Routes', r'\S+' )

   def exportPolicyIPv4Unicast( self ):
      return self.prp( 'Peer Export Policy for IPv4 Unicast Routes', r'\S+' )

   def importPolicyIPv6Unicast( self ):
      return self.prp( 'Peer Import Policy for IPv6 Unicast Routes', r'\S+' )

   def exportPolicyIPv6Unicast( self ):
      return self.prp( 'Peer Export Policy for IPv6 Unicast Routes', r'\S+' )

   def nImportApply( self ):
      return int( self.getNumPrp( 'n_import_apply' ) )

   def nSoftIn( self ):
      return int( self.getNumPrp( 'n_soft_in' ) )

   def nSoftOut( self ):
      return int( self.getNumPrp( 'n_soft_out' ) )

   def _echoWithdrawals( self ):
      return re.search(
            r'Echo Withdrawals:\s+IPv4 (?P<ipv4>\d+)\s+IPv6 (?P<ipv6>\d+)',
            self.attributesText_, re.M | re.S )

   @Tac.memoize
   def echoWithdrawalsIpv4( self ):
      '''IPv4 NLRIs Echo (special) withdrawals sent to a peer. We don't
      yet expose this via 'show ip|ipv6 bgp nei <>'''
      m = self._echoWithdrawals()
      if m:
         return int( m.group( 'ipv4' ) )
      return -1

   @Tac.memoize
   def echoWithdrawalsIpv6( self ):
      '''IPv6 NLRIs Echo (special) withdrawals sent to a peer. We don't
      yet expose this via 'show ip|ipv6 bgp nei <>'''
      m = self._echoWithdrawals()
      if m:
         return int( m.group( 'ipv6' ) )
      return -1

   @Tac.memoize
   def echoWithdrawals( self ):
      '''IPv4, IPv6 NLRIs Echo (special) withdrawals sent to a peer. We don't
      yet expose this via 'show ip|ipv6 bgp nei <>'''
      m = self._echoWithdrawals()
      if m:
         return ( int( m.group( 'ipv4' ) ), int( m.group( 'ipv6' ) ) )
      return ( -1, -1 )

   def _bgpPeerPrp( self, propName, bgpPropName, bgpPropType=str ):
      '''Parse the common pattern of '<Prp>:\\s+(<Attr + Value>[\t\n])+' in
      bgp_peer_dump, where attr-value pairs are delimited by tabs. Examples:
      'Inbound IPv4|v6 prefix counts', 'Input|Output messages', etc.
      '''
      prop = self.prp( propName, r'([\w+\.\ \-]+[\t]?)+\n' )
      if not prop:
         return None
      # pylint thinks self.prp returns a list, pylint: disable-msg=E1103 
      bgpProps = prop.strip().split( '\t' )
      for bgpProp in bgpProps:
         m = re.search( bgpPropName + r'\s+(.*)', bgpProp )
         if m:
            return bgpPropType( m.group( 1 ) )
      return None

   def updateMessageTime( self ):
      prps = [ 'Last Outbound Update Time', 'Outbound Eor Time' ]
      return [ float( self.prp( p, r'\S+' ) ) for p in prps ]

   def recvMsgQueueSize( self ):
      '''Returns a tuple with first element as # of pending messages and second
      as total number of bytes'''
      m = re.search( r'Message Queue: #(?P<size>\d+) elements (?P<bytes>\d+) bytes',
                     self.attributesText_, re.S )
      if not m:
         return (0, 0)
      return ( int(m.group('size')), int(m.group('bytes')) )

   def recvMsgQueueFull( self ):
      '''Returns True if peer receive message queue is full, False otherwise'''
      m = re.search( r'Message Queue: .* Full: (?P<full>\S+)',
                     self.attributesText_, re.S )
      if not m:
         return False
      return m.group( 'full' ) == 'yes'

   def failedConnectionAttempts( self ):
      '''Returns the number of failed connection attempts'''
      return int( self.getNumPrp( 'Failed connection attempts' ) )

class BgpAdjRibOut( RibdDumpAttributeParser ):
   '''
   Parses GII 'show bgp dump ribout <peer> addr/mask' command output (produced
   by bgp_dump_adj_rib_out_rth)

   Optional @peerdict specifies collection with bro_bit is key and peer address
   as value

   An rto and wrto in output of show bgp dump ribout <peer> <prefix> appears as
   follows ('\' at the end of line is just to keep pylint happy)

   GateD-bs321.sjc.aristanetworks.com> show bgp dump ribout 1.0.0.1 30.10.17.0/24

   150 30.10.17.0/24 u     - {{adv: 0 }} [+] {{ 2 }} 40 ? (HashID 3) \
         Nexthop: 1.0.1.2  Community/AR 0xee286234 (id 1, count 1) ascount 1
   150             {{ 1 }} ==
   150             [-] {{ 0 }} 40 ? (HashID 3) Nexthop: 1.0.1.2  \
         Community/AR 0xee286234 (id 1, count 1) ascount 1 [!!W!!]
   150             [ 10 ? (HashID 4) Nexthop: 1.0.0.1  \
         Community/AR 0xee286234 (id 1, count 1) ascount 1 <Queued EchoWithdrawal> ]
   '''
   def __init__( self, rawstr, peerdict=None ):
      self.peerdict_ = peerdict
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   @Tac.memoize
   def empty( self ):
      pass

   @Tac.memoize
   def prefix( self ):
      m = re.search( r'^(?P<prefix>\S+/\S+) u\t-',
            self.attributesText_, re.S | re.M )
      if m:
         return m.group( 'prefix' )
      return None

   def _xrtoPending( self, kind ):
      bitpats = re.finditer( r'\[\%s] {{ (?P<bits>(\d+ )+)}}' % kind,
                     self.attributesText_, re.S | re.M )
      bits = []
      for match in bitpats:
         bits = bits + match.group( 'bits' ).strip().split()
      if self.peerdict_:
         return map( lambda x: self.peerdict_[ int(x) ], bits )
      return map( int, bits )

   def isWithdrawal( self ):
      '''Returns if this entp represents a regular withdrwal'''
      return re.search( r'no metrics <Queued>',
                        self.attributesText_, re.S | re.M )

   @Tac.memoize
   def rtoPending( self ):
      '''List of bro bits for peers to whom this announcement has not yet been
      sent'''
      return self._xrtoPending( '+' )

   @Tac.memoize
   def wrtoPending( self, peerdict=None ):
      '''List of bro bits for peers to whom echo withdrawals has not yet been
      sent'''
      return self._xrtoPending( '-' )

class BgpBroEntpQueue( RibdDumpAttributeParser ):
   '''
   Parses bro_entp_queue (bgp_dump_adj_rib_out) section of GII 'show bgp dump ribout
   <peer>'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   def entpQueue( self ):
      M = re.finditer( r'(?P<prefix>\S+) brth (?P<brth>\w+) '
                      'brte (?P<brte>\w+) flags (?P<flags>0x[0-9a-f]{1}) '
                      'asp (?P<asp>[0-9]+) pathid (?P<pathid>0x[0-9a-f]{1,4}?)', 
                       self.attributesText_)
      return [m.groupdict() for m in M]

class BroPolicyApplicationStatus( RibdDumpAttributeParser ):
   '''
   Parses Policy application Status section of GII 'show bgp dump ribout <peer>'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__( self, re.sub( r'^1\d0\s+', '', rawstr,
                                                      flags=re.M ) )

   def statusDict( self ):
      M = re.finditer( r'Afi (?P<Afi>\d+), Safi (?P<Safi>\d+): '
                       '<(?P<Status>.*)>\n', self.attributesText_ )
      return [ Bunch( **m.groupdict() ) for m in M ]

class BgpRibOut( RibdDumpAttributeParser ):
   '''
   Parses BGP rib out (bgp_rib_out_dump), section of GII 'show bgp dump ribout
   <peer>'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__( self, re.sub( r'^1\d0\s+', '', rawstr,
                                                      flags=re.M ) )

   def totalPeers( self ):
      m = re.search( r'RIB-Out contains (?P<total>\d+) total peers',
                            self.attributesText_, re.S | re.M )
      if m:
         return int( m.group( 'total' ) )
      return 0

   def v4EstablishedPeers( self ):
      m = re.search( r'RIB-Out contains \d+ total peers'
                     r' .(?P<v4>\d+) v4 established', self.attributesText_,
                     re.S | re.M )
      if m:
         return int( m.group( 'v4' ) )
      return 0

   def v6EstablishedPeers( self ):
      m = re.search( r'RIB-Out contains \d+ total peers'
                     r' .(?P<v4>\d+) v4 established, (?P<v6>\d+) v6 established',
                     self.attributesText_, re.S | re.M )
      if m:
         return int( m.group( 'v6' ) )
      return 0

   @Tac.memoize
   def getInSyncPeersDict( self ):
      '''Returns a dict of 'in sync' peers indexed by their bro bit. value is
      peer address'''
      pat = r'\s+(?P<peer>\S+)\s+(?P<bro_bit>\d+)\s+.Established.\s.in sync'
      lst = re.finditer( pat, self.attributesText_, re.S | re.M )
      d = {}
      for m in lst:
         d[ int( m.group('bro_bit') )] = m.group( 'peer' )
      return d

   def flashBacklogSize( self ):
      m = re.search( r'Flash backlog size: (?P<backlog>\d+)',
                     self.attributesText_, re.S | re.M )
      if m:
         return int( m.group( 'backlog' ) )
      return 0

   @Tac.memoize
   def getBgpAdjRibOutDict( self ):
      '''Returns a dict of BgpAdjRibOut instances indexed by their prefix. Value is
      the instance of the class. Here is a sample to map:
         Adj-RIB-Out: entp 2
             [ queued announcements in brackets ]
         1.0.3.0/24 u    - i (HashID 1)  pref 100
         1.0.4.0/24 u    - {{adv: rcvnbr-bro rtbit 10 }} 2 i \
               (HashID 4) Nexthop: 1.0.1.2  Nexthop: 1.0.1.2 pref 10     0

         bro_entp_queue
         '''
      ipv6ReExt = r'(%s|%s)/%s?' % ( ipv4Re, ipv6Re, maskLen6Re )
      pattern = r'\s*(%s \S.*?)\n' % ( ipv6ReExt )
      adjMatch = re.search( r'\s*Adj-RIB-Out:.*?\n.*?\n(.*?\n)\s*bro_entp_queue',
                            self.attributesText_, re.S | re.M )
      ribOutDict = {}
      if adjMatch:
         adjText = adjMatch.group( 1 )
         routes = re.finditer( pattern, adjText, re.S | re.M )
         for route in routes:
            adjRibOutInst = BgpAdjRibOut( route.group( 1 ) )
            ribOutDict[ adjRibOutInst.prefix() ] = adjRibOutInst
      return ribOutDict

   def rrClient( self ):
      return int( self.getNumPrp( 'RR Client' ) )

   def kflags( self ):
      m = re.search( r'KFlags: (?P<flagvalue>\S+) <(?P<flagstr.*?)>\n',
            self.attributesText_, re.S | re.M )
      if m:
         return ( int( m.group( 'flagvalue', 16 ) ), m.group( 'flagstr' ) )
      return (0, '' )

   def flags( self ):
      m = re.search( r'Flags: (?P<flagvalue>\S+) <(?P<flagstr>.*?)>\n',
            self.attributesText_, re.S | re.M )
      if m:
         flagsStr = m.group( 'flagstr' )
         return ( int( m.group( 'flagvalue' ), 16 ), flagsStr )
      return ( 0, '' )

   def faultFlags( self ):
      m = re.search( r'Fault Flags: (?P<flagvalue>\S+) <(?P<flagstr>.*?)>\n',
            self.attributesText_, re.S | re.M )
      if m:
         return ( int( m.group( 'flagvalue' ), 16 ), m.group( 'flagstr' ) )
      return (0, '' )

   def capabilities( self ):
      m = re.search( r'Caps: (?P<v>\S+) <(?P<s>.*?)>\n',
            self.attributesText_, re.S | re.M )
      if m:
         return ( int( m.group( 'v' ), 16 ), m.group( 's' ) )
      return (0, '' )

   def rtoTime( self ):
      return int( self.getNumPrp( 'Out Delay' ) )

   def metricOut( self ):
      return int( self.getNumPrp( 'Metric Out' ) )

   def exportLocalPref( self ):
      return int( self.getNumPrp( 'Export Localpref') )

   def nSoftOut( self ):
      return int( self.getNumPrp( 'Bro soft out count' ) )

   def policy( self ):
      m = re.search( r'Peer Policy: (?P<v>\w+) .(?P<s>\S+)\)',
            self.attributesText_, re.S | re.M )
      if m:
         return ( m.group( 'v' ), m.group( 's' ) )
      return ('', '' )

   def policyIpv4Unicast( self ):
      m = re.search( 'Peer Policy for IPv4 Unicast Routes: '
               r'(?P<v>\w+) .(?P<s>\S+)\)', self.attributesText_, re.S | re.M )
      if m:
         return ( m.group( 'v' ), m.group( 's' ) )
      return ('', '' )

   def policyIpv6Unicast( self ):
      m = re.search( 'Peer Policy for IPv6 Unicast Routes: '
               r'(?P<v>\w+) .(?P<s>\S+)\)', self.attributesText_, re.S | re.M )
      if m:
         return ( m.group( 'v' ), m.group( 's' ) )
      return ('', '' )

   def policyApplicationStatus( self ):
      m = re.search( r'Policy application status:\n\s*(?P<Status>.*?)\n\s*'
                     'RIB-Out Peer Bit Usage:', self.attributesText_, re.S | re.M )
      if m:
         return BroPolicyApplicationStatus( m.group( 'Status' ) + '\n' ).statusDict()
      return []

class SharedAdjacencyTable( RibdDumpAttributeParser ):
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self, re.sub(r'^(1\d0)?\s+', '', rawstr,
                                                    flags=re.M))

   @Tac.memoize
   def adjacencyList( self ):
      """
      There may be multiple nexthops in an adjacency. The dump for an
      adjacency starts with "Id". The dump for a next-hop entry starts with
      "NextHop".
      """
      lst = re.finditer( r"(?P<attributes>^Id.*\n(((?:^NextHop.*\n)+"
                         "(?:^Backup-NextHop.*\n)*(?:^RtEntry.*\n)*)|"
                         "((?:^NexthopGroup.*\n)+(?:^RtEntry.*\n)*))"
                         "(?:^rt_nentries.*\n)"
                         "(?:^rt_adj_ix.*\n)?"
                         "(?:^>> Counted nentries.*\n)"
                         "(?:^>> Free LGDA indices.*)"
                         "(\n)?"
                         "((?:^IS-IS Adjacency Segments Refcount.*)"
                         "(?:\n^IS-IS Adjacency Segment.*)+)?"
                         "(\n)?"
                         "((?:^IS-IS Prefix Segments Refcount.*)"
                         "(?:\n^IS-IS Prefix Segment.*)+)?)",
               self.attributesText_, re.M )
      adjList = {}
      for match in lst:
         data = match.group( "attributes" )
         adj = RtAdjacency( data )
         adjList[ adj.id() ] = adj

      tst = re.finditer( r"(?P<attributes>^Id.*\n(?:^Tunnel-ids.*\n)"
                         "(?:^TunnelId.*\n)+"
                         "((?:^NextHop.*\n)(?:^Labels.*\n))*"
                         "(?:^RtEntry.*\n)*)",
                         self.attributesText_, re.M )
      for match in tst:
         data = match.group( "attributes" )
         adj = RtAdjacency( data )
         adjList[ adj.id() ] = adj
      return adjList

   @Tac.memoize
   def numOrphans( self ):
      '''Returns number of adjacencies that have no rt_entries associated with
      them'''
      numOrphans = 0
      nhAdjDict = RtSyncTasks( self.attributesText_ ).nexthopAdjRefCountDict()
      for adj in self.adjacencyList().itervalues():
         if len( adj.rtEntries() ) == 0 and nhAdjDict.get( adj.id(), 0 ) == 0:
            numOrphans = numOrphans + 1
      return numOrphans

   @staticmethod
   def waitForNoOrphans( dut ):
      '''Waits for number of orphans to go down to zero'''
      if not dut.isRibEnabled():
         # ArBgp doesn't have the shared adjacency table
         return
      # Note that SharedAdjacencyTable is instantiated below with rtsync=True.
      # Only when set to True will numOrphans account for nhAdjRefcount.
      # Please do not set rtsync to False.
      Tac.waitFor( lambda: \
         SharedAdjacencyTable(
            dut.giiGetAdjacencyDetails( rtsync=True ) ).numOrphans() == 0,
               description="# of orphans adjacencies to be zero on %s" % dut )

   def getAdjById( self, adjId ):
      """Given @adjId, return RtAdjacency object corresponding to it. If the
      adjacency is not found, None is returned"""
      return self.adjacencyList().get( adjId )

   def getAdjListNhs( self, ipv6=False ):
      '''Return number of times each sharedNH is referenced by adjacency'''
      addrFamily = socket.AF_INET6 if ipv6 else socket.AF_INET
      nexthops = {}
      for adj in self.adjacencyList().itervalues():
         if int( adj.addressFamily() ) != addrFamily:
            continue
         nhs = adj.nextHops()
         for nh in nhs:
            nexthops[ nh ] = nexthops.get( nh, 0 ) + 1

         bnhs = adj.backupNextHops()
         for nh in bnhs:
            nexthops[ nh ] = nexthops.get( nh, 0 ) + 1
      return nexthops


class RtAdjacency( RibdDumpAttributeParser ):
   '''
   Parses 'Shared-Adjacency Table' section of 'rdc dump' or one produced
   by GII command 'show ip adjacency'
   '''
   def __init__( self, attributesText ):
      RibdDumpAttributeParser.__init__(self, attributesText)

   @Tac.memoize
   def id( self ):
      return long( self.getStringPrp( 'Id' ), 16 )

   @Tac.memoize
   def refCount( self ):
      return self.getNumPrp( 'Refs' )

   @Tac.memoize
   def ngw( self ):
      return self.getNumPrp( 'N_gw' )

   @Tac.memoize
   def flags( self ):
      return self.prp( 'flags', r'<[\w ]+>' )

   def type( self ):
      return self.getStringPrp( 'type' )

   @Tac.memoize
   def addressFamily( self ):
      return self.getNumPrp( 'af' )

   @Tac.memoize
   def adjData( self ):
      return long( self.getStringPrp( 'data' ), 16 )

   @Tac.memoize
   def nextHops( self ):
      if self.addressFamily() == '10':
         return self.getIP6Prp( '^NextHop', findAll=True )
      else:
         return self.getIPPrp( '^NextHop', findAll=True )

   @Tac.memoize
   def getLabel( self ):
      return self.getNumPrp( 'Label' )

   @Tac.memoize
   def tunnelIds( self ):
      tunnelStrs = self.getStringPrp( 'TunnelId', findAll=True )
      tunnelIds = [ hex( int( i, 16 ) ) for i in tunnelStrs ]
      return tunnelIds

   @Tac.memoize
   def nexthopGroupIds( self ):
      nexthopGroupIds = self.getIntPrp( 'NexthopGroup ID', findAll=True )
      return nexthopGroupIds

   @Tac.memoize
   def linkBandwidths( self ):
      return self.getStringPrp( 'LinkBw', findAll=True )

   @Tac.memoize
   def backupNextHops( self ):
      if self.addressFamily() == '10':
         return self.getIP6Prp( 'Backup-NextHop', findAll=True )
      else:
         return self.getIPPrp( 'Backup-NextHop', findAll=True )

   @Tac.memoize
   def backupInterfaces( self, eosIntfNames=False ):
      intfs = self.prp( 'Backup-Interface', '[^\n]+', findAll=True )
      if not intfs: return intfs
      if eosIntfNames:
         intfs = map( kernelIntfToEosIntf, intfs )
      return intfs

   @Tac.memoize
   def adjBackup( self ):
      return long( self.getStringPrp( 'Bkup' ), 16 )

   @Tac.memoize
   def interfaces( self, eosIntfNames=False ):
      intfs = self.prp( 'Interface', '[^\n]+', findAll=True )
      if not intfs: return intfs
      if eosIntfNames:
         intfs = map( kernelIntfToEosIntf, intfs )
      return intfs

   @Tac.memoize
   def interfaceIPs( self, eosIntfNames=False ):
      ips = []
      for intf in self.interfaces():
         ips.append( intf.split( "(" )[ 0 ] )
      return ips
        
   @Tac.memoize
   def rtEntries( self ):
      # We don't use getIP[6]prefixPrp()s here because they use
      # complex strict regexs and it takes a lot of time when parsing
      # something like 5k entries.
      return self.prp( 'RtEntry', r'\S+/\S+', findAll=True )

   @Tac.memoize
   def viaSet( self, eosIntfNames=True ):
      """Return full list of vias corresponding to this adjacency"""
      return map( lambda nh, ifap: RouteVia( nexthop=nh, interface=ifap ), \
            zip( self.nextHops(), self.interfaces( eosIntfNames=eosIntfNames ) ) )

   @Tac.memoize
   def isisAdjSids( self ):
      return self.prp( 'IS-IS Adjacency Segment', '[^\n]+', findAll=True )

   @Tac.memoize
   def isisAdjSidRefCount( self ):
      return self.getNumPrp( 'IS-IS Adjacency Segments Refcount' )

   @Tac.memoize
   def isisPrefixSidRefCount( self ):
      return self.getNumPrp( 'IS-IS Prefix Segments Refcount' )

   @Tac.memoize
   def isisPrefixSids( self ):
      return self.prp( 'IS-IS Prefix Segment', '[^\n]+', findAll=True )

   def __eq__( self, other ):
      """
      Returns true if two adjacencies are equal. If @id is same, then we don't
      compare other fields. If not, we compare them. In case the argument is of
      type RouteFec, we compare @id (of @fecId) and full vias
      """
      if not isinstance(other, RtAdjacency) and \
            not isinstance(other, RouteFec):
         return NotImplemented
      if isinstance(other, RtAdjacency):
         if self.id() == other.id(): return True
         return self.refCount() == other.refCount() and \
            self.type() == other.type() and \
            self.flags() == other.flags() and \
            self.nextHops() == other.nextHops() and \
            self.backupNextHops() == other.backupNextHops() and \
            self.interfaces() == other.interfaces() and \
            self.backupInterfaces() == other.backupInterfaces() and \
            sorted( self.rtEntries() ) == sorted( other.rtEntries() )
      else:
         if self.id() != other.fecId: return False
         vias = self.viaSet()
         if len( other.vias ) != len( vias ): return False
         return set( vias ) == set( other.vias )

   def __ne__( self, other ):
      if not isinstance(other, RtAdjacency) and \
            not isinstance(other, RouteFec):
         return NotImplemented
      return not self.__eq__(other)

   # Returns true if the properties of the RtAdjacency object match with
   # the optional arguments passed
   def verifyAdjProperties( self, error, refCount=None, nextHop=None, 
         backupNextHop=None, nhgIds=None ):
      def t( *args ):
         # The trick being done here is to store enough debugging information,
         # so that we have some decent information when we hit a waitFor timeout.
         error[ 0 ] = 'expected refCount: %s\n' % refCount
         error[ 0 ] += 'refCount seen: %s\n' % self.refCount()
         error[ 0 ] += 'expected nexthops: %s\n' % refCount
         error[ 0 ] += 'nexthops seen: %s\n' % self.nextHops()
         error[ 0 ] += 'backup-nexthops seen: %s\n' % self.backupNextHops()
         error[ 0 ] += 'nexthop-group IDs seen: %s\n' % self.nexthopGroupIds()
         error[ 0 ] += str( args )

      if refCount and int( self.refCount() ) != refCount:
         t( 'refCount mismatch' )
         return False
      if nextHop and set( self.nextHops() ) != set( nextHop ):
         t( 'nextHops mismatch' )
         return False
      if backupNextHop and set( self.backupNextHops() ) != set( backupNextHop ):
         t( 'backup-nextHops mismatch' )
         return False
      if nhgIds and set( self.nexthopGroupIds() ) != set( nhgIds ):
         t( 'nexthop-group IDs mismatch' )
         return False
      return True

class IsisGrInfo( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )

   @staticmethod
   def parse( content ):
      """Parses GR information from the raw content dumped by show tech ribd
      """
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      m = re.match( r".*(?P<grInfo>GR Information Begin.*?)GR Information Ends",
                    content, re.S | re.M )
      if m:
         content = m.group( 'grInfo' )
      ret = IsisGrInfo()
      grPattern = r"Graceful Restart: (?P<gracefulRestart>(\S)+), " \
                  r"Graceful Restart Helper: (?P<grHelper>(\S)+)"    
      grStatusPattern = r"Restart Status: (?P<grStatus>[^\n]+)"

      grP = re.search( grPattern, content )
      if grP:
         ret.update( **grP.groupdict() )
      grSP = re.search( grStatusPattern, content )
      if grSP:
         ret.update( **grSP.groupdict() )

      return ret

class IsisSrInfo( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )

   @staticmethod
   def parse( content ):
      """Parses SR information from the raw content dumped by show tech ribd
      """
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      convergenceString = None
      m = re.match( r".*SR(?P<convergedState>.*?)Converged.*"
                    r"(?P<srInfo>SR Book Keeper.*?)SR Information Ends",
                    content, re.S | re.M )
      if m:
         # This would be parsed either as ' Not ' or ' '
         convergenceString = m.group( 'convergedState' )
         convergenceString = convergenceString.strip()
         content = m.group( 'srInfo' )
      ret = {}
      selfOrigPrefixSegPattern = \
          r"\s+(?P<prefix>[a-f0-9\.:]+\/\d+)" \
          r"\s+(?P<sid>\d+)" \
          r"\s+(?P<indexOrLabel>[a-zA-Z]+)" \
          r"\s+(?P<type>[a-zA-Z]+)" \
          r"\s+(?P<tlvsrc>[a-zA-Z]+)" \
          r"\s+(?P<data>\S+)"
      recvPrefixSegPattern = \
          r"\s+(?P<flags>[*#+]*)(?P<prefix>[a-f0-9\.:]+\/\d+)" \
          r"\s+(?P<sid>\d+)" \
          r"\s+(?P<indexOrLabel>[a-zA-Z]+)" \
          r"\s+(?P<type>[a-zA-Z]+)" \
          r"\s+(?P<systemId>\d+.\d+.\d+.\d+)" \
          r"\s+(?P<spfgen>\d+)"

      ret[ 'srConverged' ] = True if convergenceString == '' else False
      ret[ 'recvPrefSegments' ] = {}
      ret[ 'selfOrigPrefSegments' ] = {}

      for m2 in re.finditer( selfOrigPrefixSegPattern, content, re.M ):
         ret['selfOrigPrefSegments'][m2.group( 'prefix' ) ] = \
             IsisSrInfo( **m2.groupdict() )
      for m2 in re.finditer( recvPrefixSegPattern, content, re.M ):
         ret['recvPrefSegments'][m2.group( 'prefix' ), m2.group( 'systemId' ) ] = \
             IsisSrInfo( **m2.groupdict() )

      return ret

class SyncNexthopEntries( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.instance = int( self.instance )
      self.nheId = int( self.nheId )
      self.flags = [ s.strip() for s in self.flags.split( ' ' ) if s ]

   @staticmethod
   def parse( content ):
      """Parses 'nexthop' entries from the raw content dumped by show
      tech ribd or show policy GII command returns a dict of @SyncNexthopEntries
      keyed by the nexthop address.

        Nexthop: 1.0.0.1 (f08625c0)
        Color: Count: 2 [ 100(00) 200(01) ]
        Flags: <NexthopsChanged MetricChanged> Type: default
        Resolution: Resolved      Refcount: 2
        Interface: 0(NULL)
        Last change time: Feb 11 11:11:46
        Rib: 0 Nhp: 0x12345678 Adjacency: 0x10000000d
                Instance: 1     NHE ID: 4       Pref: 1 Pref2: 0
                Metric: 0       Metric2: 0      TotalLinkBw: 20000.00
                Resolving Mask: 24
                IGP [ Nexthops | Tunnels ]:
                        1.0.205.2 LinkBw 10000.00
                        1.0.206.2 LinkBw 10000.00

      Sample raw output for Nexthop-Group entries:

        Nexthop: 102.102.102.102 (e66b2114)
        Color: Count: 2 [ 100(00) 200(01) ]
        Flags: <> Type: default
        Resolution: Resolved
        Interface: 0(NULL)
        Last change time: Oct 31 09:03:32
        Rib: 0 Nhp: e6ccd4b0 Adjacency: 0x300000000000002
                Instance: 0     NHE ID: -2      Pref: 1 Pref2: 0
                Metric: 0       Metric2: 0      TotalLinkBw: 0.00
                Resolving Mask: 32
                IGP Nexthops:
                        NexthopGroup route
                                NexthopGroup ID: 1
                                NexthopGroup ID: 2
      """
      ret = {}
      igpNhRe = r"([\dA-Fa-f.:]+[%\w\/]*)\s*LinkBw (\d+\.\d+)"
      nhgIdRe = r"NexthopGroup ID: (\d+)"
      pat = r"\s*Nexthop: (?P<nexthop>\S+).*\n" + \
            r"\s*Color: Count: (?P<colorCnt>\d+), \[(?P<colorValues>.*)\]\n" + \
            r"\s*Flags: <(?P<flags>.*)> Type: (?P<nhetype>\S+)\n" + \
            r"\s*Resolution: (?P<rrStatus>\S+).*Refcount: (?P<refCnt>\d+)\n" + \
            r"\s*Interface: (?P<interface>\S+)" + \
            r"\s*Last change time: .*\n" + \
            r"\s*Rib: 0 Nhp: (?P<nhp>\S+) Adjacency: (?P<adjId>\S+).*\n" + \
            r"\s*Instance: (?P<instance>\d+)\s+NHE ID: (?P<nheId>[-\d]+)" + \
            r"\s*Pref: (?P<pref>[-\d]+)\s+Pref2: (?P<pref2>[-\d]+).*\n" + \
            r"\s*Metric: (?P<metric>[-\d]+)\s+Metric2: (?P<metric2>[-\d]+)" + \
            r"\s*TotalLinkBw: (?P<totalLbw>\d+\.\d+)\n" + \
            r"\s*Resolving Mask: (?P<mask>\d+)?.*\n" + \
            r"\s*IGP (?P<nheType>[A-Za-z]+):.*\n" + \
            r"(?P<igpNhs>(\s*(" + igpNhRe + \
            r"|Direct.*|Blackhole.*|NexthopGroup route.*|Nil)\s*\n)+)" + \
            r"(?P<nhgIdStrs>(\s*" + nhgIdRe + r"\s*\n)*)"
      for m in re.finditer( pat, content, re.M ):
         igpNexthops = []
         linkbws = []
         nhgIds = []
         for line in m.group( 'igpNhs' ).splitlines():
            match = re.search( igpNhRe, line )
            if match:
               igpNexthops.append( match.group( 1 ) )
               linkbws.append( match.group( 2 ) )
            else:
               igpNexthops.append( line )
         for line in m.group( 'nhgIdStrs' ).splitlines():
            match = re.search( nhgIdRe, line )
            if match:
               nhgIds.append( match.group( 1 ) )
         ret[ m.group( 'nexthop' ) ] = SyncNexthopEntries( igpNexthops=igpNexthops,
                                                           linkbws=linkbws,
                                                           nhgIds=nhgIds,
                                                           **m.groupdict() )
      return ret

   def isDirectlyResolved( self ):
      '''Returns True if the nhe is resolved through connected route, else False'''
      return self.igpNexthops[ 0 ] == 'Direct interface route'

class BgpEcmpRouteHead( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.paths = int( self.paths )
      assert self.paths == len( self.nhs )

   @staticmethod
   def parse( content ):
      """Parses ECMP Route Head entries from the raw content dumped by show
      tech ribd or show policy GII command returns a list of @BgpEcmpRouteHead
         ECMP Route Head 0xef821fc0 (4 paths) 1.0.5.0/24
         nhe-list ID: 11
         #0 100 i (HashID 7) Nexthop: 1.0.0.1
         #1 100 i (HashID 8) Nexthop: 1.0.1.1
         #2 100 i (HashID 9) Nexthop: 1.0.2.1
         #3 100 i (HashID 10) Nexthop: 1.0.3.1
      """
      ret = []
      pat = r"\s*ECMP\sRoute\sHead.*\((?P<paths>\d+)\spaths\)\s+(?P<dest>\S+)\n" + \
            r"\s*nhe-list\sID:\s(?P<nhelistId>\d+)\n(?P<ecmpEntries>(\s*#.*\n)+)"
      for m in re.finditer( pat, content, re.M ):
         nhs = []
         for entry in m.group( 'ecmpEntries' ).split( '\n' ):
            em = re.search( r".*Nexthop:\s(?P<nexthop>\S+)", entry )
            if em:
               nhs.append( em.group( 'nexthop' ) )
         ret.append( BgpEcmpRouteHead( nhs=nhs, **m.groupdict() ) )

      return ret

class RtSync( RibdDumpAttributeParser ):
   '''
   Parses 'rt sync task' section of 'rdc dump' or one produced
   by GII command 'show (static|bgp) dump sync'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   def rtSetOnListTime( self ):
      return self.getFloatPrp( 'Rt set onlist time' )

   def lastFlashTime( self ):
      return self.getFloatPrp( 'Last flash time' )

   def lastJobTime( self ):
      return self.getFloatPrp( 'Last job time' )

   def convergeTime( self ):
      return self.getFloatPrp( 'Converge time' )

   def giiFlags( self ):
      return self.prp( 'GII flags', r'[ \w]+' )

   def nexthopEntries( self ):
      return SyncNexthopEntries.parse( self.raw() )

   def ecmpSyncAdds( self ):
      return int( self.getIntPrp( 'adds since last clear: ecmp sync' ) )

   def nexthopAdjRefCountDict( self ):
      '''Returns a dict of adjacency ids for rts_nh_adj embedded in Bgp/Static
      Recursive nexthop entries. Value is number of references to that
      adjacency'''
      nheDict = self.nexthopEntries()
      # Multiple _rts_nh_entrys could share same rt_sync_nexthops. As a result
      # if say two nhes use same rt_sync_nexthop, embedded adjacency ref.
      # count will be 1 - not 2. We must build a dict with distinct
      # 'rt_sync_nexthop' as key and one of the nhe that refers to them as value
      nhpToNhe = {}
      for nhe, v in nheDict.iteritems():
         nhpToNhe[v.nhp] = nhe
      adjDict = {}
      for _, nhe in nhpToNhe.iteritems():
         v = nheDict[nhe]
         adjId = int(v.adjId, 16)
         adjDict[adjId] = adjDict.get(adjId, 0) + 1
      return adjDict

   def bgpEcmpRouteHeads( self ):
      return BgpEcmpRouteHead.parse( self.raw() )

class RtSyncTasks( RibdDumpAttributeParser ):
   '''
   Parses multiple 'rt sync task dump' sections produced
   by GII command 'show ip adjacency rtsync'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   @staticmethod
   def parse( content ):
      pat = r'bgp_sync task dump:(?P<bgp>.*)static_sync task dump:(?P<static>.*)'
      ret = {}
      match = re.finditer( pat, content, re.S|re.M )         
      for m in match:
         ret[ 'bgp' ] = RtSync( m.group( 'bgp' ) )
         ret[ 'static' ] = RtSync( m.group( 'static' ) )
      return ret

   def nexthopAdjRefCountDict( self ):
      '''Returns a dict of adjacency ids for rts_nh_adj embedded in all 
      Recursive nexthop entries ( both bgp and static sync tasks ).
      Value is number of references to that adjacency'''
      adjCounter = Counter()
      tasksDump = self.parse( self.raw() )
      for task in tasksDump.itervalues():
         adjCounter += Counter( task.nexthopAdjRefCountDict() )
      return dict( adjCounter )

class Nhelist( RibdDumpAttributeParser ):
   '''
   Parses a single entry in the 'NHE-LIST' cache section of 'rdc dump'
   '''
   def __init__( self, rawstr ):
      RibdDumpAttributeParser.__init__(self,
         re.sub(r'^1\d0\s+', '', rawstr, flags=re.M))

   def refs( self ):
      return int( self.prp( 'Refs', r'\d+' ) )

   def n_gw( self ):
      return int( self.prp( 'N_gw', r'\d+' ) )

   def adj( self ):
      return int('0x' + self.prp( 'Adj', r'\S+' ), 16)

   def configOptions( self ):
      return int( self.prp( 'config-options', r'\S+' ), 16)

   def id( self ):
      return int( self.prp( 'Id', r'\d+' ) )

   def maxEcmp( self ):
      return int( self.prp( 'max-ecmp', r'\d+' ) )

   def maxPaths( self ):
      return int( self.prp( 'max-paths', r'\d+' ) )

   def maxUcmp( self ):
      return int( self.prp( 'max-ucmp', r'\d+' ) )

   def flags( self ):
      return int( self.prp( 'flags', '0x[0-9a-fA-F]+' ), 16 )

   def activeNhes( self ):
      rlist = {}
      activeNhePattern = r'^\s+ID:\s(?P<id>-{0,1}\d+),\s+NextHop:\s(?P<nexthop>\S+)'
      content = self.prp( 'Active nhes', '(' + activeNhePattern + '[^\n]*\n)+' )
      if content:
         match = re.finditer( activeNhePattern, content, flags=re.M )
         for m in match:
            rlist[ m.group( 'nexthop' ) ] = int( m.group( 'id' ) )
      return rlist

   def bgpNhes( self ):
      rlist = {}
      activeNhePattern = \
            r'^\s+NextHop:\s(?P<nexthop>\S+)\s+' \
            'IGP-Instance\(ptr\):(?P<igpInstance>\d+)\s+' \
            '\(cache\)(?P<cache>\d+)\s+LinkBw\s+(?P<lbw>\S+)\s+' \
            'tid\s+(?P<tid>\d+)'
      content = self.prp( 'pendingBehCount\s+\S+',
                          '(' + activeNhePattern + '[^\n]*\n)+' )
      if content:
         match = re.finditer( activeNhePattern, content, flags=re.M )
         for m in match:
            rlist[ m.group( 'nexthop' ) ] = m.groupdict()
      return rlist

class RoutemapChain( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.address = int( self.address, 16 )
      self.refcount = int( self.refcount )

   @staticmethod
   def parse( content ):
      """Parses routemap chains from the raw content dumped by show tech ribd
      or show policy GII command returns a dictionary of @RoutemapChain
      objects with name of the chain as key."""

      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      m = re.match( r".*(?P<rmc>Routemap Chains.*?)^Prefix.*",
                  content, re.S | re.M )
      if m:
         # Variables
         result = {}
         contentPos = []
         individualRmBlock = []

         content = m.group( 'rmc' )

         '''
         The expected Regex pattern
         Example:
            BAR refcount 1 0xf76bb548 route-map-chain <syment>
         '''
         pat = r'\s+(?P<name>\S+)\s+refcount\s+(?P<refcount>\d+)' + \
               r'\s+(?P<address>\w+)\s+route-map-chain\s<(\S+\s)*syment(\s\S+)*>'

         # Store the iterator as a list, to be used more than once
         lst = list( re.finditer( pat, content, re.S | re.M ) )

         # Store the position of start/end of matched routemaps
         for m in lst:
            contentPos.append( [ m.start() , m.end() ] )

         # Extract individual routemap text blocks
         for rmIndex in xrange( len( contentPos ) ):
            # If this is the last RM individual text block:
            if rmIndex == len( contentPos ) - 1:
               '''The LAST individual routemap text block starts from the end
               of the last matched string, until the end of the content string'''
               individualRmBlock.append( content[ contentPos[rmIndex][1] : ] )
            else:
               '''The individual routemap block starts from the end of this
               matched string, to the beginning of the next matched string'''
               individualRmBlock.append( \
                  content[ contentPos[rmIndex][1] : contentPos[rmIndex+1][0] ] \
                  )

         for idx, m in enumerate( lst ):
            '''Add the result of the match, and its dictionary of
            @RMSequences objects, to a temporary dictionary'''
            tempDict = m.groupdict()
            tempDict['sequences'] =  RMSequences.parse( individualRmBlock[idx] )
            '''Put the temporary dictionary in a @RoutemapChain object, and add
            the object to the result dictionary, with the routemap name as key'''
            result[ m.group('name') ] = RoutemapChain( **tempDict )

         return result
      else:
         return {}

class RMSequences( Bunch ):
   '''
   @RMSequences object contains:
      name, sequence number, premit or deny,
      and match/set objects.
   '''
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.seq = int( self.seq )

   @staticmethod
   def parse( content ):
      '''
      Returns a dictionary of @RMSequences objects with
         sequence numbers as keys.
      @TODO: Parse all variations of match & set clauses
      '''
      if content:
         # Variables
         contentPos = []
         individualSeqBlock = []
         result = {}

         '''
         The expected Regex pattern
         Example:
            BAR, seq 10, permit
                 import/export match (Proto: Any) <>
                        as_comm_list_set (Proto: Any): 0xef702d10
                 import/export set (Proto: Any) <>
                        AS Info: 0xef7b0688 
         '''
         pat = r'\s+(?P<name>\S+), seq\s+(?P<seq>\d+),' + \
               r'\s+(?P<permit_or_deny>\w+)\n' + \
               r'(.*\n.*?as_comm_list_set \(.*?\): (?P<commListRef>.*$)\n)?' + \
               r'(.*\n.*?AS Info: (?P<AsInfo>\w+))?'

         # Store the iterator as a list, to be used more than once
         lst = list( re.finditer( pat, content, re.M ) )

         # Store the position of start/end of matched sequences
         for m in lst:
            contentPos.append( [ m.start() , m.end() ] )

         # Extract individual sequence text blocks
         for rmIndex in xrange( len( contentPos ) ):
            # If this is the last sequence text block:
            if rmIndex == len( contentPos ) - 1:
               individualSeqBlock.append( content[ contentPos[rmIndex][1] : ] )
            else:
               individualSeqBlock.append( \
                  content[ contentPos[rmIndex][1] : contentPos[rmIndex+1][0] ] \
                  )

         for idx, m in enumerate( lst ):
            '''Add the result of the match, and all available
            match and set objects, to a temporary dictionary'''
            tempDict = m.groupdict()
            # @TODO: Put here all match or set clauses
            tempDict['matchNHPrefixlist'] =  \
               RMMatchNHPrefixlist.parse( individualSeqBlock[idx] )
            tempDict['matchDestPrefixlist'] =  \
               RMMatchDestPrefixlist.parse( individualSeqBlock[idx] )
            '''Put the temporary dictionary in a @RMSequences object, and add
            the object to the result dictionary, with the sequence number as key'''
            result[ m.group('seq') ] = RMSequences( **tempDict )

         return result
      else:
         return {}

class RMMatchNHPrefixlist( Bunch ):
   '''
   @RMMatchNHPrefixlist object contains:
      prefixlistName and advEntry.
   '''
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.advEntry = int( self.advEntry, 16 )
      self.ipv6_ = self['prefixlistName'].startswith( '.ipv6.' )
      if self.ipv6_:
         self['prefixlistName'] = self['prefixlistName'][6:]

   @staticmethod
   def parse( content ):
      '''
      It returns a single @RMMatchNHPrefixlist object.
      '''
      if content:

         '''
         The expected Regex pattern
         Example:
            import/export match (Proto: Any) <>
               nexthop-list (Proto: Any):
                  prefix-list FOO adv_entry 0xf76bb514
         '''
         pat = r'\s+import/export\smatch.*$' + \
               r'\s+nexthop-list.*$' + \
               r'\s+prefix-list\s+(?P<prefixlistName>\S+)\s+' + \
               r'adv_entry\s+(?P<advEntry>\w+)'
         mIterator = re.finditer( pat, content, re.M )
         mList = list( mIterator )
         lenList = len( mList )
         if lenList == 0:
            return {}
         else:
            assert lenList == 1
            m = mList[0]
            result = RMMatchNHPrefixlist( **m.groupdict() )
            return result
      else:
         return {}

class RMMatchDestPrefixlist( Bunch ):
   '''
   @RMMatchDestPrefixlist object contains:
      prefixlistName and advEntry.
   '''
   def __init__( self, **kwargs ):
      Bunch.__init__( self, **kwargs )
      self.advEntry = int( self.advEntry, 16 )
      self.ipv6_ = self['prefixlistName'].startswith( '.ipv6.' )
      if self.ipv6_:
         self['prefixlistName'] = self['prefixlistName'][6:]

   @staticmethod
   def parse( content ):
      '''
      It returns a single @RMMatchDestPrefixlist object.
      '''
      if content:

         '''
         The expected Regex pattern
         Example:
            import/export match (Proto: Any) <>
               dest_mask_root (Proto: Any):
                  prefix-list FOO adv_entry 0xf76bb514
         '''
         pat = r'\s+import/export\smatch.*$' + \
               r'\s+dest_mask_root.*$' + \
               r'\s+prefix-list\s+(?P<prefixlistName>\S+)\s+' + \
               r'adv_entry\s+(?P<advEntry>\w+)'
         mIterator = re.finditer( pat, content, re.M )
         mList = list( mIterator )
         lenList = len( mList )
         if lenList == 0:
            return {}
         else:
            assert lenList == 1
            m = mList[0]
            result = RMMatchDestPrefixlist( **m.groupdict() )

            return result
      else:
         return {}

class PrefixList( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.address = int( self.address, 16 )
      self.refcount = int( self.refcount )
      self.ipv6_ = self['name'].startswith( '.ipv6.' )
      if self.ipv6_:
         self['name'] = self['name'][6:]
      self.sequences = {}

   def ipv6(self):
      return self.ipv6_

   @staticmethod
   def parse( content ):
      """Parses prefix lists from the raw content dumped by show tech ribd or
      show policy GII command returns a dictionary of @PrefixList objects with
      name of the prefix list as key. For IPv6 prefix lists, key name is
      .ipv6.xxx - where xxx is name of the prefix list"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      m = re.match( r".*(?P<pfxlist>Prefix Lists.*?)^Prefix Lists End",
                     content, re.S | re.M )
      if m:
         content = m.group( 'pfxlist' )
         pat = r'\s+(?P<name>[.\w]+)\s+refcount\s+(?P<refcount>\d+)' + \
               r'\s+(?P<address>\w+)\s+dest_mask_root\s<(\S+\s)*syment(\s\S+)*>' + \
               r'\s?(?P<seqs>.*?)\n\n'
         lst = re.finditer( pat, content, re.S | re.M )
         d = {}
         for m in lst:
            plobj = PrefixList( **m.groupdict() )
            d[ m.group( 'name' )] = plobj
            seqpat = r'\s+(?P<address>\w+)\s+refcount\s+(?P<refcount>\d+)' + \
                     r'\s+(?P<prefix>[.\w:/]+)\s+seq\s+(?P<seq>\d+)' + \
                     r'\s+ge\s+(?P<ge>\d+)\s+le\s+(?P<le>\d+)\s?'
            seqs = re.finditer( seqpat, plobj.seqs, re.S | re.M )
            for m2 in seqs:
               plobj.sequences[int(m2.group('seq'))] = \
                  PrefixListEntry( **m2.groupdict() )
         return d
      else:
         return {}

class AsPathInfoTable( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.referenceAddress = int( self.referenceAddress, 16 )

   @staticmethod
   def parse( content ):
      """Parses AS Path Info Table from the raw content dumped by show tech ribd or
      show policy GII command returns a dictionary of @AsPathInfoTable objects with
      Reference of the AS Path Info structure as key."""
      contentPos = []
      individualAsBlocks = []
      asPathInfo = {}
      pat = r"(?P<asPathTable>AS Path Info Table.*)AS Path Info structures"
      # match complete AS Path Info Table
      m = re.search( pat, content, re.S | re.M )
      if m:
         content = m.group( 'asPathTable' )
         for match in re.finditer( r'Reference', content, re.S | re.M ):
            contentPos.append( match.start() )

         # Extract individual as path entry text blocks
         for rmIndex in xrange( len( contentPos ) ):
            # If this is the last as path individual text block:
            if rmIndex == len( contentPos ) - 1:
               individualAsBlocks.append( content[ contentPos[ rmIndex ] : ] )
            else:
               individualAsBlocks.append( \
                  content[ contentPos[ rmIndex ] : \
                              ( contentPos[ rmIndex + 1 ] - 1 ) ] )

         for individualAsBlock in individualAsBlocks:
            # pylint: disable-msg=W0105
            '''
            Reference: 0xef708558 ...  Flags <COMM COMM_DEL...> (0x10050004004)
                     Metric: 0       Local ID: 0.0.0.0       AS Count: 0
                     MED: 0  Local Pref: 0   MPReach: 0x0    MPUnreach: 0x0
                     Community/AR: 0xef708390, id 2, ...
                     Community/DEL: 0xef708428, id 3, ...
                     Community-list/AR: .clrm11ar
                     Community-list/DEL: .clrm11del
                     Community-list/DEL/Regex: .clrm11del_regex
            '''
            pat = r'Reference: (?P<referenceAddress>\w+)\s+' + \
                r'Refcount: (?P<refCount>\d+)\s+Owner: (?P<owner>\w+)\s+' + \
                r'Flags <(?P<ApiFlags>(\w|\s)+)?>\s+\((?P<ApiFlagBits>\w+)\)' + \
                r'(.*?Community/AR:\s+' + \
                r'(?P<clAddressAr>\w+).\s+id\s+(?P<clIdAr>\d+))?' + \
                r'(.*?Community/DEL:\s+' + \
                r'(?P<clAddressDel>\w+).\s+id\s+(?P<clIdDel>\d+))?' + \
                r'(.*?Community/DEL/Regex:\s+(?P<clAddressDelRegex>\w+)' + \
                r'.\s+count\s+(?P<clCountDelRegex>\d+).\s+refcount\s+' + \
                r'(?P<clRefcountDelRegex>\d+))?' + \
                r'(.*?Community-list/AR:\s+(?P<clNameAr>(\.|\w)+))?' + \
                r'(.*?Community-list/DEL:\s+(?P<clNameDel>(\.|\w)+))?' + \
                r'(.*?Extended (Communities|Comm Dels):\s+Id:\s+' + \
                r'(?P<extClId>\d+)\((?P<extClAddress>\w+)\))?'
            match = re.match( pat, individualAsBlock, re.S | re.M )
            if match:
               asPathInfo[ match.groupdict()[ 'referenceAddress' ] ] \
                  = AsPathInfoTable( **match.groupdict() )
         return asPathInfo
      else:
         return {}

class PrefixListEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.address = int( self.address, 16 )
      self.refcount = int( self.refcount )
      self.seq = int( self.seq )
      self.ge = int( self.ge )
      self.le = int( self.le )

class AspathListEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.address = int( self.address, 16 )
      self.seq = int( self.seq )

class AspathList( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.address = int( self.address, 16 )
      self.refcount = int( self.refcount )
      self.sequences = {}
      self.cacheSize = int( self.cacheSize ) if self.cacheSize else 0
      self.cacheClearCount = int( self.cacheClearCount ) \
                              if self.cacheClearCount else 0
      self.cacheId = int( self.cacheId ) if self.cacheId else 0

   @Tac.memoize
   def cacheEntries( self ):
      return cacheEntries( self.cache )

   @staticmethod
   def parse( content ):
      """Parses as-path access lists from the raw content dumped by show tech ribd or
      show policy GII command returns a dictionary of @AspathList objects with
      name of the as-path list as key"""
      content = re.sub(r'^1\d0\s', '', content, flags=re.M)
      m = re.match( r".*(?P<asplist>Aspath Access Lists.*?)^Aspath",
                     content, re.S | re.M )
      if m:
         content = m.group( 'asplist' )
         pat = r'\s+(?P<name>[.\w]+)\s+refcount\s+(?P<refcount>\d+)' + \
               r'\s+(?P<address>\w+)\s+asp_list\s<(\S+\s)*syment(\s\S+)*>' + \
               r'\s?(?P<seqs>.*?)' + \
               r'(?P<cache>\s*Cache (?P<cacheId>\d+) ' + \
               r'size (?P<cacheSize>\d+) ' + \
               r'cleared (?P<cacheClearCount>\d+)\n.*?Cache End)?\n\n'
         lst = re.finditer( pat, content, re.S | re.M )
         d = {}
         for m in lst:
            aplobj = AspathList( **m.groupdict() )
            d[ m.group( 'name' )] = aplobj
            seqpat = r'^\s+(?P<address>\S+)\smatch\s(?P<match>\S+)\s+' + \
                     r'origins\s+(?P<origins>\w+)\s+seq\s+(?P<seq>\d+)\s+' + \
                     r'(?P<permit_or_deny>\w+)'
            seqs = re.finditer( seqpat, aplobj.seqs, re.S | re.M )
            for m2 in seqs:
               aplobj.sequences[int(m2.group('seq'))] = \
                  AspathListEntry( **m2.groupdict() )
         return d
      else:
         return {}

LocalAsInfo = namedtuple( 'LocalAsInfo', 'localAs count loop' )
RibBgJobInfo = namedtuple( 'RibBgJobInfo', 'jobptr task jobname jobpriority flags' )

class AsPathAttrEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )

   @staticmethod
   def parse( content, pathAttrEntriesOnly=False ):
      if not pathAttrEntriesOnly:
         content = re.sub(r'^1\d0\s', '', content, flags=re.M)
         m = re.match( r".*AS Path Attribute Table\s+(?P<AsPathAttrTable>.*?)" + \
                        r"\s+Communities Table.*",
                        content, re.S | re.M )
         content = m.group( 'AsPathAttrTable' )
      if re.search( r"No as path attribute in the table",
                    content, re.S | re.M ):
         return {}
      asPathAttrRe = \
         r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
         r"(?P<hash>\d+)\s+Lengths:\s+Path\s+(?P<path>\d+)\s+" + \
         r"Seg\s+(?P<seg>\d+)\s+Path\sConfed\s+(?P<pathConfed>\d+)\s+" + \
         r"Seg\sConfed\s+(?P<segConfed>\d+)\s+.*?AS Path:\s+(?P<aspath>.*?)" + \
         r"\(HashID\s+\d+\)\s+\(Hash\s+\d+\)"
      lst = re.finditer( r"(" + asPathAttrRe + ")", content, re.S | re.M )
      attrlist = {}
      for m in lst:
         c = AsPathAttrEntry( **m.groupdict() )
         attrlist[ c.id ] = c
      return attrlist

class AsPathEntry( Bunch ):
   def __init__( self, **kwargs ):
      Bunch.__init__(self, **kwargs)
      self.id = int( self.id )
      self.refcount = int( self.refcount )
      self.hash = int( self.hash )

   @staticmethod
   def parse( content, pathEntriesOnly=False ):
      if not pathEntriesOnly:
         content = re.sub( r'^1\d0\s', '', content, flags=re.M )
         m = re.search( r"ASPaths\s+(?P<BgpAttrInfoTable>.*?)" + \
                         r"\s+Local AS number Table.*",
                        content, re.S | re.M )
         content = m.group( 'BgpAttrInfoTable' )
      if re.search( r"No path attributes in data base",
                    content, re.S | re.M ):
         return {}
      asPathRe = \
         r"Id\s+(?P<id>\d+)\s+Refs\s+(?P<refcount>\d+)\s+Hash\s+" + \
         r"(?P<hash>\d+)\s+Lengths:\s+Path\s+(?P<path>\d+),\s+Seg\s+" \
         r"(?P<seg>\d+),\s+Attr\s+(?P<attr>\d+),\s+Aggr\s+(?P<aggr>\d+),\s+" + \
         r"MED(?P<med>.*?),.*?Path:\s+AS\sPath:(?P<aspath>.*?)\s+" + \
         r"\(HashID\s+(?P<hashid>\d+)\)\s+" + \
         r"((Nexthop:\s+(?P<nexthop>\S+))|\<.*?\>|).*?\)" + \
         r"\s+(?P<astype>internal|external|confed-external|local|invalid)"
      lst = re.finditer( r"(" + asPathRe + ")", content, re.S | re.M )
      asPathList = {}
      for m in lst:
         c = AsPathEntry( **m.groupdict() )
         asPathList[ c.id ] = c
      return asPathList

class RouteHead:
   def __init__( self, dump ):
      # Common parser for ribdump & gii 'show dump route ...',
      # so sanitize the input for gii
      self.dump_ = re.sub( r'^1\d0 +', '', dump, flags=re.M )

   def state( self ):
      match = re.search( r"state <(?P<state>[\w ]+)", self.dump_, re.S | re.M )
      if match:
         return match.group( 'state' ).lstrip()
      else:
         return ''

   def routeEntries( self, address=ipv4Re, maskLen=maskLen4Re,
                     addrFamily=socket.AF_INET ):
      """
      There may be multiple routes under a route head. The dump for a route head
      starts with "A.B.C.D/LEN". The dump for a route entry starts with
      "Protocol Preference: <val>". The attributes for a route-head, are
      added to all route entries.
      """
      lst = re.finditer( r"(?P<address>" + address + ")/(?P<maskLen>" 
            + str( maskLen ) + r")\s*\n\s*(?P<attributes>.*?\n\n\n)", 
            self.dump_, re.S | re.M )
      rlist = []
      for match in lst:
         entries = re.finditer(
            r"(^\s+\S+\s+([0-9a-f]+)\s+Preference:\s+-?\d+(/-?\d+)?\s*"
            r"(AttrId: \S+)\s*(InstanceId: \S+)?\s*(Source: \S+)?\n.*?\n\n)",
                                match.group( "attributes" ), re.S | re.M )
         header = None
         for entry in entries:
            if header is None:
               header = match.group( "attributes" )[ : entry.start() ]
            data = header + entry.groups()[ 0 ]
            route = RouteEntry( match.group( 'address' ), 
                                int( match.group( 'maskLen' ) ),
                                data, addrFamily=addrFamily )
            rlist.append( route )
      return rlist

class SharedNexthopTable:
   def __init__( self, dump ):
      self.dump_ = dump

   @staticmethod
   def parseNexthopDump( dump, ipv6=False ):
      '''
      Returns a dictionary of nexthops in the shared nexthop table in gated.
      The format of this section of output in ribd dump is:
      Shared-Nexthop Table - INET(6)
      1.0.0.1(2)(addr) 1.0.0.2(1)(addr) 1.0.1.1(2)(addr) 1.0.1.2(2)(addr)
      ...
      '''
      nexthopTable = {}
      match = re.match( r".*Shared-Nexthop Table - INET%s(.*)\d+ "\
            "shared INET%s nexthops.*" \
            % ( "6" if ipv6 else "" , "6" if ipv6 else "" ),
            dump, re.S | re.M )
      if not match:
         return nexthopTable
      lst = re.finditer( r"(?P<nexthop>%s)\((?P<refcount>[0-9]+)\)" \
                  % ( ipv6Re if ipv6 else ipv4Re ),
                  match.group(1), re.S | re.M )
      for entry in lst:
         nexthopTable[ entry.group( 'nexthop' ) ] = int( entry.group( 'refcount' ) )
      return nexthopTable

   def parse( self, ipv6=False ):
      return self.parseNexthopDump( self.dump_, ipv6=ipv6 )

class Interface( Bunch ):
   numkeys = ( 'kernelIfIndex', 'eosIfIndex' )
   setkeys = ( 'changeFlags', 'stateFlags' )

   def __init__( self, **kwargs ):
      for c in Interface.numkeys:
         if c in kwargs:
            kwargs[ c ] = int( kwargs[ c ] )
      for c in Interface.setkeys:
         if c in kwargs:
            kwargs[ c ] = set( kwargs[ c ].split( ' ' ) )
      Bunch.__init__( self, **kwargs )

class RibdDump:
   def __init__( self, dump, tables=None ):
      self.dump_ = dump
      self.routingTable_ = self.routingTable() \
                           if not tables or 'rt' in tables else None
      self.communitiesTable_ = self.communitiesTable() \
                               if not tables or 'comm' in tables else None
      self.localAsInfoTable_ = self.localAsInfoTable() \
                               if not tables or 'lclas' in tables else None
      self.bgJobInfoTable_ = self.bgJobInfoTable() \
                             if not tables or 'bgjobs' in tables else None
      self.tasksTable_ = self.tasksTable() \
                         if not tables or 'tasks' in tables else None
      self.interfaceTable_ = self.interfaceTable() \
                         if not tables or 'intfs' in tables else None

   def aspathAcls( self ):
      '''Parses the as-path access-lists section of the dump and returns a
      dictionary. For example, parses ...

     /POLICY/ASPLISTS/ASPLIST(/policy/asplist acl2):
       Current values:
         *  0 Key name String:       'acl2'

         /POLICY/ASPLISTS/ASPLIST(/policy/asplist acl2)/ASPLIST_ENTRIES: No values.

          /POLICY/ASPLISTS/ASPLIST(/policy/asplist acl2)/ASPLIST_ENTRIES/
           ASPLIST_ENTRY(1/asplist/entry):
               Current values:
                 *  0 Key seq           Uint32:      1 (0x1)
                 *  1 Req name-or-regex String:      '<.*456.*>'
                 *  2 Req inline-origin Uint32:      4294967295 (0xffffffff)
                    3 Opt deny          Flag:        FALSE

          /POLICY/ASPLISTS/ASPLIST(/policy/asplist acl2)/ASPLIST_ENTRIES/
           ASPLIST_ENTRY(2/asplist/entry):
               Current values:
                 *  0 Key seq           Uint32:      2 (0x2)
                 *  1 Req name-or-regex String:      '<.*789.*>'
                 *  2 Req inline-origin Uint32:      4294967295 (0xffffffff)
                    3 Opt deny          Flag:        FALSE

      ... and returns ...

      (Pdb) dump = edut.ribdDump()
      (Pdb) pp dump.aspathAcls()
      {'acl2': {'1': {'deny': 'FALSE',
                      'inline-origin': '4294967295',
                      'name-or-regex': "'<.*456.*>'",
                      'seq': '1'},
                '2': {'deny': 'FALSE',
                      'inline-origin': '4294967295',
                      'name-or-regex': "'<.*789.*>'",
                      'seq': '2'}}}
      (Pdb)
      '''
      m = re.search(r"^/POLICY:((.*\n)+?)(^(\s+/POLICY/MAR))", self.dump_, re.M )
      assert m, "Could not find Policy dump"
      acls = {}
      lines = m.group( 1 ).split( "\n" )
      currAcl = None
      currEntry = None
      aclRegex = r'\s+/POLICY/ASPLISTS/ASPLIST\(/policy/asplist ([^\)]+)\):$'
      for line in lines:
         hdr = re.match( aclRegex, line )
         if hdr:
            currAcl = hdr.group( 1 )
            currEntry = None
            acls[ currAcl ] = {}
            continue

         entRegex = r'\s+/POLICY/ASPLISTS/ASPLIST\(/policy/asplist %s\)' \
                    r'/ASPLIST_ENTRIES/ASPLIST_ENTRY\((\d+)/asplist' \
                    r'/entry\)' % currAcl
         ent = re.match( entRegex, line )
         if ent:
            currEntry = ent.group( 1 )
            acls[ currAcl ][ currEntry ] = {}
            continue

         attrRegex = r'(Key|Req|Opt)\s+([^ ]+)\s+([^ ]+):\s+([^ ]+)'
         attr = re.search( attrRegex, line )
         if currAcl and currEntry and attr:
            acls[ currAcl ][ currEntry ][ attr.group( 2 ) ] = attr.group( 4 )

      return acls

   # This was quickly copied from ptest/RibdMemory.
   # It could use being revisited, and expanded for
   # other types of memory allocation.
   def memUsage( self ):
      m = re.search( "^Task Blocks:$\n((^$\n|^\t.*$\n)+)", self.dump_, re.M )
      assert m, "Could not find Task Blocks dump"
      usage = {}
      bucket = None
      for line in m.group( 1 ).split( "\n" ):
         hdr = re.match( r'\s+Size:\s+(\d+)', line )
         if hdr:
            bucket = int( hdr.group( 1 ) )
            usage.setdefault( bucket, {} )
            continue
         entry = re.match( r'\s+(.*)\s+Init:\s+\d+\s+Alloc:\s+(\d+)\s+Free:\s+'
                           r'(\d+)\s+InUse:\s+(\d+)', line )
         if entry:
            assert bucket is not None
            gtype = entry.group( 1 )
            inUse = int( entry.group( 4 ) )
            usage[ bucket ][ gtype ] = inUse
      m = re.search( "^Pools:$\n(.*)\n^Packet Buffer", self.dump_, re.M + re.S )
      assert m, "Could not find pools dump"
      perpage = {}
      for line in m.group( 1 ).split( "\n" ):
         hdr = re.match( r'\s+Element Size:\s+(\d+)\s+Elements Per Page:\s+(\d+)',
               line )
         if hdr:
            bucket = int( hdr.group( 1 ) )
            usage.setdefault( bucket, {} )
            perpage[ bucket ] = int( hdr.group( 2 ) )
            continue
         entry = re.match( r'\s+(.*)\s+maxfree\s+\d+\s+full/partial/free: '
                           r'(\d+)/\d+/\d+', line )
         if entry:
            gtype = entry.group( 1 ).replace( 'Lock', '' ).rstrip().replace(
                                                                     ' ', '_' )
            full = int( entry.group( 2 ) )
            usage[ bucket ][ gtype ] = full * perpage[ bucket ]
            continue
         if re.match( r'\s+(Partial Page Alloc Counts:)?[0-9 ]+$', line ):
            for i in re.findall( r'\d+', line ):
               usage[ bucket ][ gtype ] += int( i )
            continue

      return usage
      # TODO:
      # MultiPage Memory
      # Memory page usage
      # Packet Buffer Pools

   def memUsageOf( self, memUsage, poolName ):
      ''' returns allocations in use for a given poolName from the memUsage
      structure returned in memUsage() '''
      match = [ vals for vals in memUsage.values() if poolName in vals ]
      assert len( match ) == 1

      # match looks like
      # [{'bgp_ecmp_head': 0, 'evt_id_node_t': 3, 'rt_changes': 0}]
      return match[0][poolName]

   def routeEntries( self, address=ipv4Re, maskLen=maskLen4Re,
            addrFamily=socket.AF_INET ):
      """
      There may be multiple routes under a route head. The dump for a route head
      starts with "A.B.C.D/LEN". The dump for a route entry starts with
      "Protocol Preference: <val>". The attributes for a route-head, are
      added to all route entries.
      """
      lst = re.finditer( r"(?P<address>" + address + ")/(?P<maskLen>" 
            + str( maskLen ) + r")\s*\n\s*(?P<attributes>.*?\n\n\n)", 
            self.routingTable_, re.S | re.M )
      rlist = []
      for match in lst:
         entries = re.finditer(
            r"(^\s+\S+\s+([0-9a-f]+)\s+Preference:\s+-?\d+(/-?\d+)?\s*"
            r"(AttrId: \S+)\s*(InstanceId: \S+)?\s*(Source: \S+)?\n.*?\n\n)",
                                match.group( "attributes" ), re.S | re.M )
         header = None
         for entry in entries:
            if header is None:
               header = match.group( "attributes" )[ : entry.start() ]
            data = header + entry.groups()[ 0 ]
            route = RouteEntry( match.group( 'address' ), 
                                int( match.group( 'maskLen' ) ),
                                data, addrFamily=addrFamily )
            rlist.append( route )
      return rlist

   def nhelistCacheEntries( self ):
      pat = "Id.*Refs.*Hash.*Adj.*\n.*max-paths.*max-ecmp.*config-options.*" + \
            r"\n\s*flags \S+ pendingBehCount \d+\n(\s*NextHop:.*\n)+" + \
            r"\s*Active nhes.*\n(\s*ID:.*NextHop.*\n)+\s*(NHE IDs:.*)?"
      rlist = []
      for m in re.finditer( pat, self.dump_, re.M ):
         nhelist = Nhelist( m.group( 0 ) )
         rlist.append( nhelist )
      return rlist

   @Tac.memoize
   def communityEntries( self ):
      return CommunityEntry.parse( self.dump_ )

   @Tac.memoize
   def communityLists( self ):
      return CommunityList.parse( self.dump_ )

   @Tac.memoize
   def communityListEntries( self ):
      return CommunityListEntry.parse( self.dump_ )

   @Tac.memoize
   def extCommunityEntries( self ):
      return ExtCommunityEntry.parse( self.dump_ )

   @Tac.memoize
   def largeCommunityEntries( self ):
      return LargeCommunityEntry.parse( self.dump_ )

   def routeEntry( self, prefix, addrFamily=socket.AF_INET ):
      n = Subnet( prefix, addrFamily=addrFamily )
      address = str( n.address_ )
      maskLen = n.mask_.masklen_
      return self.routeEntries( address, maskLen, addrFamily=addrFamily )

   def routingTable( self ):
      m = re.match( r".*(?P<RoutingTable>Routing Tables:.*?)^Task.*",
            self.dump_, re.S | re.M ) 
      return m.group( 'RoutingTable' ) if m else ""

   def interfaceTable( self ):
      m = re.match( r".*(?P<InterfaceTable>Interfaces:.*)\s*Interface policy",
            self.dump_, re.S | re.M )
      return m.group( 'InterfaceTable' ) if m else ""

   def tasksTable( self ):
      m = re.match( r".*Tasks .+ and Timers:(?P<Tasks>.*?)^Task Blocks.*",
               self.dump_, re.S | re.M )
      if not m:
         m = re.match( r".*Selected Tasks:(?P<Tasks>.*)",
                       self.dump_, re.S | re.M )
      return m.group( 'Tasks' ) if m else ''      

   @Tac.memoize
   def getInterfaces( self ):
      intfs = {}
      lst = re.finditer(
            r"(?P<kernelIntfName>\S+)\s+Index (?P<kernelIfIndex>[0-9]+)\s+"
            r"Address\s+(?P<physAddrType>\S+)\s+(?P<physAddr>\S+)\s+"
            r"Change:\s+<(?P<changeFlags>[^>]*)>\s+"
            r"State:\s+<(?P<stateFlags>[^>]*)>\s+"
            r"EosIndex:\s+(?P<eosIfIndex>[0-9]+)\s+"
            r"EosName:\s+(?P<eosShortName>\S+)\s+",
            self.interfaceTable_, re.S | re.M )
      # There are a bunch more fields, including whole other sections of
      # interface address info; saving these for later.
      for m in lst:
         intfs[ m.groupdict()[ 'kernelIntfName' ] ] = Interface( **m.groupdict() )
      return intfs

   def taskEntries( self ):
      '''
      Task entries appear in ribd dump as follows:

        Task IF Priority 10     RtProto Direct
                n_newpolicy 0, n_reinit 0, n_flash 0, n_reinit_async 1
                Trace options:  none

        Task INET6  Priority 15  Socket 24 rcvbuf 124K sndbuf 124K  RtProto INET6
                n_newpolicy 0, n_reinit 1, n_flash 0
                Trace options:  none

        Task static_sync        Priority 51     RtProto Static  <LowPrio>
                n_newpolicy 5, n_reinit 0, n_flash 0
                Trace options:  none

      '''
      lst = re.finditer( r"Task (?P<name>\S+)\s+Priority (?P<priority>\d+)" +
                         ".*?\n"
                         r"^\s+n_newpolicy (?P<n_newpolicy>\d+), " +
                         r"n_reinit (?P<n_reinit>\d+), n_flash (?P<n_flash>\d+), " +
                         r"n_reinit_async (?P<n_reinit_async>\d+), " +
                         r"n_rth_flash (?P<n_rth_flash>\d+), " +
                         r"n_rth_filtered (?P<n_rth_filtered>\d+), " +
                         r"n_rth_filtered_last (?P<n_rth_filtered_last>\d+)" +
                         r"(\n^\s+Trace options:\s+(?P<trace_options>.*?))?" +
                         r"(\n^\s+Trace file:\s+(?P<trace_file>\S+)\s*)?\n\n",
                         self.tasksTable_, re.S | re.M )
      d = {}
      for m in lst:
         d[ m.groupdict()['name'] ] = RibdTask( **m.groupdict() )
      return d

   def getOspf2Lsas( self ):
      '''
            Extract  LSAs from rubdDump:

            Sample input:

            type RTR id 4.4.4.4 advrt 4.4.4.4 seq 0x64 [mem: eede7310]
            left eede7310 right eede7380 bitindex 0x1
            age 368 cksum 0xb1b len 36 flags < SPFIgnore>
            time now 342.467894 install -26 nextrecv 75.930933 nextsend 0.000000
            refcount 0 spfcount 7 options <E DC>
            wrapsave 0
            rtr unreach 0
            ospf route f1049348
                [OSPF2] rtentry eede82e8 cost 16777215 tag 0 pref1/2 -110/0
                ls origin 4.4.4.4 area 0.0.0.0 dest type RTR
                options E spfcount 8 flags <>
                next hops:
            router bits <>
            -> P2P id 1.0.0.1 cost 10 data 255.255.255.255 flags <Asym>
            -> TRANS id 10.1.0.2 cost 0 data 0.0.0.0 flags <Hidden>

            type RTR id 1.0.0.1 advrt 1.0.0.1 seq 0x80000000 [mem: eede7380]
            left eede7380 right eede71c0 bitindex 0x4
            age 268 cksum 0x0 len 0 flags <Hidden >
            time now 342.467894 install 74 nextrecv 0.000000 nextsend 0.000000
            refcount 0 spfcount 7 options <>
            wrapsave 0
            rtr unreach 0
            no ospf route
            router bits <>
            -> P2P id 4.4.4.4 cost 0 data 0.0.0.0 flags <Hidden>'''
      lst = re.finditer( ospf2LsaRe, self.dump_ )


      d = {}
      for m in lst:
         madvrt = m.groupdict()['advrt']
         mlsid = m.groupdict()['lsid']
         mtype = m.groupdict()['type']
         if madvrt not in d.keys():
            d[ madvrt ] = {}
         if mtype not in d[ madvrt ].keys():
            d[ madvrt ][ mtype ] = {}
         if mlsid  not in d[ madvrt ][ mtype ].keys():
            d[ madvrt ][ mtype ][ mlsid ] = Ospf2Lsas( **m.groupdict() )

      # Return a multi-dimensional dict, that can be keyed by advrt, type and lsid
      # in that order to get the LSAs. d[ advrt ][ type ][ lsid ]
      return d

   def getOspf2DCclearLsaCounts( self ):
      '''
      Extract the DC-clear lsa counts from the ribdDump

      Sample input:
      area 0.0.0.0 [mem: e88c9000]
          transit capability: 0
          area border routers: 3
          as border routers: 0
          spf count: 6
          DC-clear lsas: 0
          indication lsas: 0
      '''
      ospf2DCRegex = ( r"(\s*area (?P<areaId>\d+.\d+.\d+.\d+) \[mem:.*)(.*\n){5}"
                        "(\s*DC-clear lsas: (?P<dclsa>\d+))"
                        "\s*(indication lsas: (?P<indlsa>\d+))" )

      lst = re.finditer( ospf2DCRegex, self.dump_ )

      d = {}
      for m in lst:
         areaId = str( m.groupdict()[ 'areaId' ] )
         dcClear = int( m.groupdict()[ 'dclsa' ] )
         indlsa = int( m.groupdict()[ 'indlsa' ] )

         d[ areaId ] = { 'dcClear': dcClear, 'indlsa': indlsa }

      return d

   def getOspf2TeInfo( self ):
      '''
      Extract TE Info from a ribdDump

      Sample input:

            type OPQ10 id 1.0.0.1 advrt 1.1.1.1 seq 0x80000002 [mem: e97d20e8]
            left e97d20e8 right e97d20e8 bitindex 0x0
            age 55 cksum 0x4b63 len 124 flags <>
            time now 88.493127 install 33 nextrecv 0.000000 nextsend 0.000000
            refcount 0 spfcount 0 options <DC O>
            wrapsave 0
            rtr unreach 0
            no ospf route
            TE LSA instance 1
            Link TLV:
                Link Type TLV: MultiAccess (2)
                Link ID TLV: 1.0.0.2
                Remote IP TLV: 0.0.0.0
                Local IP TLV: 1.0.0.1
                TE Metric TLV: 10
                Maximum Bandwidth TLV: 12500000.000000
                Maximum Reservable Bandwidth TLV: 12500000.000000
                Unreserved Bandwidth TLV: 0:12500000.000000 1:12500000.000000 ...
                Class TLV: 0


            type OPQ10 id 1.0.0.1 advrt 0.0.0.1 seq 0x80000001 [mem: e910515c]
            left e910515c right e910515c bitindex 0x0
            age 23 cksum 0x648a len 28 flags <>
            time now 50.146278 install 27 nextrecv 0.000000 nextsend 0.000000
            refcount 0 spfcount 0 options <DC O>
            wrapsave 0
            rtr unreach 0
            no ospf route
            TE LSA instance 1
            Router TLV: RouterId: 0.0.0.1

      The data in the TLVs may be in any order'''

      # First need to find any type 10 LSAs, then save the advrt and id
      # to uniquely identify it
      regex = ( ospf2LsaRe + r"\s+time[^\n]*\s+refcount[^\n]*\s+wrapsave[^\n]*\s+"
                "rtr[^\n]*\s+no ospf route\s+"
                # Then check to see if it is a TE LSA
                "TE LSA instance (?P<teinst>\d+)"
                # If it is, the data in it could be a rtr or link tlv
                "(\s+("
                # If it is an rtr tlv
                "(Router TLV: RouterId: (?P<routerId>\d+\.\d+\.\d+\.\d+))|"
                # If it is a link tlv
                "(Link TLV:(\s+("
                # The data in a link tlv can be in any order (why | is used)
                "(Link Type TLV: (?P<linktype>\w+) \(\d+\))|"
                "(Link ID TLV: (?P<linkid>\d+.\d+.\d+.\d+))|"
                "(Local IP TLV:(?P<localips>( (\d+.\d+.\d+.\d+))+))|"
                "(Remote IP TLV:(?P<remoteips>( (\d+.\d+.\d+.\d+))+))|"
                "(TE Metric TLV: (?P<metric>\d+))|"
                "(Maximum Bandwidth TLV: (?P<mbw>\d+\.\d+))|"
                "(Maximum Reservable Bandwidth TLV: (?P<mrbw>\d+\.\d+))|"
                "(Unreserved Bandwidth TLV:(?P<urbw>( \d+:\d+\.\d+)+))|"
                "(Class TLV: (?P<class>\S+))|"
                "(SRLG TLV:(?P<srlgs>( \d+)+))|"
                # End data in a link tlv
                "))+)"
                # End link tlv
                "))"
                # All LSAs in the dump end with two newlines
                "\n\n" )
      lst = re.finditer( regex, self.dump_, re.M | re.S )

      d = {}
      for m in lst:
         advrt = m.groupdict()[ 'advrt' ]
         lsid = m.groupdict()[ 'lsid' ]
         if advrt not in d.keys():
            d[ advrt ] = {}
         assert lsid not in d[ advrt ].keys()
         fields = m.groupdict()

         if fields.get( 'routerId' ):
            # Must be a Router TLV
            result = { 'type' : 'rtr',
                       'rtrId' : fields[ 'routerId' ] }
         if fields.get( 'linktype' ):
            # Must be a Link TLV
            result = { 'type' : 'link' }
            # Only link type and link id are required
            result[ 'linktype' ] = fields[ 'linktype' ]
            result[ 'linkid' ] = fields[ 'linkid' ]
            if fields.get( 'localips' ):
               result[ 'localips' ] = fields[ 'localips' ].split( " " )[ 1 : ]
            if fields.get( 'remoteips' ):
               result[ 'remoteips' ] = fields[ 'remoteips' ].split( " " )[ 1 : ]
            if fields.get( 'metric' ):
               result[ 'metric' ] = int( fields[ 'metric' ] )
            if fields.get( 'mbw' ):
               result[ 'mbw' ] = fields[ 'mbw' ]
            if fields.get( 'mrbw' ):
               result[ 'mrbw' ] = fields[ 'mrbw' ]
            if fields.get( 'urbw' ):
               result[ 'urbw' ] = fields[ 'urbw' ]
            if fields.get( 'class' ):
               result[ 'class' ] = fields[ 'class' ]
            if fields.get( 'srlgs' ):
               result[ 'srlgs' ] = sorted( fields[ 'srlgs' ].split( " " )[ 1 : ] )
         d[ advrt ][ lsid ] = result

      return d

   def ospfAuthKeyHidden( self ):
      ''' Return false if auth key is not '<removed>' which indicates hidden '''
      pattern = (
         r'/OSPF_PROTO/OSPF_FLOATING_INTF\(\w+\)/\w+\(\w+\):.{1,200}key'
         r'\s*ByteStr:\s*(<\w*>)' )
      m = re.search( pattern, self.dump_, re.S )
      if m:
         return m.group( 1 ) == '<removed>'
      return False   # no auth key found, can't verify that it is hidden.

   def getBgpPeerTaskByAddr( self, addr ):
      '''Returns peer task object corresponding to @peer (address)'''
      return self.bgpPeerTasks().get( addr )

   def bgpPeerTasks( self ):
      '''Returns dictionary of peer tasks keyed by peer address'''
      d = {}
      taskEntries = self.taskEntries()
      for v in filter( lambda t: t.isBgpPeer(), taskEntries.values() ):
         d[ v.peeraddr ] = v
      return d

   def bgpListenTask( self ):
      return self.taskEntries()[ 'BGP.0.0.0.0+179' ]

   def interfaceMcBoundaries( self, ifName ):
      pat = r'\s+/MCORE/MCORE_IFS/MCORE_IF\(%s\)' % ifName + \
         r'/MCORE_IF_BOUNDARIES/MCORE_IF_BOUNDARY\((\d+.\d+.\d+.\d+/\d+)\):'
      lst = re.finditer( pat, self.content(), re.S | re.M )
      blist = []
      for m in lst:
         blist.append( m.group(1) )
      if not blist: return None
      return blist

   def deletedLogicalInterfaces( self ):
      m = re.search( r'Deleted logical interfaces(.*?)^$',
            self.dump_, re.S | re.M ) 
      if m.groups():
         d = {}
         tuples = re.findall( '\t\t\tAddr: (.*)\t Link: (.*)',
               m.group( 0 ), re.M )
         for t in tuples:
            d[ t[ 0 ] ] = t[ 1 ]
         return d
      return None

   def interfacesToIndex( self, fixIntfNames=True ):
      m = re.match( r".*Interfaces:(?P<Interfaces>.*?)^.*Interface policy:.*",
            self.dump_, re.S | re.M ) 
      pat = r'\s+(?P<name>[a-z_0-9]+)\s+Index\s+(?P<index>\d+)'
      lst = re.finditer( pat, self.content(), re.S | re.M )
      d = {}
      for m in lst:
         if fixIntfNames:
            ifname = kernelIntfToEosIntf(  m.group('name') )
         else:
            ifname = m.group('name')
         d[ ifname ] = int( m.group('index') )
      return d

   def content( self ):
      return self.dump_

   @Tac.memoize
   def communitiesTable( self ):
      if re.search( r"No communities in the table",
               self.dump_, re.S | re.M ):
         return ''
      m = re.search( r"Communities Table\s+(?P<CommunitiesTable>.*?)" + \
                     r"\sExtended Communities Table.*",
                     self.dump_, re.S | re.M )
      return m.group( 'CommunitiesTable' ) if m else ''

   @Tac.memoize
   def asPathAttrTable( self ):
      if re.search( r"No as path attribute in the table",
               self.dump_, re.S | re.M ):
         return ''
      m = re.search( r"AS Path Attribute Table\s+(?P<AsPathAttrTable>.*?)" + \
                     r"\sCommunities Table.*",
                      self.dump_, re.S | re.M )
      return m.group( 'AsPathAttrTable' )

   @Tac.memoize
   def bgpAttrInfoTable( self ):
      m = re.search( r"Task ASPaths:\s+(?P<BgpAttrInfoTable>.*?)" + \
                     r"\sLocal AS number Table.*",
                     self.dump_, re.S | re.M )
      return m.group( 'BgpAttrInfoTable' )

   @Tac.memoize
   def asPathInfoTable( self ):
      m = re.match( r".*(?P<ASPathInfoTable>AS Path Info.*?)^Task.*",
            self.dump_, re.S | re.M ) 
      if m:
         return m.group( 'ASPathInfoTable' )
      else:
         return ''

   @Tac.memoize
   def routemapChains( self ):
      return RoutemapChain.parse( self.dump_ )

   @Tac.memoize
   def prefixLists( self ):
      return PrefixList.parse( self.dump_ )

   @Tac.memoize
   def nUsedPolicyCommits( self ):
      '''Number of mio commits that resulted in 'used' policy being changed
      NOTE: Gated now avoids rt_new_policy calls when unused policy objects are
      changed'''
      m = re.match( r".*Useful policy commits:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nRtNewpolicyCalls( self ):
      '''Number of calls to rt_new_policy API, which flashes all routes in RIB'''
      m = re.match( r".*rt_new_policy calls:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nAsyncRtNewpolicyCalls( self ):
      '''Number of calls to rt_async_new_policy_job, which
      flashes all routes in RIB'''
      m = re.match( r".*rt_new_policy async calls:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group('num') )
      return 0

   @Tac.memoize
   def nTotalPolicyCommits( self ):
      '''Total number of policy commits made so far'''
      m = re.match( r".*Total policy commits:\s(?P<num>\d+)", self.dump_, re.S )
      if m:
         return int( m.group( 'num' ) )
      return 0

   @Tac.memoize
   def aspathLists( self ):
      return AspathList.parse( self.dump_ )

   def ospfRangePresent( self, areaId, rangePrefix, cost ):
      lst = re.finditer( r".*(\d+\.\d+\.\d+\.\d+\/\d+).*cost (\d+).spfcount.\d+",
            self.dump_, re.S | re.M )
      
      if lst is None:
         return False

      for match in lst:
         if ( match.group(1) == rangePrefix and 
              match.group(2) == str( cost ) ):
            return True
      return False

   def ospfNbrNotReady( self, rtrId ):
      m = re.match( r".*Task OSPF?.*router id: %s.*adjacency exchstart " \
               r"threshold - \d+.*adjacency exchstart threshold - neighbors " \
               r"not ready (?P<ngbrNotReady>\d+).*" % rtrId,
               self.dump_, re.S | re.M )
      if m and 'ngbrNotReady' in m.groupdict():
         return int( m.groupdict()[ 'ngbrNotReady' ] )
      else:
         return 0

   def getRibOutKey( self, peer ):
      # Parses 'show tech ribd' output to return the bro_key used for a given peer.
      keys = []
      lookForPeers = False
      for line in re.split(r'[\n\r]+', self.dump_ ):
         m = re.search( r'Task RIB-Out (\S+):', line)
         if m and m.group(1):
            keys.append( { 'key' : m.group(1), 'peers' : [] } )
            continue
         if re.search( 'Peers using this RIB-Out', line ):
            lookForPeers = True
            continue
         if re.search( 'Route Queue Timer', line ):
            lookForPeers = False
            continue
         if lookForPeers:
            # Append peer to the last key in the list
            # A peer entry is like this: "1.0.4.2 1 (<state>)"
            keys[-1][ 'peers' ].append( line.strip().split()[ 0 ] )
      for key in keys:
         if peer in key[ 'peers' ]:
            return key[ 'key' ]
      return None

   def ribOutKeysEqual( self, peer1, peer2 ):
      # Returns True if peers share the same non-None key.
      # False otherwise.
      key1 = self.getRibOutKey( peer1 )
      key2 = self.getRibOutKey( peer2 )
      if ( key1 is not None and key2 is not None ):
         return key1 == key2
      return False

   def ribOutKeysDistinct( self, peer1, peer2 ):
      # Returns True if peers have distinct non-None keys.
      # False otherwise.
      key1 = self.getRibOutKey( peer1 )
      key2 = self.getRibOutKey( peer2 )
      if ( key1 is not None and key2 is not None ):
         return key1 != key2
      return False

   @Tac.memoize
   def ribOuts( self ):
      '''Returns the list of BGP RIB-Outs keys (hex numbers as string)'''
      regexp = r"RIB-Out\s(0x[0-9a-f]+)\sPriority\s+[0-9]{1,3}\s+RtProto\sBGP"
      exp = re.finditer( regexp, self.dump_, re.S | re.M )
      l = []
      if exp:
         for match in exp:
            l.append( match.group(1) )
      return l

   @Tac.memoize
   def ribOut( self, key ):
      '''Returns a BgpRibOut instance given the RIB-Out key
      (hex numbers as string) if found, else None. Here is a sample to map:
Task RIB-Out 0xeafb6004:

         RIB-Out contains 1 total peers (1 v4 established, 0 v6 established, \
0 v4 lu established, 1 in sync)
         RIB-Out LinkBw: (v4 LinkBw 0.000000, v6 LinkBw 0.000000)
         RIB-Out Key:      
      ...

         No queued announcements
         
      '''
      exp = re.search( r'^Task RIB-Out %s:\s+RIB-Out contains .*?^\S' % key,
                       self.dump_, re.S | re.M )

      if exp:
         return BgpRibOut( exp.group()[:-1] )
      return None

   @Tac.memoize
   def ribOutAll( self ):
      '''Returns all the BgpRibOut instance using the RIB-Out'''
      keys = self.ribOuts()
      bgpRibOuts = []

      for key in keys:
         bgpRibOut = self.ribOut( key )
         if bgpRibOut:
            bgpRibOuts.append( bgpRibOut )

      return bgpRibOuts

   @Tac.memoize
   def ribOutPeers( self, key ):
      '''Returns IP addresses (v4/v6) of peers using the RIB-Out'''
      m = re.match(
         r'.*^Task RIB-Out %s:.*Peers using this RIB-Out:\s+(.*)' % key +
         r'\s+Route Queue', self.dump_, re.S | re.M )
      if m:
         peers_str = m.group(1)
         exp = re.finditer( r"\s?(?P<addr>[\d+.]+|[0-9a-f:]+) (?P<bit>\d+)",
                  peers_str, re.S | re.M )
         l = []
         if exp:
            for m in exp:
               l.append( RibOutPeer( **m.groupdict() ) )
         return l
      else:
         return []

   @Tac.memoize
   def ribOutAsplQueues( self, key ):
      '''Returns IP addresses (v4/v6) of peers using the RIB-Out'''
      m = re.match(
         r'.*^Task RIB-Out %s:.*Queued annoucements\s+(.*)' % key +
         r'\s+^Task', self.dump_, re.S | re.M )
      if m:
         peers_str = m.group(1)
         exp = re.finditer( r"\s+(?P<addr>[\d+.]+|[0-9a-f:]+) (?P<bit>\d+)",
                  peers_str, re.S | re.M )
         l = []
         if exp:
            for m in exp:
               l.append( RibOutPeer( **m.groupdict() ) )
         return l
      else:
         return []

   @Tac.memoize
   def localAsInfoTable( self ):
      '''
      Parse Local AS info table from ribd dump
      example:
          Local AS number Table: 10 entries
              AS: 50 count: 1 loop: 0
              AS: 100 count: 1 loop: 0
              AS: 101 count: 1 loop: 2
              ...
      '''
      m = re.match( r".*(?P<LocalAsInfoTable>Local AS number Table: " + \
                    r"\d+ entries.*?)" + \
                    r"\s+Communities Table.*",
                     self.dump_, re.S | re.M ) 
      if m:
         return m.group( 'LocalAsInfoTable' )
      else:
         return ''

   @Tac.memoize
   def localAsInfoEntries( self ):
      lst = re.finditer( r"(" + localAsInfoEntriesRe + ")",
                         self.localAsInfoTable_, re.S | re.M )
      localAsInfo = {}
      for m in lst:
         localAsInfo[ int( m.group( 2 ) ) ] = \
               LocalAsInfo(  
               localAs=int( m.group( 2 ) ), 
               count=int( m.group( 3 ) ),
               loop=int( m.group( 4 ) ) )
      return localAsInfo

   @Tac.memoize
   def bgJobInfoTable( self ):
      '''
      Parse background job table from ribd dump
      example:
      Background jobs (2):
                   ecb2f8e0 BGP_20.10.0.0.1_BGP close peer priority 8
                   ecabec20 BGP_20.10.0.0.1_BGP close peer priority 8

      If Rib is in steady state, there shouldn't be any background jobs
      enqueued. 
      '''
      m = re.match( r".*(?P<BgJobInfoTable>Background jobs.*?)" + \
                    r"\s+Flash routines:.*",
                    self.dump_, re.S | re.M ) 
      if m:
         return m.group( 'BgJobInfoTable' )
      else:
         return ''

   @Tac.memoize
   def bgJobInfoEntries( self ):
      lst = re.finditer( r"(" + bgJobInfoEntriesRe + ")",
                         self.bgJobInfoTable_, re.S | re.M )
      bgJobInfo = {}
      for m in lst:
         bgJobInfo[ m.group( 2 ) ] = RibBgJobInfo( jobptr=m.group( 2 ),
               task=m.group( 3 ), jobname=m.group( 4 ),
               jobpriority=m.group( 5 ), flags=m.group( 7 ) )
      return bgJobInfo

   @Tac.memoize
   def ribdTraceFiles( self, includeState=False ):
      '''
      Returns the names of the trace files corresponding to the ribd DUT as a list.
      The global trace file can be different from the per protocol trace file.
      We simply retrieve the dump content and parse it to get this information. We
      won't assert if we fail to do so, but just log a warning.

      NOTE: If one keeps calling ribdDump the information is typically appended.
      In that case this will fail because we are relying on the tracing information
      present only once in the dump file. We rely on the fact that callers
      will delete an existing dump file and then force a dump, hence the tracing
      information is present only once.

      The format of this section of output in ribd dump is:

Tracing:
        Global:  none
        Files:
                /home/sinaraya/workspaces/492-dev/src/../tmp/bgp-trace.txt:
                        closed  refcount 1 size 0  limit 0  files 0
                /tmp/ArosTest/ribd11.ribd-trace.txt:
                        opened  refcount 11 size 29589  limit 0  files 0
                /tmp/ArosTest/ribd12.ribd-trace.txt:
                        opened  refcount 11 size 29589  limit 0  files 0
      '''

      content = self.content()

      pattern = '^Tracing:$\n\tGlobal:.*$\n\tFiles:$\n'
      pattern += '((\t\t.*:$\n\t\t\t(closed|opened).*$\n)+|\nTask)'

      m = re.search( pattern, content, re.M )
      if m:
         fileInfo = re.findall( '\t\t(.*):$\n\t\t\t(closed|opened).*$\n',
                                m.group( 0 ), re.M )
         if includeState:
            return fileInfo
         if fileInfo:
            files = [ x[ 0 ] for x in fileInfo ]
            return files
      sys.stderr.write( 'Warning: could not find trace file information \
                         in ribdDump' )
      return None

   @Tac.memoize
   def agentxSessionSummary( self ):
      '''
      This function parses AGENTX task dump output. For example,
      "show task dump AGENTX all" GII command would produce a task dump like below.
      That's what we are trying to parse here.
      Anywhere you see a '\' to break the line, that's not part of the output itself.

      100 Selected Tasks:
      150   Task AGENTX./var/run/agentx/master  Priority 80 RtProto net-mgmt  \
            <LowPrio>
      150      n_newpolicy 0, n_reinit 0, n_flash 0, n_reinit_async 0, \
            n_rth_flash 0, n_rth_filtered 0, n_rth_filtered_last 0
      150      Trace options:  none
      150 
      150      AGENTX./var/run/agentx/master_Startup  <OneShot>
      150         last: 14:17:07.107   next: 14:17:07.107
      150 
      150 Task AGENTX./var/run/agentx/master:
      150 
      150   Memory Allocation counts:
      150   oidcount: 0
      150   strcount: 0
      150   valuecount: 0
      150   vbcount: 0
      150   pducount: 0
      150 
      150   Subtrees registered:
      150      0: .1.3.6.1.2.1.14.1
      150      1: .1.3.6.1.2.1.14.2
      150      2: .1.3.6.1.2.1.14.4
      150      3: .1.3.6.1.2.1.14.7
      150      4: .1.3.6.1.2.1.14.10
      150      5: .1.3.6.1.2.1.14.12
      150      6: .1.3.6.1.2.1.14.16
      150      7: .1.3.6.1.2.1.14.19
      150 
      150   AgentX Session:
      150   Timeout 0   Retries 0
      150   State 7(closing)  Flags 0()
      150   Restart Timer 2   ping timer 60   ping interval 0
      150   Task errno 521(Unknown error 521)  SNMP errno 0((null))
      150   AgentX Session ID a
      150 
      150   AgentX Session statistics:
      150      pdu_dropped         :     0
      150      malloc_failed       :     0
      150      open_sent           :     1
      150      open_acked          :     1
      150      close_sent          :     0
      150      close_acked         :     0
      150      register_sent       :     8
      150      register_acked      :     8
      150      unregister_sent     :     0
      150      unregister_acked    :     0
      150      indexalloc_sent     :     0
      150      indexalloc_acked    :     0
      150      indexdealloc_sent   :     0
      150      indexdealloc_acked  :     0
      150      agentcaps_sent      :     1
      150      remove_agentcaps_sent:     0
      150      ping_sent           :     0
      150      ping_acked          :     0
      150      notify_sent         :     0
      150      notify_acked        :     0
      150      restart_count       :     1
      150      get_response_sent   :     1
      150      getnext_response_sent:     0
      150      getbulk_response_sent:     0
      150      too_big             :     0
      150      testSet_rx          :     0
      150      commitSet_rx        :     0
      150      undoSet_rx          :     0
      150      cleanupSet_rx       :     0
      150      set_response_tx     :     0
      150 
      '''
      sessionSummary = {}
      output = re.sub(r'^1\d0 +', '', self.dump_, flags=re.M) #GII return code strip
      m = re.search( "^Task AGENTX.*:$\n((^$\n|^\t.*$\n)+)", output, re.M )
      if not m:
         return sessionSummary
      agentxEntry = m.group( 0 )

      lst = re.finditer( r'(?P<countType>'
                         '(oidcount|strcount|valuecount|vbcount|pducount))'
                         ': (?P<count>[0-9]+)', agentxEntry )
      for m in lst:
         sessionSummary[ m.group( 'countType' ) ] = m.group( 'count' )

      m = re.search( r'State [0-9]\((?P<state>[a-z]*)\).*', agentxEntry, re.M )
      if m:
         sessionSummary[ 'state' ] = m.group( 'state' )

      m = re.search( r'Restart Timer (?P<restart>[0-9]+)\s*'
                     'ping timer (?P<ping>[0-9]+).*', agentxEntry )
      if m:
         sessionSummary[ 'restart-timer' ] = m.group( 'restart' )
         sessionSummary[ 'ping-timer' ] = m.group( 'ping' )

      m = re.search( r'AgentX Session ID (?P<sess>[0-9a-fA-F]+)$', 
                     agentxEntry, re.M )
      if m:
         sessionSummary[ 'sessionId' ] = int( m.group( 'sess' ), 16 ) # base16

      m = re.search( r'Subtrees registered:$\n(?P<subtrees>(\t\t.*$\n)+)',
                     agentxEntry, re.M )
      if m:
         subtrees = []
         trees = m.group( 'subtrees' ).replace( '\t', '' ).split( '\n' )
         for t in trees:
            if not t:
               continue
            oid = t.split( ':' )[ 1 ]
            subtrees.append( oid.strip() )
         sessionSummary[ 'subtrees' ] = subtrees

      m = re.search( r'AgentX Session statistics:$\n(?P<stats>(\t\t.*\n)+)',
                     agentxEntry, re.M )
      if not m:
         return sessionSummary

      statistics = {}
      stats = m.group( 'stats' ).replace( '\t', '' ).split( '\n' )
      for s in stats:
         m = re.match( r"(?P<label>[A-Za-z_]+)\s*:\s*(?P<value>[0-9]+)", s )
         if not m:
            continue
         key = m.group( 'label' )
         val = m.group( 'value' )
         statistics[ key ] = int( val )
      sessionSummary[ 'statistics' ] = statistics

      return sessionSummary

   def getSentRmxList( self, nbrId ):
      '''
      Returns a list with elements as a dictionary of keys 'Type', 'LinkId'
      AdvrtId'. The corresponding values are type of LSA ( 'RTR', 'NTW', 'AIP',
      'ABR', 'AEX' ), link Id of LSA and advertising router id.
      Gets the required information from parsing the below output from 
      'show tech ribd' under Task OSPF
      sent retransmit list:
         type RTR id 1.1.1.1 advrt 1.1.1.1 seq 0x80000004 [mem: e6b9d138]
      '''
      content = self.content()
      pattern = 'Task OSPF.*neighbor id %s.*sent retransmit list:(.*)ack list' \
                % nbrId
      lsas = []
      m = re.search( pattern, content, re.S )
      if m:
         pattern = r'type (\w+) id (%s) advrt (%s) seq (\S+)' % ( ipv4Re, ipv4Re )
         rmxList = re.findall( pattern, m.group( 1 ) )
         for entry in rmxList:
            entryDict = {}
            entryDict[ 'Type' ] = entry[ 0 ]
            entryDict[ 'LinkId' ] = entry[ 1 ]
            entryDict[ 'AdvrtId' ] = entry[ 2 ]
            entryDict[ 'SeqNum' ] = entry[ 3 ]
            lsas.append( entryDict )
      return lsas

   def getOspfStatus( self, processId=None ):
      '''
      Returns True if OSPF is enabled else False.
      Gets the required information from parsing the below output from 'show tech
      ribd' and checking for enable flag.
      /OSPF_PROTO/OSPF_INST(10):
      Current values:
         * 15 Key instance                      Uint32:  10 (0xa)
           14 Opt enable                        Flag: TRUE
      '''
      content = self.content()
      if processId:
         pattern = r'/OSPF_PROTO/OSPF_INST\(%d\):.*?Opt enable\s*Flag:\s*(\w*)' \
                                                                        % processId
      else:
         pattern = r'/OSPF_PROTO/OSPF_INST\(\d+\):.*?Opt enable\s*Flag:\s*(\w*)'
      m = re.search( pattern, content, re.S )
      if m:
         return m.group( 1 )=='TRUE'
      return False

   @Tac.memoize
   def getNumOfOspfHellos( self ):
      '''
      Returns a dictionary with keys as 'Received' and 'Sent' and values as the
      number of hellos received and sent respectively.
      Gets the required information from parsing the below output from 'show tech
      ribd'
      Task OSPF (instance 1):

       router id: 0.0.0.1
       preference: 110
       ase preference: 110
       rfc1583compatability: 0
       default-information 0, always 0, metric 1, metric-type 2, route-map NULL

       default-information lsa: 0
       originated lsas: 1    received lsas: 2002

       Packets Received:
          0: Monitor                           6411: Hello                           
         93: Database Description                 0: LS Request                      
       1593: LS Update                         5102: LS Acknowledgement              

       Packets Sent:
          0: Monitor                           4273: Hello                           
         96: Database Description                31: LS Request                      
        915: LS Update                         1704: LS Acknowledgement              
      '''
      content = self.content()
      pattern = r'Task OSPF.* (\d+): Hello.* (\d+): Hello'
      noOfHellos = { 'Received' : 0, 'Sent' : 0 }
      m = re.search( pattern, content, re.S )
      if m:
         noOfHellos[ 'Received' ] = int( m.group( 1 ) )
         noOfHellos[ 'Sent' ] = int( m.group( 2 ) )
      return noOfHellos

   @Tac.memoize
   def getNumOfDDs( self ):
      '''
      Returns a dictionary with keys as 'Received' and 'Sent' and values as the
      number of DDs received and sent respectively.
      Gets the required information from parsing the output from 'show tech
      ribd under Task OSPF
      '''
      content = self.content()
      pattern = r'Task OSPF.* (\d+): Database.* (\d+): Database'
      noOfDDs = { 'Received' : 0, 'Sent' : 0 }
      m = re.search( pattern, content, re.S )
      if m:
         noOfDDs[ 'Received' ] = int( m.group( 1 ) )
         noOfDDs[ 'Sent' ] = int( m.group( 2 ) )
      return noOfDDs

   def isisIntfInfo( self ):
      return IsisIntfInfo.parse( self.dump_ )

   def nexthopTable( self, ipv6=False ):
      return SharedNexthopTable.parseNexthopDump( self.dump_, ipv6=ipv6 )

   def getPacketQueues( self ):
      '''
      Returns a dictionary with the packet queue name as the keys and
      the value is a sub dictionary describing the packet queue. The dump output
      may not have any packet queues, in which case an empty dictionary is returned.

      A sample packet queue is below (the packet delays are not parsed):
      0xebe976e0 (OSPF LS-ACK)
      Priority: 10
      Queueing behavior: Drop Older
      Length of keys: 0
      Packet limit: 65535    Limit per key: 65535
      Current keys: 0    Current packets: 0
      Max keys: 0        Max packets: 0
      Packets dropped from this queue: 0
      Packet delays:
        0.0-0.1s: 0
        0.1-0.2s: 0
        0.2-0.3s: 0
        0.3-0.4s: 0
        0.4-0.5s: 0
        0.5-0.6s: 0
        0.6-0.7s: 0
        0.7-0.8s: 0
        0.8-0.9s: 0
        0.9-0.10s: 0
        1-2s: 0
        2-3s: 0
        3-4s: 0
        4-5s: 0
        5-6s: 0
        6-7s: 0
        7-8s: 0
        8-9s: 0
        9-10s: 0
        >10s: 0
      '''
      resultDict = {}

      # find the section of the content to look at (substring)
      # without this substring, because of all the .* in the expression
      # below, this would take huge amounts of time to find nothing if
      # ospf is not enabled (no packet queues)
      try:
         content = self.content()
         indexStart = content.index( "Packet Queues:" )
         indexEnd = content.index( "Packet Buffer Pools:", indexStart )
      except ValueError:
         print "Could not generate packet queue substring"
         return resultDict


      queueSubString = content[indexStart:indexEnd]

      lst = re.finditer( r"0x[0-9a-f]+.*?\((?P<qname>.+?)\).*?" +
                    r"Priority: (?P<priority>\d+).*?" +
                    r"Queueing behavior: (?P<qbehavior>[\s\w]+)$.*?" +
                    r"Length of keys: (?P<keylen>\d+).*?" +
                    r"Packet limit: (?P<pktlimit>\d+).*?" +
                    r"Limit per key: (?P<limitperkey>\d+).*?" +
                    r"Current keys: (?P<curkeys>\d+).*?" +
                    r"Current packets: (?P<curpkts>\d+).*?" +
                    r"Max keys: (?P<maxkeys>\d+).*?" +
                    r"Max packets: (?P<maxpackets>\d+).*?" +
                    r"Packets dropped from this queue: (?P<droppedpkts>\d+)",
                          queueSubString, re.S | re.M )

      for m in lst:
         queueDict = {}
         queueDict["Priority"] = int( m.groupdict()["priority"] )
         queueDict["Queueing behavior"] = m.groupdict()["qbehavior"]
         queueDict["Length of keys"] = int( m.groupdict()["keylen"] )
         queueDict["Packet limit"] = int( m.groupdict()["pktlimit"] )
         queueDict["Limit per key"] = int( m.groupdict()["limitperkey"] )
         queueDict["Current keys"] = int( m.groupdict()["curkeys"] )
         queueDict["Current packets"] = int( m.groupdict()["curpkts"] )
         queueDict["Max keys"] = int( m.groupdict()["maxkeys"] )
         queueDict["Max packets"] = int( m.groupdict()["maxpackets"] )
         queueDict["Packets dropped from this queue"] = \
                                             int( m.groupdict()["droppedpkts"] )

         resultDict[m.groupdict()["qname"]] = queueDict

      return resultDict

   @Tac.memoize
   def adjacencyTable( self ):
      '''
      Parse Shared-Adjacency Table from ribd dump
      '''
      m = re.match( r".*(?P<AdjTable>Shared-Adjacency Table.*?)" + \
                    r"\s+NHE-LIST Cache",
                    self.dump_, re.S | re.M )
      if m:
         return SharedAdjacencyTable( m.group( 'AdjTable' ) ).adjacencyList()
      else:
         return ''

class SrTePolicyAttr( object ):
   """
   Parse a single SR TE Policy and provide APIs to access the different constituents
   """
   def __init__( self, dump ):
      self.dump_ = dump
      self.policyAttr = self.dump_.splitlines()

   def bsid( self ):
      '''
      return binding sid as an int, -1 if bsid is not present
      '''
      sid = self.policyAttr[ 0 ].split( ":" )[ 1 ].strip()
      return int( sid, 16 ) if sid != '-' else -1

   def pref( self ):
      '''
      return preference as an int
      '''
      return int( self.policyAttr[ 1 ].split( ":" )[ 1 ].strip() )

   def enlp( self ):
      '''
      return enlp as an int
      '''
      value = self.policyAttr[ 2 ].split( ":" )[ 1 ].strip()
      return None if value == '-' else int( value )

   def slwt( self ):
      '''
      return a list of 2-tuples. Each 2-tuple contains a segment list and a
      weight. Weight is -1 if not present.
      '''
      slws = []
      for slw in self.policyAttr[ 3 : ]:
         sl = re.findall( '0x[0-9a-f]+', slw )
         sl = [ int( s, 16 ) for s in sl ]
         if 'hasNonType1' in slw:
            wt = slw.split( 'Wt:' )[ 1 ].split( '' )[ 0 ].strip()
         else:
            wt = slw.split( 'Wt:' )[ 1 ].strip()
         wt = int( wt ) if wt != '-' else -1
         slws.append( ( sl, wt ) )
      return slws

class BgpSrTePolicyDedupTableStats( object ):
   """
   Parse SR-TE dedup table stats
   """
   def __init__( self, dump ):
      self.dump_ = dump
      self.stats = self.dump_.splitlines()

   def _getVal( self, idx, pat ):
      return int( re.match( '.*%s \((\d+)\)' % pat,
                            self.stats[ idx ] ).groups()[ 0 ] )

   def policyAttrTblSize( self ):
      return self._getVal( 1, 'size' )

   def policyAttrTblNumEntries( self ):
      return self._getVal( 1, 'num entries' )

   def slTblSize( self ):
      return self._getVal( 2, 'size' )

   def slTblNumEntries( self ):
      return self._getVal( 2, 'num entries' )

   def segmentTblSize( self ):
      return self._getVal( 3, 'size' )

   def segmentTblNumEntries( self ):
      return self._getVal( 3, 'num entries' )

   def getAllAsDict( self ):
      ret = {}
      ret[ 'paSize' ] = self.policyAttrTblSize()
      ret[ 'paNum' ] = self.policyAttrTblNumEntries()
      ret[ 'slSize' ] = self.slTblSize()
      ret[ 'slNum' ] = self.slTblNumEntries()
      ret[ 'segSize' ] = self.segmentTblSize()
      ret[ 'segNum' ] = self.segmentTblNumEntries()
      return ret

   def getAllAsTuple( self ):
      return ( self.policyAttrTblSize(), self.policyAttrTblNumEntries(),
               self.slTblSize(), self.slTblNumEntries(),
               self.segmentTblSize(), self.segmentTblNumEntries() )

# TODO: Move functions from RibDumpTestLib to methods of RibdDump

# Finally, a helper to run on the DUT, which returns a RibdDump.
def ribdDump( vrfName=DEFAULT_VRF, agent=None ):
   def doShowRibdDump( ns=NsLib.DEFAULT_NS ):
      try:
         contents = NsLib.runMaybeInNetNs( ns, [ '/usr/bin/RibdShowTech' ],
                                           stdout=Tac.CAPTURE,
                                           stderr=Tac.DISCARD,
                                           asRoot=True )
      except Tac.SystemCommandError as e:
         vrfErrStr = ''
         if vrfName != DEFAULT_VRF:
            vrfErrStr += ' for VRF %s' % vrfName
         if e.error in ( 1, 2 ):
            print 'Failed to connect to ribd%s' % vrfErrStr
         elif e.error == 3:
            print 'Timeout encountered while processing show tech-support ribd%s' % \
                  vrfErrStr
         raise
      return contents
   if not vrfName or vrfName == DEFAULT_VRF:
      ns = NsLib.DEFAULT_NS
   else:
      ns = "ns-" + vrfName
   contents = doShowRibdDump( ns=ns )
   if contents:
      dump = RibdDump( contents )
      return dump
   else:
      return None

# Subtract one RibdDump.memUsage() from another.
def memUsageDelta( then, now ):
   delta = {}
   for bucket in now:
      for typ in now[ bucket ]:
         prevUse = then.get( bucket, {} ).get( typ, 0 )
         inUse = now[ bucket ][ typ ]
         if inUse != prevUse:
            delta.setdefault( bucket, {} )[ typ ] = inUse - prevUse
   return delta

def printMemUsageDelta( delta, prefix=None, printAll=False ):
   print 'Size type delta'
   for bucket in sorted( delta.keys() ):
      for typ in sorted( delta[ bucket ] ):
         if delta[ bucket ][ typ ] == 0 and not printAll:
            continue
         if prefix:
            print prefix,
         print bucket, typ, delta[ bucket ][ typ ]

# Subtract one GiiPolicy object from another and return delta values as dicts
# in different attributes:
#    - delta.rmc - dict
#    - delta.prefixList - dict
#    - delta.commList - dict
#    - delta.aspathList - dict
#
# Following booleans are set to 1 if there are no leaks for respective objects
#
#    - delta.rmcNoLeak - bool
#    - delta.prefixListNoLeak - bool
#    - delta.commListNoLeak - bool
#    - delta.aspathListNoLeak - bool
#
def policyRefcountDelta( then, now ):
   delta = Bunch()
   def getDeltas( refthen, refnow ):
      deltaRef = {}
      for rm in refthen:
         if not rm in refnow:
            deltaRef[rm] = 0
         else:
            deltaRef[rm] = refnow[rm].refcount - refthen[rm].refcount
      for rm in refnow:
         if rm not in refthen:
            deltaRef[rm] = refnow[rm].refcount
      noLeak = len( deltaRef.keys() ) == 0 or \
               all( [ v == 0 for v in deltaRef.values() ] )
      return (deltaRef, noLeak)

   # RMC - Routemap Chain
   delta.rmc, delta.rmcNoLeak = \
      getDeltas( then.routemapChains(), now.routemapChains() )

   # Prefix Lists
   delta.prefixList, delta.prefixListNoLeak = \
      getDeltas( then.prefixLists(), now.prefixLists() )

   # Community Lists
   delta.commList, delta.commListNoLeak = \
      getDeltas( then.communityLists(), now.communityLists() )

   # Aspath Lists
   delta.aspathList, delta.aspathListNoLeak = \
      getDeltas( then.aspathLists(), now.aspathLists() )

   return delta

class DynPolicyTask( RibdDumpAttributeParser ):
   '''
   Parses dynamic policy task context in ribd dump. If we get
   entire show task dump, then the regex match will work or 
   else we can simply use the entire dump output for our purposes.
   '''
   def __init__( self, rawstr ):
      rawstr = re.sub( r'^1\d0\s+', '', rawstr, flags=re.M )
      dump = ''
      m = re.match( r".*(?P<dynPolicyTask>Task dp_match:.*?)^Task.*",
                    rawstr, re.S | re.M )
      if m:
         dump = m.group( 'dynPolicyTask' )
      elif 'Selected Tasks' in rawstr:
         dump = rawstr
      RibdDumpAttributeParser.__init__(self, dump )

   def getNumConfiguredPolicies( self ):
      return int( self.prp( 'Number of policies configured', '[0-9]+' ) )

   def getNumActivePolicies( self ):
      return int( self.prp( 'Number of active policies', '[0-9]+' ) )

   def getDynamicPolicies( self ):
      policies = {}
      lst = re.finditer( 
            r"Dynamic Policy (?P<policyName>\S*)\(ID: (?P<policyid>[0-9]+)\)"
            r"\s+Match: (?P<match>\S*), "
            r"IPv4 Pl: (?P<ipv4Pl>\S*), "
            r"IPv6 Pl: (?P<ipv6Pl>\S*)"
            r"\s+Active: (?P<state>\S*), "
            r"Num contributors: (?P<numContribs>\S*)",
            self.raw(), re.S | re.M )
      for match in lst:
         policy = match.groupdict()
         policyName = policy.pop( 'policyName' )
         policies[ policyName ] =  match.groupdict()

      return policies

   def getDynamicPolicy( self, policyName ):
      return self.getDynamicPolicies().pop( policyName, None )

   def getDynamicPolicyId( self, policyName ):
      policy = self.getDynamicPolicy( policyName )
      if policy:
         return policy.pop( 'policyid' )

   def getDynamicPolicyState( self, policyName ):
      policy = self.getDynamicPolicy( policyName )
      if policy:
         state = policy.pop( 'state' )
         return True if state == "True" else False

   def getNumContributors( self, policyName ):
      policy = self.getDynamicPolicy( policyName )
      if policy:
         numContribs = int( policy.pop( 'numContribs' ) )
         return numContribs
