# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac
import DesCrypt

# pylint: disable-msg=W0105
'''
   The ReversibleSecretCli originally used the SecretCliLib::Obfuscator to
   encrypt and decrypt passwords.  In encrypted form these are alpha-numeric
   without symbols.  

   A new optional 'algorithm=MD5' parameter was added to the encodeKey()
   in order to  support the common encryption key feature where 
   a common key is used for the internal encryption of the neighbor
   passwords for hidden storage in the configuration.  This algorithm produces 
   an encrypted string that contains symbols (i.e. the equal sign) and 
   therefore can't be substitued for the existing encryption without 
   impacting existing code that expects no symbols in the encrypted string. 
   This parameter defaults to 'False' so that existing code works as it
   always has.

   A new optional 'key' parameter was added so that the neighbor passwords
   can be encrypted with the key they have always used or with the new common
   key based on an operator setting. The 'key' parameter will default to none
   indicating to use the common key with MD5 or the obfuscated key.
   
   When the password has been encrypted with the MD5 Algorithm the
   '$1$' prefix was added to the encrypted password in the 4.20 release only.
   This prefix has been borrowed from the 'crypt(3)' encryption type prefix 
   convention.  When the password has been encrypted with the MD5 Algorithm 
   using the common encryption key, the '$1c$' prefix will be added 
   to the encrypted password.
   This prefix will be used by the decrypt algorithm to determine which
   method and key to use to decrypt the hidden password.
   '''
# pylint: enable-msg=W0105


# cleartext and obfuscated key
obfuscator = Tac.newInstance( "SecretCliLib::Obfuscator" )
commonKey = "ab922cf237a82cb91d2ee72f95b6edb8"
desCryptCommonMd5Prefix = '$1c$'
desCryptMd5PrefixDeprecated = '$1$'
supportedAlgorithms = [ None, 'MD5' ]  # None defaults to obfuscate method

def encodeKey( plaintext, key=None, algorithm=None ):
   hidden = ''
   if algorithm not in supportedAlgorithms:
      assert False, 'please pass supported algorithm'

   if algorithm is None:
      return obfuscator.doEncodeKey( plaintext )

   if key is None:
      key = commonKey
      hidden += desCryptCommonMd5Prefix
   hidden += DesCrypt.encrypt( key, plaintext )
   return hidden

def decodeKey( hiddentext, key=None, algorithm=None ):
   # make sure good parameters passed
   if algorithm not in supportedAlgorithms:
      raise ValueError( "please pass supported algorithm" )

   if algorithm is None:
      return obfuscator.doDecodeKey( hiddentext )

   # Determine how hiddentext was encrypted based in prefix
   # Note: this could override the passed parameter if the prefix
   #       does not match the passed algorithm parameter
   if hiddentext.startswith( desCryptCommonMd5Prefix ):
      hiddenPrefixRemoved = hiddentext[ len( desCryptCommonMd5Prefix ) : ]
      algorithm = 'MD5'
      useKey = commonKey
   elif hiddentext.startswith( desCryptMd5PrefixDeprecated ) :
      hiddenPrefixRemoved = hiddentext[ len( desCryptMd5PrefixDeprecated ) : ]
      if key is None:
         raise ValueError( "missing key parameter" )
      algorithm = 'MD5'
      useKey = key
   else:
      # no matching prefix
      useKey = key
      hiddenPrefixRemoved = None

   if hiddenPrefixRemoved:
      # Try to decrpyt with the prefix removed and the key specific
      # to the prefix.  
      try:
         return DesCrypt.decrypt( useKey, hiddenPrefixRemoved )
      except Exception:   # pylint: disable-msg=W0703
         # decrypt throws error of type Exception with argument 'decryption error'
         # we could aso get a ValueError or TypeError.  Catching all here.
         # if we get an exception we are going to assume
         # that we hit the unlikely case where an encrytped
         # password that did not use the common key has the $1C$
         # prefix as part of its encrytped form not as a prefix and
         # we will try to decrypt with the passed key
         pass

   # Decrypt with prefix determined or passed algorithm type
   if key is None:
      raise ValueError( "missing key parameter" )
   return DesCrypt.decrypt( key, hiddentext )

#-------------------------------------------------------------------------------
# Methods for server key CLI
#-------------------------------------------------------------------------------
# a Cli rule that accepts both cleartext and obfuscated key
unencryptedName_ = 'authUnencryptedPasswd'
encryptedName_ = 'authEncryptedPasswd'

def tryDecodeToken( authToken, key=None, algorithm=None, mode=None ):
   """Tries to decode a match given from authPasswdRule.
      returns None and adds an error to mode upon failure"""
   cleartext = authToken.get( unencryptedName_ )
   if cleartext:
      return cleartext
   try:
      return decodeKey( authToken[ encryptedName_ ],
                        key=key,
                        algorithm=algorithm )
   except: # pylint: disable=bare-except
      if mode:
         mode.addError( 'Invalid encrypted password' )
      else:
         raise
      return None

############################################################
# command class for the new parser
############################################################

class NotZeroOrSevenMatcher( CliMatcher.Matcher ):
   '''Takes a matcher, but filter out input that is 0 or 7'''
   def __init__( self, matcher ):
      CliMatcher.Matcher.__init__( self )
      self.matcher_ = matcher

   def match( self, mode, context, token ):
      if context.state is None and token in ( '0', '7' ):
         # first token
         return CliMatcher.noMatch
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      return self.matcher_.completions( mode, context, token )

cleartextHelpDesc = 'Unobfuscated key string'
cleartextMatcher_ = CliMatcher.StringMatcher( helpname='LINE',
                                              helpdesc=cleartextHelpDesc )
type0KwMatcher = CliMatcher.KeywordMatcher(
   '0',
   helpdesc='Indicates that the key string is not encrypted' )
type7KwMatcher = CliMatcher.KeywordMatcher(
   '7',
   helpdesc='Specifies that a HIDDEN key will follow' )
type7Matcher = CliMatcher.PatternMatcher( '[0-9][0-9]([a-fA-F0-9][a-fA-F0-9])+',
                                          helpname='LINE',
                                          helpdesc='Obfuscated key string' )
type7Node = CliCommand.Node( type7Matcher, sensitive=True )

def _popAll( args ):
   for k in ( '$CLEARTEXT', '$TYPE7KEY', '0', '7' ):
      args.pop( k, None )

def reversibleSecretCliExpression( name, cleartextMatcher=None,
                                   keyTransformer=None, obfuscatedTextMatcher=None,
                                   returnifCleartext=None ):
   # This is reversible used by TACACS+/RADIUS.
   #
   # keyTransformer is a function that converts cleartext password
   # before encoding it.
   # obfuscatedTextMatcher will accept the obfuscated key string that is passed.
   # returnifCleartext will return PLAIN password without encoding if set to True. 
   if not cleartextMatcher:
      cleartextMatcher = cleartextMatcher_
   if not obfuscatedTextMatcher:
      obfuscatedTextMatcher = type7Node

   class ReversibleSecretExpression( CliCommand.CliExpression ):
      expression = "$CLEARTEXT | " \
                   "( 0 $TYPE0KEY ) | ( 7 $TYPE7KEY )"
      data = {
         '$CLEARTEXT' : CliCommand.Node( NotZeroOrSevenMatcher( cleartextMatcher ),
                                         sensitive=True ),
         '$TYPE0KEY' : CliCommand.Node( cleartextMatcher,
                                        sensitive=True ),
         '$TYPE7KEY' : obfuscatedTextMatcher,
         '0' : type0KwMatcher,
         '7' : type7KwMatcher
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         type7key = args.get( '$TYPE7KEY' )
         if not type7key:
            cleartext = args.get( '$CLEARTEXT' ) or args.get( '$TYPE0KEY' )
            if returnifCleartext:
               args[ 'PLAIN' ] = cleartext
               _popAll( args )
            elif cleartext:
               if keyTransformer:
                  cleartext = keyTransformer( cleartext )
               type7key = encodeKey( cleartext )

         if type7key:
            args[ name ] = type7key
            _popAll( args )

   return ReversibleSecretExpression

cleartextAuthMatcher = CliMatcher.PatternMatcher( r'.{1,80}',
                                                  helpname='WORD',
                                       helpdesc='password (up to 80 chars)' )

cleartextAuthNode = CliCommand.Node( NotZeroOrSevenMatcher( cleartextAuthMatcher ),
                                     sensitive=True )

type0AuthNode = CliCommand.Node( cleartextAuthMatcher, sensitive=True )

type7AuthMatcher = CliMatcher.PatternMatcher( r'.+',
                                              helpname='WORD',
                                              helpdesc='encrypted password' )
type7AuthNode = CliCommand.Node( type7AuthMatcher, sensitive=True )

def reversibleAuthPasswordExpression( argName,
                                      unencryptedName=unencryptedName_,
                                      encryptedName=encryptedName_ ):
   # This is reversible used by BGP etc
   class ReversibleAuthPasswordExpression( CliCommand.CliExpression ):
      expression = "$CLEARTEXT | ( 0 $TYPE0KEY ) | ( 7 $TYPE7KEY )"
      data = {
         '$CLEARTEXT' : cleartextAuthNode,
         '$TYPE0KEY' : type0AuthNode,
         '$TYPE7KEY' : type7AuthNode,
         '0' : type0KwMatcher,
         '7' : type7KwMatcher
      }
      @staticmethod
      def adapter( mode, args, argsList ):
         passwd = args.get( '$CLEARTEXT' ) or args.get( '$TYPE0KEY' )
         result = None
         if passwd:
            result = { unencryptedName: passwd }
         else:
            type7Key = args.get( '$TYPE7KEY' )
            if type7Key:
               result = { encryptedName: type7Key }
         if result:
            _popAll( args )
            args[ argName ] = result

   return ReversibleAuthPasswordExpression


