# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import imp
import inspect
import os

class FeatureKeysCheckError( Exception ):
   pass

def safeRemove( filename ):
   '''Try to remove the specified file. Catch any errors. '''

   try:
      os.remove( filename )
   except ( OSError, TypeError ):
      pass

def extractAsuManifest( nextImagePath ):
   '''Extract the 'asu-manifest' file from the next image to /tmp. '''

   import zipfile
   with zipfile.ZipFile( nextImagePath ) as nextImage:
      if 'asu-manifest' not in nextImage.namelist():
         raise FeatureKeysCheckError( 'New image missing ASU manifest file.' )
      return nextImage.extract( 'asu-manifest', '/tmp' )
   
def getFeatureKeysDict( manifestFile ):
   '''Extract and return the feature keys dict from the specified asu-manifest
   file. Raise an exception if an error occurs.
   '''

   try:
      with open( manifestFile ) as manifest:
         newAsuManifest = imp.new_module( 'newAsuManifest' )
         exec( manifest.read(), newAsuManifest.__dict__ )
         return getattr( newAsuManifest, 'pStoreFeatureKeysDict', {} )
   except ( AttributeError, IOError ):
      raise FeatureKeysCheckError( 'Failed to open ASU manifest file.' )
   except SyntaxError:
      raise FeatureKeysCheckError( 'ASU manifest file corrupted.' )

def compareFeatureKeysDicts( currDict, nextDict ):
   '''Perform the feature-keys check by comparing the feature-keys dictionaries of
   the current and next image. If the check passes, return nothing. If the check
   fails, raise an exception.
   '''

   if not isinstance( currDict, dict ) or not isinstance( nextDict, dict ):
      raise FeatureKeysCheckError( 'Invalid input type.' )
   for feature, featureKeys in currDict.iteritems():
      if feature not in nextDict:
         raise FeatureKeysCheckError( "PStore manifest check fail: feature %s not"
                                      " in manifest." % repr( feature ) )
      nextKeys = nextDict[ feature ]
      diffSet = set( featureKeys ).difference( set( nextKeys ) )
      if diffSet:
         diffSetStr = "    " + ", ".join( str( key ) for key in diffSet )
         raise FeatureKeysCheckError( "PStore manifest check fail: keys in feature"
                                      " %s not supported:\n%s" % ( repr( feature ),
                                                                  diffSetStr ) )

def checkFeatureKeys( ctx ):
   '''Check that the next image supports the required features and keys for an ASU
   upgrade. The ReloadPolicy plugin context provides the next image information and
   the mode information.
   '''

   asuManifest = None
   try:
      # Get the next image feature-keys.
      if "ASU_MANIFEST_NEXT" in os.environ:
         nextFeatureKeysDict = getFeatureKeysDict(
                                 os.environ[ "ASU_MANIFEST_NEXT" ] )
      else:
         asuManifest = extractAsuManifest( ctx.nextImage )
         nextFeatureKeysDict = getFeatureKeysDict( asuManifest )

      # Get the current image feature-keys.
      if "ASU_MANIFEST_LOCAL" in os.environ:
         currFeatureKeysDict = getFeatureKeysDict(
                                 os.environ[ "ASU_MANIFEST_LOCAL" ] )
      else:
         assert ctx.mode
         import AsuUtil
         if not hasattr( AsuUtil, 'doGetFeatureKeys' ):
            raise NameError
         # Check that doGetFeatureKeys is expecting one argument.
         if not len( inspect.getargspec( AsuUtil.doGetFeatureKeys ).args ) is 1:
            raise TypeError
         currFeatureKeysDict = AsuUtil.doGetFeatureKeys( ctx.mode.entityManager )

      compareFeatureKeysDicts( currFeatureKeysDict, nextFeatureKeysDict )
      return True

   except FeatureKeysCheckError as e:
      ctx.addError( str( e ) )
      return False
   except ( ImportError, NameError, TypeError ):
      # Accounts for the cases where AsuUtil isn't found, AsuUtil doesn't contain
      # doGetFeatureKeys, or AsuUtil arguments have changed.
      ctx.addWarning( "PStore manifest check fail: could not perform feature-keys"
                      " check." )
      return None
   finally:
      safeRemove( asuManifest )

def Plugin( ctx ):
   category = [ 'ASU' ]
   ctx.addPolicy( checkFeatureKeys, category )
