# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
from collections import namedtuple

VersionParts = namedtuple( 'VersionParts', 'nums chars' )

def ignoreFlavorAndVariant( version ):
   '''Some SWIs have flavors '-2GB' or '-PDP', or '-INT' variant. We don't
   want to treat these as downgrades if the version and release numbers are
   the same. This function filters the flavor/variant out of the version string.
   '''
   flavorsOrVariants = re.findall( r'-2GB|-PDP|-INT', version )
   for flavorOrVariant in flavorsOrVariants:
      version = version.replace( flavorOrVariant, '' )
   return version

def parseVersionStr( version ):
   '''Given a version string (i.e. '4.20.1F'), parse the string and return
   a VersionParts tuple containing a list of numbers and a list of letters
   in the string.
   '''
   version = ignoreFlavorAndVariant( version )
   vNumsAndChars = filter( None, re.split( r'(\d+)|\.', version ) )
   vNums = [ int( num ) for num in vNumsAndChars if num.isdigit() ]
   vChars = [ char.lower() for char in vNumsAndChars if char.isalpha() ]
   return VersionParts( vNums, vChars )

def isBlessed( versionInfo ):
   try:
      return versionInfo.isBlessed()
   # Since this plugin will run in the context of the source release, which may be
   # older and may not have the updated VersionInfo and SwiVersion type, we fall back
   # to using the version string.
   except AttributeError:
      return '(engineering build)' not in versionInfo.version()

def checkAsuDowngrade( ctx ):
   '''Determines if the next image version is a downgrade and needs to be prevented.
   If the next version is a downgrade, add a blocking error. If the next version
   might be a downgrade, add a warning. The ReloadPolicy plugin context provides
   the version information for the current and next image, so we don't need to
   implement that here. See AID 2788 for downgrade check details.
   '''
   warnMsg = ( "Unable to compare version numbers %s and %s, therefore cannot "
               "determine if performing an SSU downgrade, which is not supported. "
               "Please proceed if you are sure it is not a downgrade."
               % ( ctx.currentVersion.version(), ctx.nextVersion.version() ) )
   errMsg = "ASU does not support downgrade from %s to %s." % (
               ctx.currentVersion.version(), ctx.nextVersion.version() )

   currVersion = parseVersionStr( ctx.currentVersion.version() )
   nextVersion = parseVersionStr( ctx.nextVersion.version() )
   if currVersion.nums < nextVersion.nums:
      return True
   elif currVersion.nums > nextVersion.nums:
      if isBlessed( ctx.nextVersion ):
         ctx.addError( errMsg )
         return False
      else:
         # Destination is an engineering build with a smaller version number.
         ctx.addWarning( warnMsg )
         return True
   elif currVersion.chars != nextVersion.chars:
      ctx.addWarning( warnMsg )
   else:
      # internalVersion = swi_version + '-' + swi_release, so this comparison is
      # equivalent to comparing swi_release. Check this if swi_versions are equal.
      if not ( ignoreFlavorAndVariant( ctx.currentVersion.internalVersion() ) ==
               ignoreFlavorAndVariant( ctx.nextVersion.internalVersion() ) ):
         ctx.addWarning( warnMsg )
   return True

def Plugin( ctx ):
   category = [ 'ASU' ]
   ctx.addPolicy( checkAsuDowngrade, category )
