# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Bogus import to generate explicit dependency on Arnet
# pkgdeps: import Arnet

from __future__ import absolute_import, division, print_function

from functools import wraps
import itertools
import RcfLibUtils
from TypeFuture import TacLazyType
import QuickTrace

qv = QuickTrace.Var
qt8 = QuickTrace.trace8

AllFunctionStatus = TacLazyType( "Rcf::AllFunctionStatus" )
publishedAfs = None

# Just a simple running counter.  This'll be used as the 'key' constructor arg when
# instantiating AET nodes.
getNewAetId = itertools.count( start=long( 1 ) )

"""This holds the set of all the 'key' values in the AET for an
Rcf::Eval::Function.
"""
currentFunctionKeySet = None

def aetNodeInstantiator( typename ):
   """All the nodes present in an AET must be part of an instantiating collection.
   So all the AET types have a U64 constructor arg called 'key'.
   This function takes care of allocating a unique id for every AET node and passing
   it to the AET node's constructor during instantiation.
   """
   collName = RcfLibUtils.collectionNameFromTypename( typename )

   @wraps( aetNodeInstantiator )
   def createHelper( *args ):
      coll = getattr( publishedAfs.aetNodes, collName )
      key = next( getNewAetId )

      if currentFunctionKeySet is not None:
         # This will apply only in breadth tests.
         currentFunctionKeySet.add( key )

      qt8( 'Alloc AET key ', qv( key ), ' for collName ', qv( collName ) )
      return coll.newMember( key, *args )

   return staticmethod( createHelper )

def aetNodeCleanup( node ):
   if node is None or publishedAfs is None:
      return

   typename = type( node ).__name__
   if not typename.startswith( 'Rcf::Eval::' ):
      return

   collName = RcfLibUtils.collectionNameFromTypename( typename )
   coll = getattr( publishedAfs.aetNodes, collName )

   qt8( 'Free AET key ', qv( node.key ), ' for collName ', qv( collName ) )

   del coll[ node.key ]

class Eval( object ):
   """ Lazy load the Tacc type we use in the AET.
   """
   MetadataType = TacLazyType( "Rcf::Eval::FunctionMetadata" )
   AndExpressionType = TacLazyType( "Rcf::Eval::AndExpression" )
   BlockStmtType = TacLazyType( "Rcf::Eval::BlockStmt" )
   FunctionType = TacLazyType( "Rcf::Eval::Function" )
   FunctionCallExpressionType = TacLazyType( "Rcf::Eval::FunctionCallExpression" )
   IfExpressionType = TacLazyType( "Rcf::Eval::IfExpression" )
   NotExpressionType = TacLazyType( "Rcf::Eval::NotExpression" )
   OrExpressionType = TacLazyType( "Rcf::Eval::OrExpression" )
   ReturnExpressionType = TacLazyType( "Rcf::Eval::ReturnExpression" )
   SequentialExpressionType = TacLazyType( "Rcf::Eval::SequentialExpression" )

   """Wrappers for instantiating the AET nodes.
   Pass the typename instead of the type self (from above), since
   aetNodeInstantiator needs only the name and it is not possible to get
   the name from the TacLazyType (it is a protected member).
   """
   AndExpression = aetNodeInstantiator( "Rcf::Eval::AndExpression" )
   BlockStmt = aetNodeInstantiator( "Rcf::Eval::BlockStmt" )
   Function = aetNodeInstantiator( "Rcf::Eval::Function" )
   FunctionCallExpression = aetNodeInstantiator(
                                    "Rcf::Eval::FunctionCallExpression" )
   IfExpression = aetNodeInstantiator( "Rcf::Eval::IfExpression" )
   NotExpression = aetNodeInstantiator( "Rcf::Eval::NotExpression" )
   OrExpression = aetNodeInstantiator( "Rcf::Eval::OrExpression" )
   ReturnExpression = aetNodeInstantiator( "Rcf::Eval::ReturnExpression" )
   SequentialExpression = aetNodeInstantiator( "Rcf::Eval::SequentialExpression" )

   # More AET nodes will be added here automatically by RcfMetadata.py

   # This is to get rid of pylint errors about using non-existant attributes.
   def __getattr__( self, key ):
      # This should never get called, since we always only access attributes that
      # have been explicitly created using setattr.
      assert False, '__getattr__ should never get invoked'

   class Bgp( object ):
      # More AET nodes will be added here automatically by RcfMetadata.py

      # This is to get rid of pylint errors about using non-existant attributes.
      def __getattr__( self, key ):
         # This should never get called, since we always only access attributes that
         # have been explicitly created using setattr.
         assert False, '__getattr__ should never get invoked'

class Arnet( object ):
   IpGenAddr = TacLazyType( "Arnet::IpGenAddr" )
   IpGenPrefix = TacLazyType( "Arnet::IpGenPrefix" )
   IntfId = TacLazyType( "Arnet::IntfId" )
