# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import RcfAst
import RcfAstVisitor
import RcfMetadata
import RcfImmediateValueHelper

RcfTypeSystem = RcfMetadata.RcfTypeSystem
BT = RcfMetadata.RcfBuiltinTypes
ValueHelper = RcfImmediateValueHelper.RcfImmediateValueHelper
valueStr = ValueHelper.valueStr

class TypeBindingPhase( RcfAstVisitor.Visitor ):
   """ Assign type to attributes and expressions, report errors when needed.

   During this phase, we try to assign types to attributes and expressions.
   This enforces that the program is sound. This phase can emit errors to
   the diag (RcfTypingError).

   In this phase, we assume that all symbols are resolved.

   Attributes:
      diag (RcfDiag): Diagnostic class where to emit errors.
      self.currentFunction (RcfFunction): the current function.

   !Rules (don't change unless discussing with authors first)

      - This implements the pristine typing system, free of shortcomings.
          To implement exceptions due to implementation constraints (say
          AET is not ready yet for a particular operation on a type), use
          the @TypeBindingPhaseOverride.

      - Define the visit methods in the order in which the AST nodes are defined.

   @author: matthieu (rcf-dev)
   """
   def __init__( self, diags ):
      """ Constructor

      Args:
         diags (RcfDiag): the diagnostic report object.
      """
      super( TypeBindingPhase, self ).__init__()
      self.diags = diags
      self.currentFunction = None

   ###########################################################################
   #                   V I S I T    M E T H O D                              #
   ###########################################################################

   def visitRoot( self, root, **kwargs ):
      for function in root.functions:
         self.visit( function )

   def visitFunction( self, function, **kwargs ):
      self.currentFunction = function
      self.visit( function.block )

   def visitBlock( self, block, **kwargs ):
      for stmt in block.stmts:
         self.visit( stmt )

   def visitIfStmt( self, ifStmt, **kwargs ):
      self.visit( ifStmt.condition )
      self.visit( ifStmt.thenBlock )
      if ifStmt.elseBlock:
         self.visit( ifStmt.elseBlock )
      # A conditon's expr must either be Boolean or Trilean
      if not ( ifStmt.condition.evalType == BT.Boolean or
               ifStmt.condition.evalType == BT.Trilean ):
         what = "if condition must evaluate to true, false, or unknown"
         self.diags.typingError( self.currentFunction, ifStmt.condition, what )

   def visitSequentialExpr( self, sequentialExpr, **kwargs ):
      if sequentialExpr.finalBool:
         self.visit( sequentialExpr.finalBool )
      sequentialExpr.evalType = BT.Trilean

   def visitCall( self, call, **kwargs ):
      call.evalType = call.symbol.retType

   def visitExternalRefOp( self, externalRefOp, **kwargs ):
      self.visit( externalRefOp.attribute )
      self.visit( externalRefOp.extRef )
      lhsType = externalRefOp.attribute.evalType
      rhsType = externalRefOp.extRef.evalType
      assert not externalRefOp.isExact or not externalRefOp.isMatchCovered
      if externalRefOp.isExact:
         op = "match_exact"
      elif externalRefOp.isMatchCovered:
         op = "match_covered"
      else:
         op = "match"
      evalType = RcfTypeSystem.resultTypeOf( lhsType, op, rhsType )
      externalRefOp.evalType = evalType
      if evalType is None:
         args = {
            "op": op,
            "attrName": externalRefOp.attribute.fullName(),
            "extType": "%s %s" % ( externalRefOp.extRef.type,
                                     externalRefOp.extRef.name ),
         }
         what = \
            "invalid operation '{op}' between '{attrName}' and '{extType}'".format(
            **args )
         self.diags.typingError( self.currentFunction, externalRefOp, what )
         # /!\ On error, Assume that the result of a match operation is a boolean,
         # so that we don't propagate the error all the way up in the expression:
         # e.g:
         #
         # prefix is 10.0.0.0/24 or med > 0 or med match prefix_list FOO
         #                                     ^~~~~~~~~~~~~~~~~~~~~~~~~
         #                                     assume eval type boolean
         #
         # This will help user focusing on the right error.
         externalRefOp.evalType = BT.Boolean

   def visitExternalRef( self, extRef, **kwargs ):
      # Assume the user did the right thing.
      if 'prefix_list' in extRef.type:
         extRef.evalType = BT.PrefixList
      elif extRef.type == 'as_path_list':
         extRef.evalType = BT.AsPathList
      elif extRef.type == "community_list":
         extRef.evalType = BT.StdCommSetList
      elif extRef.type == "ext_community_list":
         extRef.evalType = BT.ExtCommSetList

   def visitAssign( self, assign, **kwargs ):
      self.visit( assign.attribute )
      self.visit( assign.value )
      fromType = assign.value.evalType
      toType = assign.attribute.evalType

      ########################################################################
      # Check for constness first, then check the rest.                      #
      ########################################################################
      if assign.attribute.symbol.const:
         args = {
            "op": assign.op,
            "attrName": assign.attribute.fullName(),
            "val": valueStr( assign.value ),
         }
         what = "invalid operation '{op}' between {val} and read only " \
                "'{attrName}'".format( **args )
         self.diags.typingError( self.currentFunction, assign, what )
         return

      # Set the promoteToType properly as expected by the AET gen module.
      assign.value.promoteToType = RcfTypeSystem.maybePromote( fromType, toType )

      ########################################################################
      # Check whether the modify operation is supported for this type        #
      ########################################################################
      allowed = RcfTypeSystem.modifyOpAllowed( lhsType=toType,
                                               op=assign.op,
                                               rhsType=fromType )
      if not allowed and assign.value.promoteToType:
         # Try with the promotedType
         allowed = RcfTypeSystem.modifyOpAllowed(
                   lhsType=toType, op=assign.op, rhsType=assign.value.promoteToType )

      if not allowed:
         args = {
            "op": assign.op,
            "rName": valueStr( assign.value ),
            "lName": valueStr( assign.attribute ),
         }
         what = "invalid operation '{op}' between {lName} and {rName}".format(
               **args )
         self.diags.typingError( self.currentFunction, assign, what )
         return

   def visitReturn( self, returnStmt, **kwargs ):
      self.visit( returnStmt.expr )
      expType = returnStmt.expr.evalType
      retType = self.currentFunction.symbol.retType
      returnStmt.expr.promoteToType = RcfTypeSystem.maybePromote(
                                             fromType=expType, toType=retType )
      if expType != retType and not returnStmt.expr.promoteToType:
         what = "return expression must evaluate to true, false, or unknown"
         self.diags.typingError( self.currentFunction, returnStmt, what )

   def visitConstant( self, constant, **kwargs ):
      constant.evalType = ValueHelper.constantTypeToEvalType( constant.type )
      if constant.type == RcfAst.Constant.Type.asPath:
         self.visitAsPathValue( constant )
      valueHelper = ValueHelper( self.currentFunction, self.diags )
      valueHelper.validate( constant )

   def visitAsPathValue( self, asPathValue ):
      for asn in asPathValue.value:
         if asn.get( "attr" ):
            self.visit( asn[ "attr" ] )

   def visitAttribute( self, attribute, **kwargs ):
      attribute.evalType = attribute.symbol.rcfType

   def visitBinOp( self, binOp, **kwargs ):
      self.visit( binOp.lhs )
      self.visit( binOp.rhs )
      lhsType = binOp.lhs.evalType
      rhsType = binOp.rhs.evalType

      promotedType = RcfTypeSystem.maybePromote( fromType=rhsType,
                                                 toType=lhsType )
      evalType = RcfTypeSystem.resultTypeOf( lhsType, binOp.operator, rhsType )
      if evalType is None and promotedType:
         # Try with the promotedType
         evalType = RcfTypeSystem.resultTypeOf(
                                       lhsType, binOp.operator, promotedType )

      binOp.evalType = evalType
      binOp.rhs.promoteToType = promotedType

      if not evalType:
         args = {
            "op": binOp.operator,
            "lName": valueStr( binOp.lhs ),
            "rName": valueStr( binOp.rhs )
         }
         what = "invalid operation '{op}' between {lName} and {rName}".format(
               **args )
         self.diags.typingError( self.currentFunction, binOp, what )
         # /!\ On error, Assume that the result of a binary operation is a boolean,
         # so that we don't propagate the error all the way up in the expression:
         # e.g:
         #
         # prefix is 10.0.0.0/24 or prefix is 20.0.0.0/24 or prefix is 2
         #                                                   ^~~~~~~~~~~
         #                                                 assume eval type boolean
         #
         # This will help user focusing on the right error.
         binOp.evalType = BT.Boolean

   def visitNot( self, notExpr, **kwargs ):
      self.visit( notExpr.expr )
      # Not's expression must either be Boolean or Trilean
      if not ( notExpr.expr.evalType == BT.Boolean or
               notExpr.expr.evalType == BT.Trilean ):
         what = "'not' must be applied to true, false, or unknown expression"
         self.diags.typingError( self.currentFunction, notExpr, what )
      notExpr.evalType = notExpr.expr.evalType
