#!/bin/python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from TypeFuture import TacLazyType

from RcfAbstractScope import AbstractScope
import RcfMetadata
from RcfSymbol import ExternalSymbol

BT = RcfMetadata.RcfBuiltinTypes

class Scope( AbstractScope ):
   """ A scope holds a dictionnary of symbols indexed by their name.

   We define object in a scope, and we resolve them through the scopes.
   A Scope may, or may not have an enclosing scope.

   e.g:
   - global scope doesn't have an enclosing scope.
   - a function scope has the global scope enclosing it.

   Attributes:
      name (str): the name of the scope e.g 'global'.
      symbols (dict): dict of symbols defined at this scope, indexed by
                      their name
      enclosing (Scope, optional): the parent (aka 'enclosing') scope.
   """
   def __init__( self, name, enclosing=None ):
      super( Scope, self ).__init__()
      self.name = name
      self.enclosing = enclosing
      self.symbols = {}

   def define( self, symbol ):
      """ Define a symbol in this scope.

      This function will also set the scope (self) where this symbol is defined.

      Args:
         symbol (Symbol): the symbol to define.
      """
      symbol.definitionScope = self
      self.symbols[ symbol.name ] = symbol

   def resolve( self, reference ):
      """ Find the symbol associated to this name, in this scope, or
      in its enclosing scope (recursively).

      Args:
         reference (str): the name of the symbol to find.

      Returns:
         the symbol (Symbol) if found, None otherwise.
      """
      symbol = self.symbols.get( reference )
      if symbol is None:
         if self.enclosing:
            return self.enclosing.resolve( reference )
      return symbol

   def getScopeName( self ):
      return self.name

   def getEnclosingScope( self ):
      return self.enclosing

class BlockScope( Scope ):
   """ A block scope is created to keep track of Rcf block and their nesting.
   """
   def __init__( self, enclosingScope ):
      super( BlockScope, self ).__init__( name="Local", enclosing=enclosingScope )

class GlobalScope( Scope ):
   """ The Global scope is unique in Rcf and has no enclosing scope.
   """
   def __init__( self ):
      super( GlobalScope, self ).__init__( name="Global", enclosing=None )

class ExternalScope( object ):
   """ Class that we use to lookup whether an external symbol is defined or not.

   Attributes:
      aclListConfig: Tacc entity holding the collections of ACLs defined in Sysdb.
   """
   AclCommunityListType = TacLazyType( "Acl::CommunityListType" )

   def __init__( self, aclListConfig ):
      self.aclListConfig = aclListConfig
      self.dispatchOnType = {
            'prefix_list_v4': self.resolvePrefixListV4,
            'prefix_list_v6': self.resolvePrefixListV6,
            'as_path_list': self.resolveAsPathList,
            'community_list': self.resolveStdCommunityList,
            'ext_community_list': self.resolveExtCommunityList,
      }

   def define( self, symbol ):
      raise TypeError( "Rcf does not define external symbols" )

   def resolve( self, extRef ):
      """ Check whether a given external ref is defined in Sysdb or not.

      Args:
         externalRef (Ast.ExternalRef): the external reference to check.

      Returns:
         true if extRef is defined already, false otherwise.
      """
      if not self.aclListConfig:
         return False
      return self.dispatchOnType[ extRef.type ]( extRef )

   def resolvePrefixListV4( self, extRef ):
      utf8Name = extRef.name.encode( 'utf8' )
      assert self.aclListConfig
      if not utf8Name in self.aclListConfig.prefixList:
         return None
      typeOptions = [ BT.PrefixList ]
      return ExternalSymbol( name=utf8Name, rcfTypeOptions=typeOptions, node=extRef )

   def resolvePrefixListV6( self, extRef ):
      utf8Name = extRef.name.encode( 'utf8' )
      assert self.aclListConfig
      if not utf8Name in self.aclListConfig.ipv6PrefixList:
         return None
      typeOptions = [ BT.PrefixList ]
      return ExternalSymbol( name=utf8Name, rcfTypeOptions=typeOptions, node=extRef )

   def resolveAsPathList( self, extRef ):
      utf8Name = extRef.name.encode( 'utf8' )
      assert self.aclListConfig
      if not utf8Name in self.aclListConfig.pathList:
         return None
      typeOptions = [ BT.AsPathList ]
      return ExternalSymbol( name=utf8Name, rcfTypeOptions=typeOptions, node=extRef )

   def resolveAnyCommunityList( self, extRef, commListEntryTypes ):
      """ Check whether a given external ref exists in AclList Config and contains
      entries for the commList types we are about.

      Args:
         extRef (Ast.ExternalRef): the external reference to check.
         commListEntryTypes (python set): RCF builtin extref types we care about

      Returns:
         ExternalSymbol object if the requested commList was found in AclListConfig
         with entries of the type(s) we requested, None otherwise.
      """
      utf8Name = extRef.name.encode( 'utf8' )
      assert self.aclListConfig
      if not utf8Name in self.aclListConfig.communityList:
         # no name found, not defined
         return None
      commList = self.aclListConfig.communityList[ utf8Name ]
      typeOptions = self.communityListRcfTypes( commList )
      if not typeOptions & commListEntryTypes:
         # we found the name, but there is no entries in this
         # community list for the comm list types we care about.
         return  None
      return ExternalSymbol( name=utf8Name,
                             rcfTypeOptions=typeOptions,
                             node=extRef )

   def resolveStdCommunityList( self, extRef ):
      stdCommListEntryTypes = {
         BT.StdCommSetList,
         BT.StdRegexCommSetList
      }
      return self.resolveAnyCommunityList( extRef, stdCommListEntryTypes )

   def resolveExtCommunityList( self, extRef ):
      extCommListEntryTypes = {
         BT.ExtCommSetList,
         BT.ExtRegexCommSetList
      }
      return self.resolveAnyCommunityList( extRef, extCommListEntryTypes )

   def communityListRcfTypes( self, communityList ):
      types = set()
      for entry in communityList.communityListEntry.itervalues():
         if entry.listType == self.AclCommunityListType.communityStandard:
            types.add( BT.StdCommSetList )
         elif entry.listType == self.AclCommunityListType.communityExpanded:
            types.add( BT.StdRegexCommSetList )
         elif entry.listType == self.AclCommunityListType.extCommunityStandard:
            types.add( BT.ExtCommSetList )
         elif entry.listType == self.AclCommunityListType.extCommunityExpanded:
            types.add( BT.ExtRegexCommSetList )
         else:
            continue # no support
      return types

