# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from functools import partial

from RcfAbstractScope import AbstractScope

class AbstractType( object ):
   """ An Abstract type symply has a name.
   """
   def getName( self ):
      raise NotImplementedError

class Symbol( object ):
   """ A symbol in Rcf has a name and an associated type.

   Each symbol points to the node in the AST where it is defined.

   Note that a Symbol is not a type.

   Attributes:
      name (str): name of the symbol
      rcfType (RcfType, NIY): rcf type of this symbol
      node (RcfAST Node): AST Node defining this symbol
      const (bool): whether this symbol is const or not
      aetType (Tacc): the AET type for this symbol
      routeMapFeature (dict): dict of value indexed by RouteMap feature
                              (attribute name of Routing::Policy::RouteMapFeatures)
   """
   def __init__( self, name, rcfType=None, node=None, const=False,
                 aetType=None, routeMapFeatures=None ):
      self.name = name
      self.rcfType = rcfType
      self.node = node
      self.const = const
      self.aetType = aetType
      self.routeMapFeatures = routeMapFeatures

   def updateRouteMapFeatures( self, routeMapFeature ):
      if self.routeMapFeatures:
         for feature, value in self.routeMapFeatures.iteritems():
            setattr( routeMapFeature, feature, value )

ConstSymbol = partial( Symbol, const=True )

class ScopeSymbol( Symbol, AbstractScope ):
   """ Super symbols, that can act as a scope itself.

   e.g: function, struct etc..

   * Why not inherit from both Symbol and Scope?

     To keep multiple inheritance sane, we should only inherit from
     one concrete class. Following this rule helps reasoning
     more clearly about the class hierarchy, at the cost of minimal
     code duplication:

     ScopedSymbol is a Symbol that can "act-as" a Scope.

   * Why not inherit from Scope instead of Symbol?

     Functionnaly, we want to treat this more like a super symbol
     rather that a super scope. A struct, for instance, is a super
     symbol, but a dumb down scope.
   """

   def __init__( self, name, rcfType=None, node=None, enclosingScope=None ):
      Symbol.__init__( self, name=name, rcfType=rcfType, node=node )
      self.enclosingScope = enclosingScope
      self.members = {}

   def define( self, symbol ):
      """ Define a symbol within this symbol.

      Args:
         symbol (RcfSymbol.Symbol): the symbol we want to include.
      """
      symbol.definitionScope = self
      self.members[ symbol.name ] = symbol

   def getScopeName( self ):
      return self.name

   def getEnclosingScope( self ):
      return self.enclosingScope

   def resolve( self, reference ):
      raise NotImplementedError

class ExternalSymbol( Symbol ):
   """ Special symbol for external references (e.g: asPathList, prefixList etc...)

   This symbol is special in the sense that a given name may carry many types.
   For instance,  an "community" can either be standard, or regex.

   In the type binding phase, we leverage this extra inforrmation to select
   one type in this list, depending on its use case.

   @Attributes:
      rcfTypeOptions: Set of RcfBuiltin types. When the list only has one
                      type, we automatically set the symbol's rcfType to
                      this one type, otherwise, the rcfType is left to None.
   """

   def __init__( self, name, rcfTypeOptions, node ):
      super( ExternalSymbol, self ).__init__( name=name )
      if len( rcfTypeOptions ) == 1:
         self.rcfType = next( iter( rcfTypeOptions ) )
         self.rcfTypeOptions = None
      else:
         self.rcfTypeOptions = rcfTypeOptions


class Function( ScopeSymbol ):
   """ Function are a special kind of ScopeSymbol, with a name and
   a return type.
   """
   def __init__( self, name, rcfType, retType, node, enclosingScope ):
      super( Function, self ).__init__( name=name,
                                        rcfType=rcfType,
                                        node=node,
                                        enclosingScope=enclosingScope )
      self.retType = retType

   def resolve( self, reference ):
      """ Resolve a reference (string) in this scope, and above.

      Args:
         reference (str): a string from the rcf code like "foo".

      Retunrs:
         The symbol associated to this reference according to the scope
         search, or None.
      """
      symbol = self.members.get( reference )
      if symbol is None:
         if self.enclosingScope:
            return self.enclosingScope.resolve( reference )
      return symbol

class Type( Symbol, AbstractType ):
   """ Simple Rcf Type for Simple attributes.

   Note that a Type is a Symbol.

   Attributes:
      aetOpTypes (dict): dict of AET type indexed by operation.
   """
   def __init__( self, name, aetOpTypes=None ):
      self.name = name
      self.aetOpTypes = aetOpTypes

   def getName( self ):
      return self.name

class Enum( Type ):
   """ An Rcf Type used to represent Enums like Origin.

   Attributes:
      valueDict (dict): dict of enum name to enum value.
   """
   def __init__( self, name, valueDict, aetOpTypes=None ):
      super( Enum, self ).__init__( name=name, aetOpTypes=aetOpTypes )
      self.valueDict = valueDict
