#!/bin/python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The RCF compiler is data driven and the RcfMetadata.yaml file provides all the
metadata that the compiler uses to figure out the following:
* The set of builtin types in the language
* The set of BGP and other protocol symbols that are available for use in RCF
* The RCF language type system rules

When adding a new BGP or other protocol attribute (common) or when adding a new
RCF builtin type (less common), the information needs to be added to that yaml
file and this file rarely needs to be modified.

Please contact satishm@ or rcf-dev@ if there are any questions.
"""

from collections import defaultdict
import yaml
import RcfAst
from RcfSymbol import (
      ConstSymbol,
      Enum,
      Symbol,
      Type,
)
import RcfTypeFuture
import Toggles.RcfLibToggleLib  # pylint: disable=unused-import

# Some common helpers
def getRcfType( name ):
   return getattr( RcfBuiltinTypes, name, None )

def registerAetType( name ):
   # name could be <typename> or Bgp.<typename>
   if name is None:
      return
   if name.startswith( 'Bgp' ):
      cls = RcfTypeFuture.Eval.Bgp
      attrName = name.split( '.' )[ 1 ]
      aetTypename = 'Rcf::Eval::Bgp::%s' % attrName
   else:
      cls = RcfTypeFuture.Eval
      attrName = name
      aetTypename = 'Rcf::Eval::%s' % name

   lazyType = RcfTypeFuture.TacLazyType( aetTypename )
   setattr( cls, '%sType' % attrName, lazyType )
   setattr( cls, attrName, RcfTypeFuture.aetNodeInstantiator( aetTypename ) )

def getAetType( name ):
   if name is None:
      return None
   if name.startswith( 'Bgp' ):
      cls = RcfTypeFuture.Eval.Bgp
      attrName = name.split( '.' )[ 1 ]
   else:
      cls = RcfTypeFuture.Eval
      attrName = name

   aetType = getattr( cls, attrName, None )
   if not aetType:
      registerAetType( name )

   return getattr( cls, attrName )

def toggleEnabled( toggleName ):
   if toggleName is None:
      return True
   toggleFn = getattr( Toggles.RcfLibToggleLib, 'toggle%sEnabled' % toggleName )
   return toggleFn()

class RcfKeywords( object ):
   keywords = set()

   @staticmethod
   def addKeyword( keyword ):
      RcfKeywords.keywords.add( keyword )

   @staticmethod
   def isKeyword( keyword ):
      return keyword in RcfKeywords.keywords

class RcfBuiltinTypes( object ):
   """This class holds all the builtin types (like Int, Boolean etc) in RCF.
   """
   # Fields here will be populated by the metadata processor, based on the
   # contents of RcfMetadata.yaml.

   enumTypes = []
   enumStringToTypename = {}

   # This is to get rid of pylint errors about using non-existant attributes.
   def __getattr__( self, key ):
      # This should never get called, since we always only access attributes that
      # have been explicitly created using setattr.
      assert False, '__getattr__ should never get invoked'

   @staticmethod
   def addBuiltinType( name, aetOpTypes=None ):
      internalName = '<%s>' % name
      value = Type( internalName, aetOpTypes=aetOpTypes )
      setattr( RcfBuiltinTypes, name, value )

   @staticmethod
   def addBuiltinEnumType( name, valueDict, aetOpTypes ):
      internalName = '<%s>' % name
      value = Enum( internalName, valueDict=valueDict, aetOpTypes=aetOpTypes )
      setattr( RcfBuiltinTypes, name, value )
      setattr( RcfAst.Constant.Type, name, name )
      RcfBuiltinTypes.enumTypes.append( name )

      for enumKeyword in valueDict:
         RcfKeywords.addKeyword( enumKeyword )
         RcfBuiltinTypes.enumStringToTypename[ enumKeyword ] = name

class RcfBuiltinSymbols( object ):
   """This class holds all the builtin symbols (like med, prefix etc) in RCF.
   """
   builtInAttributes = []

   @staticmethod
   def registerSymbol( name, const, rcfType, aetType, routeMapFeatures=None ):
      SymbolType = ConstSymbol if const else Symbol
      symbol = SymbolType( name, rcfType=getRcfType( rcfType ),
                           aetType=getAetType( aetType ),
                           routeMapFeatures=routeMapFeatures )
      RcfBuiltinSymbols.builtInAttributes.append( symbol )

      for symbol in name.split( '.' ):
         RcfKeywords.addKeyword( symbol )

class RcfTypeSystem( object ):
   """This class implements the RCF type system.
   """
   # allowedImplicitConversions[ ( fromType, toType ) ] will be set to True, if
   # fromType can be implicitly converted to toType.
   allowedImplicitConversions = defaultdict( lambda: False )

   # allowedModifyOps[ ( type, op ) ] will be set to True, if 'op' is allowed
   # for 'type'
   allowedModifyOps = defaultdict( lambda: False )

   # resultTypes[ ( lhsType, op, rhsType ) ] will be set to the appropriate
   # resultType, if '<lhsType> <op> <rhsType>' is allowed.
   resultTypes = defaultdict( lambda: None )

   # aetTypes[ ( lhsType, op, rhsType ) ] will be set to the appropriate tuple of aet
   # node types if the aet types are specified for a particular tuple of stmt types.
   # Additionally, a list of ctor args (or None) can be set for the rhsAetType ctor.
   aetTypes = defaultdict( lambda: None )

   @staticmethod
   def getAstNodeType( astNode ):
      return astNode.promoteToType if astNode.promoteToType else astNode.evalType

   @staticmethod
   def addAllowedImplicitConversion( fromTypeName, toTypeName ):
      fromType = getRcfType( fromTypeName )
      toType = getRcfType( toTypeName )
      RcfTypeSystem.allowedImplicitConversions[ ( fromType, toType ) ] = True

   @staticmethod
   def implicitlyConvertible( fromType, toType ):
      """Returns a bool indicating whether fromType can be implicitly converted
      to toType. fromType and toType are types available inside RcfBuiltinTypes.
      """
      return RcfTypeSystem.allowedImplicitConversions[ ( fromType, toType ) ]

   @staticmethod
   def addAllowedModifyOp( lhsTypename, op, rhsTypename ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      RcfTypeSystem.allowedModifyOps[ ( lhsType, op, rhsType ) ] = True

   @staticmethod
   def modifyOpAllowed( lhsType, op, rhsType ):
      """Tells whether the given assign operation is allowed on the given type.
      """
      return RcfTypeSystem.allowedModifyOps[ ( lhsType, op, rhsType ) ]

   @staticmethod
   def maybePromote( fromType, toType ):
      """Compatibility exists if fromType is same as toType or if fromType
      is implicitly convertible to toType.
      Return: toType, if implicit conversion exists
      """
      return toType if RcfTypeSystem.implicitlyConvertible( fromType, toType ) \
             else None

   @staticmethod
   def addAllowedConditionOp( lhsTypename, op, rhsTypename ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      RcfTypeSystem.resultTypes[ ( lhsType, op, rhsType ) ] = RcfBuiltinTypes.Boolean

   @staticmethod
   def resultTypeOf( lhsType, op, rhsType ):
      """ Get the result type of a binary operation.
      Returns the result type of this operation over these operands.
      Returns None if this operation is not allowed on these operands.
      """
      return RcfTypeSystem.resultTypes[ ( lhsType, op, rhsType ) ]

   @staticmethod
   def addAetTypesForStmt( lhsTypename, op, rhsTypename, lhsAetTypename,
                           opAetTypename, rhsAetTypename, rhsAetCtorArgs ):
      lhsType = getRcfType( lhsTypename )
      rhsType = getRcfType( rhsTypename )
      lhsAetType = getAetType( lhsAetTypename )
      opAetType = getAetType( opAetTypename )
      rhsAetType = getAetType( rhsAetTypename )
      RcfTypeSystem.aetTypes[ ( lhsType, op, rhsType ) ] = (
            lhsAetType, opAetType, rhsAetType, rhsAetCtorArgs )

   @staticmethod
   def aetTypesOf( lhsType, op, rhsType ):
      """ Get the aet types for a modification or condition (lhs, op, rhs).
      Return: (lhsAetType, opAetType, rhsAetType) if aet types are specified for stmt
      """
      return RcfTypeSystem.aetTypes[ ( lhsType, op, rhsType ) ]

class RcfMetadataProcessor( object ):
   def __init__( self ):
      with open( '/usr/share/Rcf/RcfMetadata.yaml' ) as f:
         self.metadata = yaml.load( f.read() )
      self.allBuiltinTypes = self.metadata[ 'RcfTypes' ]
      self.allBuiltinAttributes = self.metadata[ 'Attributes' ]
      self.allTypingRules = self.metadata[ 'TypingRules' ]
      self.otherAetTypes = self.metadata[ 'OtherAetTypes' ]
      self.otherKeywords = self.metadata[ 'OtherKeywords' ]

      self.processAllBuiltinTypes()
      self.processAllBuiltinAttributes()
      self.processAllTypingRules()
      self.processAllAetTypes()
      self.processAllKeywords()

   def processOperatorsDict( self, operatorsMetadata ):
      aetOpTypes = {}
      for opName, aetTypeName in operatorsMetadata.iteritems():
         registerAetType( aetTypeName )
         aetOpTypes[ opName ] = getAetType( aetTypeName )

         if opName[ 0 ].isalpha():
            RcfKeywords.addKeyword( opName )

      return aetOpTypes

   def processOperatorsList( self, operatorsMetadata ):
      aetOpTypes = {}
      for typeAndOp in operatorsMetadata:
         typ = typeAndOp.split( '.' )[ 0 ]
         opName = typeAndOp.split( '.' )[ 1 ]
         aetOpTypes[ opName ] = getRcfType( typ ).aetOpTypes[ opName ]
      return aetOpTypes

   def processOperatorToAetType( self, operatorsMetadata ):
      if isinstance( operatorsMetadata, dict ):
         return self.processOperatorsDict( operatorsMetadata )
      else:
         assert isinstance( operatorsMetadata, list )
         return self.processOperatorsList( operatorsMetadata )

   def processBuiltinType( self, typename, metadata ):
      aetOpTypes = None

      operatorsMetadata = metadata.get( 'OperatorToAetType' )
      if operatorsMetadata:
         aetOpTypes = self.processOperatorToAetType( operatorsMetadata )

      RcfBuiltinTypes.addBuiltinType( name=typename, aetOpTypes=aetOpTypes )

   def processAllEnumTypes( self, metadata ):
      aetOpTypes = self.processOperatorToAetType( metadata[ 'OperatorToAetType' ] )
      for typename, valueDict in metadata[ 'EnumTypes' ].iteritems():
         RcfBuiltinTypes.addBuiltinEnumType(
               name=typename, valueDict=valueDict, aetOpTypes=aetOpTypes )

   def processAllBuiltinTypes( self ):
      for entry in self.allBuiltinTypes:
         if isinstance( entry, dict ):
            assert len( entry ) == 1
            typename = entry.keys()[ 0 ]
            metadata = entry[ typename ]
            if typename == 'Enum':
               self.processAllEnumTypes( metadata )
            else:
               self.processBuiltinType( typename, metadata )
         else:
            RcfBuiltinTypes.addBuiltinType( name=entry )

   def fullname( self, namespace, attrname ):
      if namespace:
         return '%s.%s' % ( namespace, attrname )
      else:
         return attrname

   def processAttribute( self, attrname, metadata, namespace=None,
                         inheritedRouteMapFeatures=None ):
      effectiveRouteMapFeatures = inheritedRouteMapFeatures or {}
      fullname = self.fullname( namespace, attrname )

      aetType = metadata.get( 'aetType' )
      if aetType is not None:
         # aetType would be None for namespaces like 'igp' which exist only to
         # define attrbutes like igp.tag.
         registerAetType( aetType )
         effectiveRouteMapFeatures.update(
                           metadata.get( 'routeMapFeatures' ) or {} )
         RcfBuiltinSymbols.registerSymbol(
               fullname, const=metadata.get( 'const' ),
               rcfType=metadata[ 'rcfType' ], aetType=aetType,
               routeMapFeatures=effectiveRouteMapFeatures )

      subattrs = metadata.get( 'Subattributes' )
      if subattrs is not None:
         self.processAttributes(
                     subattrs, namespace=fullname,
                     inheritedRouteMapFeatures=effectiveRouteMapFeatures )

   def processAttributes( self, metadata, namespace=None,
                          inheritedRouteMapFeatures=None ):
      for name, value in metadata.iteritems():
         if toggleEnabled( value.get( 'toggle' ) ):
            self.processAttribute(
                name, value, namespace=namespace,
                inheritedRouteMapFeatures=inheritedRouteMapFeatures )

   def processAllBuiltinAttributes( self ):
      self.processAttributes( self.allBuiltinAttributes )

   def processImplicitConversions( self ):
      for key, values in self.allTypingRules[ 'ImplicitConversions' ].iteritems():
         if not isinstance( values, list ):
            values = [ values ]
         for value in values:
            RcfTypeSystem.addAllowedImplicitConversion( fromTypeName=key,
                                                        toTypeName=value )

   def processAllowedOperations( self, sectionName, updateFn ):
      for entry in self.allTypingRules[ sectionName ]:
         if isinstance( entry, dict ):
            key = entry.keys()[ 0 ]
            toggleName = entry[ key ].get( 'toggle' )
            lhsAetTypename = entry[ key ].get( 'lhsAetType' )
            opAetTypename = entry[ key ].get( 'opAetType' )
            rhsAetTypename = entry[ key ].get( 'rhsAetType' )
            rhsAetCtorArgs = entry[ key ].get( 'rhsAetCtorArgs' )
         else:
            key = entry
            toggleName = None
            lhsAetTypename = None
            opAetTypename = None
            rhsAetTypename = None
            rhsAetCtorArgs = None

         if not toggleEnabled( toggleName ):
            continue

         lhsTypename, op, rhsTypename = key.split()
         if lhsTypename == 'Enum':
            for enumTypename in RcfBuiltinTypes.enumTypes:
               updateFn( enumTypename, op, enumTypename )
         else:
            updateFn( lhsTypename, op, rhsTypename )

         if lhsAetTypename and opAetTypename:
            RcfTypeSystem.addAetTypesForStmt( lhsTypename,
                                              op,
                                              rhsTypename,
                                              lhsAetTypename,
                                              opAetTypename,
                                              rhsAetTypename,
                                              rhsAetCtorArgs )

         if op[ 0 ].isalpha():
            RcfKeywords.addKeyword( op )

   def processAllowedConditionOperations( self ):
      self.processAllowedOperations( 'AllowedConditionOperations',
                                     RcfTypeSystem.addAllowedConditionOp )

   def processAllowedModifyOperations( self ):
      self.processAllowedOperations( 'AllowedModifyOperations',
                                     RcfTypeSystem.addAllowedModifyOp )

   def processAllTypingRules( self ):
      self.processImplicitConversions()
      self.processAllowedConditionOperations()
      self.processAllowedModifyOperations()

   def processAllAetTypes( self ):
      for name in self.otherAetTypes:
         registerAetType( name )

   def processAllKeywords( self ):
      for keyword in self.otherKeywords:
         RcfKeywords.addKeyword( keyword )

metadataProcessor = RcfMetadataProcessor()

# TODO:
# * Need a way to define toggles for a specific combination of (attribute, operation)
#    * eg. block 'next_hop = ...' based on the RcfNextHopSet toggle, without blocking
#      it for other IpAddress based types.
# * Rename RcfBuitinType.Prefix to RcfBuitinType.IpPrefix.
