# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import antlr4

import Tracing
import QuickTrace
from RcfAstGen import genAst
from RcfDiag import RcfDiag
from RcfLexer import (
      InputStream,
      RcfLexer,
   )
from RcfParser import RcfParser
import RcfSymbolTableGen

t0 = Tracing.t0
qv = QuickTrace.Var
qt8 = QuickTrace.trace8


class RcfLintRequest( object ):
   """ A lint request.
   """
   def __init__( self, rcfCodeString, strictMode ):
      self.rcfCodeString = rcfCodeString
      self.strictMode = strictMode

class RcfLintResult( object ):
   """Output from the RCF compiler

   Arguments
   ---------
   rcfLintRequest : RcfLintRequest
      The lint request to which this is a response
   success : bool
      Whether the linting was successful or not
   errorList : (list of str) or (None)
   warningList : (list of str) or (None)
      None if compilation was successful. List of error strings otherwise.
   """
   def __init__( self, rcfLintRequest, success, errorList=None, warningList=None ):
      self.rcfLintRequest = rcfLintRequest
      self.success = success
      self.errorList = errorList
      self.warningList = warningList

class RcfLinter( object ):
   """ Rcf Linter instance

   aclConfig : Acl::AclListConfig
      This contains the list of all configured IPv4 and IPv6 prefix lists,
      community lists, etc.
   """
   def __init__( self, aclConfig ):
      self.aclConfig = aclConfig

   @staticmethod
   def failureResult( request, diag ):
      errorList = diag.allErrorStrList() if diag.hasErrors() else None
      warningList = diag.allWarningStrList() if diag.hasWarnings() else None
      return RcfLintResult( request,
                            success=False,
                            errorList=errorList,
                            warningList=warningList )

   def lint( self, request ):
      """ Lint the given request.

      Args:
         request: RcfLintRequest: the request to lint.

      Returns:
         RcfLintResult
      """
      diag = RcfDiag( strict=request.strictMode )

      # Step 1: Got token stream
      lexer = RcfLexer( InputStream( request.rcfCodeString ) )
      lexer.removeErrorListeners()
      lexer.addErrorListener( diag )
      tokenStream = antlr4.CommonTokenStream( lexer )

      # Step 2: Perform parsing
      parser = RcfParser( tokenStream )
      parser.removeErrorListeners()
      parser.addErrorListener( diag )
      parseTree = parser.rcf()

      if diag.hasErrors():
         return self.failureResult( request, diag )

      # Step 3: Build AST
      rcfAst = genAst( parseTree, diag )

      # Step 4: Run the semantic analysis
      RcfSymbolTableGen.runSemanticAnalysis( rcfAst, diag, self.aclConfig )

      if diag.hasErrors():
         return self.failureResult( request, diag )
      elif diag.hasWarnings():
         return RcfLintResult( request, success=True,
                               warningList=diag.allWarningStrList() )
      else:
         return RcfLintResult( request, success=True )
