# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from collections import defaultdict

class Callgraph( object ):
   """ Directed graph of callers and callees

   Each vertex in this graph is a function name, and arcs (directed) represent
   the caller -> callee relationship.
   """
   def __init__( self, data=None ):
      self._calleesByCaller = data or defaultdict( set )

   def copy( self ):
      return Callgraph( data=self._calleesByCaller.copy() )

   def add( self, caller, callee ):
      self._calleesByCaller[ caller ].add( callee )

   def delete( self, caller ):
      del self._calleesByCaller[ caller ]

   def callees( self, caller ):
      # here we are returning a dict of { caller, [callees] }
      return self._calleesByCaller.get( caller, {} )

   def callers( self ):
      return self._calleesByCaller.iterkeys()

def findStronglyConnected( callgraph ):
   """ Tarjan's algorithm for finding strongly connected components. A strongly
   connected component of a graph is one where every vertex is reachable from every
   other vertex. The algorithm runs in linear time.

   Attributes:
      callgraph (default dict): dictionary of callers:[callees] generated from the
      symbol table
   """
   result = []
   index = {}
   # represents the smallest index of any node known to be
   # reachable from v through v's DFS subtree
   lowlink = {}
   stack = []
   indexCounter = [ 0 ] # numbers each node ( funcName ) uniquely

   def _stronglyConnected( funcName ):
      """ Sets the unique index for the func name to the smallest unused index
      """
      index[ funcName ] = indexCounter[ 0 ]
      lowlink[ funcName ] = indexCounter[ 0 ]
      indexCounter[ 0 ] += 1
      stack.append( funcName )
      successors = callgraph.callees( funcName )
      for successor in successors:
         if successor not in index:
            # Successor has not yet been visited; recurse on it
            _stronglyConnected( successor )
            lowlink[ funcName ] = min( lowlink[ funcName ], lowlink[ successor ] )
         elif successor in stack:
            # the successor is in the stack and therefore
            # in the current strongly connected component
            lowlink[ funcName ] = min( lowlink[ funcName ], index[ successor ] )

      # if funcName is a rootNode, pop the stack and generate scc
      if lowlink[ funcName ] == index[ funcName ]:
         connected_component = []
         while True:
            successor = stack.pop()
            connected_component.append( successor )
            if successor == funcName:
               break
         # store the result
         result.append( connected_component )
   for func in callgraph.callers():
      if func not in index:
         _stronglyConnected( func )
   return result # returns a set of lists that represent srongly connected components

def removeFunc( callgraph, target ):
   """ Remove target function from the callgraph
   """
   if callgraph.callees( target ): # check that
      callgraph.delete( target )
      for nbrFunc in callgraph.callers():
         callgraph.callees( nbrFunc ).discard( target )

def subgraph( callgraph, vertices ):
   """ Get the subgraph of the function callgraph induced
    by the set of function names (vert)
   """
   sub = Callgraph()
   for v in vertices:
      sharedSet = callgraph.callees( v ) & vertices
      for call in sharedSet:
         sub.add( v, call )
   return sub

# disable msg to accomidate example graph
# pylint: disable-msg=W1401
def findAllCycles( callgraph ):
   """ Yield each elementary cycle within the callgraph. An elementary
   cycle is defined as a cycle in a graph who's verticies ( and by extention,
   its edges ) are used at most once in the cycle. With exception is the last vertex
   which signifies a cycle. By contrast a simple cycle is defined as a cycle where
   no edge appears nore than once but verticies may be repreated.

   For example, take the graph:

   A -- B
   |  / | \
   | /  |  C
   F -- D /
    \  /
     E

   One elementary cycle for this would be A > B > C > D > E > F > A
   A simple but not elementary cycle would be: A > B > C > D > E > F > B > D > F > A
   """
   def _unblock( thisFuncName, blocked, B ):
      toUnblock = set( [ thisFuncName ] )
      while toUnblock:
         currFunc = toUnblock.pop()
         if currFunc in blocked:
            blocked.remove( currFunc )
            toUnblock.update( B[ currFunc ] )
            B[ currFunc ].clear()

   components = findStronglyConnected( callgraph )
   cyclesFound = []
   # pylint: disable-msg=R1702
   while components:
      component = components.pop() # currently investigated strongly connected comp
      startFunc = component.pop()
      path = [ startFunc ]
      blocked = set()
      closed = set()
      blocked.add( startFunc )
      B = defaultdict( set )
      stack = [ ( startFunc, list( callgraph.callees( startFunc ) ) ) ]
      while stack: # walk spanning tree to find cycles
         visitingFunc, calledFuncs = stack[ -1 ] # using last-in
         if calledFuncs:
            nextFunc = calledFuncs.pop()
            if nextFunc == startFunc: # found a cycle
               # append a copy of the path instead of reference
               cyclesFound.append( list( path ) )
               closed.update( path )
            elif nextFunc not in blocked:
               path.append( nextFunc )
               stack.append( ( nextFunc, list( callgraph.callees( nextFunc ) ) ) )
               closed.discard( nextFunc )
               blocked.add( nextFunc )
               continue
         if not calledFuncs: # no callees
            if visitingFunc in closed:
               _unblock( visitingFunc, blocked, B )
            else:
               for nbrFunc in callgraph.callees( visitingFunc ):
                  if visitingFunc not in B[ nbrFunc ]:
                     B[ nbrFunc ].add( visitingFunc )
            stack.pop()
            path.pop()
      removeFunc( callgraph, startFunc )
      # by this time, component has start func removed, component may be empty
      H = subgraph( callgraph, set( component ) )
      components.extend( findStronglyConnected( H ) )
   return cyclesFound
