# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import RcfAst
from RcfMetadata import (
      RcfBuiltinTypes,
      RcfTypeSystem,
      getRcfType,
   )
import RcfTypeFuture as Rcf

class RcfImmediateValueValidationHelper( object ):
   """ Validate that a given constant (value) is in the expected range of
   acceptable values.
   """
   def __init__( self, currentFunction, diags ):
      self.currentFunction = currentFunction
      self.diags = diags

   def validate( self, constant ):
      dispatchMap = {
         RcfAst.Constant.Type.prefix: self._validatePrefix,
         RcfAst.Constant.Type.ipAddress: self._validateIpAddress,
         RcfAst.Constant.Type.asPath: self._validateAsPath,
      }

      # If there is no validation function defined for this type, validation is
      # not required.
      validator = dispatchMap.get( constant.type )
      if validator:
         validator( constant )

   def _validatePrefix( self, prefixValue ):
      encodedValue = prefixValue.value.encode( 'utf8' )
      try:
         Rcf.Arnet.IpGenPrefix( encodedValue )
      except ( ValueError, IndexError ):
         fmt = { 'p': encodedValue }
         msg = "invalid IP prefix '{p}'"
         self.diags.typingError( self.currentFunction, prefixValue,
                                 msg.format( **fmt ) )

   def _validateIpAddress( self, value ):
      encodedValue = value.value.encode( 'utf8' )
      try:
         Rcf.Arnet.IpGenAddr( encodedValue )
      except ( ValueError, IndexError ):
         fmt = { 'nh': encodedValue }
         msg = "invalid IP address '{nh}'"
         self.diags.typingError( self.currentFunction, value,
                                 msg.format( **fmt ) )

   def _validateAsPath( self, asPathValue ):
      for asn in asPathValue.value:
         if not asn.get( 'attr' ):
            continue

         asnVal = asn.values()[ 0 ]
         typ = RcfTypeSystem.getAstNodeType( asnVal )
         if typ not in [ RcfBuiltinTypes.Int, RcfBuiltinTypes.AsNumber ]:
            fmt = { 'attr': asnVal.name }
            msg = "invalid AS number '{attr}'"
            self.diags.typingError( self.currentFunction, asPathValue,
                                    msg.format( **fmt ) )

class RcfImmediateValueToStringHelper( object ):
   @staticmethod
   def asDotStr( value ):
      high = value >> 16
      low = 0x0000FFFF & value
      return '%d.%d' % ( high, low )

   @staticmethod
   def asPathStr( path ):
      """Returns the string representation of an AsPath value.
      Arguments:
         path (dictionary <AS Path element type : element value>): AS path to print
      """
      strList = []
      for element in path:
         elemType, elemVal = element.popitem()
         if elemType == 'as_dot':
            strList.append(
                  RcfImmediateValueToStringHelper.asDotStr( elemVal.value ) )
         elif elemType == 'int':
            strList.append( str( elemVal.value ) )
         elif elemType == 'attr':
            strList.append( str( elemVal.name ) )

      return ' '.join( strList )

   @staticmethod
   def valueStr( valueNode ):
      """Returns the string representation of a value (const or attribute).
      Arguments:
         valueNode (Rcf.Constant or Rcf.Attribute): the value node.
      """
      isExpectedInstance = isinstance( valueNode, (
                                       RcfAst.Attribute,
                                       RcfAst.ExternalRef,
                                       RcfAst.Constant ) )
      assert isExpectedInstance, "Unexpected value type"
      if isinstance( valueNode, RcfAst.Attribute ):
         val = valueNode.fullName()
      elif isinstance( valueNode, RcfAst.ExternalRef ):
         val = '%s %s' % ( valueNode.type, valueNode.name )
      elif valueNode.type is RcfAst.Constant.Type.asDot:
         val = RcfImmediateValueToStringHelper.asDotStr( valueNode.value )
      elif valueNode.type is RcfAst.Constant.Type.asPath:
         val = RcfImmediateValueToStringHelper.asPathStr( valueNode.value )
      elif valueNode.type is RcfAst.Constant.Type.empty:
         val = 'empty'
      elif valueNode.type is RcfAst.Constant.Type.none:
         val = 'none'
      else:
         val = str( valueNode.value )

      return "'%s'" % val

class RcfImmediateValueHelper( object ):
   def __init__( self, currentFunction, diags ):
      self.currentFunction = currentFunction
      self.diags = diags
      self.validationHelper = RcfImmediateValueValidationHelper(
                                                self.currentFunction, self.diags )

   def validate( self, constant ):
      self.validationHelper.validate( constant )

   @staticmethod
   def getEnumType( enumString ):
      typename = RcfBuiltinTypes.enumStringToTypename.get( enumString )
      return getattr( RcfAst.Constant.Type, typename ) if typename is not None \
                                                       else None

   @staticmethod
   def getInterfaceType( value ):
      """Returns the interface constant type if the input conforms to an
         Arnet::IntfId, otherwise returns none.
      Arguments:
         value (str): the string representation of interface, short or long
      """
      value = str( value )
      try:
         Rcf.Arnet.IntfId( value )
         return RcfAst.Constant.Type.interface
      except ( ValueError, IndexError ):
         pass
      try:
         Rcf.Arnet.IntfId.fromShortName( value )
         return RcfAst.Constant.Type.interface
      except ( ValueError, IndexError ):
         pass
      return None

   @staticmethod
   def valueStr( valueNode ):
      return RcfImmediateValueToStringHelper.valueStr( valueNode )

   @staticmethod
   def constantTypeToEvalType( typ ):
      mapping = {
         RcfAst.Constant.Type.integer: RcfBuiltinTypes.Int,
         RcfAst.Constant.Type.boolean: RcfBuiltinTypes.Boolean,
         RcfAst.Constant.Type.trilean: RcfBuiltinTypes.Trilean,
         RcfAst.Constant.Type.prefix: RcfBuiltinTypes.Prefix,
         RcfAst.Constant.Type.asPath: RcfBuiltinTypes.AsPathImmediate,
         RcfAst.Constant.Type.asDot: RcfBuiltinTypes.AsDot,
         RcfAst.Constant.Type.interface: RcfBuiltinTypes.Interface,
         RcfAst.Constant.Type.ipAddress: RcfBuiltinTypes.IpAddress,
         RcfAst.Constant.Type.empty: RcfBuiltinTypes.Empty,
         RcfAst.Constant.Type.none: RcfBuiltinTypes.None,
      }
      evalType = mapping.get( typ )
      if evalType is None:
         # This can happen for enum types
         evalType = getRcfType( typ )
      return evalType

class AetValueGen( object ):
   aetTriStateNameToValueMap = {
      'unknown': Rcf.Eval.TriStateBoolValueType.RcfUnknown,
      'true': Rcf.Eval.TriStateBoolValueType.RcfTrue,
      'false': Rcf.Eval.TriStateBoolValueType.RcfFalse,
   }

   def __init__( self, aetGenVisitor ):
      self.aetGenVisitor = aetGenVisitor
      self.valueTypeToAetBuild = {
         RcfBuiltinTypes.AsPath: self.buildAsPathValue,
         RcfBuiltinTypes.Boolean: self.buildTriStateBoolValue,
         RcfBuiltinTypes.Int: self.buildIntValue,
         RcfBuiltinTypes.Prefix: self.buildPrefixValue,
         RcfBuiltinTypes.Community: self.buildCommunity,
         RcfBuiltinTypes.ExtCommunity: self.buildExtCommunity,
         RcfBuiltinTypes.Trilean: self.buildTriStateBoolValue,
         RcfBuiltinTypes.AsNumber: self.buildIntValue,
         RcfBuiltinTypes.IpAddress: self.buildIpAddressValue,
         RcfBuiltinTypes.Interface: self.buildInterfaceValue,
         RcfBuiltinTypes.Empty: self.buildEmpty,
         RcfBuiltinTypes.None: self.buildNone,
      }

      for enumTypename in RcfBuiltinTypes.enumTypes:
         enumType = getRcfType( enumTypename )
         self.valueTypeToAetBuild[ enumType ] = self.buildEnumValue

   def build( self, value ):
      return self.valueTypeToAetBuild[ self.typeOf( value ) ]( value )

   @classmethod
   def typeOf( cls, node ):
      return node.evalType if node.promoteToType is None else node.promoteToType

   def buildIpAddressValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetIpAddress = self.aetGenVisitor.visit( value )
      else:
         constant = value
         val = constant.value
         pfx = Rcf.Arnet.IpGenAddr( str( val ) )
         errMsg = "IP address immediate value cannot be %s, try 'none'" % str( val )
         assert not pfx.isAddrZero, errMsg
         aetIpAddress = Rcf.Eval.IpAddressImmediate( pfx )
      return aetIpAddress

   def buildInterfaceValue( self, value ):
      """Returns the immediate Interface representation of a value.
         The value must conform to an Arnet::IntfId. We assume it will
         as the AST phase should already have verified this.
      Arguments:
         value (str): the string reperesentation of the itnerface, short or long
      """
      interfaceString = str( value.value )
      intfId = None
      try:
         intfId = Rcf.Arnet.IntfId( interfaceString )
      except ( ValueError, IndexError ):
         pass
      try:
         intfId = Rcf.Arnet.IntfId.fromShortName( interfaceString )
      except ( ValueError, IndexError ):
         pass
      aetInterface = Rcf.Eval.InterfaceImmediate( intfId )
      return aetInterface

   def buildAsPathValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         asnVal = self.aetGenVisitor.visit( value )
         return Rcf.Eval.IntList( asnVal, None )
      elif value.evalType is RcfBuiltinTypes.Int or \
           value.evalType is RcfBuiltinTypes.AsNumber or \
           value.evalType is RcfBuiltinTypes.AsDot:
         asnVal = self.buildIntValue( value )
         return Rcf.Eval.IntList( asnVal, None )
      else:
         # build int list
         constant = value
         nextVal = None
         asnVal = None
         # We build the IntList objects representing an AsPath backwards
         for asn in reversed( constant.value ):
            if asn.has_key( 'int' ):
               asnVal = self.buildIntValue( asn[ 'int' ] )
            elif asn.has_key( 'as_dot' ):
               asnVal = self.buildIntValue( asn[ 'as_dot' ] )
            elif asn.has_key( 'kw' ):
               # empty
               assert asn[ 'kw' ].value == 0
               assert asn[ 'kw' ].type == RcfAst.Constant.Type.asPath
               return None
            elif asn.has_key( 'attr' ):
               asnVal = self.aetGenVisitor.visit( asn[ 'attr' ] )
            else:
               # TODO: last_as, auto support
               assert False, ( "ASN type not supported in RCF aet" )
            intListVal = Rcf.Eval.IntList( asnVal, nextVal )
            nextVal = intListVal
         return nextVal

   def buildIntValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetInt = self.aetGenVisitor.visit( value )
      else:
         constant = value
         aetInt = Rcf.Eval.IntImmediate( constant.value )
      return aetInt

   def buildPrefixValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetPrefix = self.aetGenVisitor.visit( value )
      else:
         constant = value
         val = constant.value
         pfx = Rcf.Arnet.IpGenPrefix( str( val ) )
         aetPrefix = Rcf.Eval.IpPrefixImmediate( pfx )
      return aetPrefix

   def buildEnumValue( self, value ):
      if isinstance( value, RcfAst.Attribute ):
         aetNode = self.aetGenVisitor.visit( value )
      else:
         typ = value.type
         val = value.value
         aetNode = Rcf.Eval.IntImmediate( getRcfType( typ ).valueDict[ val ] )
      return aetNode

   def buildCommunity( self, value ):
      errorMsg = "No immediate value support for community yet"
      assert isinstance( value, RcfAst.ExternalRef ), errorMsg
      extRef = value
      return self.aetGenVisitor.visit( extRef )

   def buildExtCommunity( self, value ):
      errorMsg = "No immediate value support for ext_community yet"
      assert isinstance( value, RcfAst.ExternalRef ), errorMsg
      extRef = value
      return self.aetGenVisitor.visit( extRef )

   def buildTriStateBoolValue( self, value, **kwargs ):
      constant = value
      aetTriStateValue = AetValueGen.aetTriStateNameToValueMap[ constant.value ]
      aetTriStateBoolImmediate = Rcf.Eval.TriStateBoolImmediate( aetTriStateValue )
      return aetTriStateBoolImmediate

   def buildEmpty( self, value ):
      # We do not support building empty values in a generic sense. Existing uses
      # of 'empty' have specific rhsAetType and rhsAetCtorArgs in RcfMetadata.yaml.
      assert False, ( "Building 'empty' value generically is not supported" )

   def buildNone( self, value ):
      # We do not support building 'none' values.
      assert False, ( "Building 'none' value generically is not supported" )
