#!/bin/python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import RcfAst

class IVisitor( object ):
   """ Visitor interface to override.
   """
   def visit( self, node, **kwargs ):
      raise NotImplementedError

   def __call__( self, node, **kwargs ):
      return node.accept( self, **kwargs )

class Visitor( IVisitor ):
   """ Rcf Visitor interface to visit the Abstract Syntax Tree (AST).
   """
   def __init__( self ):
      self.dispatchMap = {
         RcfAst.Root: self.visitRoot,
         RcfAst.Function: self.visitFunction,
         RcfAst.Block: self.visitBlock,
         RcfAst.IfStmt: self.visitIfStmt,
         RcfAst.SequentialExpr: self.visitSequentialExpr,
         RcfAst.Call: self.visitCall,
         RcfAst.ExternalRefOp: self.visitExternalRefOp,
         RcfAst.ExternalRef: self.visitExternalRef,
         RcfAst.Assign: self.visitAssign,
         RcfAst.Return: self.visitReturn,
         RcfAst.Constant: self.visitConstant,
         RcfAst.Attribute: self.visitAttribute,
         RcfAst.BinOp: self.visitBinOp,
         RcfAst.Not: self.visitNot,
      }

   def visit( self, node, **kwargs ):
      return self.dispatchMap[ type( node ) ]( node, **kwargs )

   def visitRoot( self, root, **kwargs ):
      pass

   def visitFunction( self, function, **kwargs ):
      pass

   def visitBlock( self, block, **kwargs ):
      pass

   def visitIfStmt( self, ifStmt, **kwargs ):
      pass

   def visitSequentialExpr( self, sequentialExpr, **kwargs ):
      pass

   def visitCall( self, call, **kwargs ):
      pass

   def visitExternalRefOp( self, externalRefOp, **kwargs ):
      pass

   def visitExternalRef( self, extRef, **kwargs ):
      pass

   def visitAssign( self, assign, **kwargs ):
      pass

   def visitReturn( self, returnStmt, **kwargs ):
      pass

   def visitConstant( self, constant, **kwargs ):
      pass

   def visitAttribute( self, attribute, **kwargs ):
      pass

   def visitBinOp( self, binOp, **kwargs ):
      pass

   def visitNot( self, notExpr, **kwargs ):
      pass
