#!/bin/python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" AST definition.

This module holds the definition of the RCF AST, and visitor to walk it.

!Rules (don't change unless discussing with authors first)

   - Follow the open/closed principle: The AST data model should be closed
     for modification, and open for extension.

   - Don't change AST attributes, don't add AST nodes, don't implement
     static, class, or member function: use a dedicated visitor instead.

   @author: matthieu (rcf-dev)
"""

from __future__ import absolute_import, division, print_function

class Node( object ):
   """ Abstract Syntax Tree node base type.

   This node holds all the information common to every node in the AST.

   Attributes:
      context (Antlr Context): the antlr context from parse-tree.
      evalType (RcfType): type evaluated during AST construction.
      scope (Scope): scope where the node sits in the symbol table.
      promoteToType (RcfType): Rcf type destination, if the semantic analysis
                               deemed to promote.
   """
   def __init__( self, context ):
      self.context = context
      self.evalType = None
      self.scope = None
      self.promoteToType = None

   def accept( self, visitor, **kwargs ):
      """ Accept the visitor with its arguments, and return its result.

      Args:
         visitor (Visitor): implements visit function.
         **kwargs: any named argument to be forwarded.
      """
      return visitor.visit( self, **kwargs )

class Root( Node ):
   """ Rcf ROOT AST node.

   The RCF Root holds a list of functions (class Function)

   Attributes:
      functions (list[ Function ]): list of function definitons.
   """
   def __init__( self, context, functions ):
      super( Root, self ).__init__( context )
      self.functions = functions

class Function( Node ):
   """ Rcf Function node

   A function has a name, and an execution block.

   Attributes:
      name (str): function name.
      block (Block): function block definiton.
      symbol (RcfSymbol): function symbol after definition.
   """
   def __init__( self, context, name, block ):
      super( Function, self ).__init__( context )
      self.name = name
      self.block = block
      self.symbol = None

class Block( Node ):
   """ Rcf block

   A block represents a list of instruction. We can find block
   in functions, and if statements.

   Attributes:
      stmts (list[ Nodes ]): list of statements.
   """
   def __init__( self, context, stmts ):
      super( Block, self ).__init__( context )
      self.stmts = stmts

class IfStmt( Node ):
   """ Rcf If statement

   Represents a branch in the AST.

   Attributes:
      condition (Node): the condition AST expression.
      thenBlock (Block): branch when the condition is true.
      elseBlock (Block, optional): branch when the condition is false.
   """
   def __init__( self, context, condition, thenBlock, elseBlock=None ):
      super( IfStmt, self ).__init__( context )
      self.condition = condition
      self.thenBlock = thenBlock
      self.elseBlock = elseBlock

class SequentialExpr( Node ):
   """ Rcf Sequential expression (of the form 'a() ?? b() ?? c()')

   Attributes:
      funcCalls: List of 'Call' nodes for each function being called.
      finalBool: Currently, the finalBool holds only the RcfAst.Constant
                 corresponding to the final true/false literal.  If the final
                 literal is 'unknown', it is parsed/accepted, but discarded in the
                 AST.  Also, if the final term is a function call, it is part
                 of 'funcCalls' above and 'finalBool' is None.
   """
   def __init__( self, context, funcCalls, finalBool ):
      super( SequentialExpr, self ).__init__( context )
      self.funcCalls = funcCalls
      self.finalBool = finalBool

class Call( Node ):
   """ Rcf call statement

   A call references another function by its name, and it's symbol in the AST.
   The function symbol itself has a reference to the AST node of this function
   defintion in the AST. (functionSymbol is an RCF Symbol).

   The symbol is resolved in later phases, after AST creation.

   Attributes:
      funcName (str): function name.
      funcSymbol (Symbol): function symbol in the symbol table.
   """
   def __init__( self, context, funcName ):
      super( Call, self ).__init__( context )
      self.funcName = funcName
      self.symbol = None

class ExternalRefOp( Node ):
   """ Compare external represents a comparison with an external entity through
   a 'match', 'match_covered' or 'match-exact' condition.

   Attributes:
      attribute (Attribute): the Attribute (AST node) on which the match operates.
      isExact (bool): whether it is an exact match or not.
      isMatchCovered (bool): whether we are using 'match_covered' instead of match.
         Only one of isExact or isMatchCovered is allowed to be true.
      extRef (ExternalRef): the external reference AST node.
      op (string): string representation of the operation
   """
   def __init__( self, context, attribute, isExact, isMatchCovered, extRef ):
      super( ExternalRefOp, self ).__init__( context )
      self.attribute = attribute
      self.isExact = isExact
      self.isMatchCovered = isMatchCovered
      self.extRef = extRef
      assert not isExact or not isMatchCovered
      if isExact:
         self.op = "match_exact"
      elif isMatchCovered:
         self.op = "match_covered"
      else:
         self.op = "match"

class ExternalRef( Node ):
   """ External refrence

   Represents an external reference to another Cli construct.

   e.g prefix_list_v4 NETFLIX

   Attriubtes:
      name (str): the name of the external construct (e.g NETFLIX).
      etype (str): the type of the external construct (e.g: prefix_list_v4).
      refFromAssign (bool): whether this ext ref was reference from an assignment
                            (set a the resolution phase)
      isIpV4 (bool/None): whether the current external ref refers to an IPv4
                          or None if it doesn't apply.
                             e.g:
                               prefix_list_v4 -> true
                               prefix_list_v6 -> false
                               community_list -> None
   """
   def __init__( self, context, name, etype, refFromAssign=False ):
      super( ExternalRef, self ).__init__( context )
      self.name = name
      self.utf8name = self.name.encode( "utf8" )
      self.type = etype
      self.refFromAssign = refFromAssign
      self.ipVersionned = (
         any( ipv in etype for ipv in ( 'v4', 'v6' ) )
      )
      self.isIpV4 = 'v4' in etype if self.ipVersionned else None

class Assign( Node ):
   """ Rcf assignment node.

   e.g: med += 4;

   Attributes:
      attribute (Attribute): AST node Attribute.
      value (Constant): AST node Constant.
      op (string): operation (e.g '+=' or '-=', or '=')
   """
   def __init__( self, context, attribute, value, op ):
      super( Assign, self ).__init__( context )
      self.attribute = attribute
      self.value = value
      self.op = op

class Return( Node ):
   """ Rcf assignment node.

   e.g: return POLICYA() and med >= 10;

   Attributes:
      attribute (Attribute): Attribute (AST Node).
      expr (Node): AST subtree expression.
   """
   def __init__( self, context, expr ):
      super( Return, self ).__init__( context )
      self.expr = expr

class Constant( Node ):
   """ Rcf constant (a.k.a 'Immediate' value )

   e.g: 42, or 'unknown'

   Attributes:
      value (?): holds the value of the constant.
      ctype (Constant.Type): the constant type.
   """

   class Type( object ):
      integer = 'integer'
      boolean = 'boolean'
      trilean = 'trilean'
      prefix = 'prefix'
      asPath = 'aspath'
      asDot = 'asDot'
      interface = 'interface'
      ipAddress = 'ipAddress'
      empty = 'empty'
      none = 'none'
      # More will be automatically added here (eg. enums like Origin) when the
      # metadata is processed during initialization.

   def __init__( self, context, value, ctype ):
      super( Constant, self ).__init__( context )
      self.value = value
      self.type = ctype

class Attribute( Node ):
   """ Rcf Attribute node

   Represents an Attribute (e.g: med, prefix, or community.len) in Rcf.

   Attributes:
      name (str): the attribute name (e.g 'prefix').
      symbol (RcfSymbol): Associated RcfSymbol (set when resolved).
   """
   def __init__( self, context, name ):
      super( Attribute, self ).__init__( context )
      self.name = name
      self.symbol = None

   def fullName( self ):
      return self.name

class BinOp( Node ):
   """ Binary Operation in Rcf

   e.g: true and false
        ^~~~ ^~~ ^~~~~
         lhs  op  rhs

   Attributes:
      operator (string): e.g 'and', 'or'.
      rhs (Node): right hand side node.
      lhs (Node): left hand side node.
      isBooleanOp (bool): whether this BinOp is and/or
   """
   def __init__( self, context, operator, lhs, rhs ):
      super( BinOp, self ).__init__( context )
      self.operator = operator
      self.rhs = rhs
      self.lhs = lhs
      self.isBooleanOp = (
         any( bop == operator for bop in ( "and", "or" ) )
      )

class Not( Node ):
   """ Rcf boolean inverter

   Simply holds an expression (Ast Node), and inverses it's
   boolean value:
   e.g:  not ( false and true )
             ^~~~~~~~~~~~~~~~~~ (expr)

   Attributes:
      expr (Node): Rcf expression AST tree.
   """
   def __init__( self, context, expr ):
      super( Not, self ).__init__( context )
      self.expr = expr
