# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import traceback

import Agent
import BothTrace
import Cell
from RcfCompiler import (
   RcfCompiler,
   RcfCompileRequest,
)
import Logging
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'RcfAgent' )
bt0 = BothTrace.tracef0

MAX_U32 = 0xFFFFFFFF

Logging.logD( id="RCF_UNEXPECTED_COMPILATION_ERROR",
              severity=Logging.logError,
              format="An unexpected processing error occurred while compiling "
                     "the RCF configuration.",
              explanation="An unexpected processing error occurred while compiling "
                          "the RCF configuration."
                          "When this occurs, no RCF functions will be applied "
                          "on the affected router.",
              recommendedAction="Revert to a previously working version of RCF "
                                "config, and report this error to Arista's "
                                "customer support team."
)

Logging.logD( id="RCF_STARTUP_CONFIG_COMPILATION_ERROR",
              severity=Logging.logError,
              format="The RCF code in the startup config failed to compile.",
              explanation="The RCF code in the startup config failed to compile. "
                          "This is most likely due to a syntax error in the saved "
                          "RCF code. When this occurs, no RCF functions will be "
                          "applied on the affected router. "
                          "For more details, run 'show router rcf errors startup'.",
              recommendedAction="Revert to a previously working version of RCF code"
)

def agentName():
   return 'Rcf'

redundancyStatus = None
redundancyStatusReactor = None

class RcfAgentConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Rcf::Config"

   def __init__( self, notifier, aclConfig, rcfAllFunctionStatus, rcfStatus ):
      bt0( "RcfAgentConfigReactor: init" )
      Tac.Notifiee.__init__( self, notifier )
      self.rcfCompiler = RcfCompiler( aclConfig, rcfAllFunctionStatus )
      self.aclConfig = aclConfig
      self.rcfAllFunctionStatus = rcfAllFunctionStatus
      self.rcfStatus = rcfStatus

   # if Sysdb rcfText changes, compile it in lax mode and commit to AllFunctionStatus
   @Tac.handler( "rcfText" )
   def handleRcfTextChange( self ):
      bt0( "RcfAgentConfigReactor: text changed" )
      rcfText = self.notifier_.rcfText
      compileRequest = RcfCompileRequest( rcfText, strictMode=False )
      try:
         compileResult = self.rcfCompiler.compile( compileRequest )
         if compileResult and compileResult.success:
            compileResult.publish( self.rcfAllFunctionStatus,
                                   self.rcfStatus )
            self.rcfStatus.active = True
         else:
            # Cli Published invalid rcfText in RcfConfig.
            # The only reason we can get there is if this happened in the context
            # of startup config, where we would publish invalid code in Config
            # regardless of whether it compiles fine or not.
            bt0( "RcfAgentConfigReactor: AET generation failed!" )
            errorStr = "\n".join( compileResult.errorList )
            self.rcfStatus.lastStartupCompilationError = errorStr
            self.rcfStatus.active = False
            # pylint: disable=undefined-variable
            Logging.log( RCF_STARTUP_CONFIG_COMPILATION_ERROR )
      except Exception: # swallow everything # pylint: disable=broad-except
         bt0( 'RCF compiler error' )
         bt0( traceback.format_exc() )
         bt0( 'RCF text' )
         bt0( rcfText )
         # pylint: disable=undefined-variable
         Logging.log( RCF_UNEXPECTED_COMPILATION_ERROR )
         self.rcfStatus.active = False

   # Handle cleanup
   @Tac.handler( "enabled" )
   def handleCleanup( self ):
      bt0( "RcfAgentConfigReactor: rcfConfig.enabled changed" )
      enabled = self.notifier_.enabled
      if enabled:
         bt0( "RcfAgentConfigReactor: kicking text reeval" )
         self.handleRcfTextChange()
         self.rcfStatus.enabled = True
      else:
         bt0( "RcfAgentConfigReactor: going down, cleanup" )
         self.rcfAllFunctionStatus.aet.clear()
         self.rcfStatus.functionNames.clear()
         self.rcfStatus.active = False
         self.rcfStatus.enabled = False

class RedundancyStatusReactor( Tac.Notifiee ):
   """Reacts to redundancy/status.mode changing. This is needed to
   bump up the agent2AgentmountChangedCounter when the supervisor
   becomes the active supervisor.
   The bump up is done by blindly invoking the callback function.
   """
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatusEntity, callback ):
      Tac.Notifiee.__init__( self, redStatusEntity )
      self.callback = callback

   @Tac.handler( "mode" )
   def handleRedundancyMode( self ):
      self.callback()


class RcfAgent( Agent.Agent ):
   """ Rcf Agent reacts to ConfigAgent writing valid Rcf text and compiles it
   down to its AET form, that ArBgp can run.

   RcfAgent writes to the AllFunctionStatus's AET collection, and ArBgp directly
   reads updates from it through an Agent2Agent mount.

   This agent is managed by Launcher, and will only run when RcfConfig's RcfText
   (that ConfigAgent writes) is not empty.
   """
   agentDirName = agentName()
   def __init__( self, entityManager ):
      bt0( "Initialize RcfAgent" )
      self.sysname = entityManager.sysname()
      bt0( "Starting agent %s in system %s" % ( agentName(), self.sysname ) )
      self.entityManager = entityManager
      self.rcfAllFunctionStatus = None
      self.rcfConfig = None
      self.aclConfig = None
      self.rcfStatus = None
      self.status_ = None
      self.configReactor_ = None
      self.local = None
      Agent.Agent.__init__( self, entityManager, agentName=agentName() )

   def doInit( self, entityManager ):
      bt0( "RcfAgent: doInit" )
      global redundancyStatus

      agent2AgentRcfDir = Tac.root[ self.sysname ].mkdir( 'Agent2AgentRcf' )
      self.rcfAllFunctionStatus = agent2AgentRcfDir.newEntity(
                                    "Rcf::AllFunctionStatus",
                                    "RcfAllFunctionStatus"
                                 )

      mg = entityManager.mountGroup()
      redundancyStatus = mg.mount( Cell.path( 'redundancy/status' ),
                                 'Redundancy::RedundancyStatus',
                                 mode='r' )
      self.rcfConfig = mg.mount( 'routing/rcf/config',
                                 'Rcf::Config',
                                 mode='rS' )
      self.rcfStatus = mg.mount( 'routing/rcf/status_ci',
                                 'Rcf::Status',
                                 mode='w' )
      self.aclConfig = mg.mount( 'routing/acl/config',
                                 'Acl::AclListConfig',
                                 mode='r' )

      mg.close( self.doMountComplete )

   def isActiveSupervisor( self ):
      global redundancyStatusReactor
      # redundancyStatus got mounted. If a reactor for this is not created yet,
      # create one now.
      if not redundancyStatusReactor:
         redundancyStatusReactor = RedundancyStatusReactor( redundancyStatus,
                                                            self.doMountComplete )
      # In a dual-sup situation, only proceed if we are the active supervisor
      if redundancyStatus.mode != "active":
         bt0( "RcfAgent: isActiveSupervisor: no" )
         return False
      bt0( "RcfAgent: isActiveSupervisor: yes" )
      return True

   def doMountComplete( self ):
      bt0( "RcfAgent: doMountComplete" )
      if not self.isActiveSupervisor():
         return
      # If the mount changed counter is non-zero, ConfigAgent has restarted. Since
      # the RCF AETs are not in Sysdb, they do not survive a ConfigAgent restart.
      # Therefore, if there is any RCF config in Sysdb, we have to recompile and
      # recommit the AETs by triggering the python reactor.
      if not hasattr( self.entityManager, "rcfConfigReactor" ):
         bt0( "RcfAgent: initialize rcfConfigReactor" )
         self.entityManager.rcfConfigReactor = RcfAgentConfigReactor(
                                                   self.rcfConfig,
                                                   self.aclConfig,
                                                   self.rcfAllFunctionStatus,
                                                   self.rcfStatus )

      self.entityManager.rcfConfigReactor.handleRcfTextChange()
      # At this point, we know /<sysname>/Agent2Agent/RcfAllFunctionStatus has been
      # created in RcfAgent, so we bump up the counter in rcfStatus to kick ArBgp.
      if self.rcfStatus.agent2AgentMountChangedCounter == MAX_U32:
         bt0( "RcfAgent: agent2AgentMountChangedCounter wrapped" )
         self.rcfStatus.agent2AgentMountChangedCounter = 1
      else:
         self.rcfStatus.agent2AgentMountChangedCounter += 1
      bt0( "RCF agent2AgentMountChangedCounter %i" %
            self.rcfStatus.agent2AgentMountChangedCounter )

      self.rcfStatus.enabled = True

def main():
   container = Agent.AgentContainer( [ RcfAgent ] )
   container.runAgents()

