# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import RcfTypeFuture
import Tac

def getType( name ):
   return Tac.Type( 'Rcf::Eval::%s' % name )

class AetCleanupHelper( object ):
   """This class helps cleanup the AET nodes from the instantiating collections.
   It has helper functions for the core RCF language constructs.  For the generated
   AET types, it uses python/TACC reflection using node.attributes and cleans up
   the embedded nodes there.

   Public functions:
   * cleanup( node )

   The rest are internal functions.
   """
   def __init__( self ):
      self.typeMap = {
         getType( 'Function' ): self._cleanupFunction,
         getType( 'AndExpression' ): self._cleanupAndExpression,
         getType( 'OrExpression' ): self._cleanupOrExpression,
         getType( 'NotExpression' ): self._cleanupNotExpression,
         getType( 'IfExpression' ): self._cleanupIfExpression,
         getType( 'ReturnExpression' ): self._cleanupReturnExpression,
         getType( 'BlockStmt' ): self._cleanupLinkedList,
         getType( 'SequentialExpression' ): self._cleanupLinkedList,
         getType( 'IntList' ): self._cleanupLinkedList,
         getType( 'UIntList' ): self._cleanupLinkedList,
      }

      self.attrsToConsider = [ 'lhs', 'rhs', 'start', 'end', 'val' ]

   def cleanup( self, node ):
      if node is None:
         return

      func = self.typeMap.get( type( node ) )
      if func:
         func( node )

      # Take care of cleaning up the nodes contained inside the generated
      # matcher/setter nodes (eg. lhs/rhs in IntIs).
      if hasattr( node, 'attributes' ):
         for attr in self.attrsToConsider:
            if hasattr( node, attr ):
               self.cleanup( getattr( node, attr ) )

      RcfTypeFuture.aetNodeCleanup( node )

   def _cleanupLinkedList( self, head ):
      # The number of statements in a 'Block' could be big. We don't want each
      # stmt in a block or sequential expression to be cleaned up recursively
      # from the previous entry in the linked list. Instead, we collect all the
      # statements into a list and just cleanup the 'entry' and 'next' in each
      # element in the list.
      stmts = []
      cur = head
      while cur is not None:
         stmts.append( cur )
         cur = cur.next

      for stmt in stmts:
         self.cleanup( stmt.entry )
         # stmt.next is also part of stmts and will be processed next.
         # Don't call self.cleanup( node.next ), to avoid recursive calls.
         RcfTypeFuture.aetNodeCleanup( stmt.next )

      RcfTypeFuture.aetNodeCleanup( head )

   def _cleanupFunction( self, node ):
      self.cleanup( node.body )

   def _cleanupAndExpression( self, node ):
      self.cleanup( node.lhs )
      self.cleanup( node.rhs )

   def _cleanupOrExpression( self, node ):
      self.cleanup( node.lhs )
      self.cleanup( node.rhs )

   def _cleanupNotExpression( self, node ):
      self.cleanup( node.expr )

   def _cleanupIfExpression( self, node ):
      self.cleanup( node.condition )
      self.cleanup( node.thenBlock )
      self.cleanup( node.elseBlock )

   def _cleanupReturnExpression( self, node ):
      self.cleanup( node.expr )
