#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from QosTypes import * # pylint: disable-msg=W0401
import Tracing
import random
from operator import itemgetter

# Explicit dependency so the preinit profile can use FabricIntf types
# pkgdeps: library FabricIntf

__defaultTraceHandle__ = Tracing.Handle( 'QosCli' )
t0 = Tracing.trace0

QOS_CLI_TIMEOUT = 60
QOS_PMAP_COUNTER_TCAM_PROFILE = 'qos-pmap-counter'
QOS_PMAP_COUNTER_WITHOUT_DSCP_ACTION = 'qos-pmap-counter-without-dscp-action'
QOS_PMAP_DROP_PRECEDENCE_TCAM_PROFILE = 'qos-pmap-drop-precedence'
VXLAN_ROUTING_QOS_TCAM_PROFILE = 'vxlan-routing-qos'

coppStaticClassName = [ tacCMapNm.coppBpdu, 
                        tacCMapNm.coppLacp, 
                        tacCMapNm.coppLldp,
                        tacCMapNm.coppArp, 
                        tacCMapNm.coppSelfIpTc6To7,
                        tacCMapNm.coppSelfIp, 
                        tacCMapNm.coppIpmcRsvd, 
                        tacCMapNm.coppIpmcMiss,
                        tacCMapNm.coppL3DestMiss, 
                        tacCMapNm.coppIgmp, 
                        tacCMapNm.coppTc3To5, 
                        tacCMapNm.coppTc6To7, 
                        tacCMapNm.coppGlean, 
                        tacCMapNm.coppL3SlowPath, 
                        tacCMapNm.coppSflow, 
                        tacCMapNm.coppL3Ttl1, 
                        tacCMapNm.coppArpResolver,
                        tacCMapNm.coppIpLocking,
                        tacCMapNm.coppAclLog,
                        tacCMapNm.coppDrop, 
                        tacCMapNm.coppDefault,
                        tacCMapNm.coppIpBroadcast,
                        tacCMapNm.coppIpUnicast,
                        tacCMapNm.coppIpMc,
                        tacCMapNm.coppUnicastArp,
                        tacCMapNm.coppL3Ttl0,
                        tacCMapNm.coppIpv6Nd,
                        tacCMapNm.coppL2Rsvd,
                        tacCMapNm.coppPimPtp,
                        tacCMapNm.coppOspfIsis,
                        tacCMapNm.coppL2Broadcast,
                        tacCMapNm.coppL2Unicast,
                        tacCMapNm.coppL3LpmOverflow,
                        tacCMapNm.coppLinkLocal,
                        tacCMapNm.coppMcastSnoop,
                        tacCMapNm.coppBgp,
                        tacCMapNm.coppVrrp,
                        tacCMapNm.coppMlag,
                        tacCMapNm.coppBfd,
                        tacCMapNm.coppPim,
                        tacCMapNm.coppVxlanVtepLearn,
                        tacCMapNm.coppVxlanEncap,
                        tacCMapNm.coppMplsTtl01,
                        tacCMapNm.coppMplsLabelMiss,
                        tacCMapNm.coppPtp,
                        tacCMapNm.coppMod,
                        tacCMapNm.coppMvrp,
                        tacCMapNm.coppMtu,
                        tacCMapNm.coppEgressAclLog,
                        tacCMapNm.coppCvx,
                        tacCMapNm.coppCvxHeartbeat,
                        tacCMapNm.coppMirroring,
                        tacCMapNm.coppPtpSnoop,
                        tacCMapNm.coppMacLearn,
                        tacCMapNm.coppDot1xMba,
                        tacCMapNm.coppCfm,
                        tacCMapNm.coppCfmSnoop,
                        tacCMapNm.coppArpInspect,
                        tacCMapNm.coppMplsArpSuppress,
                        tacCMapNm.coppProtocolSnoop,
                        tacCMapNm.coppDefaultSnoop,
                        tacCMapNm.coppNat,
                        tacCMapNm.coppLdp,
                        tacCMapNm.coppRsvp,
                        tacCMapNm.coppBfdPtp,
                        tacCMapNm.coppL3Rsvd,
                        tacCMapNm.coppDot1xVxlan,
                        tacCMapNm.coppAcllogSflow,
                        ]

builtInClassMapNames = {
   tacClassMapBuiltIn.mlagControl : [ tacClassMapMatchBuiltIn.matchMlagControl,
                                      'Packets from MLAG peer' ],
   tacClassMapBuiltIn.layer2Control : [ tacClassMapMatchBuiltIn.matchLayer2Control,
                                 'Any layer2 control packets like LLDP,STP,LACP' ],
   tacClassMapBuiltIn.mlagControlHB : [
      tacClassMapMatchBuiltIn.matchMlagControlHeartbeat,
      'MLAG heartbeat packets from MLAG peer' ],
   tacClassMapBuiltIn.lldp : [ tacClassMapMatchBuiltIn.matchLldp, 'LLDP packets' ],
   tacClassMapBuiltIn.lacp : [ tacClassMapMatchBuiltIn.matchLacp, 'LACP packets' ],
   tacClassMapBuiltIn.bpdu : [ tacClassMapMatchBuiltIn.matchBpdu, 'BPDU packets' ],
   tacClassMapBuiltIn.pvst : [ tacClassMapMatchBuiltIn.matchPvst, 'PVST packets' ],
   tacClassMapBuiltIn.mstp : [ tacClassMapMatchBuiltIn.matchMstp, 'MSTP packets' ],
   tacClassMapBuiltIn.vxlanEncap : [ tacClassMapMatchBuiltIn.matchVxlanEncapsulation,
                              'Packets that require software VXLAN encapsulation' ],
   tacClassMapBuiltIn.vxlanVtepLearn : [ tacClassMapMatchBuiltIn.matchVxlanVtepLearn,
                          'Packets from remote VTEP for learning the VTEP' ],
   tacClassMapBuiltIn.ptp : [ tacClassMapMatchBuiltIn.matchPtp, 'PTP packets' ],
   tacClassMapBuiltIn.selfIpAll : [ tacClassMapMatchBuiltIn.matchSelfIpAll,
                     'Routed packet destined to any IP address of this switch' ],
   tacClassMapBuiltIn.selfIpHigh : [ tacClassMapMatchBuiltIn.matchSelfIpHighPriority,
                               'Routed IP packets with high DSCP/VLAN priority ' + \
                               'destined to any IP address of this switch' ],
   tacClassMapBuiltIn.selfIpLow : [ tacClassMapMatchBuiltIn.matchSelfIpLowPriority,
                              'Routed IP packets with low DSCP/VLAN priority ' + \
                              'destined to any IP address of this switch' ],
   tacClassMapBuiltIn.arpNeeded : [ tacClassMapMatchBuiltIn.matchArpNeeded,
                    'Packets requiring ARP resolution' ],
   tacClassMapBuiltIn.unicastRouteMiss : [
      tacClassMapMatchBuiltIn.matchUnicastRouteMiss,
                            'IP unicast routed packets encountered a miss in ' + \
                            'route lookup' ],
   tacClassMapBuiltIn.mcastRouteMiss : \
      [ tacClassMapMatchBuiltIn.matchMulticastRouteMiss,
        'IP multicast routed packets encountered a miss in route lookup' ],
   tacClassMapBuiltIn.unicastRouteOverflow : \
      [ tacClassMapMatchBuiltIn.matchUnicastRouteOverflow,
        'Routed IP packets lookup fail due to route table overflow' ],
   tacClassMapBuiltIn.routedIpOptions : [
      tacClassMapMatchBuiltIn.matchRoutedIpOptions,
                           'Routed IP packets with options' ],
   tacClassMapBuiltIn.bfd : [ tacClassMapMatchBuiltIn.matchBfd,
             'Packets from any BFD' ],
   tacClassMapBuiltIn.layer3Control : [ tacClassMapMatchBuiltIn.matchLayer3Control,
                        'Any layer3 control packets like BGP,OSPF,ISIS' ],
   tacClassMapBuiltIn.selfBgp : [ tacClassMapMatchBuiltIn.matchSelfBgp,
                  'Packets from any BGP peer' ],
   tacClassMapBuiltIn.ospf : [ tacClassMapMatchBuiltIn.matchOspf,
              'OSPF control packets' ],
   tacClassMapBuiltIn.ttlException : [ tacClassMapMatchBuiltIn.matchTtlException,
                       'Routed IP packets with expired TTL' ],
   tacClassMapBuiltIn.ipv6Nd : [ tacClassMapMatchBuiltIn.matchIpv6nd,
                 'IPv6 neighbor discovery packets' ],
   tacClassMapBuiltIn.isis : [ tacClassMapMatchBuiltIn.matchIsis,
              'IS-IS protocol packets' ],
   tacClassMapBuiltIn.vrrp : [ tacClassMapMatchBuiltIn.matchVrrp,
              'VRRP protocol packets' ],
   tacClassMapBuiltIn.pim : [ tacClassMapMatchBuiltIn.matchPim,
             'PIM protocol packets' ],
   tacClassMapBuiltIn.dhcp : [ tacClassMapMatchBuiltIn.matchDhcp,
              'DHCP packets' ],
   tacClassMapBuiltIn.linkLocalMcast : [
      tacClassMapMatchBuiltIn.matchLinkLocalMulticast,
                              'IP link local multicast packets destined ' + \
                              'to 224.0.0.x' ],
   tacClassMapBuiltIn.mplsRouteMiss : [ tacClassMapMatchBuiltIn.matchMplsRouteMiss,
                         'MPLS packets encountered a miss in the MPLS label ' + \
                         'lookup' ],
   tacClassMapBuiltIn.natMiss : [ tacClassMapMatchBuiltIn.matchNatMiss,
                  'IP packets requiring NAT encountered a miss on translation ' + \
                  'entry' ],
   tacClassMapBuiltIn.unicastRpfFailure : [
      tacClassMapMatchBuiltIn.matchUnicastRpfFailure,
                             'Routed IP uRPF failure packets' ],
   tacClassMapBuiltIn.igmp : [ tacClassMapMatchBuiltIn.matchIgmp,
                               'IGMP protocol packets' ],
   tacClassMapBuiltIn.arp : [ tacClassMapMatchBuiltIn.matchArp,
                              'ARP protocol packets' ],
   tacClassMapBuiltIn.layer3SlowPath : [ tacClassMapMatchBuiltIn.matchLayer3SlowPath,
                                         'Layer3 exception packets' ],
   tacClassMapBuiltIn.mvrp : [ tacClassMapMatchBuiltIn.matchMvrp,
                               ' Mvrp packets' ],
   tacClassMapBuiltIn.macSourceMiss : [ tacClassMapMatchBuiltIn.matchMacSourceMiss,
                                        'Source mac address miss' ],
   tacClassMapBuiltIn.ipBroadcast : [ tacClassMapMatchBuiltIn.matchIpBroadcast,
                                      'IP Broadcast packets' ],
   tacClassMapBuiltIn.layer2Broadcast : [
      tacClassMapMatchBuiltIn.matchLayer2Broadcast,
      'Packets destined to broadcast MAC address' ],
   tacClassMapBuiltIn.selfIcmp : [ tacClassMapMatchBuiltIn.matchSelfIcmp,
                                   'ICMP packets destined to switch' ],
   tacClassMapBuiltIn.cfm : [ tacClassMapMatchBuiltIn.matchCfm,
                              'Cfm packets' ],
   tacClassMapBuiltIn.cvx : [ tacClassMapMatchBuiltIn.matchCvx,
                              'Packets destined to Cvx Client/Controller port' ],
   tacClassMapBuiltIn.dot1xMBA : [ tacClassMapMatchBuiltIn.matchDot1xMBA,
                                   'Dot1x MBA packets' ],
   tacClassMapBuiltIn.natTcpFlags : [ tacClassMapMatchBuiltIn.matchNatTcpFlags,
                                      'NAT tcp flags packets' ],
}

CliError = { 
   'cannotDeleteCMap' : 'Cannot delete static class maps',
   'cannotReorder' : 'Cannot reorder static classes',
   'cMapNotSupported' : 'Static class %s is not supported on this platform',
   'maxClassError' : 'Cannot add more than %d classes in a policy map',
   'cannotConfigureCMap' : 'Cannot modify static class maps',
   'cannotConfigureDropClass' : 'Cannot modify internal static class %s',
   'dynamicClassError' : 'Only pre-defined static class maps can be '\
      'attached to this policy',
   'lagPolicingNotSupported' : 'Policing on Lag not supported on this platform',
   'cannotAddBuiltInCmap' : 'Built-in class not supported on this platform',
   'cannotDeleteBuiltInCmap' : 'Built-in class cannot be deleted',
   'cannotDeleteClassDefault' : 'Class-default cannot be deleted',
   }

# CLI keywords that are controlled by a flag. These are currently disabled
cliDirectionOutputSupported = False

coppMapType = tacClassMapType.mapControlPlane
coppMapTypeCli = tacClassMapType.mapCoppCli
qosMapType = tacClassMapType.mapQos
qosMapTypeCli = tacClassMapType.mapQosCli
pdpMapType = tacClassMapType.mapPdp
# All types of Policy/Class Maps
mapTypes = [ qosMapType, pdpMapType ]

try:
   import QosCoppLib # pylint: disable-msg=unused-import
except ImportError as _e:
   # QosCoppLib doesn't exist
   if _e.message != 'No module named QosCoppLib':
      raise _e

pmapQosActionTypes = [ tacActionType.actionSetCos, 
                       tacActionType.actionSetDscp, 
                       tacActionType.actionSetDrop, 
                       tacActionType.actionSetTc,
                       tacActionType.actionSetDropPrecedence ]

pmapCoppActionTypes = [ tacActionType.actionSetShape, 
                        tacActionType.actionSetBandwidth ]

tempShapeRate = Tac.Value( 'Qos::ShapeRate' )
invalidShapeRate = Tac.Value( 'Qos::ShapeRate' )

tempGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
fabricIntfName = "Fabric0/255"
fabricShortName = "Fabric"

#------------------------------------------------------------------------------------
# Base class which every Qos module will inherit from
#------------------------------------------------------------------------------------
class QosModuleBase( object ):
   def __init__( self, qosHwStatus ):
      self.qosHwStatus = qosHwStatus

   def isDefaultIntfConfig( self, intfConfig ):
      assert NotImplementedError

#------------------------------------------------
# Qos Module Registration dictionary
#------------------------------------------------
_registeredQosModules = {}

def registerQosModule( moduleName, moduleObj ):
   assert isinstance( moduleObj, QosModuleBase )
   assert moduleName not in _registeredQosModules
   _registeredQosModules[ moduleName ] = moduleObj

#------------------------------------------------
# Functions to check for defaultAttrs
#------------------------------------------------
def isDefaultCos( cos, qosHwStatus ):
   defaultCos = ( tacCos.invalid, qosHwStatus.defaultCos )
   return cos in defaultCos

def isDefaultDscp( dscp, qosHwStatus ):
   defaultDscp = ( tacDscp.invalid, qosHwStatus.defaultDscp )
   return dscp in defaultDscp

def isDefaultDscpRewriteMap( dscpRewriteMapName ):
   return dscpRewriteMapName == tacDscpRewriteMapName.defaultMapName

def isDefaultCosToTcProfile( cosToTcProfileName ):
   return cosToTcProfileName == tacCosToTcProfileName.defaultProfileName

def isDefaultTcToCosMap( tcToCosMapName ):
   return tcToCosMapName == tacTcToCosMapName.defaultMapName

def isDefaultTrustMode( trustMode ):
   defaultTrustMode = ( tacTrustMode.invalid, )
   return trustMode in defaultTrustMode

def isDefaultShapeRate( cfgShapeRate ):
   return ( ( cfgShapeRate.rate == tacShapeRateVal.invalid ) and
            ( cfgShapeRate.unit == tacShapeRateUnit.shapeRateKbps ) and
            not cfgShapeRate.shared )

def isDefaultGuaranteedBw( cfgGuaranteedBw ):
   return ( cfgGuaranteedBw.bw == tacGuaranteedBwVal.invalid )

def isDefaultSchedulerGroupName( cfgSchedulerGroupName ):
   return not cfgSchedulerGroupName

def isDefaultTxQueueConfig( txQueueConfig ):
   return ( ( txQueueConfig.priority == tacTxQueuePriority.priorityInvalid ) and
            ( txQueueConfig.shapeRate.rate == tacShapeRateVal.invalid ) and
            ( txQueueConfig.guaranteedBw.bw == tacGuaranteedBwVal.invalid ) and
            ( txQueueConfig.bandwidth == tacPercent.invalid ) and
            ( txQueueConfig.ecnConfig is None ) and
            ( txQueueConfig.nonEctConfig is None ) and
            ( not txQueueConfig.delayEcnEnabled ) and
            ( txQueueConfig.wredConfig is None ) and
            ( len( txQueueConfig.dropThresholds ) == 0 ) and
            ( txQueueConfig.latencyThreshold == tacLatencyThreshold() ) )

def isDefaultTxQueueIntfConfig( intfConfig ):
   for txQueueConfig in intfConfig.txQueueConfig.itervalues():
      if not isDefaultTxQueueConfig( txQueueConfig ):
         return False
   return True

def isDefaultIntfConfig( qosHwStatus, intfConfig ):
   assert intfConfig

   # For the modules which have already been registered
   for moduleObj in _registeredQosModules.itervalues():
      if not moduleObj.isDefaultIntfConfig( intfConfig ):
         return False

   # For the modules which have not been registered
   return ( isDefaultTrustMode( intfConfig.trustMode ) and
            isDefaultCos( intfConfig.defaultCos, qosHwStatus ) and
            isDefaultDscp( intfConfig.defaultDscp, qosHwStatus ) and
            isDefaultDscpRewriteMap( intfConfig.dscpRewriteMapName ) and
            isDefaultCosToTcProfile( intfConfig.cosToTcProfileName ) and
            isDefaultTcToCosMap( intfConfig.tcToCosMapName ) and
            isDefaultTcToCosMap( intfConfig.cpuTcToCosMapName ) and
            isDefaultSchedulerGroupName( intfConfig.schedulerGroupName ) and
            isDefaultTxQueueIntfConfig( intfConfig ) )

#------------------------------------------------
# End to Functions to check for defaultAttrs
#------------------------------------------------

def qosTimeoutWarning():
   return "The QoS configuration is still being programmed into hardware. "\
       "The system might be busy for a while."

def getCurrentTimeForConfigUpdate():
   return Tac.now()

def shapeRateUnitFromEnum( unit ):
   if unit == tacShapeRateUnit.shapeRateKbps:
      return 'kbps'
   elif unit == tacShapeRateUnit.shapeRatePps:
      return 'pps'
   else:
      return ''

def guaranteedBwUnitFromEnum( unit ):
   if unit == tacGuaranteedBwUnit.guaranteedBwKbps:
      return 'kbps'
   elif unit == tacGuaranteedBwUnit.guaranteedBwPps:
      return 'pps'
   elif unit == tacGuaranteedBwUnit.guaranteedBwPercent:
      return 'percent'
   else:
      return 'unspecified'

def shapeRateUnitToEnum( unit ):
   assert( unit == 'kbps' or  unit == 'pps' )
   if unit == 'kbps':
      return tacShapeRateUnit.shapeRateKbps
   else:
      return tacShapeRateUnit.shapeRatePps

def guaranteedBwUnitToEnum( unit ):
   assert unit in [ 'kbps', 'pps', 'percent' ]
   if unit == 'kbps':
      return tacGuaranteedBwUnit.guaranteedBwKbps
   elif unit == 'pps':
      return tacGuaranteedBwUnit.guaranteedBwPps
   elif unit == 'percent':
      return tacGuaranteedBwUnit.guaranteedBwPercent
   else:
      assert False

def schedModeEnumToShortStr( mode ):
   if mode == tacTxQueuePriority.priorityInvalid:
      return 'SP'
   elif mode == tacTxQueuePriority.priorityRoundRobin:
      return 'RR'
   else:
      return '-'

def schedModeEnumToStr( mode ):
   if mode == tacTxQueuePriority.priorityInvalid:
      return 'strictPriority'
   elif mode == tacTxQueuePriority.priorityRoundRobin:
      return 'roundRobin'
   else:
      return 'unspecified'


def isLagPort( name ):
   return name.startswith( "Port-Channel" )

def isSvi( name ):
   return name.startswith( "Vlan" )

def isVxlanIntf( name ):
   return name.startswith( "Vxlan" )

def isDot1qTunnel( intfInfo ):
   return intfInfo[ 'switchportMode' ] == 'dot1q-tunnel' or \
          intfInfo.get( 'selective-qinq', False )

def isCoppStaticClassSupported( qosAclConfig, qosSliceHwStatus, cmapName ):
   if qosAclConfig is None or qosSliceHwStatus is None:
      return True

   coppStaticClassSupported = False
   cpType = classMapCpType( qosAclConfig, cmapName )
   if cpType == tacClassMapCpType.cmapCpStatic:
      cpStaticType = classMapCpStaticType( qosAclConfig, cmapName )
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         if cpStaticType in sliceHwStatus.coppStaticClass.keys():
            coppStaticClassSupported = True
   return coppStaticClassSupported

# verifies if any Lag port is part of service policy
def isLagInServicePolicy( qosInputConfig, qosInputProfileConfig, pMapName ):
   spConfig = qosInputConfig.servicePolicyConfig
   for key, spObj in spConfig.items():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds.items():
            if isLagPort( intf[ 0 ] ):
               return True
   spConfig = qosInputProfileConfig.servicePolicyConfig
   for key, spObj in spConfig.items():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds.items():
            if isLagPort( intf[ 0 ] ):
               return True
   return False

# verifies if any SVI is part of service policy
def isSviInServicePolicy( qosInputConfig, qosInputProfileConfig, pMapName ):
   spConfig = qosInputConfig.servicePolicyConfig
   for key, spObj in spConfig.items():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds.items():
            if isSvi( intf[ 0 ] ):
               return True
   spConfig = qosInputProfileConfig.servicePolicyConfig
   for key, spObj in spConfig.items():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds.items():
            if isSvi( intf[ 0 ] ):
               return True
   return False

def isVxlanIntfInServicePolicy( qosInputConfig, qosInputProfileConfig, pMapName ):
   spConfig = qosInputConfig.servicePolicyConfig
   for key, spObj in spConfig.iteritems():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds:
            if isVxlanIntf( intf ):
               return True
   spConfig = qosInputProfileConfig.servicePolicyConfig
   for key, spObj in spConfig.iteritems():
      if key.pmapName == pMapName:
         for intf in spObj.intfIds:
            if isVxlanIntf( intf ):
               return True
   return False

# verifies if police is configured in given policy-map
def isPolicerInPmap( qosAclConfig, pMapName ):
   if qosAclConfig is None or \
      pMapName not in qosAclConfig.pmapType[ qosMapType ].pmap.members():
      return False
   pmap = qosAclConfig.pmapType[ qosMapType ].pmap[ pMapName ]
   if pmap.classActionDefault.policer:
      return True
   for cmap in pmap.classAction:
      if pmap.classAction[ cmap ].policer:
         return True
   return False

def isL2ParamsMatchInPmap( qosAclConfig, pMapName, checkFor=None ):
   """
   @brief@
   This function verifies if L2Params match is configured in policy-maps.
   If 'checkFor' is None, it will check for l2Params match option.
   Otherwise, it would check for that specific match values in the match option
   Allowed items in checkFor: [ 'vlan', 'cos', 'vlan inner' ]
   """
   if ( qosAclConfig is None or
        pMapName not in qosAclConfig.pmapType[ qosMapType ].pmap ):
      return False
   matchType = None
   matchIsPresent = False
   pmap = qosAclConfig.pmapType[ qosMapType ].pmap[ pMapName ]
   for cmapName in pmap.classAction:
      cmap = qosAclConfig.cmapType[ qosMapType ].cmap
      if cmapName in cmap and cmap[ cmapName ].match:
         matchType = cmap[ cmapName ].match.keys()[ 0 ]
         if matchType == 'matchL2Params':
            matchValue = cmap[ cmapName ].match[ matchType ]
            if checkFor:
               if ( ( 'vlan' in checkFor and matchValue.vlanValue is None ) or
                    ( 'cos' in checkFor and matchValue.cosValue is None ) or
                    ( 'vlan inner' in checkFor and matchValue.innerVlanValue
                      is None ) ):
                  break
            matchIsPresent = True
   return matchIsPresent

# returns true if pMapName can be applied to the VTI
def isPmapSupportedForVxlan( qosAclConfig, pMapName ):
   if qosAclConfig is None or \
      pMapName not in qosAclConfig.pmapType[ qosMapType ].pmap.members():
      return True
   pmap = qosAclConfig.pmapType[ qosMapType ].pmap[ pMapName ]
   for cmapName in pmap.classAction:
      cmap = qosAclConfig.cmapType[ qosMapType ].cmap
      cmapInfo = cmap.get( cmapName )
      if cmapInfo and cmapInfo.match.keys():
         matchType = cmapInfo.match.keys()[ 0 ]
         if matchType == 'matchL2Params':
            if cmap[ cmapName ].match[ matchType ].vlanValue:
               return True
         return False
   return True

# verify if match mpls traffic-class is configured in given policy-map
def isMplsTrafficClassMatchInPmap( qosAclConfig, pMapName ):
   if qosAclConfig is None or \
      pMapName not in qosAclConfig.pmapType[ qosMapType ].pmap.members():
      return False
   matchType = None
   pmap = qosAclConfig.pmapType[ qosMapType ].pmap[ pMapName ]
   for cmapName in pmap.classAction:
      cmap = qosAclConfig.cmapType[ qosMapType ].cmap
      if cmapName in cmap and cmap[ cmapName ].match.keys():
         matchType = cmap[ cmapName ].match.keys()[ 0 ]
         if matchType == 'matchMplsTrafficClass':
            return True
   return False

def isDefaultClass( mapType, clMap ):
   if isinstance( clMap, str ):
      cmapName = clMap
   else:
      cmapName = clMap.name
   if mapType == coppMapType:
      if cmapName == tacCMapNm.coppDefault:
         return True
   else:
      if cmapName == tacCMapNm.classDefault:
         return True
   return False

def defaultClassName( mapType ):
   if mapType == coppMapType:
      return tacCMapNm.coppDefault
   else:
      return tacCMapNm.classDefault

def coppRateShapeAndBandwidth( qosHwStatus ):
   if qosHwStatus.coppKbpsSupported:
      return( tacActionRateTypeKbps.min, tacActionRateTypeKbps.max )
   elif qosHwStatus.coppPpsSupported:
      return( tacActionRateTypePps.min, tacActionRateTypePps.max )
   else:
      return( 0, ( tacActionRateType.invalid - 1 ) )

def isPolicyShapeRateInvalid( rate ):
   return rate == tacActionRateType.invalid

def isPolicyShapeRateNotCfgd( rate ):
   return rate == tacActionRateType.noValue

def createPMapAndCMapTypes( cfg ):
   for t in mapTypes:
      if t not in cfg.pmapType.keys():
         cfg.newPmapType( t )

   for t in mapTypes:
      if t not in cfg.cmapType.keys():
         cfg.newCmapType( t )

def classMapCpStaticType( qosAclConfig, cmapName ):
   if cmapName in qosAclConfig.cmapType[ coppMapType ].cmap:
      return qosAclConfig.cmapType[ coppMapType ].cmap[ cmapName ].cpStaticType
   elif isDefaultClass( coppMapType, cmapName ) == True:
      return tacClassMapCpStaticType.cmapCpStaticDefault
   return ""

def classMapCpType( qosAclConfig, cmapName ):
   cmapType = coppMapType
   if cmapName in qosAclConfig.cmapType[ cmapType ].cmap:
      return qosAclConfig.cmapType[ cmapType ].cmap[ cmapName ].cpType
   elif isDefaultClass( coppMapType, cmapName ) == True:
      return tacClassMapCpType.cmapCpStatic
   return tacClassMapCpType.cmapCpDynamic

def coppStaticClassFromHwStatus( qosSliceHwStatus ):
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      # Iterate over the slices and find out the first slice
      # with static CoPP classes present
      if len( sliceHwStatus.coppStaticClass ):
         return sliceHwStatus.coppStaticClass
   return {} 

def coppStaticClassesByPrio( qosSliceHwStatus ):
   coppStaticClasses = coppStaticClassFromHwStatus( qosSliceHwStatus )
   prioAndTypes = map( None, [ i.prio for i in coppStaticClasses.values() ],
                       coppStaticClasses.keys() )
   prioAndTypes.sort( reverse=True )
   return prioAndTypes

def deleteClassPrio( classPrio, cmapName ):
   for classPrioChosen in classPrio.values():
      if classPrioChosen.cmapName == cmapName:
         deletionAt = classPrioChosen.index
         break
   numOfClasses = len( classPrio)
   for i in range( deletionAt, numOfClasses ):
      # shift values in class prio upwards by 1
      classPrio[ i ].cmapName = \
          classPrio[ i + 1 ].cmapName
   del classPrio[ numOfClasses ]

def classPresent( qosAclSliceHwStatus, key, mapType, clMap=None ):
   if mapType not in qosAclSliceHwStatus.pmapType:
      return False
   if not qosAclSliceHwStatus.pmapType[ mapType ].pmap:
      return False
   if key not in qosAclSliceHwStatus.pmapType[ mapType ].pmap:
      return False

   pmapHwStatus = qosAclSliceHwStatus.pmapType[ mapType ].pmap[ key ]
   if clMap:
      if isinstance( clMap, str ):
         if clMap not in pmapHwStatus.cmap:
            return False
      else:
         if clMap.name not in pmapHwStatus.cmap:
            return False
   return True

def coppDynamicClassPresent( qosAclHwStatus, qosAclSliceHwStatus,
                             key, mapType, clMap=None ):
   if qosAclHwStatus.coppDynamicClassSupported == False or \
      mapType == 'mapQos' :
      return False
   return classPresent( qosAclSliceHwStatus, key, mapType, clMap )

def coppPerPortClassPresent( qosAclHwStatus, qosAclSliceHwStatus,
                             key, mapType, pMap=None, clMap=None ):
   if qosAclHwStatus.intfCoppSupported == False or \
      mapType == 'mapQos' or pMap.name == tacPMapNm.coppName:
      return False
   return classPresent( qosAclSliceHwStatus, key, mapType, clMap )

def policyQosClassPresent( qosAclSliceHwStatus, key, mapType, clMap=None ):
   if mapType != 'mapQos':
      return False
   return classPresent( qosAclSliceHwStatus, key, mapType, clMap )

def programmedPolicyMapHwStatus( qosHwStatus, qosAclHwStatus, 
                                 qosSliceHwStatus, qosAclSliceHwStatus,
                                 pmapType, pmapName, 
                                 direction=None ):
   hwStatus = qosSliceHwStatus
   if qosAclSliceHwStatus:
      hwStatus = qosAclSliceHwStatus

   if hwStatus is not None:
      pmapTypePtr = hwStatus.pmapType.get( pmapType )
      if pmapTypePtr is not None:
         inPolicyStatus = None
         outPolicyStatus = None
         if direction == None:
            d = tacDirection.input
            key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", d, pmapName )
            inPolicyStatus = pmapTypePtr.pmap.get( key )
            d = tacDirection.output
            key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", d, pmapName )
            outPolicyStatus = pmapTypePtr.pmap.get( key )
         elif direction == tacDirection.input:
            d = direction
            key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", d, pmapName )
            inPolicyStatus = pmapTypePtr.pmap.get( key )
         elif direction == tacDirection.output:
            d = direction
            key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", d, pmapName )
            outPolicyStatus = pmapTypePtr.pmap.get( key )
         if inPolicyStatus is not None and outPolicyStatus is not None:
            if( ( inPolicyStatus.prgmStatus != \
                     tacPMapHwPrgmStatus.hwPrgmStatusSuccess ) or \
                   ( outPolicyStatus.prgmStatus != \
                        tacPMapHwPrgmStatus.hwPrgmStatusSuccess ) ):
               return tacPMapHwPrgmStatus.hwPrgmStatusFail
            else:
               return tacPMapHwPrgmStatus.hwPrgmStatusSuccess
         elif inPolicyStatus is not None:
            return inPolicyStatus.prgmStatus
         elif outPolicyStatus is not None:
            return outPolicyStatus.prgmStatus
   return tacPMapHwPrgmStatus.hwPrgmStatusInProgress

def programmedPMapHwStatus( qosHwStatus, qosAclHwStatus, 
                            qosSliceHwStatusDir,
                            qosAclSliceHwStatusDir,
                            pmapType, pmapName, 
                            direction=None ):
   if direction is None:
      direct = tacDirection.input
   else:
      direct = direction
   hwStatusDir = qosSliceHwStatusDir
   key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", direct, pmapName )

   for qosAclSliceHwStatus in qosAclSliceHwStatusDir.itervalues():
      if coppDynamicClassPresent( qosAclHwStatus, qosAclSliceHwStatus, 
                                  key, pmapType ) or pmapType == 'mapQos' or \
         ( pmapType == 'mapControlPlane' and 
           qosHwStatus.coppActionPolicerSupported and
           pmapName != tacPMapNm.coppName ) or pmapType == 'mapPdp':
         hwStatusDir = qosAclSliceHwStatusDir
         break
   
   nStatusSuccess = 0
   nStatusFail = 0
   nStatusInProgress = 0
   sliceHwStatus = None
   aclSliceHwStatus = None

   for hwStatus in hwStatusDir.itervalues():
      if hwStatusDir == qosSliceHwStatusDir:
         sliceHwStatus = hwStatus
         aclSliceHwStatus = None
      else:
         sliceHwStatus = None
         aclSliceHwStatus = hwStatus
      prgmdPMapHwStatus = programmedPolicyMapHwStatus( qosHwStatus, 
                                                       qosAclHwStatus,
                                                       sliceHwStatus,
                                                       aclSliceHwStatus,
                                                       pmapType, pmapName,
                                                       direction=direct )
      if prgmdPMapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
         nStatusSuccess += 1
      elif prgmdPMapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusFail:
         nStatusFail += 1
      elif prgmdPMapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusInProgress:
         nStatusInProgress += 1

   t0( " nStatusSuccess:", nStatusSuccess,
       " nStatusFail:", nStatusFail ,
       " nStatusInProgress :", nStatusInProgress )

   if nStatusFail != 0:
      return tacPMapHwPrgmStatus.hwPrgmStatusFail
   elif nStatusInProgress != 0:
      return tacPMapHwPrgmStatus.hwPrgmStatusInProgress
   elif nStatusSuccess != 0:
      return tacPMapHwPrgmStatus.hwPrgmStatusSuccess

def classNameFromCpStaticType( cpStaticType ):
   if cpStaticType == tacClassMapCpStaticType.cmapCpStaticInvalid:
      return None
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticBpdu:
      return tacCMapNm.coppBpdu
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticLacp:
      return tacCMapNm.coppLacp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticLldp:
      return tacCMapNm.coppLldp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticArp:
      return tacCMapNm.coppArp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticSelfIp:
      return tacCMapNm.coppSelfIp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticSelfIpTc6To7:
      return tacCMapNm.coppSelfIpTc6To7
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpmcRsvd:
      return tacCMapNm.coppIpmcRsvd
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpmcMiss:
      return tacCMapNm.coppIpmcMiss
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3DestMiss:
      return tacCMapNm.coppL3DestMiss
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIgmp:
      return tacCMapNm.coppIgmp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticTc3To5:
      return tacCMapNm.coppTc3To5
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticTc6To7:
      return tacCMapNm.coppTc6To7
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticGlean:
      return tacCMapNm.coppGlean
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3SlowPath:
      return tacCMapNm.coppL3SlowPath
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3Ttl1:
      return tacCMapNm.coppL3Ttl1
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticSflow:
      return tacCMapNm.coppSflow
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticAclLog:
      return tacCMapNm.coppAclLog
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticArpResolver:
      return tacCMapNm.coppArpResolver
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpLocking:
      return tacCMapNm.coppIpLocking
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticDrop:
      return tacCMapNm.coppDrop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticDefault:
      return tacCMapNm.coppDefault
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpUnicast:
      return tacCMapNm.coppIpUnicast
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpMc:
      return tacCMapNm.coppIpMc
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpBroadcast:
      return tacCMapNm.coppIpBroadcast
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticUnicastArp:
      return tacCMapNm.coppUnicastArp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3Ttl0:
      return tacCMapNm.coppL3Ttl0
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticIpv6Nd:
      return tacCMapNm.coppIpv6Nd
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL2Rsvd:
      return tacCMapNm.coppL2Rsvd
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticPimPtp:
      return tacCMapNm.coppPimPtp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticOspfIsis:
      return tacCMapNm.coppOspfIsis
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL2Broadcast:
      return tacCMapNm.coppL2Broadcast
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL2Unicast:
      return tacCMapNm.coppL2Unicast
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3LpmOverflow:
      return tacCMapNm.coppL3LpmOverflow
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticLinkLocal:
      return tacCMapNm.coppLinkLocal
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMcastSnoop:
      return tacCMapNm.coppMcastSnoop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticBgp:
      return tacCMapNm.coppBgp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticVrrp:
      return tacCMapNm.coppVrrp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMlag:
      return tacCMapNm.coppMlag
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticBfd:
      return tacCMapNm.coppBfd
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticPim:
      return tacCMapNm.coppPim
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticVxlanEncap:
      return tacCMapNm.coppVxlanEncap
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticVxlanVtepLearn:
      return tacCMapNm.coppVxlanVtepLearn
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMplsTtl01:
      return tacCMapNm.coppMplsTtl01
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMplsLabelMiss:
      return tacCMapNm.coppMplsLabelMiss
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticPtp:
      return tacCMapNm.coppPtp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticPtpSnoop:
      return tacCMapNm.coppPtpSnoop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMod:
      return tacCMapNm.coppMod
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMvrp:
      return tacCMapNm.coppMvrp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMtu:
      return tacCMapNm.coppMtu
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticEgressAclLog:
      return tacCMapNm.coppEgressAclLog
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticCvx:
      return tacCMapNm.coppCvx
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticCvxHeartbeat:
      return tacCMapNm.coppCvxHeartbeat
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMirroring:
      return tacCMapNm.coppMirroring
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMacLearn:
      return tacCMapNm.coppMacLearn
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticDot1xMba:
      return tacCMapNm.coppDot1xMba
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticCfm:
      return tacCMapNm.coppCfm
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticCfmSnoop:
      return tacCMapNm.coppCfmSnoop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticArpInspect:
      return tacCMapNm.coppArpInspect
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticMplsArpSuppress:
      return tacCMapNm.coppMplsArpSuppress
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticProtocolSnoop:
      return tacCMapNm.coppProtocolSnoop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticDefaultSnoop:
      return tacCMapNm.coppDefaultSnoop
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticNat:
      return tacCMapNm.coppNat
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticLdp:
      return tacCMapNm.coppLdp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticRsvp:
      return tacCMapNm.coppRsvp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticBfdPtp:
      return tacCMapNm.coppBfdPtp
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticL3Rsvd:
      return tacCMapNm.coppL3Rsvd
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticDot1xVxlan:
      return tacCMapNm.coppDot1xVxlan
   elif cpStaticType == tacClassMapCpStaticType.cmapCpStaticAcllogSflow:
      return tacCMapNm.coppAcllogSflow

   else:
      assert False, \
         'No matching className for cpStaticType {}'.format( cpStaticType )

def programmedPMapHwStatusFromEnum( pmapHwStatus ):
   if pmapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
      return 'Successful'
   elif pmapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusFail:
      return 'Failed'
   elif pmapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusInProgress:
      return 'InProgress'

def strFromEnum( matchCondition ):
   if matchCondition == 'matchConditionAny':
      return 'match-any'
   else:
      return 'unspecified'

def rateUnitFromEnum( rateUnit ):
   if rateUnit == 'rateUnitPps':
      return 'pps'
   elif rateUnit == 'rateUnitbps':
      return 'bps'
   elif rateUnit == 'rateUnitKbps':
      return 'kbps'
   elif rateUnit == 'rateUnitMbps':
      return 'mbps'

def matchFromEnum( match ):
   if match == 'matchIpAccessGroup':
      return 'ip access-group'
   elif match == 'matchIpv6AccessGroup':
      return 'ipv6 access-group'
   elif match == 'matchL2Params':
      return [ 'vlan', 'cos', 'vlan inner' ]
   elif match == 'matchDscpEcn':
      return [ 'dscp', 'ecn', 'dscpEcn' ]
   elif match == 'matchMplsTrafficClass':
      return 'mpls-traffic-class'
   elif match == 'matchMacAccessGroup':
      return 'mac access-group'

def matchToEnum( match ):
   if match == 'ip access-group':
      return 'matchIpAccessGroup'
   elif match == 'ipv6 access-group':
      return 'matchIpv6AccessGroup'
   elif match == 'vlan' or match == 'cos' or match == 'vlan inner':
      return 'matchL2Params'
   elif match == 'dscpEcn' or match == 'dscp' or match == 'ecn':
      return 'matchDscpEcn'
   elif match == 'mpls-traffic-class':
      return 'matchMplsTrafficClass'
   elif match == 'mac access-group':
      return 'matchMacAccessGroup'

def mapTypeToEnum( mapType ):
   if mapType == 'control-plane':
      return coppMapType
   elif mapType == 'copp':
      return coppMapTypeCli
   elif mapType == 'qos':
      return qosMapType
   elif mapType == 'quality-of-service':
      return qosMapTypeCli
   elif mapType == 'pdp':
      return pdpMapType
   else:
      assert False, "Undefined mapType"

def mapTypeFromEnum( mapType ):
   if mapType == coppMapType:
      return 'control-plane'
   elif mapType == coppMapTypeCli:
      return 'copp'
   elif mapType == qosMapType:
      return 'qos'
   elif mapType == qosMapTypeCli:
      return 'quality-of-service'
   elif mapType == pdpMapType:
      return 'pdp'
   else:
      assert False, "Undefined mapType"

def directionToEnum( direction ):
   if direction == 'input':
      return 'input'
   elif direction == 'output':
      return 'output'

def directionFromEnum( direction ):
   if direction == 'input':
      return 'input'
   elif direction == 'output':
      return 'output'

def actionToEnum( action ):
   if action == 'cos':
      return tacActionType.actionSetCos
   elif action == 'dscp':
      return tacActionType.actionSetDscp
   elif action == 'drop':
      return tacActionType.actionSetDrop
   elif action == 'shape':
      return tacActionType.actionSetShape
   elif action == 'bandwidth':
      return tacActionType.actionSetBandwidth
   elif action == 'traffic-class':
      return tacActionType.actionSetTc
   elif action == 'drop-precedence':
      return tacActionType.actionSetDropPrecedence

def actionFromEnum( action ):
   if action == tacActionType.actionSetCos:
      return 'cos'
   elif action == tacActionType.actionSetDscp:
      return 'dscp'
   elif action == tacActionType.actionSetDrop:
      return 'drop'
   elif action == tacActionType.actionSetShape:
      return 'shape'
   elif action == tacActionType.actionSetBandwidth:
      return 'bandwidth'
   elif action == tacActionType.actionSetTc:
      return 'traffic-class'
   elif action == tacActionType.actionSetDropPrecedence:
      return 'drop-precedence'

def burstUnitFromEnum( burst ):
   if burst == tacBurstUnit.burstUnitBytes:
      return 'bytes'
   elif burst == tacBurstUnit.burstUnitKBytes:
      return 'kbytes'
   elif burst == tacBurstUnit.burstUnitMBytes:
      return 'mbytes'
   elif burst == tacBurstUnit.burstUnitPackets:
      return 'packets'

def burstUnitSymbolFromEnum( burst ):
   if burst == tacBurstUnit.burstUnitBytes:
      return 'B'
   elif burst == tacBurstUnit.burstUnitKBytes:
      return 'KB'
   elif burst == tacBurstUnit.burstUnitMBytes:
      return 'MB'
   else:
      assert False, 'No symbol for burst unit %s' % ( burst )

def matchOptionFromEnum( matchOption ):
   if matchOption == tacMatchOption.matchIpAccessGroup:
      return 'ip'
   elif matchOption == tacMatchOption.matchIpv6AccessGroup:
      return 'ipv6'
   elif matchOption == tacMatchOption.matchL2Params:
      return [ 'vlan', 'cos', 'vlan inner' ]
   elif matchOption == tacMatchOption.matchDscpEcn:
      return [ 'dscp', 'ecn', 'dscpEcn' ]
   elif matchOption == tacMatchOption.matchMplsTrafficClass:
      return 'mpls-traffic-class'
   elif matchOption == tacMatchOption.matchMacAccessGroup:
      return 'mac'
   else:
      return None

def matchOptionToEnum( matchOption ):
   if matchOption == 'ip':
      return tacMatchOption.matchIpAccessGroup
   elif matchOption == 'ipv6':
      return tacMatchOption.matchIpv6AccessGroup
   elif matchOption == 'l2Params':
      return tacMatchOption.matchL2Params
   elif matchOption == 'dscpEcn' or matchOption == 'dscp' or \
        matchOption == 'ecn':
      return tacMatchOption.matchDscpEcn
   elif matchOption == 'mpls-traffic-class':
      return tacMatchOption.matchMplsTrafficClass
   elif matchOption == 'mac':
      return tacMatchOption.matchMacAccessGroup
   elif matchOption in builtInClassMapNames:
      return tacMatchOption.matchBuiltIn
   else:
      return None

matchOptionsList = [ 'matchIpAccessGroup',
                     'matchIpv6AccessGroup',
                     'matchL2Params',
                     'matchDscpEcn',
                     'matchMplsTrafficClass',
                     'matchMacAccessGroup',
                     'matchBuiltIn' ]

def matchOptionOtherToEnum( matchOption ):
   try:
      otherOptionsList = list( matchOptionsList )
      otherOptionsList.remove( matchOption )
      return otherOptionsList
   except KeyError:
      return None

def builtInClassMatchTypeToEnum( matchType ):
   try:
      return builtInClassMapNames[ matchType ][ 0 ]
   except KeyError:
      return None

# ------------------------------------------
#   Policy map routines
# ------------------------------------------
def pmapIs( cfg, pmapName, pmapType ):
   pmType = cfg.pmapType[ pmapType ]
   pmap = pmType.newPmap( pmapName, pmType.type )
   return pmap

def pmapDel( cfg, pmapName, pmapType ):
   del cfg.pmapType[ pmapType ].pmap[ pmapName ]

def pmapClassActionAndPrioIs( pmap, cmapName, mapType, classPrio, addAction=True ):
   if addAction == True:
      # Create the class action
      pmap.classAction.newMember( cmapName )
   numberOfClasses = len( classPrio )
   i = numberOfClasses + 1
   cmapPresent = False
   for prio in classPrio:
      if classPrio[ prio ].cmapName == cmapName:
         cmapPresent = True
         break
   if not cmapPresent:
      classPrio.newMember( i )
      classPrio[ i ].cmapName = cmapName

class Policer( object ):
   def __init__( self, cir, cirUnit, bc, bcUnit,
                 pir=0, pirUnit=None, be=0, beUnit=None, policerName=None,
                 yellowAction=True ):
      self.cir = cir
      self.cirUnit = cirUnit
      self.bc = bc
      self.bcUnit = bcUnit
      self.policerMode = 'trTCM' if pir else 'committed'
      self.pir = pir
      self.pirUnit = pirUnit
      self.be = be
      self.beUnit = beUnit
      self.policerName = policerName
      self.redAction = [ tacActionType.actionSetDrop, 1 ]
      self.yellowAction = None
      if yellowAction:
         action = random.choice( [ [ tacActionType.actionSetDropPrecedence, None ],
                                 [ tacActionType.actionSetDscp,
                                    random.randint( 0, 64 ) ] ] )
         self.yellowAction = ( action if self.policerMode == 'trTCM' else
                               self.redAction )
   def _setUnits( self, classAction ):
      classAction.policer.cirUnit = self.cirUnit
      classAction.policer.bcUnit = self.bcUnit
      if self.policerMode == 'trTCM':
         classAction.policer.pirUnit = self.pirUnit
         classAction.policer.beUnit = self.beUnit

   def _setPolicerActions( self, classAction ):
      if self.yellowAction is not None:
         classAction.policer.yellowActions.newMember( self.yellowAction[ 0 ] )
         if self.yellowAction[ 1 ]:
            classAction.policer.yellowActions[ self.yellowAction[ 0 ] ].value = \
                self.yellowAction[ 1 ]
      classAction.policer.redActions.newMember( self.redAction[ 0 ] )
      classAction.policer.redActions[ self.redAction[ 0 ] ].value = 1

   def setPolicer( self, classAction ):
      policerCfgName = classAction.name
      if self.policerName is not None:
         policerCfgName = self.policerName
      classAction.policer = ( policerCfgName, self.cir, self.bc, self.pir, self.be )
      classAction.policer.named = self.policerName is not None
      self._setUnits( classAction )
      self._setPolicerActions( classAction )

   def __repr__( self ):
      policerStr = ( "CIR: %s( %s ), BC: %s( %s )" % ( self.cir, self.cirUnit,
                                                       self.bc, self.bcUnit ) )
      if self.policerName is not None:
         policerStr = "NAME: " + self.policerName + "\n" + policerStr
      if self.policerMode == 'trTCM':
         policerStr += ( "\nPIR: %s( %s ), BE: %s( %s )" % ( self.pir, self.pirUnit,
                                                             self.be, self.beUnit ) )
      return policerStr

def policerIs( pmap, mapType, cmapName, cir, cirUnit, bc, bcUnit,
               pir=0, pirUnit=None, be=0, beUnit=None, policerName=None,
               yellowAction=True ):
   if ( mapType == qosMapType and cmapName == tacCMapNm.classDefault ) or \
      ( mapType == coppMapType and cmapName == tacCMapNm.coppDefault ):
      classAction = pmap.classActionDefault
   elif cmapName not in pmap.classAction.keys():
      classAction = pmap.classAction.newMember( cmapName )
   else:
      classAction = pmap.classAction[ cmapName ]
   
   policerObj = Policer( cir, cirUnit, bc, bcUnit, pir, pirUnit, be, beUnit,
                         policerName=policerName, yellowAction=yellowAction )
   policerObj.setPolicer( classAction )
   t0( "Created policer.\n class:", cmapName, "\n", str( policerObj ) )

def policerDel( pmap, mapType, cmapName ):
   if ( mapType == qosMapType and cmapName == tacCMapNm.classDefault ) or \
      ( mapType == coppMapType and cmapName == tacCMapNm.coppDefault ):
      pmap.classActionDefault.policer = None
   else:
      pmap.classAction[ cmapName ].policer = None

# Basic checks to verify that all classes in src are in dst
# and vice versa.
def identicalPolicyMap( src, dst ):
   def classActionsMatch( ca1, ca2 ):
      if ( ca1 and not ca2 ) or \
             ( ca2 and not ca1 ):
         # Return false if either is None.
         return False
      ca1Keys = set( ca1.policyAction )
      ca2Keys = set( ca2.policyAction )
      if ca1Keys != ca2Keys:
         return False
      for actionType in ca1.policyAction:
         if actionType not in ca2.policyAction:
            return False
         if actionType in \
                [ tacActionType.actionSetShape, tacActionType.actionSetBandwidth ]:
            if ca1.policyAction[ actionType ].rate.rateUnit != \
                   ca2.policyAction[ actionType ].rate.rateUnit:
               return False
            if ca1.policyAction[ actionType ].rate.val != \
                   ca2.policyAction[ actionType ].rate.val:
               return False
         elif actionType in [ tacActionType.actionSetDscp, 
                              tacActionType.actionSetCos, 
                              tacActionType.actionSetDrop, 
                              tacActionType.actionSetDropPrecedence, 
                              tacActionType.actionSetTc ]:
            if ca1.policyAction[ actionType ].value != \
                   ca2.policyAction[ actionType ].value:
               return False

      if ca1.policer is None and ca2.policer:
         return False
      if ca1.policer and ca2.policer is None:
         return False
      if ca1.policer and ca2.policer:
         if ( ca1.policer.cir != ca2.policer.cir or
              ca1.policer.cirUnit != ca2.policer.cirUnit or
              ca1.policer.bc != ca2.policer.bc or
              ca1.policer.bcUnit != ca2.policer.bcUnit or
              ca1.policer.pir != ca2.policer.pir or
              ca1.policer.pirUnit != ca2.policer.pirUnit or
              ca1.policer.be != ca2.policer.be or
              ca1.policer.beUnit != ca2.policer.beUnit or
              ca1.policer.named != ca2.policer.named or
              ca1.policer.name != ca2.policer.name ):
            return False
      return True

   if src.shared != dst.shared:
      return False
   srcKeys = src.classAction.keys()
   dstKeys = dst.classAction.keys()
   if sorted( srcKeys ) != sorted( dstKeys ):
      return False
   # Number of dynamic classes is same
   if len( src.classPrio ) != len( dst.classPrio ):
      return False
   # Class Priorities are unchanged.
   for p, classPrio in sorted( src.classPrio.iteritems() ):
      if classPrio.cmapName != dst.classPrio[ p ].cmapName:
         return False
   # Class's action parameters are unchanged ( both static/dynamic classes )
   for key, srcCA in src.classAction.iteritems():
      dstCA = dst.classAction.get( key )
      if not dstCA or not classActionsMatch( srcCA, dstCA ):
         return False
   # Separate check for classDefault
   srcClassDef = src.classActionDefault
   dstClassDef = dst.classActionDefault
   if not classActionsMatch( srcClassDef, dstClassDef ):
      return False

   return True

# Returns a list of policy map types supported for copy operation
# Currently only 'qos' type is supported
def copyablePMapTypes():
   return [ qosMapType ]

# ------------------------------------------
#   Copp Policy map routines
# ------------------------------------------
def coppStaticClass( cmapName ):
   if cmapName in coppStaticClassName:
      return True
   return False
   
def coppPMapClassDefaultIs( pmap, cmapType, static=False, addAction=False,
                            rateUnit=tacRateUnit.rateUnitInvalid ):
   cmapName = tacCMapNm.coppDefault
   if pmap.classDefault == None:
      pmap.classDefault = ( cmapName, cmapType )
      match = pmap.classDefault.match.newMember( 'matchIpAccessGroup' )
      match.strValue = 'default'
   cmDef = pmap.classDefault
   if static:
      cmDef.cpType = 'cmapCpStatic'
      cmDef.cpStaticType = 'cmapCpStaticDefault'
   if pmap.classActionDefault == None:
      pmap.classActionDefault = ( cmapName, )
   if addAction:
      actionShapeIs( pmap, cmapName, cmapType, 
                     tacActionRateType.invalid, rateUnit )
      actionBandwidthIs( pmap, cmapName, cmapType,
                         tacActionRateType.invalid, rateUnit )

def coppPMapDynamicClassIs( pmap, cmapName, mapType ):
   pmapClassActionAndPrioIs( pmap, cmapName, mapType, pmap.classPrio )

def coppPMapStaticClassIs( pmap, cmapName, mapType ):
   pmapClassActionAndPrioIs( pmap, cmapName, mapType, pmap.coppStaticClassPrio )

def coppStaticClassIs( cfg, pmap, cmapName, cpStaticType, maximum, minimum, 
                       rateUnit=tacRateUnit.rateUnitInvalid ):
   cmapIs( cfg, cmapName, coppMapType, 'cmapCpStatic', cpStaticType )
   coppPMapStaticClassIs( pmap, cmapName, coppMapType )
   actionShapeIs( pmap, cmapName, coppMapType, maximum, rateUnit )
   actionBandwidthIs( pmap, cmapName, coppMapType, minimum, rateUnit )

def coppDynamicClassIs( cfg, pmap, cmapName, aclName, maximum, minimum, 
                        rateUnit=tacRateUnit.rateUnitInvalid, aclType='ip', 
                        createCMap=True ):
   if createCMap:
      cmapIs( cfg, cmapName, coppMapType, 'cmapCpDynamic', 'cmapCpStaticInvalid' )

   matchOpt = matchOptionToEnum( aclType )
   matchIs( cfg, cmapName, coppMapType, matchOpt, aclName )
   coppPMapDynamicClassIs( pmap, cmapName, coppMapType )
   if maximum == None:
      maximum = tacActionRateType.noValue
   if minimum == None:
      minimum = tacActionRateType.noValue
   actionShapeIs( pmap, cmapName, coppMapType, maximum, rateUnit )
   actionBandwidthIs( pmap, cmapName, coppMapType, minimum, rateUnit )
   
# ------------------------------------------
#   Class map routines
# ------------------------------------------
def cmapIs( cfg, cmapName, cmapType, cpType, cpStaticType ):
   cmType = cfg.cmapType[ cmapType ]
   cm = cmType.newCmap( cmapName, cmType.type )
   cm.cpType = cpType
   cm.cpStaticType = cpStaticType
   cm.matchCondition = 'matchConditionAny'
   return cm

# ------------------------------------------
#   Class map routines
# ------------------------------------------
def cmapDel( cfg, cmapName, cmapType ):
   del cfg.cmapType[ cmapType ].cmap[ cmapName ]

# ------------------------------------------
#   Action routines
# ------------------------------------------
def actionShapeIs( pmap, cmapName, mapType, shapeVal, rateUnit ):
   assert( shapeVal != None )
   if isDefaultClass( mapType, cmapName ) == True:
      classAction = pmap.classActionDefault
   else:
      classAction = pmap.classAction[ cmapName ]
   assert classAction != None
   actions = classAction.policyAction
   if tacActionType.actionSetShape not in actions:
      actions.newMember( tacActionType.actionSetShape )
   shape = actions[ tacActionType.actionSetShape ]
   if shape.rate == None:
      shape.rate = ()
   rate = shape.rate
   rate.val = shapeVal
   rate.rateUnit = rateUnit

def actionBandwidthIs( pmap, cmapName, mapType, bandwidthVal, rateUnit ):
   assert( bandwidthVal != None )
   if isDefaultClass( mapType, cmapName ) == True:
      classAction = pmap.classActionDefault
   else:
      classAction = pmap.classAction[ cmapName ]
   assert classAction != None
   actions = classAction.policyAction
   if tacActionType.actionSetBandwidth not in actions:
      actions.newMember( tacActionType.actionSetBandwidth )
   bandwidth = actions[ tacActionType.actionSetBandwidth ]
   if bandwidth.rate == None:
      bandwidth.rate = ()
   rate = bandwidth.rate
   rate.val = bandwidthVal
   rate.rateUnit = rateUnit

# ------------------------------------------
#   Match routines
# ------------------------------------------
def matchIs( cfg, cmapName, mapType, matchType, matchValue ):
   cm = cfg.cmapType[ mapType ].cmap[ cmapName ]
   assert cm != None
   m = cm.match.newMember( matchType )
   if matchType == 'matchIpAccessGroup'or \
         matchType == 'matchIpv6AccessGroup' or \
         matchType == 'matchL2Params' or \
         matchType == 'matchDscpEcn' or \
         matchType == 'matchMplsTrafficClass' or \
         matchType == 'matchMacAccessGroup':
      m.strValue = matchValue
   return m

# ------------------------------------------
#   Service policy routines
# ------------------------------------------
def servicePolicyIs( cfg, mapType, direction, pmapName, no=False ):
   key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
   if not no:
      if key not in cfg.servicePolicyConfig:
         cfg.servicePolicyConfig.newMember( key )
   else:
      if key in cfg.servicePolicyConfig:
         del cfg.servicePolicyConfig[ key ]

# ------------------------------------------
#   Pdp Service policy routines
# ------------------------------------------
def builtInClassIs( cfg, cmapName ):
   cmapType = cfg.cmapType[ pdpMapType ]
   cm = cmapType.newCmap( cmapName, cmapType.type )
   cm.cpType = 'cmapCpNone'
   matchOption = matchOptionToEnum( cmapName )
   cm.match.newMember( matchOption )
   cm.match[ matchOption ].builtInClassMatchValue = builtInClassMatchTypeToEnum(
      cmapName )
   cm.match[ matchOption ].strValue = cmapName
   return cm

def createBuiltInClasses( cfg ):
   for builtInClassName in builtInClassMapNames.keys():
      builtInClassIs( cfg, builtInClassName )
      
# ------------------------------------------
#   Copp Service policy routines
# ------------------------------------------
def coppServicePolicyIs( cfg, no=False ):
   servicePolicyIs( cfg, coppMapType, 'input', tacPMapNm.coppName, no=no )

def createDefaultStaticClass( cfg, pmap ):
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppBpdu, 'cmapCpStaticBpdu', 
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppLldp, 'cmapCpStaticLldp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppLacp, 'cmapCpStaticLacp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppArp, 'cmapCpStaticArp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3Ttl1, 'cmapCpStaticL3Ttl1',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3SlowPath, 'cmapCpStaticL3SlowPath',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3DestMiss, 'cmapCpStaticL3DestMiss',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIgmp, 'cmapCpStaticIgmp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpmcRsvd, 'cmapCpStaticIpmcRsvd',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpmcMiss, 'cmapCpStaticIpmcMiss',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppSelfIpTc6To7, 
                      'cmapCpStaticSelfIpTc6To7',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppSelfIp, 'cmapCpStaticSelfIp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppTc6To7, 'cmapCpStaticTc6To7',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppTc3To5, 'cmapCpStaticTc3To5',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppGlean, 'cmapCpStaticGlean',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppSflow, 'cmapCpStaticSflow',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppAclLog, 'cmapCpStaticAclLog',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppArpResolver, 
                      'cmapCpStaticArpResolver',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpLocking, 
                      'cmapCpStaticIpLocking',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppDrop, 'cmapCpStaticDrop',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpBroadcast,
                      'cmapCpStaticIpBroadcast',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpUnicast, 'cmapCpStaticIpUnicast',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpMc, 'cmapCpStaticIpMc',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppUnicastArp, 'cmapCpStaticUnicastArp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3Ttl0, 'cmapCpStaticL3Ttl0',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppIpv6Nd, 'cmapCpStaticIpv6Nd',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL2Broadcast, 
                      'cmapCpStaticL2Broadcast',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL2Unicast, 
                      'cmapCpStaticL2Unicast',
                      tacActionRateType.invalid, tacActionRateType.invalid ) 

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3LpmOverflow,
                      'cmapCpStaticL3LpmOverflow',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL2Rsvd, 'cmapCpStaticL2Rsvd',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppPimPtp, 'cmapCpStaticPimPtp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppOspfIsis, 'cmapCpStaticOspfIsis',
                      tacActionRateType.invalid, tacActionRateType.invalid )
  
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppLinkLocal, 'cmapCpStaticLinkLocal',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMcastSnoop, 
                      'cmapCpStaticMcastSnoop',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppBgp, 'cmapCpStaticBgp',
                      tacActionRateType.invalid, tacActionRateType.invalid ) 

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppVrrp, 'cmapCpStaticVrrp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMlag, 'cmapCpStaticMlag',
                      tacActionRateType.invalid, tacActionRateType.invalid ) 
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppBfd, 'cmapCpStaticBfd',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppPim, 'cmapCpStaticPim',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppVxlanEncap, 
                      'cmapCpStaticVxlanEncap',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMplsTtl01, 
                      'cmapCpStaticMplsTtl01',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMplsLabelMiss, 
                      'cmapCpStaticMplsLabelMiss',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMtu, 
                      'cmapCpStaticMtu',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppVxlanVtepLearn,
                      'cmapCpStaticVxlanVtepLearn',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppPtp, 'cmapCpStaticPtp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppPtpSnoop, 'cmapCpStaticPtpSnoop',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMod, 'cmapCpStaticMod',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMvrp, 'cmapCpStaticMvrp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppEgressAclLog, 
                      'cmapCpStaticEgressAclLog',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppCvx, 'cmapCpStaticCvx',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppCvxHeartbeat,
                      'cmapCpStaticCvxHeartbeat',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMirroring, 'cmapCpStaticMirroring',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMacLearn, 'cmapCpStaticMacLearn',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppDot1xMba, 'cmapCpStaticDot1xMba',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppCfm, 'cmapCpStaticCfm',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppCfmSnoop, 'cmapCpStaticCfmSnoop',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppArpInspect, 'cmapCpStaticArpInspect',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppMplsArpSuppress,
                      'cmapCpStaticMplsArpSuppress',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppProtocolSnoop,
                      'cmapCpStaticProtocolSnoop',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppDefaultSnoop,
                      'cmapCpStaticDefaultSnoop', 
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppNat,
                      'cmapCpStaticNat',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppLdp, 'cmapCpStaticLdp',
                      tacActionRateType.invalid, tacActionRateType.invalid )
   
   coppStaticClassIs( cfg, pmap, tacCMapNm.coppRsvp, 'cmapCpStaticRsvp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppBfdPtp, 'cmapCpStaticBfdPtp',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppL3Rsvd, 'cmapCpStaticL3Rsvd',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppDot1xVxlan, 'cmapCpStaticDot1xVxlan',
                      tacActionRateType.invalid, tacActionRateType.invalid )

   coppStaticClassIs( cfg, pmap, tacCMapNm.coppAcllogSflow,
                      'cmapCpStaticAcllogSflow',
                      tacActionRateType.invalid, tacActionRateType.invalid )


   coppPMapClassDefaultIs( pmap, coppMapType, static=True, addAction=True )


valLenToSIUnitPrefix = { 
                        #  (Len, unitsPrefixInSI )
                           ( 9, 'G'),
                           ( 6, 'M'),
                           ( 3, 'k'),
                       }

def cliConvertValToSIUnit( value, reqPrecision ):
   pfx = ''
   length = len( str( value ) )
   value = float( value )
   combinedLen = length + reqPrecision 

   for valLen, unitsPfx in \
       sorted( valLenToSIUnitPrefix, key=itemgetter(0), reverse=True ):
      if combinedLen > valLen:
         pfx = unitsPfx
         value = value / 10 ** valLen
         break

   valueStr = '{0:.1f}'.format( value ) #formatting the value to one decimal
   return valueStr, pfx
      
def cliConvertShapeRateToSIUnit( shapeRate, reqPrecision=0 ):
   if shapeRate == invalidShapeRate:
      return '-', ''
   else:
      if shapeRate.unit == tacShapeRateUnit.shapeRateKbps:
         ppsOrBpsStr  = 'bps'
         shapeRateVal = shapeRate.rate * 1000
      else:
         ppsOrBpsStr  = 'pps'
         shapeRateVal = shapeRate.rate

      shapeRateValStr, pfx = cliConvertValToSIUnit( shapeRateVal, reqPrecision) 
      return shapeRateValStr, pfx + ppsOrBpsStr

def cliConvertBothShapeRatesToSIUnit( cfgShapeRate, operShapeRate ):
   unitsStr = 'err'
   cfgPrecision = 0
   operPrecision = 0

   if operShapeRate != invalidShapeRate and cfgShapeRate != invalidShapeRate:
      operShapeRateValLen = len( str( operShapeRate.rate ) ) 
      cfgShapeRateValLen = len( str( cfgShapeRate.rate ) )
      if operShapeRateValLen > cfgShapeRateValLen:
         cfgPrecision = operShapeRateValLen - cfgShapeRateValLen
      else:
         operPrecision = cfgShapeRateValLen - operShapeRateValLen

   cfgShapeRateStr, cfgShapeRateSIUnits = \
      cliConvertShapeRateToSIUnit( cfgShapeRate, cfgPrecision )

   operShapeRateStr, operShapeRateSIUnits = \
      cliConvertShapeRateToSIUnit( operShapeRate, operPrecision )

   if ( operShapeRateSIUnits == cfgShapeRateSIUnits ) or \
         ( operShapeRate == invalidShapeRate and cfgShapeRate != invalidShapeRate ):
      unitsStr = cfgShapeRateSIUnits
   elif ( operShapeRate != invalidShapeRate and cfgShapeRate == invalidShapeRate ):
      # Oper is valid but config is invalid, can happen in port-channel's 
      # member port case etc
      unitsStr = operShapeRateSIUnits

   return cfgShapeRateStr, operShapeRateStr, unitsStr

def cliConvertGuaranteedBwToSIUnit( guaranteedBw, reqPrecision=0 ):
   if guaranteedBw == invalidGuaranteedBw:
      return '-', ''
   else:
      if guaranteedBw.unit == tacGuaranteedBwUnit.guaranteedBwKbps:
         ppsOrBpsOrPercentStr  = 'bps'
         guaranteedBwVal = guaranteedBw.bw * 1000
      elif guaranteedBw.unit == tacGuaranteedBwUnit.guaranteedBwPps:
         ppsOrBpsOrPercentStr  = 'pps'
         guaranteedBwVal = guaranteedBw.bw
      else:
         ppsOrBpsOrPercentStr = '%'
         guaranteedBwVal = guaranteedBw.bw

      guaranteedBwValStr, pfx = cliConvertValToSIUnit( guaranteedBwVal, 
                                                       reqPrecision )
      return guaranteedBwValStr, pfx + ppsOrBpsOrPercentStr

def cliConvertBothGuaranteedBwsToSIUnit( cfgGuaranteedBw, operGuaranteedBw ):
   unitsStr = 'err'
   cfgPrecision = 0
   operPrecision = 0

   if operGuaranteedBw != invalidGuaranteedBw and \
         cfgGuaranteedBw != invalidGuaranteedBw:
      operGuaranteedBwValLen = len( str( operGuaranteedBw.bw ) ) 
      cfgGuaranteedBwValLen = len( str( cfgGuaranteedBw.bw ) )
      if operGuaranteedBwValLen > cfgGuaranteedBwValLen:
         cfgPrecision = operGuaranteedBwValLen - cfgGuaranteedBwValLen
      else:
         operPrecision = cfgGuaranteedBwValLen - operGuaranteedBwValLen

   cfgGuaranteedBwStr, cfgGuaranteedBwSIUnits = \
     cliConvertGuaranteedBwToSIUnit( cfgGuaranteedBw, cfgPrecision )

   operGuaranteedBwStr, operGuaranteedBwSIUnits = \
     cliConvertGuaranteedBwToSIUnit( operGuaranteedBw, operPrecision )

   if ( operGuaranteedBwSIUnits == cfgGuaranteedBwSIUnits ) or \
         ( operGuaranteedBw == invalidGuaranteedBw and \
         cfgGuaranteedBw != invalidGuaranteedBw ):
      unitsStr = cfgGuaranteedBwSIUnits
   elif operGuaranteedBw != invalidGuaranteedBw and \
         cfgGuaranteedBw == invalidGuaranteedBw:
      # Oper is valid but config is invalid, can happen in port-channel's 
      # member port case etc
      unitsStr = operGuaranteedBwSIUnits

   return cfgGuaranteedBwStr, operGuaranteedBwStr, unitsStr

def convertTrustModeModelToStr( trustmode ):
   if trustmode == 'dscpTrusted':
      return 'DSCP'
   elif trustmode == 'cosTrusted':
      return 'COS'
   else:
      return 'UNTRUSTED'

def convertTrustModeModelFromStr( trustmode ):
   if trustmode == 'DSCP':
      return 'dscpTrusted'
   elif trustmode == 'COS':
      return 'cosTrusted'
   else:
      return 'unTrusted'

def compareAttributeToDefault( qosHwStatus, attribute, value, invalid ):
   hwValue = getattr( qosHwStatus, attribute ) 
   if value == invalid or value == hwValue: 
      return str( hwValue ).upper()
   return str( value ).upper() 

def convertShapeValueToPoliceValue( shape ):
   # Per port policer cir is half of shape rate in pps, bc is 1/5 of cir
   # in packet. If shape rate is invalid or no-value, use 50000 pps as
   # default. Also, converted policer is in pps unit.  return ( cir,
   # cirUnit, bc, bcUnit )
   cir = 100000
   if shape.rate.val != tacActionRateType.invalid and \
          shape.rate.val != tacActionRateType.noValue and \
          shape.rate.rateUnit != tacRateUnit.rateUnitInvalid:
      # Convert bps to pps assuming 64B packet, plus 20B overhead, hence:
      #    1 bps = 1 / ( 84 * 8 ) = 1 / 672 pps
      # Also, make sure min cir is 1 pps
      if shape.rate.rateUnit == tacRateUnit.rateUnitbps:
         cir = ( shape.rate.val + 671 ) / 672
      elif shape.rate.rateUnit == tacRateUnit.rateUnitKbps:
         cir = ( shape.rate.val * 1000 + 671 ) / 672
      elif shape.rate.rateUnit == tacRateUnit.rateUnitMbps:
         cir = ( shape.rate.val * 1000000 + 671 ) / 672
      else:
         assert shape.rate.rateUnit == tacRateUnit.rateUnitPps
         cir = shape.rate.val

   return ( ( cir + 1 ) / 2, tacRateUnit.rateUnitPps,
            ( cir + 4 ) / 5, tacBurstUnit.burstUnitPackets )

def setupPdpHwConfig( qosInputConfig, intfs ):
   pdpHwConfig = qosInputConfig.newEntity( 'Qos::Input::Config', 'pdpHwConfig' )
   pdpHwConfig.agentName = 'pdpHwConfig'
   pdpHwConfig.configPriority = 90
   pdpHwConfig.intfConfigPriority = 90
   defaultPdpSpKey = Tac.newInstance( 'Qos::ServicePolicyKey', pdpMapType,
                                      tacDirection.input,
                                      tacPMapNm.defaultPdpPmapName )
   defaultSpConfig = pdpHwConfig.servicePolicyConfig.newMember( defaultPdpSpKey )

   if not hasattr( intfs, '__iter__' ):
      intfs = [ intfs ]
   for intf in intfs:
      defaultSpConfig.intfIds[ intf ] = True
   Tac.runActivities( 1 )
   return pdpHwConfig

def createDefaultPdpClassConfig( pmap, className, priority, cir, bc, pir,
                                 be, cirUnit=tacRateUnit.rateUnitKbps,
                                 bcUnit=tacBurstUnit.burstUnitKBytes,
                                 yellowAction=False, count=False ):
   pmap.classPrio.newMember( priority )
   pmap.classPrio[ priority ].cmapName = className
   pmap.classAction.newMember( className )
   if count:
      # Count action
      pmap.classAction[ className ].count = True
   else:
      # Police action
      policerIs( pmap, pdpMapType, className, cir, cirUnit, bc,
                 bcUnit, pir=pir, be=be, yellowAction=yellowAction )

def createDefaultPdpDefaultClassConfig( pmap, matchOption, matchStr,
                                        count=True ):
   pmap.classDefault = ( tacCMapNm.classDefault, pdpMapType )
   match = pmap.classDefault.match.newMember( matchOption )
   match.strValue = matchStr
   pmap.classActionDefault = ( tacCMapNm.classDefault, )
   pmap.classActionDefault.count = count

# Given a string range in the form of 'x-y' or single value, 'x',
# it will parse return set containing min value and max value
# of the range in integer type
def rangeToMinMaxValue( rangeStr ):
   parts = [ int( p ) for p in rangeStr.split( '-' ) ]
   if len( parts ) == 1:
      lbound = parts[ 0 ]
      ubound = lbound
   else:
      lbound = parts[ 0 ]
      ubound = parts[ 1 ]
   return( lbound, ubound )

def validDpConfigValues():
   return [ 1, 2 ]

def tcDpToCosSupportedDpValues():
   return [ 0, 1, 2 ]

def generateTcDpPair( tc, dp ):
   return Tac.ValueConst( "Qos::TcDpPair", tc, dp )

def bsToBytes( bs, bsUnit ):
   if bsUnit == tacBurstUnit.burstUnitBytes:
      bsBytes = bs
   elif bsUnit == tacBurstUnit.burstUnitKBytes:
      bsBytes = bs * 1000
   elif bsUnit == tacBurstUnit.burstUnitMBytes:
      bsBytes = bs * 1000 * 1000
   return bsBytes
