# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import ReversibleDict
import Tac

ConnectorKey = Tac.Type( "Pseudowire::ConnectorKey" )
ConnectorType = Tac.Type( "Pseudowire::ConnectorType" )
ConnIndexType = Tac.Type( "Pseudowire::ConnectorIndex" )
connIndex0 = Tac.Value( "Pseudowire::ConnectorIndex", 0 )
connIndex1 = Tac.Value( "Pseudowire::ConnectorIndex", 1 )
FlowLabelMode = Tac.Type( 'Pseudowire::FlowLabelMode' )
FlowLabelType = Tac.Type( 'Pseudowire::FlowLabelType' )
flDisabled = FlowLabelType( 0x0000 )
flReceive = FlowLabelType( 0x4000 )
flTransmit = FlowLabelType( 0x8000 )
flBoth = FlowLabelType( 0xc000 )
RouteKey = Tac.Type( 'Mpls::RouteKey' )
RouteMetric = Tac.Value( 'Mpls::RouteMetric' )

vlanIdAny = Tac.Type( "Bridging::VlanId" ).invalid

flEnumValDisabled = Tac.enumValue( 'Pseudowire::FlowLabelMode',
                                   FlowLabelMode.disabled )
flEnumValTransmit = Tac.enumValue( 'Pseudowire::FlowLabelMode',
                                   FlowLabelMode.transmit )
flEnumValReceive = Tac.enumValue( 'Pseudowire::FlowLabelMode',
                                   FlowLabelMode.receive )
flEnumValBoth = Tac.enumValue( 'Pseudowire::FlowLabelMode',
                                FlowLabelMode.both )

flEnumToVal = { FlowLabelMode.disabled: flEnumValDisabled,
                FlowLabelMode.transmit: flEnumValTransmit,
                FlowLabelMode.receive: flEnumValReceive,
                FlowLabelMode.both: flEnumValBoth }

flValToEnum = ReversibleDict( flEnumToVal ).reverse()

flValCapabilityMsg = {
   flEnumValDisabled: "none",
   flEnumValTransmit: "transmit",
   flEnumValReceive: "receive",
   flEnumValBoth: "transmit and receive",
}

flValUsedMsg = {
   flEnumValDisabled: "no",
   flEnumValTransmit: "transmitted",
   flEnumValReceive: "received",
   flEnumValBoth: "transmitted and received",
}

PwPingCcType = Tac.Type( 'Pseudowire::PwPingCcType' )
PwPingCvType = Tac.Type( 'Pseudowire::PwPingCvType' )

pwPingCcEnumToVal = { PwPingCcType.cw:
                        Tac.enumValue( 'Pseudowire::PwPingCcType',
                                       PwPingCcType.cw ),
                      PwPingCcType.ra:
                        Tac.enumValue( 'Pseudowire::PwPingCcType',
                                       PwPingCcType.ra ),
                      PwPingCcType.ttlExp:
                        Tac.enumValue( 'Pseudowire::PwPingCcType',
                                       PwPingCcType.ttlExp ) }

pwPingCvEnumToVal = { PwPingCvType.icmpPing: Tac.enumValue(
                                                'Pseudowire::PwPingCvType',
                                                PwPingCvType.icmpPing ),
                      PwPingCvType.lspPing: Tac.enumValue(
                                                'Pseudowire::PwPingCvType',
                                                PwPingCvType.lspPing ) }

# These are the inverse dicts for pwPingCcEnumToVal and pwPingCvEnumToVal
pwPingCcValToEnum = { v: k for k, v in pwPingCcEnumToVal.iteritems() }
pwPingCvValToEnum = { v: k for k, v in pwPingCvEnumToVal.iteritems() }

pwPingCcMsg = {
   PwPingCcType.cw: "Control word",
   PwPingCcType.ra: "Router alert label",
   PwPingCcType.ttlExp: "TTL expiry",
}
assert pwPingCcEnumToVal.keys() == pwPingCcMsg.keys()

pwPingCvMsg = {
   PwPingCvType.icmpPing: "ICMP ping",
   PwPingCvType.lspPing: "LSP ping",
}
assert pwPingCvEnumToVal.keys() == pwPingCvMsg.keys()

undefStr = "-"

def powersOfTwo( val ):
   '''
   Returns every set bit of i as its power of 2
   Example:
      powersOfTwo( 7 ) yields ( 1, 2, 4 )
   '''
   p = 1
   while val:
      if val & 1:
         yield p
      p <<= 1
      val >>= 1

def strVccv( val, values, msgs ):
   if not val:
      return undefStr
   msg = []
   for i in powersOfTwo( val ):
      enum = values.get( i )
      # use msg string if known, else use the raw hex value
      msg.append( msgs.get( enum, hex( i ) ) )
   return ", ".join( msg )

def getInterfaceConnectorKey( intf, dot1qTag=None ):
   if dot1qTag is None:
      dot1qTag = vlanIdAny
   intfName = intf.name if hasattr( intf, "name" ) else intf
   connectorKeyName = ConnectorKey.localConnectorKeyName( intfName, dot1qTag )
   key = ConnectorKey()
   key.localConnectorKeyIs( connectorKeyName )
   return key

def getInterfaceConnector( intf, dot1qTag, connName ):
   connectorKey = getInterfaceConnectorKey( intf, dot1qTag=dot1qTag )
   return Tac.Value( "Pseudowire::CliConnector",
                     connectorKey=connectorKey,
                     name=connName )

def getSubinterfaceConnector( subintf, connName ):
   connectorKey = getInterfaceConnectorKey( subintf )
   return Tac.Value( "Pseudowire::CliConnector", connectorKey=connectorKey,
                     name=connName )

def getLdpConnectorKey( pwName ):
   key = ConnectorKey()
   key.ldpConnectorKeyIs( pwName )
   return key

def getMplsLdpConnector( pwName, connName ):
   connectorKey = getLdpConnectorKey( pwName )
   return Tac.Value( "Pseudowire::CliConnector", connectorKey=connectorKey,
                     name=connName )

def getBgpConnectorKey( vpwsName, pwName ):
   key = ConnectorKey()
   key.bgpConnectorKeyIs( vpwsName, pwName )
   return key

def getBgpConnector( vpwsName, pwName, connName ):
   connectorKey = getBgpConnectorKey( vpwsName, pwName )
   return Tac.Value( "Pseudowire::CliConnector", connectorKey=connectorKey,
                     name=connName )

def getMplsRouteKey( label ):
   return RouteKey.fromLabel( label )
