#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# PLX EEPROM version is stored in register 0x20c ( User Test Pattern )
# We check this register in NorCalInit to test whether the version matches
# the one from the EOS image.
#
# EEPROM version in register 20Ch has the following format
# 0x10013475
# 0x1001 is the version
# 0x3475 is the pci vendor id of arista networks ( using it as signature )
#

import Pci
import PciUtil
import Tac
import FpgaUtil

def plxPrint( text, console=True ):
   print text
   if console:
      FpgaUtil.printToConsole( text + "\n" )


def busId( deviceID,
           vendorID = 0x10b5,
           entryNum = 0 ):
   devices = Pci.allDevicesById ( Pci.Id( vendorID, deviceID ) )
   if devices != [None]:
      devices = [ dev for dev in devices
                  if dev.address_.slot == 0 and dev.address_.function == 0 ]
      devices = sorted( devices, key=lambda x: str(x))
      if len(devices) > entryNum:
         return str( devices[ entryNum ])
   return "None"

eepromFileMap = {}
eepromFileMap[ ( 'eaglepeak', 'lc' ) ] = { 'pci': busId(0x8725),
                                           'chip' : '8725',
                                           'smbus' : '/scd/0/3/0x3A',
                                           'textFile' : 
                                           'PlxPex8725-epk-lc-3a.eeprom.txt',
                                           'binaryFile' : 
                                           'PlxPex8725-epk-lc-3a.eeprom.bin' }
eepromFileMap[ ( 'eaglepeak', 'fc' ) ] = { 'pci': busId(0x8725, entryNum = 1),
                                           'chip' : '8725',
                                           'smbus' : '/scd/0/3/0x38',
                                           'textFile' : 
                                           'PlxPex8725-epk-fc-38.eeprom.txt',
                                           'binaryFile' : 
                                           'PlxPex8725-epk-fc-38.eeprom.bin' }
eepromFileMap[ ( 'oak', 'lc' ) ] = { 'pci': busId(0x8749),
                                     'chip' : '8749',
                                     'smbus' : '/scd/0/3/0x38',
                                     'textFile' : 'PlxPex8749-oak-ca.eeprom.txt',
                                     'binaryFile' : 'PlxPex8749-oak-ca.eeprom.bin' }
eepromFileMap[ ( 'oak', 'fc' ) ] = { 'pci': busId(0x8725),
                                     'chip' : '8725',
                                     'smbus' : '/scd/0/3/0x3A',
                                     'textFile' : 'PlxPex8725-oak-ca.eeprom.txt',
                                     'binaryFile' : 'PlxPex8725-oak-ca.eeprom.bin' }
eepromFileMap[ ( 'oldfaithful', 'lc' ) ] = {
   'pci': busId(0x8749), 
   'chip' : '8749',
   'smbus' : '/scd/0/3/0x3A',
   'textFile' : 'PlxPex8749-old-faithful-lc-3a-ca.eeprom.txt',
   'binaryFile' : 'PlxPex8749-old-faithful-lc-3a-ca.eeprom.bin'
}
eepromFileMap[ ( 'oldfaithful', 'fc' ) ] = {
   'pci': busId(0x8725), 
   'chip' : '8725',
   'smbus' : '/scd/0/3/0x38',
   'textFile' : 'PlxPex8725-old-faithful-fc-38-ca.eeprom.txt',
   'binaryFile' : 'PlxPex8725-old-faithful-fc-38-ca.eeprom.bin'
}
eepromFileMap[ ( 'greatfountain', 'lc' ) ] = {
   'pci': busId( 0x8749, entryNum=1 ),
   'chip' : '8749',
   'smbus' : '/scd/0/3/0x3A',
   'textFile' : 'PlxPex8749-great-fountain-lc-3a.eeprom.txt',
   'binaryFile' : 'PlxPex8749-great-fountain-lc-3a.eeprom.bin'
}
eepromFileMap[ ( 'greatfountain', 'fc' ) ] = {
   'pci': busId( 0x8749 ),
   'chip' : '8749',
   'smbus' : '/scd/0/3/0x38',
   'textFile' : 'PlxPex8749-great-fountain-fc-38.eeprom.txt',
   'binaryFile' : 'PlxPex8749-great-fountain-fc-38.eeprom.bin'
}

eepromFilePath = '/usr/share/plx-pcie-drivers/'
SupePlxTypes = [ 'lc', 'fc' ]
ARISTA_VENDOR_ID = 0x3475


# 
# readEepromVersionFromTextFile - reads eeprom version from text file in filesystem
# 
def readEepromVersionFromTextFile( infile ):
   with open( infile ) as f:
      for line in f.readlines():
         if '0x0020c' in line.lower():
            version = int( line.split()[1], 16 )
            versionMagic = version & 0xFFFF
            versionEeprom = ( version >> 16 ) & 0xFFFF
            if versionMagic == ARISTA_VENDOR_ID:
               print ( "version from file %s 0x%04x"  %
                          ( infile, versionEeprom ) )
               return versionEeprom
   print ( "No version from file %s"  % ( infile ) )
   return 0

#
# readEepromVersionFromHw - reads eeprom version from given PCI bus/dev/function
#
def readEepromVersionFromHw( dev ):
   versionEeprom = 0
   version = PciUtil.read32( dev, 0, 0x20c )
   versionMagic = version & 0xFFFF
   if versionMagic == ARISTA_VENDOR_ID:
      versionEeprom = ( version >> 16 ) & 0xFFFF
      if versionEeprom == 0xFFFF:
         print ( "invalid eeprom version, possibly corrupt eeprom")
      else:
         print ( "valid arista eeprom version:0x%04x 0x%08x dev %s" % ( 
            versionEeprom, version, dev ) )
   else:
      print ( "invalid arista eeprom version:0x%08x dev %s" % ( version, dev ) )
   return versionEeprom

#
# validEeprom - checks if eeprom is corrupted, empty or valid
#
def validEeprom( chip, dev ):
   out = Tac.run( [ 'PlxEeprom.py', 'valid', chip, dev ], 
                  stdout=Tac.CAPTURE )
   return out

#
# validEepromDevice - checks if eeprom is for valid device
#
def validEepromDevice( dev ):
   eepDevId = PciUtil.read32( dev, 0, 0x210 )
   if eepDevId :
      devId = PciUtil.read32( dev, 0, 0 )
      if devId != eepDevId:
         print ("WARNING: eeprom device id %x different from device %x" % (
               eepDevId, devId ))

#
# updatePlxEeprom - validates eeprom versions and upgrades if necessary
#
def updatePlxEeprom( platform, force=False, console=True ):
   doReboot = False
   if platform in ( 'oak', 'eaglepeak', 'oldfaithful', 'greatfountain' ):
      print ("Checking eeprom for platform - %s " % platform)
   else:
      print ("Not checking eeprom for platform - %s " % platform)
      return False

   for supePlx in SupePlxTypes:
      textFile = ( eepromFilePath + 
                   eepromFileMap[ ( platform, supePlx ) ] [ 'textFile' ] )
      binaryFile = ( eepromFilePath + 
                     eepromFileMap[ ( platform, supePlx ) ] [ 'binaryFile' ] )
      dev = eepromFileMap[ ( platform, supePlx ) ][ 'pci' ]
      smbusAddress = eepromFileMap[ ( platform, supePlx ) ] [ 'smbus' ]
      chip = eepromFileMap[ ( platform, supePlx ) ] [ 'chip' ]
      versionInImage = readEepromVersionFromTextFile( textFile )
      versionInHw = readEepromVersionFromHw( dev )
      validEepromDevice( dev )
      validTest = validEeprom( chip, smbusAddress )
      if "Valid" in validTest:
         if force or versionInImage != versionInHw:
            plxPrint( "version 0x%x(%s) != hw 0x%x(%s)" %
                      ( versionInImage, binaryFile, versionInHw, dev ), console )
         else:
            print( "versionInImage 0x%x(%s) matches versionInHw 0x%x(%s)" %
                         ( versionInImage, binaryFile, versionInHw, dev ))
            continue
      elif "Corrupted" in validTest:
         plxPrint( "version 0x%x(%s) != hw 0x%x(%s)" % \
                   ( versionInImage, binaryFile, versionInHw, dev ), console )
         plxPrint(
            " PLX EEPROM corrupted, fixing - Please do not power off supervisor ",
            console )
         Tac.run( [ 'PlxEeprom.py', 'format', chip, smbusAddress ] )
         if not force:
            plxPrint( " PLX EEPROM format completed ", console )
         doReboot = True
         continue

      if not force:
         plxPrint(
            " PLX EEPROM upgrade in progress - Please do not power off supervisor ",
            console )

      try:
         Tac.run( [ 'PlxEeprom.py', 'program', chip, smbusAddress, binaryFile ] )
         if not force:
            plxPrint( " PLX EEPROM upgrade completed ", console )
      except Tac.SystemCommandError:
         try:
            if not force:
               plxPrint( " PLX EEPROM formatting....", console )
            Tac.run( [ 'PlxEeprom.py', 'format', chip, smbusAddress ] )
            Tac.run( [ 'PlxEeprom.py', 'program', chip, smbusAddress, binaryFile ] )
            if not force:
               plxPrint( " PLX EEPROM upgrade completed ", console )
         except Tac.SystemCommandError:
            plxPrint(
               " PLX EEPROM upgrade failed - try rebooting ", console )
      doReboot = True
   return doReboot

