#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import

import ctypes
from fcntl import ioctl
import os
import re
from six import string_types
import time

#
# Those values and declarations are transliteration of the linux kernel headers.
# see /usr/include/linux/i2c.h and /usr/include/linux/i2c-dev.h
#

I2C_RDWR        = 0x0707
I2C_FUNCS       = 0x0705
I2C_SLAVE       = 0x0703
I2C_SLAVE_FORCE = 0x0706
I2C_SLAVE       = 0x0703
I2C_SMBUS       = 0x0720 # SMBus-level access

# smbus_access read or write markers
I2C_SMBUS_READ  = 1
I2C_SMBUS_WRITE = 0

# SMBus transaction types (size parameter in the above functions)
#   Note: these no longer correspond to the (arbitrary) PIIX4 internal codes!

I2C_SMBUS_QUICK             = 0
I2C_SMBUS_BYTE              = 1
I2C_SMBUS_BYTE_DATA         = 2
I2C_SMBUS_WORD_DATA         = 3
I2C_SMBUS_PROC_CALL         = 4
I2C_SMBUS_BLOCK_DATA        = 5
I2C_SMBUS_I2C_BLOCK_BROKEN  = 6
I2C_SMBUS_BLOCK_PROC_CALL   = 7 # SMBus 2.0
I2C_SMBUS_I2C_BLOCK_DATA    = 8

I2C_FUNC_I2C                    = 0x00000001
I2C_FUNC_10BIT_ADDR             = 0x00000002
I2C_FUNC_PROTOCOL_MANGLING      = 0x00000004 # I2C_M_{REV_DIR_ADDR,NOSTART,..}
I2C_FUNC_SMBUS_PEC              = 0x00000008
I2C_FUNC_SMBUS_BLOCK_PROC_CALL  = 0x00008000 # SMBus 2.0
I2C_FUNC_SMBUS_QUICK            = 0x00010000
I2C_FUNC_SMBUS_READ_BYTE        = 0x00020000
I2C_FUNC_SMBUS_WRITE_BYTE       = 0x00040000
I2C_FUNC_SMBUS_READ_BYTE_DATA   = 0x00080000
I2C_FUNC_SMBUS_WRITE_BYTE_DATA  = 0x00100000
I2C_FUNC_SMBUS_READ_WORD_DATA   = 0x00200000
I2C_FUNC_SMBUS_WRITE_WORD_DATA  = 0x00400000
I2C_FUNC_SMBUS_PROC_CALL        = 0x00800000
I2C_FUNC_SMBUS_READ_BLOCK_DATA  = 0x01000000
I2C_FUNC_SMBUS_WRITE_BLOCK_DATA = 0x02000000
I2C_FUNC_SMBUS_READ_I2C_BLOCK   = 0x04000000 # I2C-like block xfer
I2C_FUNC_SMBUS_WRITE_I2C_BLOCK  = 0x08000000 # w/ 1-byte reg. addr.

I2C_SMBUS_BLOCK_MAX = 32

I2C_FD_RETRY_LIMIT = 10

class i2c_smbus_data( ctypes.Union ):
   _fields_ = [ ( 'byte', ctypes.c_uint8 ),
                ( 'word', ctypes.c_uint16 ),
                ( 'block', ctypes.c_uint8 * ( I2C_SMBUS_BLOCK_MAX + 2 ) ) ]

class i2c_smbus_ioctl_data( ctypes.Structure ):
   _fields_ = [ ( 'read_write', ctypes.c_uint8 ),
                ( 'command', ctypes.c_uint8 ),
                ( 'size', ctypes.c_int ),
                ( 'data', ctypes.POINTER( i2c_smbus_data ) ) ]

#
# End of declarations
#

class I2cBusNotFoundError( Exception ):
   pass

def i2cBusIdFromName( name ):
   nameRe = re.compile( name )
   i2cRoot = '/sys/class/i2c-dev'

   for i2cDir in os.listdir( i2cRoot ):
      busNameFile = os.path.join( i2cRoot, i2cDir, "name" )
      with open( busNameFile ) as f:
         if nameRe.match( f.read().strip() ):
            return int( i2cDir.split( '-' )[ 1 ] )

   raise I2cBusNotFoundError( 'No i2c bus matches %s' % name )

class SMBus( object ):
   def __init__( self, bus, force=False ):
      self.bus = None
      if isinstance( bus, string_types ):
         self.bus = i2cBusIdFromName( bus )
      else:
         self.bus = bus
      self.fd = -1
      self.slave_command = I2C_SLAVE_FORCE if force else I2C_SLAVE
      self.address = None

      # open the i2c fd
      self.open()

   def setAddress( self, address ):
      if address == self.address:
         return

      if ioctl( self.fd, self.slave_command, address ) == -1:
         raise IOError( "failed to set address on i2c bus" )

      self.address = address

   def open( self ):
      filename = "/dev/i2c-%d" % self.bus
      retryCount = 0
      while retryCount < I2C_FD_RETRY_LIMIT:
         if not os.path.exists( filename ):
            time.sleep( 1 )
         retryCount += 1

      self.fd = os.open( filename, os.O_RDWR )
      if self.fd == -1:
         raise IOError( "SMBus: failed to open %s" % filename )

   def close( self ):
      os.close( self.fd )

   def recvByte( self, address ):
      self.setAddress( address )
      data = self.smbusAccess( I2C_SMBUS_READ, 0x00, I2C_SMBUS_BYTE )
      return data.contents.byte & 0xff

   def read8( self, address, command ):
      self.setAddress( address )
      data = self.smbusAccess( I2C_SMBUS_READ, command, I2C_SMBUS_BYTE_DATA )
      return data.contents.byte & 0xff

   def read16( self, address, command ):
      self.setAddress( address )
      data = self.smbusAccess( I2C_SMBUS_READ, command, I2C_SMBUS_WORD_DATA )
      return data.contents.word & 0xffff

   def readBlock( self, address, command, size ):
      assert size <= I2C_SMBUS_BLOCK_MAX
      self.setAddress( address )
      _data = i2c_smbus_data()
      _data.byte = size # pylint: disable=attribute-defined-outside-init
      data = self.smbusAccess( I2C_SMBUS_READ, command, I2C_SMBUS_I2C_BLOCK_DATA,
            _data )
      return bytearray( data.contents.block[ 1 : 1 + data.contents.block[ 0 ] ] )

   def sendByte( self, address, byte ):
      self.setAddress( address )
      self.smbusAccess( I2C_SMBUS_WRITE, byte, I2C_SMBUS_BYTE )

   def write8( self, address, command, data ):
      self.setAddress( address )
      _data = i2c_smbus_data()
      _data.byte = data # pylint: disable-msg=W0201
      self.smbusAccess( I2C_SMBUS_WRITE, command, I2C_SMBUS_BYTE_DATA, _data )

   def write16( self, address, command, data ):
      self.setAddress( address )
      _data = i2c_smbus_data()
      _data.word = data # pylint: disable-msg=W0201
      self.smbusAccess( I2C_SMBUS_WRITE, command, I2C_SMBUS_WORD_DATA, _data )

   def writeBlock( self, address, command, data ):
      assert len( data ) <= I2C_SMBUS_BLOCK_MAX
      self.setAddress( address )
      _data = i2c_smbus_data()
      _data.byte = len( data ) # pylint: disable-msg=W0201
      view = memoryview(_data.block)
      view[ 1 : _data.byte + 1 ] = data
      self.smbusAccess( I2C_SMBUS_WRITE, command, I2C_SMBUS_I2C_BLOCK_DATA, _data )
      return _data.byte

   # pylint: disable-msg=W0201
   def smbusAccess( self, read_write, command, size, data=None ):
      smbusArg = i2c_smbus_ioctl_data()
      smbusArg.read_write = read_write
      smbusArg.command = command
      smbusArg.size = size
      if not data:
         data = i2c_smbus_data()
      smbusArg.data = ctypes.pointer( data )
      if ioctl( self.fd, I2C_SMBUS, smbusArg ) == -1:
         raise IOError( "failed to send data on the i2c bus" )
      return smbusArg.data

