#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, print_function

import Tac

# EthIntf is an ugly place to have this file, but PhyEee depends on EthIntf and both
# packages use functions in this file.
def phyConfigStatusAtPos( sysdbRoot, phyTopology, intfId, phyPos,
                          firstLaneOnly=False ):
   '''Given the root phy topology object, returns all phy configs and statuses that
   are associated with the intfId at the specified phy position. There can be
   multiple phy objects due to the organization of some 40G-only ports where each
   lane has its own phy object.

   firstLaneOnly: Whether or not only the first lane's phy config and status object
   should be returned for 40G-only ports with multiple phy statuses.

   Returns a list of ( phyIntfConfig, phyIntfStatus ) pairs.'''
   phyData = []
   intfItself = [ intfId ]
   laneIntfs = [ "%s/%d" % ( str( intfId ), lane )
                 for lane in range( 1, 2 if firstLaneOnly else 5 ) ]
   # First, try to find a phy config and status object for the specified intfId.
   # If none are found, it may be a 40G-only interface that has multiple phy
   # statuses. Assume that a 40G-only Ethernet interface EthernetX may have phy
   # topology entries named EthernetX/1, EthernetX/2, EthernetX/3, EthernetX/4. This
   # organization only appears for 40G-only ports.
   for possibleIntfs in ( intfItself, laneIntfs ):
      for possibleIntf in possibleIntfs:
         try:
            Tac.Value( "Arnet::EthIntfId", possibleIntf )
         except IndexError:
            # if not an Ethernet interface, skip.
            continue
         phyIntfStatuses = phyTopology.phyIntfStatuses.get( possibleIntf )
         # Entry has to exist for the specified interface and position.
         if phyIntfStatuses and phyPos in phyIntfStatuses.phyIntfData:
            phyIntfConfig, phyIntfStatus = phyTopologyConfigStatus(
               sysdbRoot, phyIntfStatuses.phyIntfData[ phyPos ] )
            # If we either have both a phyConfig and phyStatus or a phyStatus and
            # the phyStatus tells us it doesn't have a config, add what we have
            # to the results list. Note that phyConfig may be returned as None.
            if ( phyIntfConfig and phyIntfStatus ) or \
               ( phyIntfStatus and getattr( phyIntfStatus, 'ignoreConfig', False ) ):
               phyData.append( ( phyIntfConfig, phyIntfStatus ) )
      if phyData:
         # if valid phy information is found, no need to look further.
         break
   return phyData

def phyTopologyConfigStatus( sysdbRoot, phyIntfData ):
   '''Given a Hardware::PhyTopology::PhyIntfData object, returns the phy intf config
   and phy intf status corresponding to it if they are valid. If they don't exist or
   are stale, ( None, None ) is returned.'''
   if phyIntfData.statusPath:
      assert phyIntfData.configPath
      phyIntfStatus = None
      phyIntfConfig = None
      try:
         phyIntfStatus = sysdbRoot.entity[ phyIntfData.statusPath ]
         phyIntfConfig = sysdbRoot.entity[ phyIntfData.configPath ]
      except KeyError:
         pass
      else:
         if phyIntfStatus.generation.valid and \
            phyIntfStatus.generation == phyIntfConfig.generation:
            return phyIntfConfig, phyIntfStatus
      # If we couldn't retrive one of the entities, check if we at least have a
      # phyStatus that indicates it doesn't have a phyConfig.
      if phyIntfStatus and phyIntfStatus.generation.valid and \
            getattr( phyIntfStatus, 'ignoreConfig', False ):
         return None, phyIntfStatus

   return None, None

def mountPhyTopology( em ):
   mg = em.mountGroup()
   mg.mount( 'hardware/phyChip/config', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phy/status', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phy/config', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/phyChip/status', 'Tac::Dir', 'ri' )
   phyTopology = mg.mount( 'hardware/phy/topology/allPhys',
                           'Hardware::PhyTopology::AllPhyIntfStatuses', 'r' )
   mg.close( blocking=True )
   if not hasattr( mountPhyTopology, 'hardwarePhyConfigDirReactor' ):
      mountPhyTopology.hardwarePhyConfigDirReactor = \
         Tac.newInstance( "Sysdb::DirMounter", "hardware/phy/config", "?",
                          "Tac::Dir", "ri", em.cEntityManager(),
                          Tac.activityManager.clock )
      mountPhyTopology.hardwarePhyConfigDirReactor.start()
      # BUG215494 Need to add logic to wait for archer mounts to complete if any.
      # Wait for all phy config directories that are currently present to be mounted.
      # Paths in deferredMountPaths are dequeued when a mount is performed, not when
      # the mount completes, so also wait on all mounts being complete.
      try:
         Tac.waitFor( lambda:
               not mountPhyTopology.hardwarePhyConfigDirReactor.deferredMountPaths
                      and em.cEntityManager().allMountsComplete(),
                      # sleep if in notification handler, otherwise run activities.
                      sleep=not Tac.activityManager.inExecTime.isZero,
                      description="phy data structures to be available",
                      timeout=60.0 )
      except Tac.Timeout:
         # Ok to proceed best we can if phy directories are not all mounted.
         pass
   return phyTopology
