#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import string
import Tac, EntityManager, Cell, Tracing, PyClient
import MdioUtil
import os, sys, ctypes

traceHandle = Tracing.Handle( 'PhyDiagLib' )

t0 = traceHandle.trace0
t7 = traceHandle.trace7
t9 = traceHandle.trace9


#-----------------------------------------------------------------------------
#
# Random helper stuff
#
#-----------------------------------------------------------------------------
# BUG1596: This is just a real pain.

class EntityMounter( object ):

   phyDirName = "hardware/phy"
   collDirName = "generic/all"
   
   configTypename = "Hardware::Phy::ConfigPtrDir"
   statusTypename = "Hardware::Phy::StatusPtrDir"

   def _mountConfigStatus( self ):
      """Mount the config and status tree of the hardware/phy area.
      
      # XXX roth -- if I recall, we need to start by mounting these tree with the
      # correct perms, before we start traversing the generic collections.
      """
      
      t0( "Mounting config and status trees" )
      try:
         mg = self.entMan_.mountGroup()
         if os.environ.get( "ABUILD" ):
            mg.mount( os.path.join( self.phyDirName,
                                    "config" ),
                      "Tac::Dir", "ri" )
         else:
            _ = PyClient.PyClient( "ar", "Sysdb" ).agentRoot().entity[
                  os.path.join( self.phyDirName, "config" ) ]
         mg.mount( os.path.join( self.phyDirName,
                                 "status" ),
                   "Tac::Dir", "ri" )
         mg.close( blocking=True )
      except EntityManager.MountError:
         ssw = sys.stderr.write
         ssw( "Failed to mount config and status\n")
         ssw( "Are the agents running? ('service ProcMgr start')\n" )
         ssw( "(If you recently restarted ProcMgr, try again in a few secs...)\n" )
         sys.exit( 1 )
      
   def _mountPhyDir( self ):

      t0( "Starting mount of generic phys" )
      try:
         mg = self.entMan_.mountGroup()
         if os.environ.get( "ABUILD" ):
            self.config_ = mg.mount( os.path.join( self.phyDirName,
                                                   "config",
                                                   self.collDirName ),
                                     self.configTypename, "r" )
         else:
            self.config_ = PyClient.PyClient( "ar", "Sysdb" ).agentRoot().entity[
               os.path.join( self.phyDirName, "config", self.collDirName ) ]
         self.status_ = mg.mount( os.path.join( self.phyDirName,
                                                "status",
                                                self.collDirName ),
                                  self.statusTypename, "r" )
         mg.close( blocking=True )
      except EntityManager.MountError:
         ssw = sys.stderr.write
         ssw( "Failed to mount {config,status}/%s\n" % self.collDirName )
         ssw( "Are the agents running? ('service ProcMgr start')\n" )
         ssw( "(If you recently restarted ProcMgr, try again in a few secs...)\n" )
         sys.exit( 1 )

   def _mountPciDeviceStatusDir( self ):

      t0( "Starting mount of pciDeviceStatusDir" )
      try:
         mg = self.entMan_.mountGroup()
         mg.mount( 'cell/%d/hardware/pciDeviceStatusDir' % Cell.cellId(),
                   'Hardware::PciDeviceStatusDir', 'r' )
         mg.close( blocking=True )
      except EntityManager.MountError:
         ssw = sys.stderr.write
         ssw( "Failed to mount pciDeviceStatusDir\n" )
         ssw( "Are the agents running? ('service ProcMgr start')\n" )
         ssw( "(If you recently restarted ProcMgr, try again in a few secs...)\n" )
         sys.exit( 1 )

   def __init__( self, entMan ):

      self.entMan_ = entMan

      self._mountConfigStatus()
      self._mountPhyDir()
      self._mountPciDeviceStatusDir()

   @classmethod
   def mountEntities( cls, entMan ):
      mounter = cls( entMan )
      return ( mounter.config_, mounter.status_, )

class SliceEntityMounter( EntityMounter ):
   """Entity mounter that is slice-aware.

   This version mounts the entities for only a given slice.
   """

   collDirName = "generic/slice/%d"
   
   def __init__( self, entMan, sliceId ):

      self.sliceId_ = sliceId
      super( SliceEntityMounter, self ).__init__( entMan )

   def _mountPhyDir( self ):

      t0( "Starting mount of per-slice generic phys" )

      collDirName = self.collDirName % self.sliceId_
      
      try:
         mg = self.entMan_.mountGroup()
         if os.environ.get( "ABUILD" ):
            self.config_ = mg.mount( os.path.join( self.phyDirName,
                                                   "config",
                                                   collDirName ),
                                     self.configTypename, "r" )
         else:
            self.config_ = PyClient.PyClient( "ar", "Sysdb" ).agentRoot().entity[
               os.path.join( self.phyDirName, "config", collDirName ) ]
         self.status_ = mg.mount( os.path.join( self.phyDirName,
                                                "status",
                                                collDirName ),
                                  self.statusTypename, "r" )
         mg.close( blocking=True )
      except EntityManager.MountError:
         ssw = sys.stderr.write
         ssw( "Failed to mount {config,status}/%s\n" % collDirName )
         ssw( "Are the agents running? ('service ProcMgr start')\n" )
         ssw( "(If you recently restarted ProcMgr, try again in a few secs...)\n" )
         sys.exit( 1 )

   @classmethod
   def mountSliceEntities( cls, entMan, sliceId ):
      mounter = cls( entMan, sliceId )
      return ( mounter.config_, mounter.status_, )

def getPhys( entMan, sliceId=None ):
   """Return a unified phy config and status object.

   If we specified a slice ID, use that to restrict which phys are returned.
   """

   # BUG7838 -- it's ugly to hardcode the "FixedSystem" sliceId here.
   if ( sliceId is None ) or ( sliceId == "FixedSystem" ):
      config, status = EntityMounter.mountEntities( entMan )
   else:
      config.status = SliceEntityMounter.mountSliceEntities( entMan,
                                                             sliceId=sliceId )

   return config, status

def cmp_phy( p1, p2 ):
   """Friendly comparison for phy names."""
   
   while True:

      i1 = 0
      while p1 and p1[0:1] in string.digits:
         i1 *= 10
         i1 += ord( p1[0] ) - ord( '0' )
         p1 = p1[1:]

      i2 = 0
      while p2 and p2[0:1] in string.digits:
         i2 *= 10
         i2 += ord( p2[0] ) - ord( '0' )
         p2 = p2[1:]

      c = cmp( i1, i2 )
      if c:
         return c

      if p1 and p2:
         c1 = p1[0:1]
         p1 = p1[1:]
         c2 = p2[0:1]
         p2 = p2[1:]
         c = cmp( c1, c2 )
         if c:
            return c
      else:
         return cmp( p1, p2 )

def cmp_phyConfig( p1, p2 ):
   return cmp_phy( p1.name, p2.name )

# default implementation of mountEntities, uses the phy generic interface
mountEntities = EntityMounter.mountEntities


#-----------------------------------------------------------------------------
#
# Helpers to instantiate Phy diags objects
#
#-----------------------------------------------------------------------------
def newPhy( config, status ):
   # Right now, this is pretty lame -- it assumes a Clause 45 PHY --
   # but eventually, we should be able to figure out whether we're
   # talking to a Clause 22 or Clause 45 PHY by looking at the Aham.
   # (There's a Clause22/Clause45 bit in there somewhere.)
   return Clause45Phy( config, status )


#-----------------------------------------------------------------------------
#
# Basic PHY wrapper object.
#
#-----------------------------------------------------------------------------
class Phy( object ):
   def __init__( self, config, status ):
      self.config_ = config
      self.status_ = status
      self.name = config.name
      self.mdio_ = Mdio( config )

   def phyStateIs( self, phyState, blocking=True ):
      """Set the phyState attr in the PhyStatus. Note that this update
      is not EntityLogged to Sysdb until activities have run and given
      the EntityLog a chance to flush. Therefore, if this is being
      called in a diags script just before existing, someone needs to
      ensure that the EntityLog gets flushed.

      By default, this function blocks until the EntityLog is flushed,
      but if you are about to update a bunch of PHYs, it might be
      better to do this 'by hand', i.e., set blocking=False when
      calling this method on a bunch of objects, and then calling
      Tac.flushEntityLogs after the loop."""
      self.status_.phyState = phyState
      if blocking:
         Tac.flushEntityLog()

   def intfName( self ):
      """Return the CLI-visible name of the interface corresponding to
      this PHY."""
      return str( self.status_.intfId )

   def hwReset( self ):
      return self.config_.reset.asserted

   def hwResetIs( self, reset ):
      t0( self.config_.name, " reset = ", reset )
      self.config_.reset.asserted = reset

   def read( self, devAddr, regAddr ):
      """Read a single register."""
      return self.mdio_.read( devAddr, regAddr )

   def readBulk( self, addrList ):
      """Read a whole bunch of MDIO registers."""
      return self.mdio_.readBulk( addrList )

   def startRead( self, addr ):
      """Start Read in asynchronous mode"""
      return self.mdio_.startRead( addr )
  
   
   def write( self, devAddr, regAddr, data ):
      """Write a single register."""
      self.mdio_.write( devAddr, regAddr, data )
   

   def startWrite( self, addr, data ):
      """Write in asynchronous mode"""
      return self.mdio_.startWrite( addr, data )

   def writeBulk( self, data ):
      """Write 'data', a list of tuples of (addr,data), to the device."""
      self.mdio_.writeBulk( data )

   # XXX_LWR: These next two "startXXX" attrs are a bit gross, but
   #          they enable an important performance improvement when
   #          programming the SPI flashes of Teranetics PHYs... I'll
   #          figure out how to make this a bit more palatable later.
   def startWriteBulk( self, data ):
      """Ask the MDIO agent to start writing 'data', a list of tuples
      of (addr,data), to the device, and return the (probably)
      in-progress AhamRequest to the caller, who can wait on it. The
      idea is that the caller can fire off a bunch of these all at
      once, and then wait for them all at once."""
      return self.mdio_.startWriteBulk( data )

   def startWriteBlobletByName( self, blobletName ):
      """Ask the Mdio Agent to start writing the named block, and
      return the (probably) in-progress AhamRequest back to the
      caller, who can then wait on it."""
      return self.mdio_.startWriteBlobletByName( blobletName )

   def rmw( self, devAddr, regAddr, mask, data ):
      """Read-modify-write a single register."""
      self.mdio_.rmw( devAddr, regAddr, mask=mask, new=data )

   def diagsModeIs( self, diagsMode ):
      """Put the PHY into the specified diagsMode. Note that if the
      PHY is already in that same diags mode, it is up to our caller
      to decide if they want to yank it out of that diags mode and
      back into it."""
      self.config_.diagsMode = diagsMode
      self.config_.diagsModeEpoch += 1

   def diagsModeStatusReflectsConfig( self ):
      """
      Returns True if the diags mode status reflects the config, else
      False.
      """
      s = self.status_
      c = self.config_
      return ( ( s.diagsMode == c.diagsMode ) and
               ( s.diagsModeEpoch == c.diagsModeEpoch ) )

   def present( self ):
      raise NotImplementedError

   def reset( self ):
      raise NotImplementedError

   def loopbackIs( self, loopbackMode ):
      """Puts the PHY into the specified loopback mode."""
      self.config_.loopback = loopbackMode

   def loopback( self ):
      """Returns the current loopback mode."""
      return self.config_.loopback

   def loopbackApplied( self ):
      """
      Returns True if the loopback mode status reflects the config,
      else False.
      """
      return self.config_.loopback == self.status_.loopback

   def loopbackSupported( self ):
      """Returns the types of loopbacks supported by the PHY."""
      pass

   def usingInterrupts( self ):
      """ returns name of interrupt the phy is using """
      if not self.config_.interruptCtrl:
         return None
      t9( "phy " + self.config_.name + " using irq " + 
             self.config_.interruptCtrl.fullName )
      return self.config_.interruptCtrl.fullName

   def interruptCount( self ):
      """ returns whether the scd detects an interrupt from the phy
      """
      assert self.config_.interruptCtrl
      return self.status_.interruptCount
      

class Clause45Phy( Phy ):
   def __init__( self, config, status ):
      Phy.__init__( self, config, status )


   def present( self ):
      """
      Returns True if the PHY is present, else False. Test for
      presence by reading the 'device present' bits from the PMA/PMD
      status 2 register (reg 1.0008).  See the IEEE 802.3-2005, Clause
      45.2.1.7 specification for additional details.
      """
      actual = self.mdio_.read( 0x1, 0x0008 )
      return ( ( actual >> 14 ) & 0x3 ) == 0x2


   def reset( self ):
      """
      Returns True if the PHY is in reset, else False.

      Uses standard PMA/PMD and PCS register sets.
      """
      val = self.mdio_.read( 0x1, 0x0000 )
      t7( self.name, ": read 1.0000 = %04x" % val )
      return ( ( val >> 15 ) & 0x1 ) == 0x1

   def asyncWrite( self, mmdAddr, regAddr, data ):
      addr = ( mmdAddr << 16 ) + regAddr
      return self.startWrite( addr, data )


   def asyncRead( self, mmdAddr, regAddr ):
      addr = ( mmdAddr << 16 ) + regAddr 
      return self.startRead( addr )


   def phyState( self ):
      return self.status_.phyState


   def hwResetCount( self ):
      return self.status_.resetCount


class Clause22Phy( Phy ):
   def __init__( self, config, status ):
      Phy.__init__( self, config, status )

   # XXX_LWR: TODO: fill in stuff for Clause 22 PHYs.

   def present( self ):
      raise NotImplementedError

   def reset( self ):
      raise NotImplementedError

#-----------------------------------------------------------------------------
#
# MDIO helper cruft
#
# XXX_LWR: this should probably live in Arbus and/or Arsys?
#
#-----------------------------------------------------------------------------
class Mdio( object ):
   """MDIO interface for one PHY"""
   def __init__( self, phyConfig ):
      self.helper_ = None
      self.sock_ = None

      # XXX_LWR: I really don't need the PhyConfig -- I just need the
      #          Aham.
      self.phyConfig_ = phyConfig

      # Generate a MDIO helper for the PHY
      ahamDesc = phyConfig.ahamDesc

      self.helper_ = MdioUtil.MdioAgentClient( ahamDesc )


   def read( self, devAddr, register ):
      try:
         actual = self.helper_.read16( devAddr, register )
         t9( "read %x %s (%x.%x)" % ( actual[0], self.phyConfig_.name,
                                      devAddr, register ) )
         return actual[0]
      except MdioUtil.MdioError, e:
         print "Exception occured when reading from %s (%x.%x) [%s]" % (
            self.phyConfig_.name, devAddr, register, e )
         sys.exit( 1 )


   def startRead( self, addr ):
      ( req, buf ) = createReadRequest( self.phyConfig_, [ addr ] )
      req.state = "started"
      return ( req, buf[0] )


   def readBulk( self, addrList ):
      (req, buf) = createReadRequest( self.phyConfig_, addrList )
      _doMdioTransaction( req )
      return buf

   def write( self, devAddr, register, value ):
      try:
         self.helper_.write16( devAddr, register, [value] )
      except MdioUtil.MdioError, e:
         print "Exception occured when reading from %s (%x.%x) [%s]" % (
            self.phyConfig_.name, devAddr, register, e)
         sys.exit( 1 )


   def startWrite( self, addr, value ):
      ( req, buf ) = createWriteRequest( self.phyConfig_, [ ( addr, value ) ] )
      req.state = "started"
      return ( req, buf[0] )


   def writeBulk( self, array ):
      ( req, _ ) = createWriteRequest( self.phyConfig_, array )
      _doMdioTransaction( req )


   def startWriteBulk( self, array ):
      ( req, buf ) = createWriteRequest( self.phyConfig_, array )
      req.state = "started"
      return ( req, buf )


   def startWriteBlobletByName( self, blobletName ):
      req = self.phyConfig_.ahamDesc.aham.newRequest
      req.nextOp = Tac.Value( "Hardware::AhamRequest::OpDesc",
                              op="write", bloblet=blobletName, addr=0 )
      req.state = "started"
      return req


   def rmw( self, devAddr, register, mask, new ):
      """read-modify-write an MDIO register.  'mask' is a bitmask of which
         bits to change, and 'new' holds the new values for those bits.

         NOTE: No lock is taken to protect the register during
               the operation."""

      existing = self.read( devAddr, register )
      result = ( (existing & ~mask) | # existing bit to keep at 1
                 (mask & new) )       # change bit to 1
      self.write( devAddr, register, result )


   def close( self ):
      # This was copied from BusUtilCommaon.unpauseAgentIfNecessary to
      # avoid adding a dependency on Scd.
      if self.sock_:
         self.sock_.send( "unpause\r\n" )
         # wait for the agent to acknowledge
         self.sock_.recv( 10 )

def populateWriteRequest( req, txns ):
   """Populate an MDIO Write AhamRequest ('req') with the transactions
   in 'txns', a list of (addr,data) tuples."""
   
   buf = []
   for i in range( len( txns ) ):
      buf.append( ctypes.create_string_buffer( 2 ) )

   i = 0
   for (addr, data) in txns:
      b = buf[ i ]
      b[ 0 ] = chr( data & 0xff )
      b[ 1 ] = chr( ( data & 0xff00 ) >> 8 )
      t0( "Processing transaction %d: 0x%05x 0x%04x (0x%04x)" % \
          ( i, addr, data, int( ord( b.raw[0:1] ) ) ) )
      i += 1
      bufAddr = ctypes.addressof( ctypes.pointer( b ).contents )
      req.nextOp = Tac.Value( "Hardware::AhamRequest::OpDesc",
                              op="write", addr=addr, data=bufAddr,
                              count=1, elementSize=2 )

   # Hold on to this buf until the MDIO transaction has finished!!
   return buf


def createWriteRequest( phyConfig, txns ):
   req = phyConfig.ahamDesc.aham.newRequest
   buf = populateWriteRequest( req, txns )
   return ( req, buf )


def populateReadRequest( req, addrList ):
   buf = []
   for i in range( len( addrList ) ):
      buf.append( ctypes.create_string_buffer( 2 ) )

   i = 0
   for addr in addrList:
      if not isinstance( addr, int ):
         addr = int( addr, 16 )
      b = buf[ i ]
      t0( "Processing transaction %d: 0x%05x" % ( i, addr ) )
      bufAddr = ctypes.addressof( ctypes.pointer( b ).contents )
      req.nextOp = Tac.Value( "Hardware::AhamRequest::OpDesc",
                              op="read", addr=addr, data=bufAddr,
                              count=1, elementSize=2 )
      i += 1

   # Hold on to this buf until the MDIO transaction has finished!!
   return buf


def createReadRequest( phyConfig, addrList ):
   req = phyConfig.ahamDesc.aham.newRequest
   buf = populateReadRequest( req, addrList )
   return ( req, buf  )


def printMdioRequest( buf, addrList ):
   i = 0
   for addr in addrList:
      b = buf[ i ]
      data = ( int( ord( b.raw[ 1 ] ) ) << 8 ) + ( int( ord( b.raw[ 0 ] ) ) )
      print "%05x:    %04x" % ( addr, data )
      i += 1


def _doMdioTransaction( req ):
   req.state = "started"
   assert req.status == "inProgress"

   # Doing bulk MDIO transactions is often latency-sensitive (we need
   # to read or write a whole ton of registers, usually), and is only
   # ever done from diags scripts, so dial down the maxDelay to 1ms.
   Tac.waitFor( lambda: req.status != "inProgress",
                description="MDIO transaction to finish",
                maxDelay=0.001 )

   if req.status == "succeeded":
      t0( "MDIO transaction succeeded!" )
   else:
      print "ERROR: MDIO transaction failed!"


# XXX_LWR: this is a hack for bringup -- "mdiobulkClean" uses it.
def writeBulkArray( phyConfig, array ):
   """Expects 'array' to be a list of tuples of (addr,data)."""
   ( req, _ ) = createWriteRequest( phyConfig, array )
   _doMdioTransaction( req )


def writeBulk( phyConfig, txnsTxt ):
   """Expects 'txnsTxt' to be ASCII, in format 'addr,data'."""
   txns = []
   for t in txnsTxt:
      s = t.split( "," )
      addr = int( s[ 0 ], 16 )
      data = int( s[ 1 ], 16 )
      txns.append( ( addr, data ) )

   ( req, _ ) = createWriteRequest( phyConfig, txns )

   _doMdioTransaction( req )


def readBulk( phyConfig, addrList ):
   (req, buf) = createReadRequest( phyConfig, addrList )
   _doMdioTransaction( req )
   printMdioRequest( buf, addrList )

# ----------------------------------------------------------------------------
#
# Functions to load and write blobs and bloblets
#
# ----------------------------------------------------------------------------
class BlobProperties( object ):
   def __init__( self, version, blobletCount ):
      self.version = version
      self.blobletCount = blobletCount

def blobProps( blobFilename, blobGroupName, blobName ):

   # If it's a python MdioBloblet, the ops are a list of (addr,data)
   # tuples pointed to by the "sequence" attr, and if it's a C++
   # Mdio::Bloblet, the MdioOps are an indexedAttr of Mdio::MdioOp
   # Value types named "op".
   if blobFilename.endswith( ".so" ):
      Tac.dlopen( blobFilename )
      blobDir = Tac.singleton( "Mdio::BlobDir" )
      blob = blobDir.blobGroup[ blobGroupName ].blob[ blobName ]
      bps = BlobProperties( version=blob.version,
                            blobletCount=len( blob.bloblet.keys() ) )
      return bps


   # Strip ".py" (really, ".py*") from the module name -- this lets me
   # tab-expand the filename on the command line.
   modName = blobFilename
   dotPy = blobFilename.find( ".py" )
   if dotPy != -1:
      modName = blobFilename[ : dotPy ]

   sys.path.append( os.getcwd() )
   try:
      import __builtin__
      blobModule = __builtin__.__import__( modName )
   except ImportError:
      print
      print "Problems importing '%s' -- is there a typo?" % modName
      sys.exit( 1 )

   allBlobs = blobModule.allBlobs()
   blobletList = None
   for (blobGroup, blobDict) in allBlobs:
      if blobGroup != blobGroupName:
         continue
      # Found the blob group!
      ( bn, blobletList ) = blobDict.get( blobName, (None, None) )

   if blobletList is None:
      print "ERROR: blob file '%s' does not contain blob group '%s', blob '%s'" \
            % ( blobFilename, blobGroupName, blobName )
      sys.exit( 1 )

   bp = BlobProperties( version=bn, blobletCount=len( blobletList ) )

   return bp


def writeBlobletByName( phys, blobletName ):
   rd = {}
   for p in phys:
      rd[ p.name ] = p.startWriteBlobletByName( blobletName )

   def allDone( rd ):
      for req in rd.values():
         if req.status == "inProgress":
            return False
      return True

   # Note: using a maxDelay of 1ms is pretty CPU-intensive, but
   #       chances are that the machine isn't doing much useful work
   #       until the blob has been written to the PHY.
   Tac.waitFor( lambda: allDone( rd ), timeout=60,
                description="bloblet '%s' to be written" % blobletName,
                maxDelay=0.001 )


def writeBlobByName( phys, blobletCount, blobFileName, blobGroupName,
                     blobName, busyPhysFn=None, showProgress=False ):

   assert busyPhysFn is None or callable( busyPhysFn )
   todo = blobletCount
   blobNum = 1
   for _ in range( todo ):
      if showProgress:
         print ( "%d" % todo ),
         sys.stdout.flush()
      # Bloblet names are of the form:
      #
      # <python module name>:<blob group name>:<blob name>:<bloblet key>
      blobletKey = blobNum - 1
      blobletName = "%s:%s:%s:%d" % \
                    ( blobFileName , blobGroupName , blobName , blobletKey )
      writeBlobletByName( phys, blobletName )
      if busyPhysFn:
         # The caller can pass in a PHY-specific function that returns
         # the list of PHYs that have not yet completed writing the
         # given bloblet.
         while busyPhysFn( phys ):
            if showProgress:
               print ".",
               sys.stdout.flush()
      blobNum += 1
      todo -= 1

