# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Afetch
import Agent
import PcsApiModels
import ReversibleSecretCli
import Tac
import Tracing
import UrlMap
import UwsgiAaa
import UwsgiConstants
import atexit
import os
import re
import simplejson

from ApiBaseModels import ModelJsonSerializer
from GenericReactor import GenericReactor
from UwsgiRequestContext import (
      HttpBadRequest,
      HttpException,
      HttpForbidden,
      HttpNotFound,
      UwsgiRequestContext
)

# Method 'api' is abstract in class 'PcsApiBase' but is not overridden
# in derived classes
# pylint: disable-msg=W0223

t0 = Tracing.trace0
t2 = Tracing.trace2
t6 = Tracing.trace6
# Only for periodic and frequent calls
t8 = Tracing.trace8

HTTP_OK = 200

# When PcsApiHandler is instantiated as Agent, the PcsHttpAgent won't be available to
# accept HTTP requests. PcsApiHandler itself sends requests to NSX e.g. full-sync
# as part of init and NSX responds by sending HTTP POST requests to CVX separately
# and may hit 404 errors.
# Since the activity loop is run as a separate thread, the workaround is to let
# PcsApiHandler defer making the HTTP requests NSX by an arbitrary time so that those
# calls will be made in the activity thread and the agent will be able to server
# HTTP requests in the main thread
AGENT_INIT_TIME = 1

ApiType = Tac.Type( "Pcs::ApiType" )

class PcsControllerApi( object ):
   apiPrefix = "/policy/api/v1"
   fullSyncApi = "/infra/sites/default/enforcement-points/%s?action=full-sync"
   ipSetApi = "/api/v1/ip-sets"
   groupNotificationWatcherApi = ( "/api/v1/notification-watchers/%s/notifications?"
                                   "action=%s" )
   groupSubscribe = "add_uri_filters"
   groupUnsubscribe = "delete_uri_filters"
   groupApiPrefix = apiPrefix + "/infra/domains/default/groups/%s"
   groupMembersApi = ( "/members/ip-addresses?enforcement_point_path="
                       "/infra/sites/default/enforcement-points/default" )

class PcsApiCaller( object ):
   def __init__( self, config ):
      self.config = config
      root = Tac.root.entity.get( '%s/%s' % ( Tac.sysname(),
                                              agentClass.__name__ ) )
      if root:
         self.afetchClient = Afetch.Client( baseDir=root )
      else:
         self.afetchClient = Afetch.Client()

   def urlPrefix( self ):
      proto = os.environ.get( "TEST_PROTO", "https" )
      return "%s://%s:%d" % ( proto, self.config.controllerIp, self.config.port )

   def contentType( self ):
      contentHeader = {
         'Content-type' : 'application/json',
         'Accept' : 'application/json'
      }
      return contentHeader

   def authenticate( self, req ):
      req.authType = Afetch.AuthType.authBasic
      req.username = self.config.username
      password = self.config.password
      try:
         password = ReversibleSecretCli.decodeKey( self.config.password )
      except TypeError:
         # Password is in plain-text
         pass
      req.password = password

   def stop( self, requestKey ):
      if not requestKey or ( requestKey not in
                             self.afetchClient.requestDir.request ):
         return
      self.afetchClient.stop( requestKey )

   def requestExists( self, method, uri ):
      uri = self.urlPrefix() + uri
      reqKey = Tac.Value( 'Afetch::RequestKey', method, uri )
      return reqKey in self.afetchClient.requestDir.request

   def request( self, method, uri, body='', callback=None ):
      t0( "Http Request", method, uri, body )
      uri = self.urlPrefix() + uri
      headers = {}
      headers.update( self.contentType() )
      req = self.afetchClient.request( method, uri, callback=callback,
                                       headers=headers, running=False )
      req.body = body
      # If controller is unavailable or request failed temporarily, we should retry
      # every 3 mins. Currrently, the default is 30 mins which is too long
      req.timeoutRefresh = 3 * 60
      req.debug = True
      req.sslVerifyPeer = False
      req.pinnedPublicKey = self.config.thumbprint
      self.authenticate( req )
      self.afetchClient.start( req )
      return req.requestKey

class PcsApiBase( object ):
   """All classes that handle a particular URL should be derived from this class"""
   def __init__( self, apiCaller, perApiState ):
      self.apiCaller = apiCaller
      self.perApiState = perApiState
      # The decorator UrlMap.urlHandler won't register the actual object. So, this is
      # a hack to replace the preregistered method for the url with a tuple of the
      # object and the method name. This helps PcsApiHandler in calling the method on
      # the real object
      uri = UrlMap.UrlToRegex()( self.api )
      for httpType, reqFunc in UrlMap.urlMap.iteritems():
         if uri in reqFunc:
            f = UrlMap.urlMap[ httpType ][ uri ]
            if type( f ) is tuple:
               # The class is reinstantiated because of config change
               continue
            t0( "Updating", httpType, self.api, self, f.__name__ )
            UrlMap.urlMap[ httpType ][ uri ] = ( self, f.__name__ )

   @property
   def api( self ):
      raise NotImplementedError

   # returns False if it is known that an error response was received by CVX for this
   # ( method + api ) request.
   def getState( self, method, uri ):
      if ( self.perApiState.method == method and uri in self.perApiState.uri ):
         return self.perApiState.success
      else:
         return True

   def logState( self, method, uri, err, errorStr ):
      self.perApiState.method = method
      self.perApiState.uri = uri
      self.perApiState.success = not err
      self.perApiState.detailString = errorStr
      self.perApiState.utcTimestamp = Tac.utcNow()

   def request( self, method, uri, body='', callback=None ):
      return self.apiCaller.request( method, uri, body, callback )

   def handleResponse( self, response ):
      t6( "Received response for ", response.requestKey )
      t6( "finished :", response.finished(), "lastError :", response.lastError,
          "statusCode :", response.statusCode )
      t6( "header :", response.header.items() )
      t6( "body :", response.body )
      if response.statusCode / 100 == 2:
         # This will remove the request from Afetch. Otherwise, Afetch keeps
         # processing the request periodically irrespective of the status code
         # of the response.
         self.apiCaller.stop( response.requestKey )
      if response.statusCode == 412 or response.statusCode == 409:
         self.handleResponseConflict()
      self.logState( response.requestKey.method, response.requestKey.uri,
                     response.statusCode / 100 != 2, response.lastErrorMessage )

   def handleResponseConflict( self ):
      pass # Derived class can override this to handle conflict responses.

class PcsFullSyncClient( PcsApiBase ):
   api = ""

   def __init__( self, config, sectionDir, apiCaller, perApiState ):
      PcsApiBase.__init__( self, apiCaller, perApiState )
      self.config_ = config
      self.sectionDir_ = sectionDir
      self.syncRequestTimer_ = Tac.ClockNotifiee( self.sendRequest,
                                           timeMin=Tac.now() + AGENT_INIT_TIME )
      # To handle the case of NSX being configured with policies after
      # CVX is configured
      self.syncCompleteTimer_ = Tac.ClockNotifiee( self.checkFullSyncComplete )
      self.retryTime_ = 30
      # For testing
      self.retries = 0

   def constructBody( self ):
      d = {}
      d[ "connection_info" ] = {}
      d[ "connection_info" ][ "resource_type" ] = "CvxConnectionInfo"
      return simplejson.dumps( d )

   def sendRequest( self ):
      uri = ( PcsControllerApi.apiPrefix +
              PcsControllerApi.fullSyncApi % self.config_.endpointName )
      self.request( Afetch.Method.POST, uri,
                    body=self.constructBody(),
                    callback=self.handleResponse )
      self.syncCompleteTimer_.timeMin = Tac.now() + self.retryTime_

   def checkFullSyncComplete( self ):
      if self.sectionDir_.section:
         # If there's at least one policy, then it means CVX received or is
         # receiving the policies
         t0( "Full-sync is complete. Resending request is not needed." )
         return
      t0( "Resending full-sync request" )
      self.sendRequest()
      self.retries += 1

class PcsRuleTranslator( PcsApiBase ):
   api = "/pcs/v1/section/{sectionId}[/]"

   def __init__( self, sectionDir, apiStatus, groupDir, apiCaller, perApiState ):
      PcsApiBase.__init__( self, apiCaller, perApiState )
      self.sectionDir = sectionDir
      self.apiStatus = apiStatus
      self.groupDir = groupDir
      self.initialSectionList = self.sectionDir.section.keys()

   @UrlMap.urlHandler( ( 'DELETE', ), api )
   def handleDelete( self, requestContext, sectionId ):
      t0( "Received policy delete request for section", sectionId )
      if sectionId not in self.sectionDir.section:
         t0( "Section not present" )
         self.raiseException( HttpNotFound, "Section not found", 'DELETE',
                              requestContext, False )
      del self.sectionDir.section[ sectionId ]
      return '{}'

   def raiseException( self, exceptionType, msg, method, rc, fullSyncCheck=True ):
      if fullSyncCheck and not self.apiStatus.policyFullSyncComplete:
         self.handleEndOfFullSync( rc )
      t0( msg )
      self.logState( method, self.api, True, msg )
      raise exceptionType( msg )

   def doFullSyncCheck( self, rc ):
      method = 'POST'
      if rc.getHeader( 'HTTP_FULL_SYNC' ) == 'true':
         index = rc.getHeader( 'HTTP_SECTION_INDEX' )
         size = rc.getHeader( 'HTTP_SECTION_LIST_SIZE' )
         if not ( index and size ):
            msg = "Did not receive SECTION_%s as part of FULL_SYNC" % (
                     "LIST_SIZE" if not size else "INDEX" )
            self.raiseException( HttpBadRequest, msg, method, rc, False )
         try:
            index = int( index )
            size = int( size )
         except ValueError:
            msg = "Received invalid value for SECTION_INDEX/SECTION_LIST_SIZE header"
            t0( "INDEX", index, "LIST_SIZE", size )
            self.raiseException( HttpBadRequest, msg, method, rc, False )
      elif not self.apiStatus.policyFullSyncComplete:
         self.raiseException( HttpForbidden, "CVX is in full-sync window", method,
                              rc, False )

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext, sectionId ):
      method = 'POST'
      t0( "Received policy for section", sectionId )
      self.doFullSyncCheck( requestContext )
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      t8( data )
      model = PcsApiModels.SectionModel()
      model.populateModelFromJson( data )
      if sectionId != model.getModelField( 'id' ).value:
         msg = "SectionId of URL and JSON data don't match for " + sectionId
         self.raiseException( HttpBadRequest, msg, method, requestContext )
      section = self.sectionDir.newSection( sectionId )
      try:
         model.toSysdb( section )
      except ValueError:
         msg = "Parsing policy section %s failed" % sectionId
         self.raiseException( HttpBadRequest, msg, method, requestContext )
      # Handle unresolved nsgroups
      for rule in section.rule.itervalues():
         for target, typ in rule.source.iteritems():
            if typ == 'group' and target not in self.groupDir.group:
               group = self.groupDir.newGroup( target )
               # Maintain references to the group
               group.section.add( section )
         for target, typ in rule.destination.iteritems():
            if typ == 'group' and target not in self.groupDir.group:
               group = self.groupDir.newGroup( target )
               group.section.add( section )
      # If this POST call was in response to a full-sync request, handle the
      # delta of sections at the end
      if sectionId in self.initialSectionList:
         self.initialSectionList.remove( sectionId )
      if not self.apiStatus.policyFullSyncComplete:
         self.handleEndOfFullSync( requestContext )
      self.logState( method, self.api, False, "Processed " + sectionId )
      return '{}'

   def handleEndOfFullSync( self, requestContext ):
      sectionIndex = int( requestContext.getHeader( 'HTTP_SECTION_INDEX' ) )
      sectionListSize = int( requestContext.getHeader( 'HTTP_SECTION_LIST_SIZE' ) )
      if sectionIndex != sectionListSize:
         # It's a little dicey because CVX can receive policies out of order.
         # But, we don't know NSX behavior when http request for one of the
         # policies fails or times out for any reason. Need to clarify if NSX
         # retries requests that failed right away or if it handles such failed
         # requests after sending the rest of the policies.
         t0( "%d/%d - Not end of full sync" % ( sectionIndex, sectionListSize ) )
         return
      for sectionId in self.initialSectionList:
         del self.sectionDir.section[ sectionId ]
      # Cleanup unknown groups
      policyNsGroups = set( [] )
      for section in self.sectionDir.section.itervalues():
         for rule in section.rule.itervalues():
            policyNsGroups.update( [ k for k, v in rule.source.iteritems()
                                       if v == 'group' ] )
            policyNsGroups.update( [ k for k, v in rule.destination.iteritems()
                                       if v == 'group' ] )
      obsoleteGroups = set( self.groupDir.group.keys() ) - policyNsGroups
      t0( "Obsolete groups", obsoleteGroups )
      for grp in obsoleteGroups:
         del self.groupDir.group[ grp ]
      # Let Pcs agent know if we synced everything from NSX
      self.setSyncComplete()

   def setSyncComplete( self ):
      self.apiStatus.policyFullSyncComplete = True
      self.apiStatus.controllerSyncComplete = not self.groupDir.group

   def doCleanup( self ):
      self.groupDir.group.clear()
      self.sectionDir.section.clear()

class PcsEpDbTagReactor( Tac.Notifiee ):
   notifierTypeName = "Pcs::TagInfo"

   def __init__( self, tag, callback ):
      t6( "Tag reactor : %s" % tag.name )
      Tac.Notifiee.__init__( self, tag )
      self.callback_ = callback
      self.tag_ = tag

   @Tac.handler( 'host' )
   def handleHostInTagInfo( self, key ):
      t2( "%s, %s in handleHostInTagInfo" % ( self.notifier_.name, key ) )
      self.callback_( self.tag_.name )

class PcsEpDbHostReactor( Tac.Notifiee ):
   notifierTypeName = "Pcs::HostInfo"

   def __init__( self, host, callback ):
      t6( "Host reactor : %s" % host.ipAddr )
      Tac.Notifiee.__init__( self, host )
      self.callback_ = callback
      self.host_ = host

   @Tac.handler( 'tag' )
   def handleTagInHostInfo( self, key ):
      t2( "%s, %s in handleTagInHostInfo" % ( self.notifier_.ipAddr, key ) )
      self.callback_( self.host_.ipAddr )

class PcsTagIpSetManager( PcsApiBase, Tac.Notifiee ):
   api = ""
   notifierTypeName = "Pcs::PolicyEndpointDb"

   def __init__( self, policyEpDb, controllerTagDb, apiCaller, perApiState ):
      Tac.Notifiee.__init__( self, policyEpDb )
      PcsApiBase.__init__( self, apiCaller, perApiState )
      self.policyEpDb = policyEpDb
      self.controllerTagDb = controllerTagDb
      self.controllerTagDbSynced = False
      self.epHostReactor = Tac.collectionChangeReactor(
         policyEpDb.host, PcsEpDbHostReactor,
         reactorArgs=( self.handleHostUpdate, ) )
      self.epTagReactor = Tac.collectionChangeReactor(
         policyEpDb.tag, PcsEpDbTagReactor,
         reactorArgs=( self.handleTagUpdate, ) )
      self.syncTimer_ = Tac.ClockNotifiee( self.maybeFetchFromController,
                                           timeMin=Tac.now() + AGENT_INIT_TIME )

   def maybeFetchFromController( self ):
      # if controllerTagDb is empty, then fetch all ipsets from
      # controller
      if not self.controllerTagDb.ipset:
         t2( "Making a ipset GET request to check ipset content with controller" )
         self.request( Afetch.Method.GET,
                       PcsControllerApi.ipSetApi,
                       callback=self.processIpSetGetResponse )
      else:
         self.controllerTagDbSynced = True

   def processIpSetGetResponse( self, response ):
      self.handleResponse( response )
      if not response.body:
         t0( "Empty body for response to Ip Set GET" )
         return
      data = simplejson.loads( response.body )

      model = PcsApiModels.IpSetResultModel()
      model.populateModelFromJson( data )
      model.toSysdb( self.controllerTagDb )
      self.controllerTagDbSynced = True
      # Do the real processing now
      self.handleFullSwitchIpSets()

   def constructBody( self, tagName, ipAddrs, ipsetId="", rev=0 ):
      d = {}
      d[ 'display_name' ] = "Arista_CVX_" + tagName
      d[ 'ip_addresses' ] = [ str( ipAddr ) for ipAddr in ipAddrs ]
      d[ 'resource_type' ] = "CvxConnectionInfo"
      if ipsetId:
         d[ 'id' ] = ipsetId
      d[ 'tags' ] = [ { 'scope' : 'cvx', 'tag' : tagName } ]
      d[ '_revision' ] = rev
      return simplejson.dumps( d )

   @Tac.handler( 'initialSwitchMountsComplete' )
   def handleFullSwitchIpSets( self ):
      if not self.policyEpDb.initialSwitchMountsComplete:
         t0( "Initial switch mounts not complete, dont reconcile yet" )
         return
      # we can start comparing policyEpDb with controllerTagDb
      # Set( policyEpDb && controllerTagDb ) =  do nothing for same ip address sets
      # Set( policyEpDb' && controllerTagDb ) =  send delete to controller
      # Set( policyEpDb && controllerTagDb' ) = send update/add to controller
      policyTags = set( self.policyEpDb.tag.keys() )
      controllerTags = { ipset.tag for ipset in self.controllerTagDb.ipset.values() }
      allTags = policyTags | controllerTags
      deleteSet = allTags - policyTags
      t0( "Tags for deletion", deleteSet )

      updateSet = policyTags - controllerTags
      for ipset in self.controllerTagDb.ipset.values():
         if ipset.tag in policyTags:
            if set( ipset.ipAddr.keys() ) != set(
                  self.policyEpDb.tag[ ipset.tag ].host.keys() ):
               t2( "Controller IpAddrs,", ipset.ipAddr.keys(),
                   " policyEp IpAddrs", self.policyEpDb.tag[ ipset.tag ].host.keys()
               )
               updateSet.add( ipset.tag )
      t0( "Tags for update", updateSet )

      for ipset in self.controllerTagDb.ipset.values():
         if ipset.tag in deleteSet:
            t6( "Deleting tag", ipset.tag )
            self.doDeleteTag( ipset.id )
         elif ipset.tag in updateSet:
            t6( "Updating tag", ipset.tag )
            self.doUpdateTag( ipset.id, ipset.revision, ipset.tag )
            updateSet.remove( ipset.tag )
      for tag in updateSet:
         t6( "Updating tag", tag )
         self.doUpdateTag( "", 0, tag )

   def handleDeleteResponse( self, response ):
      self.handleResponse( response )
      # Extract the id
      ipsetid = re.split( '[/ ?]', response.requestKey.uri )[ -2 ]
      if response.statusCode / 100 == 2:
         t0( "Controller deleted tag successfully", ipsetid )
      else:
         # We did attempt it, so we will still delete it locally so we dont keep
         # repeating delete request. If this manifests as a controller policy to be
         # applied to non-existent interfaces, the ACL entries will not be applied to
         # those interfaces.
         t0( "Controller unable to delete tag", ipsetid )
         # fallthrough
      t0( "Deleting tag with id", ipsetid, "from controller tag DB" )
      if self.controllerTagDb.ipset.get( ipsetid ):
         del self.controllerTagDb.ipset[ ipsetid ]

   def doDeleteTag( self, ipsetid ):
      if not ipsetid:
         return
      delim = os.environ.get( "MOCK_CONTROLLER_DELIMITER", "/" )
      api = ( PcsControllerApi.ipSetApi + delim + ipsetid + '?force=true' )
      t0( "Making Afetch DELETE ipset call for", ipsetid )
      self.request( Afetch.Method.DELETE, api, body="",
                    callback=self.handleDeleteResponse )

   def doUpdateTag( self, ipsetid, revision, tagName ):
      if not tagName:
         t0( "tagName is mandatory to update the controller ip set" )
         return
      if not self.controllerTagDbSynced:
         t0( "Waiting for GET response to be processed" )
         return

      if ipsetid:
         delim = os.environ.get( "MOCK_CONTROLLER_DELIMITER", "/" )
         api = ( PcsControllerApi.ipSetApi + delim + ipsetid )
         if self.apiCaller.requestExists( Afetch.Method.PUT, api ):
            t6( "Postponing update for", ipsetid )
            return
         if ( not self.policyEpDb.tag.get( tagName ) or
              not self.controllerTagDb.ipset.get( ipsetid ) ):
            t0( "Tag %s has been deleted, no update necessary" % tagName )
            return
         # make a request only if ipAddresses have changed.
         if set( self.policyEpDb.tag[ tagName ].host.keys() ) == set(
               self.controllerTagDb.ipset[ ipsetid ].ipAddr.keys() ):
            t0( "IP addresses are in sync, no further updates necessary" )
            # It is possible that we have sent out multiple PUT updates, one of which
            # is accepted, rest are returned with error code ( because of revision
            # mismatch ). If we are finally in sync, we should clear perApiState
            # failures for this Tag only.
            if self.getState( Afetch.Method.PUT, ipsetid ) == False:
               self.logState( Afetch.Method.PUT,
                              api,
                              False,
                              "Controller and CVX are in sync" )
            return
         # this is an update of ip addresses only

         t0( "Making Afetch PUT ipset call for", ipsetid )
         body = self.constructBody( tagName,
                  self.policyEpDb.tag[ tagName ].host.keys(),
                  ipsetId=ipsetid, rev=revision )
         self.request( Afetch.Method.PUT, api, body=body,
                       callback=self.handleUpdatePutResponse )
      else:
         # the controller does not know about this tag
         # if there are no hosts yet, don't make the call
         if tagName not in self.policyEpDb.tag:
            t0( "Tag", tagName, "not present in controller endpoint database." )
            return
         if not self.policyEpDb.tag[ tagName ].host:
            t0( "Not making POST ip set call: no known hosts yet for ", tagName )
            return

         # Afetch requestKey is a combination of Method + URI. Modify the URI with a
         # fragment identifier to make the key unique for a POST call per tag.
         # The fragment identifier is a client side anchor selector, and the server
         # should ignore it.
         api = PcsControllerApi.ipSetApi + '#' + tagName
         if self.apiCaller.requestExists( Afetch.Method.POST, api ):
            t8( "Not making a POST call, as request for %s exists in requests queue"
                % tagName )
            return
         t2( "Making Afetch POST ipset call for", tagName )
         self.request( Afetch.Method.POST, api, body=self.constructBody(
            tagName, self.policyEpDb.tag[ tagName ].host.keys() ),
                       callback=self.handleUpdatePostResponse )

   def handleUpdatePutResponse( self, response ):
      self.handleResponse( response )
      # Extract the id
      ipsetid = response.requestKey.uri.split( '/' )[ -1 ]
      if not response.statusCode:
         t0( "Expected a valid status code" )
         return
      if response.statusCode / 100 != 2:
         t0( "Ip-Set PUT response", response.statusCode, " return" )
         # Do not automatically keep retrying. Make a new request with the latest
         # ip addr set and revision.
         self.apiCaller.stop( response.requestKey )
      else:
         self.handleUpdateResponse( response, ipsetid )

      # Updates might have been pending while this response was awaited.
      ipset = self.controllerTagDb.ipset.get( ipsetid )
      if not ipset:
         t0( "Ipset ", ipsetid, "has been deleted" )
         return
      self.doUpdateTag( ipset.id, ipset.revision, ipset.tag )

   def handleUpdatePostResponse( self, response ):
      self.handleResponse( response )
      if not response.statusCode:
         t0( "Expected a valid status code" )
         return

      if response.statusCode / 100 != 2:
         t0( "Ip-Set POST response", response.statusCode, " return" )
         return

      # Extract the id
      if not response.body:
         t0( "Empty response body when handling POST response" )
         return
      data = simplejson.loads( response.body )
      ipsetid = data.get( 'id' )  # pylint: disable=E1103
      if not ipsetid:
         t0( "Expected id in POST response not found" )
         return
      self.handleUpdateResponse( response, ipsetid )
      ipset = self.controllerTagDb.ipset.get( ipsetid )
      if ipset:
         # if there are new updates while we waited for the post response.
         self.doUpdateTag( ipset.id, ipset.revision, ipset.tag )

   def handleUpdateResponse( self, response, ipsetid ):
      t0( "Controller updated tag", ipsetid )
      ipset = self.controllerTagDb.ipset.get( ipsetid )
      if not response.body:
         t0( "Expected response to contain IP and tag information, was empty" )
         return

      data = simplejson.loads( response.body )
      if not ipset:
         # by contract, there needs to be only 1 tag in the response
         # pylint: disable=E1103
         if not data.get( 'tags' ) or not data.get( 'tags' )[ 0 ].get( 'tag' ):
            t0( "Response does not contain tags, ignoring response" )
            return
         else:
            tagName = data[ 'tags' ][ 0 ][ 'tag' ]
         if not data.get( '_revision' ):
            t0( "Expected revision, assuming 0" )
         revision = data.get( '_revision', 0 )
         # pylint: enable=E1103
         ipset = self.controllerTagDb.newIpset( ipsetid, revision, tagName )
      model = PcsApiModels.IpSetModel()
      model.populateModelFromJson( data )
      model.toSysdb( ipset ) # populates IP Addresses as in response.
      ipset.parseComplete = True

   @Tac.handler( 'tag' )
   def handleTagUpdate( self, key ):
      t2( "Handling tag update for ", key )
      if not self.policyEpDb.initialSwitchMountsComplete:
         t2( "Initial switch mounts not complete, ignore tag update" )
         return

      ipsetid, revision = "", 0
      for ipset in self.controllerTagDb.ipset.values():
         if ipset.tag == key:
            ipsetid = ipset.id
            revision = ipset.revision
            t6( "Existing id for tag", ipset.tag, "is ", ipsetid, "at", revision )
            break
      if key in self.notifier().tag:
         t2( "Tag ", key, " added to to TagInfo collection" )
         # schedule an update to the Controller
         self.doUpdateTag( ipsetid, revision, key )
      else:
         self.doDeleteTag( ipsetid )

   @Tac.handler( 'host' )
   def handleHostUpdate( self, key ):
      t0( "Handling host update for ", key )
      if not self.policyEpDb.initialSwitchMountsComplete:
         t0( "Initial switch mounts not complete, ignore host update" )
         return

      if key in self.notifier().host:
         t0( "IpAddr ", key, " added to to HostInfo collection" )
         # schedule an update to the Controller for all tags associated with this
         # host
         updateTagSet = set( self.policyEpDb.host[ key ].tag.keys() )
         for ipset in self.controllerTagDb.ipset.values():
            if ipset.tag in updateTagSet:
               self.doUpdateTag( ipset.id, ipset.revision, ipset.tag )
               updateTagSet.remove( ipset.tag )
         for tag in updateTagSet:
            self.doUpdateTag( "", 0, tag )
      else:
         for ipset in self.controllerTagDb.ipset.values():
            if key in ipset.ipAddr:
               t0( "An existing ipset id for ", key, " is ", ipset.id, "at revision",
                   ipset.revision )
               self.doUpdateTag( ipset.id, ipset.revision, ipset.tag )

   def handleResponseConflict( self ):
      # In observed cases, a conflict response indicates revision mismatch.
      # The only way to recover is to re-get
      t0( "Handling error response indicating server conflict" )
      self.controllerTagDbSynced = False
      self.controllerTagDb.ipset.clear()
      self.maybeFetchFromController()

class PcsGroupResolver( PcsApiBase, Tac.Notifiee ):
   notifierTypeName = "Pcs::PolicyGroupDir"
   api = "/pcs/v1/nsgroup/notification[/]"
   notifId = "group.change_notification"

   def __init__( self, config, groupDir, sectionDir, apiStatus,
                 apiCaller, perApiState ):
      PcsApiBase.__init__( self, apiCaller, perApiState )
      Tac.Notifiee.__init__( self, groupDir )
      self.config = config
      self.groupDir = groupDir
      self.sectionDir = sectionDir
      self.apiStatus = apiStatus
      self.apiCaller = apiCaller
      self.syncTimer_ = Tac.ClockNotifiee( self.sync,
                                           timeMin=Tac.now() + AGENT_INIT_TIME )
      self.retryTimer_ = Tac.ClockNotifiee( self.retryRequests,
                                           timeMin=Tac.endOfTime )
      self.defaultRetryTime = 180
      self.requestsToRetry = {} # key is reqKey, val is groupName pending resolution
      self.fullSyncCheckInterval_ = 10
      self.syncCompleteTimer_ = Tac.ClockNotifiee( self.checkFullSyncComplete,
                                           timeMin=Tac.now() )
      # Dict of groupname and the ipset currently being processed. Gets cleared once
      # the group is resolved.
      self.groupIpSet = {}

   def nudgeRetryClock( self ):
      t2( "Nudging retry clock" )
      self.retryTimer_.timeMin = Tac.now() + self.defaultRetryTime

   def retryRequests( self ):
      t2( "Retrying any pending group resolution requests",
          self.requestsToRetry.values() )
      if not self.requestsToRetry:
         t2( "No more requests to retry " )
         return
      reqs = self.requestsToRetry.copy()
      self.requestsToRetry.clear()
      for reqKey, groupName in reqs.iteritems():
         group = self.groupDir.get( groupName )
         if group and group.resolved:
            t2( "group name", groupName, "already resolved" )
         else:
            t2( "Retrying group resolution for", groupName )
            self.apiCaller.stop( reqKey )
            # Retry the exact query. See if the query is part of a sequence of
            # queries to find all the hosts that are part of the group.
            cursor = None
            m = re.search( r'cursor=(\d+)', reqKey.uri )
            if m:
               cursor = m.group( 1 )
            self.doResolveGroup( groupName, cursor )

   def sync( self ):
      for name in self.groupDir.iterkeys():
         self.handleGroup( name )

   def subscribeApi( self, name ):
      return ( ( PcsControllerApi.groupNotificationWatcherApi + "#%s" ) %
                  ( self.config.notificationUuid,
                    PcsControllerApi.groupSubscribe,
                    name ) )

   def unsubscribeApi( self, name ):
      return ( ( PcsControllerApi.groupNotificationWatcherApi + "#%s" ) %
                  ( self.config.notificationUuid,
                    PcsControllerApi.groupUnsubscribe,
                    name ) )

   @Tac.handler( 'group' )
   def handleGroup( self, name ):
      if name not in self.groupDir:
         # Withdraw pending group notification and resolution requests, if any
         reqKey = Tac.Value( 'Afetch::RequestKey', Afetch.Method.POST,
                             self.subscribeApi( name ) )
         self.apiCaller.stop( reqKey )
         api = ( PcsControllerApi.groupApiPrefix % name +
                 PcsControllerApi.groupMembersApi )
         reqKey = Tac.Value( 'Afetch::RequestKey', Afetch.Method.GET, api )
         self.apiCaller.stop( reqKey )
         self.doUnregisterGroupForNotifications( name )
         return

      self.doRegisterGroupForNotifications( name )
      self.doResolveGroup( name )

   def registrationBody( self, name ):
      d = {}
      d[ "notification_id" ] = self.notifId
      # TODO charanjith : It's efficient to send all groups at once maybe once an
      # one or all sections are processed
      d[ "uri_filters" ] = [ PcsControllerApi.groupApiPrefix % name ]

      return simplejson.dumps( d )

   def doRegisterGroupForNotifications( self, name ):
      self.request( Afetch.Method.POST, self.subscribeApi( name ),
                    body=self.registrationBody( name ),
                    callback=self.handleResponse )

   def doUnregisterGroupForNotifications( self, name ):
      self.request( Afetch.Method.POST, self.unsubscribeApi( name ),
                    body=self.registrationBody( name ),
                    callback=self.handleResponse )

   def doResolveGroup( self, name, cursor=None ):
      uri = ( ( PcsControllerApi.groupApiPrefix % name ) +
              PcsControllerApi.groupMembersApi )
      if cursor:
         uri = uri + '&cursor=%s' % cursor
      reqKey = self.request( Afetch.Method.GET, uri,
                             callback=self.processIpSet )
      t2( "Group resolution requested for", name )
      self.requestsToRetry[ reqKey ] = name

   def processIpSet( self, response ):
      self.handleResponse( response )
      # Extract name from ../group/<name>/members/ip-addresses
      url = response.requestKey.uri.split( "/members/ip-addresses" )[ 0 ]
      groupName = url.split( '/' )[ -1 ]
      if groupName not in self.groupDir:
         return
      t2( "Group resolution response for", groupName )
      self.nudgeRetryClock() # schedule retry after initial set of responses.
      if not response.body:
         t0( "Empty body in response for group resolution", groupName )
         return
      if response.statusCode != 0 and response.requestKey in self.requestsToRetry:
         t2( "Received response for request", groupName )
         del self.requestsToRetry[ response.requestKey ]
      data = simplejson.loads( response.body )
      # Process the ipSet we received for this request.
      model = PcsApiModels.GroupIpSetModel()
      model.populateModelFromJson( data )
      group = self.groupDir.get( groupName )
      if groupName in self.groupIpSet:
         self.groupIpSet[ groupName ] += data[ "results" ]
      else:
         self.groupIpSet[ groupName ] = data[ "results" ]
      # Set the resolution state of the group to False so that all the hosts are
      # added before any ACLs are generated for them. Else, Pcs starts generating
      # ACLs as soon as it sees hosts under a group.
      group.resolved = False
      model.toSysdb( group )
      # If the response contains the field 'cursor', then there is more data to be
      # fetched. Requery NSX-T passing in 'cursor' as an additional argument to the
      # GET request.
      if "cursor" in data:
         self.doResolveGroup( groupName, cursor=data[ "cursor" ] )
         return
      group.resolved = True
      # Cleanup the dead IPs if the group is fully resolved.
      for ip in group.ipAddr:
         if ( ip.stringValue not in self.groupIpSet[ groupName ] and
              # Cover the /32 case
              ip.ipGenAddr.stringValue not in self.groupIpSet[ groupName ] ):
            del group.ipAddr[ ip ]
      del self.groupIpSet[ groupName ]

   def checkFullSyncComplete( self ):
      # Notify Pcs agent that the full sync is complete
      if ( self.apiStatus.policyFullSyncComplete and
           not self.apiStatus.controllerSyncComplete ):
         self.apiStatus.controllerSyncComplete = all(
               [ g.resolved for g in self.groupDir.group.itervalues() ] )
      t2( "Full-sync status: policy=%s group=%s" % (
            self.apiStatus.policyFullSyncComplete,
            self.apiStatus.controllerSyncComplete ) )
      if self.apiStatus.controllerSyncComplete:
         self.syncCompleteTimer_.timeMin = Tac.endOfTime
      else:
         self.syncCompleteTimer_.timeMin = Tac.now() + self.fullSyncCheckInterval_

   @UrlMap.urlHandler( ( 'POST', 'DELETE' ), api )
   def handleGroupChangeNotification( self, requestContext ):
      method = requestContext.getRequestType()
      self.logState( method, self.api, False, "Received group change notification" )
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      t8( data )
      model = PcsApiModels.GroupNotificationModel()
      model.populateModelFromJson( data )
      refreshNeeded = model.getModelField( 'refresh_needed' ).value
      groupCount = model.getModelField( 'result_count' ).value
      changedGroupUris = model.uris( self.notifId )
      if changedGroupUris:
         for uri in changedGroupUris:
            method = requestContext.getRequestType()
            if type( uri ) is dict:
               assert 'operation' in uri
               method = uri[ 'operation' ]
               assert 'uri' in uri
               groupName = uri.get( 'uri' ).split( '/' )[ -1 ]
            else:
               groupName = uri.split( '/' )[ -1 ]
            if groupName not in self.groupDir:
               return
            if method in ( 'POST', 'UPDATE' ):
               self.doResolveGroup( groupName )
            elif method == 'DELETE':
               # NSX doesn't send DELETE notification unless none of the sections
               # have references to the group. So, we'll rely on NSX's correctness
               # and skip internal checks on our current sections
               del self.groupDir.group[ groupName ]
            else:
               # For testing
               assert False, "Unrecognized method"
            t0( "Processed", method, "for", groupName )
      elif refreshNeeded and groupCount == 0:
         t0( "Resolving all existing groups" )
         for name in self.groupDir:
            self.doResolveGroup( name )
      return '{}'

class PcsStatusResponder( PcsApiBase ):
   api = "/pcs/v1/section/{sectionId}/status[/]"

   def __init__( self, policyStatusDir, perApiState ):
      PcsApiBase.__init__( self, None, perApiState ) # ApiCaller is unnecessary
      self.policyStatusDir = policyStatusDir

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleStatusCall( self, requestContext, sectionId ):
      t0( "Received status request for section", sectionId )
      section = self.policyStatusDir.get( sectionId )
      if not section:
         self.logState( 'GET', self.api, '404', "Section not found" )
         raise HttpNotFound( "Section not found" )
      model = PcsApiModels.StatusModel()
      model.fromSysdb( section )
      self.logState( 'GET', self.api, None, "For %s" % sectionId )
      return simplejson.dumps( model, cls=ModelJsonSerializer )

class PcsHitCountResponder( PcsApiBase ):
   api = "/pcs/v1/section/{sectionId}/hit-count[/]"

   def __init__( self, policyStatusDir, perApiState ):
      PcsApiBase.__init__( self, None, perApiState )
      self.policyStatusDir = policyStatusDir

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleHitCountCall( self, requestContext, sectionId ):
      t0( "Received hit-count request for section", sectionId )
      section = self.policyStatusDir.get( sectionId )
      if not section:
         self.logState( 'GET', self.api, '404', "Section not found" )
         raise HttpNotFound( "Section not found" )
      model = PcsApiModels.HitCountModel()
      model.fromSysdb( section )
      self.logState( 'GET', self.api, None, "For %s" % sectionId )
      return simplejson.dumps( model, cls=ModelJsonSerializer )

class PcsConfigReactor( Tac.Notifiee ):
   """A TACC notifiee that reacts to changes in the Pcs agent configuration."""

   notifierTypeName = "Pcs::Config"

   def __init__( self, config, apiStatus ):
      t0( "PcsConfigReactor init" )
      Tac.Notifiee.__init__( self, config )
      self.apiStatus_ = apiStatus
      self.config_ = config
      self.checkConfigReady()

   @Tac.handler( "enabled" )
   def handleEnabled( self ):
      t0( "Config enabled changed to", self.notifier_.enabled )
      self._handleConfig()

   @Tac.handler( "endpointName" )
   def handleEndpointName( self ):
      t0( "Config endpointName changed to", self.notifier_.endpointName )
      self._handleConfig()

   @Tac.handler( "controllerIp" )
   def handleIpAddr( self ):
      t0( "ControllerIp changed to", self.notifier_.controllerIp )
      self._handleConfig()

   @Tac.handler( "port" )
   def handlePort( self ):
      t0( "Port changed to", self.notifier_.port )
      self._handleConfig()

   @Tac.handler( "username" )
   def handleUsername( self ):
      t0( "Username changed to", self.notifier_.username )
      self._handleConfig()

   @Tac.handler( "password" )
   def handlePassword( self ):
      t0( "Password changed to", self.notifier_.password )
      self._handleConfig()

   @Tac.handler( "thumbprint" )
   def handleThumbprint( self ):
      t0( "Thumbprint changed to", self.notifier_.thumbprint )
      self._handleConfig()

   @Tac.handler( "notificationUuid" )
   def handleNotifUuid( self ):
      t0( "NotificationUuid changed to", self.notifier_.notificationUuid )
      self._handleConfig()

   def _handleConfig( self ):
      # When config changes, we should start all over again. So, set ready=False to
      # trigger cleanup of all API handlers
      self.apiStatus_.ready = False
      self.checkConfigReady()

   def checkConfigReady( self ):
      configReady = bool( self.config_.enabled and
                          self.config_.username and
                          self.config_.endpointName and
                          self.config_.controllerIp and
                          self.config_.thumbprint and
                          self.config_.notificationUuid and
                          self.config_.port )
      t0( "%s configuration" % ( "Sufficient" if configReady else "Insufficient" ) )
      self.apiStatus_.ready = configReady


class ShutdownReactor( object ):
   def __init__( self, pcsConfig, controllerStatus, clusterStatus, httpConfig,
                 apiStatus ):
      self.pcsConfig = pcsConfig
      self.controllerStatus = controllerStatus
      self.clusterStatus = clusterStatus
      self.httpConfig = httpConfig
      self.apiStatus = apiStatus
      self.pcsReactor = GenericReactor( self.pcsConfig, [ 'enabled' ],
                                        self.handleEnabled )
      self.cvxReactor = GenericReactor( self.controllerStatus, [ 'enabled' ],
                                        self.handleEnabled )
      self.clusterReactor = GenericReactor(
                                 self.clusterStatus, [ 'isStandaloneOrLeader' ],
                                 self.handleEnabled )
      self.httpReactor = GenericReactor( self.httpConfig, [ 'enabled' ],
                                        self.handleEnabled )
      self.handleEnabled()

   def handleEnabled( self, notifiee=None ):
      t0( "pcs=%s cvx=%s leader=%s httpd=%s" %
          ( self.pcsConfig.enabled, self.controllerStatus.enabled,
            self.clusterStatus.isStandaloneOrLeader, self.httpConfig.enabled ) )
      enabled = True
      if not ( self.pcsConfig.enabled and self.controllerStatus.enabled and
               self.clusterStatus.isStandaloneOrLeader ):
         self.apiStatus.ready = False
         enabled = False
      elif not self.httpConfig.enabled:
         # If Httpd is shutdown, we'll shutdown PcsHttpAgent but continue to run
         # Pcs service. So, cleanup shouldn't be done
         enabled = False
      self.apiStatus.enabled = enabled

class PcsApiHandler( Agent.Agent ):
   def __init__( self, em ):
      Agent.Agent.__init__( self, em, agentName="PcsHttpAgent" )
      self.fullSyncClient = None
      self.notifRegister = None
      self.ipSetManager = None
      self.groupResolver = None
      self.ruleTranslator = None
      self.statusResponder = None
      self.hitcountResponder = None
      self.configReactor = None
      self.readyReactor = None
      self.shutdownReactor = None
      self.apiCaller = None
      self.genApiState = None

   # Parameters differ from overridden 'doInit' method
   # pylint: disable-msg=arguments-differ
   def doInit( self, em ):
      # pylint: disable-msg=W0201
      mg = em.mountGroup()
      self.config = mg.mount( "pcs/config", "Pcs::Config", "r" )
      self.apiStatus = mg.mount( "pcs/apiStatus", "Pcs::ApiStatus", "w" )
      self.policyEpDb = mg.mount( "pcs/controller/endpoint",
                                 "Pcs::PolicyEndpointDb", "r" )
      self.policyStatus = mg.mount( "pcs/controller/status",
                                    "Pcs::PolicyStatusDir", "r" )
      self.sectionDir = mg.mount( "pcs/input/http/section",
                                  "Pcs::PolicySectionDir", "w" )
      self.groupDir = mg.mount( "pcs/input/http/group",
                                "Pcs::PolicyGroupDir", "wS" )
      self.controllerTagDb = mg.mount( "pcs/input/http/ipset",
                                       "Pcs::IpSetSyncDir", "wS" )
      self.ipStatus = mg.mount( "ip/status", "Ip::Status", "r" )
      self.controllerStatus = mg.mount( "controller/status",
                                        "Controllerdb::Status", "r" )
      self.clusterStatus = mg.mount( "controller/cluster/statusDir",
                                     "ControllerCluster::ClusterStatusDir", "r" )
      self.httpConfig = mg.mount( "mgmt/capi/config", "HttpService::Config", "r" )

      def _mountsComplete():
         self.aaaManager = UwsgiAaa.UwsgiAaaManager( em.sysname() )
         self.genApiState = self.apiStatus.newPerApiState( ApiType.general )
         self.readyReactor = GenericReactor( self.apiStatus, [ 'ready' ],
                                             self.handleReady, callBackNow=True )
         self.shutdownReactor = ShutdownReactor(
                                    self.config, self.controllerStatus,
                                    self.clusterStatus.status[ 'default' ],
                                    self.httpConfig.service[ 'pcs' ],
                                    self.apiStatus )
         self.configReactor = PcsConfigReactor( self.config, self.apiStatus )

      mg.close( _mountsComplete )

   def handleReady( self, notifiee=None ):
      t0( "handleReady", self.apiStatus.ready )
      if self.apiStatus.ready:
         self.apiCaller = PcsApiCaller( self.config )
         self.ruleTranslator = PcsRuleTranslator(
            self.sectionDir, self.apiStatus, self.groupDir, self.apiCaller,
            self.apiStatus.newPerApiState( ApiType.ruleTranslator ) )
         self.ipSetManager = PcsTagIpSetManager(
            self.policyEpDb, self.controllerTagDb, self.apiCaller,
            self.apiStatus.newPerApiState( ApiType.tagUpdater ) )
         self.groupResolver = PcsGroupResolver(
            self.config, self.groupDir, self.sectionDir, self.apiStatus,
            self.apiCaller, self.apiStatus.newPerApiState( ApiType.groupResolver ) )
         self.statusResponder = PcsStatusResponder(
            self.policyStatus,
            self.apiStatus.newPerApiState( ApiType.statusResponder ) )
         self.hitcountResponder = PcsHitCountResponder(
            self.policyStatus,
            self.apiStatus.newPerApiState( ApiType.hitCountResponder ) )
         self.apiStatus.policyFullSyncComplete = False
         self.apiStatus.controllerSyncComplete = False
         self.fullSyncClient = PcsFullSyncClient(
            self.config, self.sectionDir, self.apiCaller,
            self.apiStatus.newPerApiState( ApiType.fullSync ) )
      else:
         self.doCleanup()

   def doCleanup( self ):
      self.fullSyncClient = None
      self.notifRegister = None
      self.ipSetManager = None
      self.groupResolver = None
      if self.ruleTranslator:
         self.ruleTranslator.doCleanup()
         self.ruleTranslator = None
      self.statusResponder = None
      self.hitcountResponder = None
      self.apiCaller = None
      self.apiStatus.perApiState.clear()
      self.apiStatus.policyFullSyncComplete = False
      self.apiStatus.controllerSyncComplete = False
      self.controllerTagDb.ipset.clear()

   def processRequest( self, request ):
      userCtx = None
      urlPath = '*'
      try:
         if not ( self.aaaManager and getattr( self.apiStatus, 'ready', False ) ):
            raise HttpForbidden( "CVX not ready for handling HTTP requests" )
         requestContext = UwsgiRequestContext( request, self.aaaManager )
         url = requestContext.getParsedUrl()
         userCtx = requestContext.authenticate()
         requestType = requestContext.getRequestType()
         func, urlArgs = UrlMap.getHandler( requestType, url )
         urlPath = url.path
         t8( "Got request for url", urlPath, "with args", urlArgs )
         if func is None:
            raise HttpNotFound( "Invalid endpoint or method requested" )
         obj, funcName = func
         t8( "Calling ", funcName )
         result = getattr( obj, funcName )( requestContext, **urlArgs )
      except HttpException as e:
         t0( "processRequest HttpException : %s", e.message )
         errMsg = simplejson.dumps( { 'error' : e.message } )
         msg = e.message
         if '[' in msg:
            # Strip the header
            msg = msg.split( '[' )[ 0 ]
         self.genApiState.uri = urlPath
         self.genApiState.success = False
         self.genApiState.detailString = msg
         self.genApiState.utcTimestamp = Tac.utcNow()
         return ( "%s %s" % ( e.code, e.name ), "application/json",
                  e.additionalHeaders, errMsg )
      else:
         return ( '200 OK', 'application/json', None, result )
      finally:
         if userCtx:
            requestContext.deauthenticate( userCtx )

agentClass = PcsApiHandler

class PcsHttpAgent( object ):
   def __init__( self ):
      self.container_ = Agent.AgentContainer( [ agentClass ],
                                              passiveMount=True,
                                              agentTitle="PcsHttpAgent" )
      self.container_.startAgents()
      # When agent makes >1k HTTP requests in parallel, a new socket FD that's
      # monitored in the activity loop can have a large FD value. Tacc fwk
      # asserts if fd > 1k when select is used. Use epoll as a workaround.
      Tac.activityManager.useEpoll = True
      # Run ActivityLoop in a separate thread in order to serve HTTP requests
      # in the main thread
      Tac.activityThread().start( daemon=True )
      Tac.waitFor( lambda: len( self.container_.agents_ ) > 0,
                   maxDelay=AGENT_INIT_TIME / 10,
                   sleep=True, description="agent warmup" )
      self.agent_ = self.container_.agents_[ 0 ]
      # Just exit when killed, abandoning sockets buffer is fine
      atexit.register( lambda: os._exit( 0 ) ) # pylint: disable-msg=protected-access

   def __call__( self, request, start_response ):
      ( reponseCode, contentType, headers, body ) = \
            self.agent_.processRequest( request )
      headers = headers or []
      headers.append( ( 'Content-type', contentType ) )
      if body:
         headers.append( ( 'Content-length', str( len( body ) ) ) )
      start_response( reponseCode, UwsgiConstants.DEFAULT_HEADERS + headers )
      return [ body ]
