#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SmbusUtil
import PLSmbusUtil
import Tac
import socket

def agentIsRunning( agentName="PlutoSmbus" ):
   try:
      Tac.run( [ "pgrep", agentName ], stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return False
   return True

#
# Convenient wrapper class around PLSmbusUtil
#
# PLSmbusClient *indirectly* read/write from/to PCI by sending socket
# requests to PlutoSmbus daemon.
#
# In other words, it's a *client* of PlutoSmbus daemon.
#
# This class requires PlutoSmbus daemon running.
#
# If PlutoSmbus daemon is not running, PLSmbusClient's read/write requests will
# fail.
#

# pylint: disable-msg=W0223
class PLSmbusClient( SmbusUtil.BaseSmbusHelper ):
   def __init__( self, accelId, busId, deviceId, addrSize, pciAddress, backend ):
      SmbusUtil.BaseSmbusHelper.__init__( self )
      assert accelId >= 0
      assert 0 <= busId <= 0xf
      self.accelId = accelId
      self.busId = busId
      self.deviceId = deviceId
      self.addrSize = addrSize
      self.pciAddress = pciAddress
      self.backend = backend
      self.sock = None
      self.pci = None

      self.connect()

   def connect( self ):
      if self.pciAddress and agentIsRunning() and not self.sock:
         self.pci = PLSmbusUtil.encodePCIAddress( self.pciAddress )
         self.sock = PLSmbusUtil.connect()

   def read( self, address, count, accessSize, raw=False, pec=False ):
      self.connect()
      result = PLSmbusUtil.read( self.sock, self.pci, self.accelId, self.busId,
                                 self.deviceId, address, count=count,
                                 backend=self.backend )
      if count == 1:
         result = [ PLSmbusUtil.stringToInt( result ) ]
      else:
         result = [ PLSmbusUtil.stringToInt( result[ i ] ) \
                    for i in range( len( result ) ) ]
      return result

   def write( self, address, data, accessSize, stride=0, pec=False ):
      self.connect()
      dataStr = "".join( data )
      PLSmbusUtil.write( self.sock, self.pci, self.accelId, self.busId,
                         self.deviceId, address, dataStr, backend=self.backend )

#
# Convenient wrapper function to construct raw PCI helper using SmbusUtil,
# with reasonable default values for most Pluto systems.
#
# PLSmbusRawPciHelper *directly* reads/writes to PCI.
#
# Thus, this class requires PlutoSmbus NOT running.
#
# If PlutoSmbus daemon is running, PLSmbusRawPciHelper's read/write requests will
# fail.
#
def PLSmbusRawPciHelper( accelId, busId, deviceId, addrSize,
                         readDelayMs='delay50ms', writeDelayMs='delay50ms',
                         busTimeout='busTimeout1000ms',
                         writeNoStopReadCurrent=False, smbusAddrSpacing=0x80,
                         smbusAgentId=None, pciAddress=None ):
   factory = SmbusUtil.Factory()
   device = factory.device( accelId, busId, deviceId, addrSize,
                            readDelayMs=readDelayMs, writeDelayMs=writeDelayMs,
                            busTimeout=busTimeout,
                            writeNoStopReadCurrent=writeNoStopReadCurrent,
                            smbusAddrSpacing=smbusAddrSpacing,
                            smbusAgentId=smbusAgentId,
                            pciAddress=pciAddress )
   return device

#
# Convenient wrapper class around both classes (PLSmbusClient & PLSmbusRawPciHelper)
# above.
#
# PLSmbusPciDevice can both *directly* or *indirectly* reads/writes to PCI:
#
# 1. If PlutoSmbus is running, PLSmbusPciDevice sends PCI read/write requests to
# PlutoSmbus daemon for the daemon to proceed.
#
# 2. If PlutoSmbus is NOT running, PLSmbusPciDevice directly reads/writes to PCI
# using raw PCI helper functions defined in SmbusUtil.
#
# 3. If PlutoSmbus daemon changes its state (from "running" to "NOT running" and
# vice versa), PLSmbusPciDevice automatically adjusts its actions between the 2
# ways listed above accordingly.
#
# This class can be used in scenarios where the code must be adaptive to
# PlutoSmbus's running or NOT running status. If PlutoSmbus's status is expected
# to persist, it's probably better to use PLSmbusClient or PLSmbusRawPciHelper
# classes above.
#
class PLSmbusPciDevice( PLSmbusClient ):
   def __init__( self, accelId, busId, deviceId, addrSize, pciAddress, backend ):
      super( PLSmbusPciDevice, self ).__init__( accelId, busId, deviceId,
                                                addrSize, pciAddress, backend )
      self.rawPciHelper = PLSmbusRawPciHelper( accelId, busId, deviceId,
                                               addrSize, pciAddress=pciAddress )

   def read( self, address, count, accessSize, raw=False, pec=False ):
      if agentIsRunning():
         try:
            return super( PLSmbusPciDevice, self ).read( address, count, accessSize,
                                                         raw, pec )
         except socket.error:
            pass
      return self.rawPciHelper.read( address, count, accessSize, raw, pec )

   def write( self, address, data, accessSize, stride=0, pec=False ):
      if agentIsRunning():
         try:
            super( PLSmbusPciDevice, self ).write( address, data, accessSize,
                                                          stride, pec )
            return
         except socket.error:
            pass
      self.rawPciHelper.write( address, data, accessSize, stride, pec )
