#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import OrderedDict
import hashlib
import MgmtSecuritySslStatusSm
import Tac, SuperServer, os, shutil, QuickTrace
import Tracing

t1 = Tracing.trace1
__defaultTraceHandle__ = Tracing.Handle( 'HscOvsdb' )
qv = QuickTrace.Var
qt1 = QuickTrace.trace1

defaultTimeout = 20

ovsPkiBin = 'ovs-pki'
ovsdbToolBin = 'ovsdb-tool'

# This is the main implementation of the OVSDB server SuperServerPlugin.  Two,
# services, OvsdbServer and TestOvsdbServer inherit from these classes and provide
# the methods for starting OVSDB server itself.  All other logic remains in this
# file.

def checkDir( dirName ):
   if not os.path.isdir( dirName ):
      os.makedirs( dirName )

def runInShell( argv, cwd=None ):
   cmd = ' '.join( argv )
   t1( 'runInShell ( %s ): %s' % ( cwd, cmd ) )
   output = None
   try:
      output = Tac.run( argv, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, 
                        cwd=cwd )
   except Tac.Timeout:
      t1( "Timeout running '%s'" % ( cmd, ) )
      raise
   except Tac.SystemCommandError, e:
      t1( "Error running '%s'" % ( cmd, ) )
      t1( e )
      raise
   return output

def mapOvsdbLogLevel( ovsdbLogLevel, level ):
   if level == ovsdbLogLevel.disabled:
      return 'OFF'
   elif level == ovsdbLogLevel.emergency:
      return 'EMER'
   elif level == ovsdbLogLevel.error:
      return 'ERR'
   elif level == ovsdbLogLevel.warning:
      return 'WARN'
   elif level == ovsdbLogLevel.information:
      return 'INFO'
   elif level == ovsdbLogLevel.debug:
      return 'DBG'
   else:
      assert False, 'Invalid OVSDB log level %s'  % ( level )

MGMT_SECURITY_SSL_CONSTANTS = Tac.Type( "Mgmt::Security::Ssl::Constants" )
SslFeature = Tac.Type( "Mgmt::Security::Ssl::SslFeature" )

class OvsdbConfigNotifieeBase( SuperServer.LinuxService ):
   ''' Reactor to the 'hsc/config' and 'hsc/cliconfig' objects that
   setup the ovsdb-server.'''

   notifierTypeName = 'Hsc::CliConfig'
   FILE_REP_MODE = 'inotify'
   FILE_REP_REQ_NAME = 'HscOvsdb'

   def __init__( self, serviceName, hscConfig, hscCliConfig, hscStatus,
                 sslConfig, sslStatus, cvxConfig, cvxStatus,
                 fileRepStatus, clusterStatusDir, haFiles, dirPrefix='',
                 active=True ):
      self.started = False
      self.restart = False
      self.failedStarts = 0
      self.hscConfig  = hscConfig
      self.config = None
      self.cliConfig = hscCliConfig
      self.status = hscStatus
      self.sslConfig = sslConfig
      self.sslStatus = sslStatus
      self.cvxConfig = cvxConfig
      self.cvxStatus = cvxStatus
      self.clusterStatusDir = clusterStatusDir
      self.path = { }
      self.ovsdbLogLevel = Tac.Type( 'Hsc::OvsdbLogLevel' )
      self.options = None
      self.active = active
      self.dirPrefix = dirPrefix
      self.managers    = ''
      self.newManagers = False
      self.persistDatabase = self.cliConfig.persistDatabase
      self.haFiles = haFiles
      self.ovsdbStarted = False
      self.fileRepStatus = fileRepStatus
      self.controllerFileReplicationSm = None
      self.sslReactor = None

      self.initPath()

      os.environ[ 'OVS_RUNDIR' ] = self.hscConfig.ovsdbDataDir

      if self.serviceProcessWarm():
         self.started = True
         self.starts = 1

      SuperServer.LinuxService.__init__(
            self, serviceName, 'ovsdb-server', hscCliConfig,
            self.path[ 'ovsdbServerConf' ] )
      self.initSslReactor()


   def isHscRunnable( self ):
      return 'default' in self.clusterStatusDir.status and \
            self.clusterStatusDir.status [ 'default' ].isStandaloneOrLeader

   def onSwitchover( self, active ):
      self.active = active
      self.initPath()
      self.initSslReactor()
      self._maybeRestartService()

   def initDatabasePath( self ):
      dataDir = ( self.hscConfig.ovsdbBackupDir if self.persistDatabase 
                  else self.hscConfig.ovsdbRunDir )
      self.hscConfig.ovsdbDataDir = dataDir
      self.path[ 'db' ] = dataDir + 'vtep.db'

   def initPath( self ):
      runDir = self.dirPrefix + '/var/run/openvswitch/'
      logDir = self.dirPrefix + '/var/log/openvswitch/'
      certDir = self.dirPrefix + '/persist/secure/openvswitch/'
      shareDir = '/usr/share/openvswitch/'
      backupDir = self.dirPrefix + '/mnt/flash/openvswitch/'
      configDir = self.dirPrefix + '/etc/sysconfig/'
      if self.active:
         self.hscConfig.ovsdbRunDir = runDir
         self.hscConfig.ovsdbLogDir = logDir
         self.hscConfig.ovsdbCertDir = certDir
         self.hscConfig.ovsdbBackupDir = backupDir

         self.hscConfig.ovsdbSocketFile = runDir + 'db.sock'

         self.hscConfig.ovsdbCertFile = certDir + 'ovsdbclient-cert.pem'
         self.path[ 'cert' ] = self.hscConfig.ovsdbCertFile

         self.hscConfig.ovsdbPidFile = runDir + 'ovsdb-server.pid'

      self.path[ 'ctl' ] = runDir + 'ovsdb-server.ctl'
      self.path[ 'schema' ] = shareDir + 'vtep.ovsschema'
      self.path[ 'caCert' ] = certDir + 'ovsdbserver-ca.pem'
      self.path[ 'privKey' ] = certDir + 'ovsdbclient-privkey.pem'
      self.path[ 'ovsdbServerConf' ] = configDir + 'ovsdb-server'
      self.path[ 'ovsdb-server-log' ] = logDir + 'ovsdb-server.log'
      self.path[ 'ovs-pki-log' ] = certDir + 'ovs-pki.log'

      self.initDatabasePath()

      for key, val in self.path.iteritems():
         t1( '%s = %s' % ( key, val ) )

   def initSslReactor( self ):
      if self.cliConfig.sslProfile:
         self.sslReactor = OvsdbSslReactor( self.sslConfig, self.sslStatus,
                                            self.cliConfig.sslProfile,
                                            self.cliConfig, self )
      else:
         if self.sslReactor:
            self.sslReactor.close()
            self.sslReactor = None
            self.cleanupCerts()
            self.sync()

   def serviceEnabled( self ):
      if not ( self.active and self.cliConfig.enableOvsdb ):
         enabled = False
      else:
         enabled = ( self.cliConfig.enableAgent and self.cvxStatus.enabled and \
                     self.isHscRunnable() ) \
                   or ( self.started and self.status.running )
         if enabled and self.cliConfig.sslProfile:
            status = self._getProfileStatus()
            enabled = bool( status and status.state == "valid"
                  and status.certKeyPath )
      t1( 'serviceEnabled: %s' % ( enabled, ) )
      return enabled

   def serviceProcessWarm( self ):
      if not self.serviceEnabled():
         # As far as SuperServer is concerned, the agent is warm if it isn't
         # enabled.
         warm = True
      else:
         running = self.isRunning()
         t1( 'Running: %s, ovsdbSocketFile: %s' % ( running,
            self.hscConfig.ovsdbSocketFile ) )
         if running and os.path.exists( self.hscConfig.ovsdbSocketFile ):
            t1( 'ovsdb-server is running and socket file is present' )
            warm = True
         else:
            warm = False

      t1( 'serviceProcessWarm: %s' % ( warm, ) )
      return warm

   def addExtraOvsdbOptions( self ):
      raise NotImplementedError

   def _getProfileStatus( self ):
      if self.sslReactor and self.sslReactor.profileStatus_:
         return self.sslReactor.profileStatus_
      return None

   def _getExtraSslSettings( self ):
      status = self._getProfileStatus()
      tlsVersionMap = [ ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1, 'TLSv1' ),
                        ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1_1, 'TLSv1.1' ),
                        ( MGMT_SECURITY_SSL_CONSTANTS.tlsv1_2, 'TLSv1.2' ) ]
      useCa = False
      fipsMode = False
      tlsVersions = zip( *tlsVersionMap )[ 1 ]
      ciphers = MGMT_SECURITY_SSL_CONSTANTS.defaultCipherSuite()
      if status and status.state == "valid" and status.certKeyPath:
         fipsMode = status.fipsMode
         tlsVersions = []
         for mask, version in tlsVersionMap:
            if status.tlsVersion & mask:
               tlsVersions.append( version )
         if status.trustedCertsPath:
            useCa = True
         ciphers = status.cipherSuite
      return fipsMode, tlsVersions, useCa, ciphers

   def conf( self ):
      managers = ''
      for target in self.cliConfig.managerAddress.keys():
         managerIp = target.split( ':' )[ 1 ]
         managerPort = target.split( ':' )[ 2 ]
         managers += '%s:%s ' % ( managerIp, managerPort )

      if managers != self.managers:
         self.newManagers = True
         self.managers    = managers

      if self.cliConfig.persistDatabase != self.persistDatabase:
         # service should restart and the changes to the database path 
         # will take place
         self.persistDatabase = self.cliConfig.persistDatabase
         self.initDatabasePath()

      fipsMode, tlsVersions, useCa, ciphers = self._getExtraSslSettings()

      self.options = [
         self.path[ 'db' ],
         '--pidfile=%s' % ( self.hscConfig.ovsdbPidFile ),
         '--unixctl=%s' % ( self.path[ 'ctl' ], ),
         '--no-chdir',
         '--remote=punix:%s' % ( self.hscConfig.ovsdbSocketFile, ),
         '--remote=db:hardware_vtep,Global,managers',
         '--private-key=%s' % self.path[ 'privKey' ],
         '--certificate=%s' % self.hscConfig.ovsdbCertFile,
         '--ssl-protocols=%s' % ','.join( tlsVersions ),
         '--ssl-ciphers=%s' % ciphers,
         '-vANY:CONSOLE:%s' % ( mapOvsdbLogLevel( self.ovsdbLogLevel,
            self.cliConfig.logConsole ), ),
         '-vANY:SYSLOG:%s' % ( mapOvsdbLogLevel( self.ovsdbLogLevel,
            self.cliConfig.logSyslog ), ),
         '-vANY:FILE:%s' % ( mapOvsdbLogLevel( self.ovsdbLogLevel,
            self.cliConfig.logFile ), ) ]

      if useCa:
         self.options.append( '--ca-cert=%s' % self.path[ 'caCert' ] )
      else:
         self.options.append( '--bootstrap-ca-cert=%s' % self.path[ 'caCert' ] )

      if fipsMode:
         self.options.append( '--enable-fips' )

      self.addExtraOvsdbOptions()

      if self.cliConfig.logFile != self.ovsdbLogLevel.disabled:
         self.options += [ '--log-file=%s' % ( self.path[ 'ovsdb-server-log' ], ) ]

      settings = OrderedDict( [ ( 'PIDFILE', self.hscConfig.ovsdbPidFile ),
                                ( 'OPTIONS', "'%s'" % ' '.join( self.options ) ) ] )

      # Add the manager information to the config to cause OVSDB to restart when
      # the manager is changed.  This will allow the bootstrap file to be cleaned
      # up when the agent is stopped to ensure the connection succeeds to the new
      # manager. Same with SSLPROFILE.
      settings[ 'MANAGER' ] = self.managers
      status = self._getProfileStatus()
      if status:
         h = hashlib.md5()
         # pylint: disable=E1101
         h.update( self._readFile( status.certKeyPath ) )
         h.update( self._readFile( status.trustedCertsPath ) )
         settings[ 'SSLPROFILE' ] = self.cliConfig.sslProfile
         settings[ 'SSLSTATUS' ] = status.state
         settings[ 'SSLCERTSHASH' ] = h.hexdigest()

      value = '\n'.join( '{}={}'.format( k, v ) for k, v in settings.items() )
      t1( 'conf:\n%s' % ( value, ) )
      return value

   def cleanupCerts( self ):
      t1( 'cleanupCerts' )

      if os.path.exists( self.hscConfig.ovsdbCertFile ):
         os.remove( self.hscConfig.ovsdbCertFile )

      if os.path.exists( self.path[ 'privKey' ] ):
         os.remove( self.path[ 'privKey' ] )

      if os.path.exists( self.path[ 'caCert' ] ):
         os.remove( self.path[ 'caCert' ] )

   def cleanupRunDir( self ):
      t1( 'cleanupRunDir' )

      if os.path.exists( self.hscConfig.ovsdbPidFile ):
         os.remove( self.hscConfig.ovsdbPidFile )

      if os.path.exists( self.path[ 'ctl' ] ):
         os.remove( self.path[ 'ctl' ] )

      if os.path.exists( self.hscConfig.ovsdbSocketFile ):
         os.remove( self.hscConfig.ovsdbSocketFile )

   def cleanupVtepDb( self ):

      if os.path.exists( self.path[ 'db' ] ):
         t1( 'Removing vtep.db' )
         qt1( 'Hsc:: Removing vtep.db' )
         os.remove( self.path[ 'db' ] )

   def createVtepDb( self ):
      t1( 'Create vtep.db' )
      qt1( 'Hsc:: Create vtep.db' )

      db = self.path[ 'db' ]
      schema = self.path[ 'schema' ]

      # verify db and schema versions
      if os.path.exists( db ):
         dbVersion = None
         schemaVersion = None
         needsConversion = False
         try:
            dbVersion = self.runWithLogging( [ ovsdbToolBin, 'db-version', db ] )
            schemaVersion = self.runWithLogging( [ ovsdbToolBin, 'schema-version',
                                                   schema ] )
         except Tac.SystemCommandError:
            t1( 'DB cannot be read; re-creating db' )
         else:
            t1( 'dbVersion=%s schemaVersion=%s' % ( dbVersion, schemaVersion ) )

            # Verify that the global table hasn't been truncated
            queryOutput = self.runWithLogging( [ ovsdbToolBin, 'query', db,
               '["hardware_vtep", '
               '{ "op": "select", "table":"Global", "where":[] }]' ] )

            if 'ERR|I/O error:' in queryOutput:
               t1( 'Global Table has been truncated, and needs to be recreated' )

            elif dbVersion == schemaVersion:
               t1( 'Not re-creating vtep.db; '
                   'the DB version and schema version match' )
               return
            else:
               needsConversion = True
         # try to convert the db 
         if needsConversion:
            t1( 'Try to convert db to newer schema' )
            try:
               self.runWithLogging( [ ovsdbToolBin, 'convert', db, schema ] )
               return
            except Tac.SystemCommandError:
               t1( 'DB cannot be converted; re-creating db' )

      self.cleanupVtepDb()
      qt1( 'Hsc:: Creating vtep.db using ovsdb-tool' )
      # create vtep.db
      self.runWithLogging( [ ovsdbToolBin, 'create', db, schema ] )

      # insert global table into database
      trans = '[ "hardware_vtep", {"row":{},"table":"Global","op":"insert"} ]'
      self.runWithLogging( [ ovsdbToolBin, 'transact', db, trans ] )

   def isRunning( self ):
      running = False

      if os.path.exists( self.hscConfig.ovsdbPidFile ):
         with open( self.hscConfig.ovsdbPidFile, 'r' ) as pidFile:
            pid = pidFile.readline()
            t1( 'PID of ovsdb-server: %s' % pid[ :-1 ] )

         if pid != '' and os.path.exists( '/proc/%s' % ( pid[ :-1 ], ) ):
            t1( 'PID %s exists' % pid[ :-1 ] )
            running = True

      return running

   def generateCertificates( self, certDir ):
      t1( 'generateCertificates %s ' % ( certDir ) )
      self.runWithLogging( [ ovsPkiBin, 'init', '--force',
                             '--dir=%s' % ( certDir, ) ], certDir )
      self.runWithLogging( [ ovsPkiBin, 'req+sign', 'ovsdbclient', '--force',
                             '--dir=%s' % ( certDir, ) ], certDir )
      qt1( 'Hsc:: Certificates are generated' )

   def runWithLogging( self, argv, cwd=None ):
      if self.cliConfig.logFile != self.ovsdbLogLevel.disabled:
         if argv[ 0 ] == ovsPkiBin:
            argv.append( '--log=%s' % ( self.path[ 'ovs-pki-log' ], ) )
      return runInShell( argv, cwd=cwd )

   def _readFile( self, path ):
      if not os.path.exists( path ):
         return ''
      with open( path ) as f:
         return f.read()

   def importCertificates( self, status ):
      cert = self._readFile( self.hscConfig.ovsdbCertFile )
      certKey = self._readFile( status.certKeyPath )
      if certKey != cert:
         shutil.copy( status.certKeyPath, self.path[ 'privKey' ] )
         shutil.copy( status.certKeyPath, self.hscConfig.ovsdbCertFile )
         qt1( "Hsc:: Certificate/Key pair are imported" )
      if not status.trustedCertsPath:
         return
      caCert = self._readFile( self.path[ 'caCert' ] )
      caCertProfile = self._readFile( status.trustedCertsPath )
      if caCertProfile != caCert:
         shutil.copy( status.trustedCertsPath, self.path[ 'caCert' ] )
         qt1( "Hsc:: CA Certificate is imported" )

   # This function should only be called while ovsdb-server is stopped
   def checkConfig( self ):
      t1( 'checkConfig newManager=%s started=%s restart=%s' % ( self.newManagers,
         self.started, self.restart ) )
      qt1( 'Hsc:: checkConfig newManager', qv( self.newManagers ), 'started',
            qv( self.started ), 'restart', qv( self.restart ) )

      if self.started:
         return

      checkDir( os.path.dirname( self.hscConfig.ovsdbSocketFile ) )
      checkDir( self.hscConfig.ovsdbRunDir )
      checkDir( self.hscConfig.ovsdbLogDir )

      # Just check persist here.  Local should already exist from the image.
      checkDir( self.hscConfig.ovsdbDataDir )
      checkDir( self.hscConfig.ovsdbBackupDir )

      # Cleanup any leftover service files if OVSDB server is not running.  An
      # abnormal exit may leave files behind and we determine whether the
      # service is warm based on these files.
      running = self.isRunning()

      if not running:
         self.cleanupRunDir()

      self.createVtepDb()

      certDir = self.hscConfig.ovsdbCertDir
      checkDir( certDir )
      if self.cliConfig.sslProfile:
         status = self._getProfileStatus()
         assert status and status.state == "valid" and status.certKeyPath
         self.importCertificates( status )
      elif ( not os.path.exists( self.hscConfig.ovsdbCertFile ) or
             not os.path.exists( self.path[ 'privKey' ] ) ):
         self.generateCertificates( certDir )

      # If the manager address has changed and no CA is configured in the
      # SSL profile, remove the bootstrap CA to ensure we get the one
      # associated with the new manager. This will also remove the bootstrap
      # file upon boot.
      status = self._getProfileStatus()
      if ( self.newManagers and os.path.exists( self.path[ 'caCert' ] ) and
           not ( status and status.trustedCertsPath ) ):
         os.remove( self.path[ 'caCert' ] )
         qt1( "Hsc:: Bootstrap is cleared" )

   def startOvsdb( self ):
      raise NotImplementedError

   def stopOvsdb( self ):
      raise NotImplementedError

   def startService( self ):
      t1( 'startService started=%s restart=%s starts=%s ' \
          'failedStarts=%s newManager=%s' % ( self.started,
          self.restart, self.starts, self.failedStarts, self.newManagers ) )
      qt1( 'Hsc:: startService started', qv( self.started ), 'restart',
            qv( self.restart ), 'starts', qv( self.starts ), 'failedStarts',
            qv( self.failedStarts ), 'newManager', qv( self.newManagers ) )

      if not self.started:
         try:
            self.checkConfig()
         except Exception: # pylint: disable-msg=W0703
            t1( 'checkConfig failed. Do not start ovsdb-server and try again later' )
            import traceback
            tb = traceback.format_exc()
            t1( tb )
            self.failedStarts += 1
            return

         # Note that at this point the DB, cert files, etc. are guaranteed to exist
         # from the self.checkConfig() call above.  If that fails for any reason it
         # returns above and won't reach here.
         #
         # This call will attempt to start ovsdb-server as a separate process but
         # does not wait for it to start running.  It is possible that the process
         # will end up not starting for some reason (like if it can't create its PID
         # file due to a full file-system, etc.).  In that case the background
         # SuperServer health check will detect that it isn't running and we will end
         # up in the elif case below.
         self.startOvsdb()

         t1( "Calling vtep.db handler" )
         self.handleVtepDb()

         self.started = True
      elif not self.isRunning():
         # If ovsdb-server failed to start or died unexpectedly this will attempt
         # to restart it.  We will get here when startService is called from
         # _checkServiceHealth() after the 'service ovsdb-server status' call
         # fails.
         t1( 'ovsdb-server failed to start or died; restarting' )
         self.failedStarts += 1
         self.startOvsdb()

      if not self.restart:
         self.starts += 1

   def stopService( self ):
      t1( 'stopService started=%s restart=%s stops=%s' % ( self.started,
          self.restart, self.stops ) )
      qt1( 'Hsc:: stopService started', qv( self.started ), 'restart',
            qv( self.restart ), 'stops', qv( self.stops ) )

      self.stopOvsdb()

      t1( 'Calling vtep.db handler' )
      self.handleVtepDb()

      self.started = False

      if not self.restart:
         self.stops += 1

   def restartService( self ):
      t1( 'restartService started=%s starts=%s failedStarts=%s stops=%s '
          'restarts=%s' % ( self.started, self.starts, self.failedStarts,
          self.stops, self.restarts ) )
      qt1( 'Hsc:: restartService started', qv( self.started ), 'starts',
            qv( self.starts ), 'failedStarts', qv( self.failedStarts ),
            'stops', qv( self.stops ), 'restarts', qv( self.restarts ) )

      self.restart = True

      # The restart must be done using the functions in here rather than using the
      # Linux service restart to ensure that all config changes are applied
      # correctly.
      self.stopService()
      self.startService()

      self.restart = False
      self.restarts += 1

   def registerFileReplication( self ):
      if self.controllerFileReplicationSm:
         t1( "FileReplicationSm is already initialized" )
         return

      t1( "Creating new controllerFileReplicationSm" )
      clusterStatus = self.clusterStatusDir.status[ 'default' ]
      self.controllerFileReplicationSm = Tac.newInstance(
            "ControllerCluster::ControllerFileReplicationSm", self.fileRepStatus,
            clusterStatus )

      t1( "Registering HscOvsdb to file replication" )
      requester = self.controllerFileReplicationSm.config.requester
      requesterConfig = requester.newMember(
            OvsdbConfigNotifieeBase.FILE_REP_REQ_NAME,
            OvsdbConfigNotifieeBase.FILE_REP_MODE )
      for aFile in self.haFiles:
         if aFile in self.path:
            srcPath = self.path[ aFile ]
            dstPath = srcPath
            t1( "Adding %s for replication" % srcPath )
            qt1( 'Hsc:: Adding', qv( os.path.basename( srcPath ) ), 'to replicate' )
            requesterConfig.file.newMember( srcPath, dstPath )

   def unregisterFileReplication( self ):
      if self.controllerFileReplicationSm:
         t1( "Unregistering HscOvsdb from file replication" )
         requester = self.controllerFileReplicationSm.config.requester
         del requester[ OvsdbConfigNotifieeBase.FILE_REP_REQ_NAME ]

         t1( "Deleting ReplicationSm" )
         qt1( "Hsc:: Deleting ReplicationSm" )
         self.controllerFileReplicationSm = None

   def handleFileReplication( self ):
      t1( "handleFileReplication is called" )
      clusterStatus = self.clusterStatusDir.status[ 'default' ]

      assert not clusterStatus.isStandalone, "handleFileReplication should be "\
            "called only in cluster mode"

      t1( "ovsdbdStarted: %s, isStandaloneOrLeader: %s" % ( self.ovsdbStarted,
         clusterStatus.isStandaloneOrLeader ) )
      qt1( "Hsc:: handleFileReplication ovsdbdStarted", qv( self.ovsdbStarted ),
            "isStandaloneOrLeader", qv( clusterStatus.isStandaloneOrLeader ) )

      # The source files need to be present when we create FileReplicationSm. So, we
      # make sure both ovsdb is administritavely up and the node is Master
      if self.ovsdbStarted and clusterStatus.isStandaloneOrLeader:
         t1( "Registering file replication" )
         self.registerFileReplication()
      else:
         t1( "Unregistering file replication" )
         self.unregisterFileReplication()

   @Tac.handler( 'sslProfile' )
   def handleSslProfile( self ):
      t1( 'OvsdbConfigNotifieeBase.handleSslProfile' )
      self.initSslReactor()

   def handleVtepDb( self ):
      # When a Cvx runs in Standalone Mode, we only delete vtep.db file if Hsc is
      # disabled in the CLI.
      # For Cluster Mode, we don't delete vtep.db on a follower if its Hsc is made
      # disabled while Cvx is still enabled. This ensures that if Hsc is made enabled
      # later on this particular follower, no special mechanism is neeeded to copy
      # the vtep.db file from the Master. Note that the Master syncs vtep.db file
      # only when there is an 'inotify' event or a follower joins the cluster.
      # As Cvx is kept enabled during the entire disabling and re-enabling of Hsc
      # service on this follower, the Master doesn't notice any change in the
      # follower's state (because, there is no cluster join event). When Hsc is
      # enabled again on this follower, it gets the file only at the next 'inotify'
      # event. If this follower becomes Master with an empty vtep.db file, then there
      # will be network churns. However, we delete vtep.db file when Cvx is disabled
      # on the follower. Note that when Cvx is re-enabled on it there will be a new
      # cluster join event and the follower will receive the latest vtep.db file from
      # the Master.
      # For the Master, when we disable Hsc service we delete vtep.db file. This
      # eventually deletes the file on each of the active followers. However,
      # as disabling Cvx in the Master will elect a new Cvx Master, we don't do a
      # cluster wide cleanup of vtep.db.
      # Also, if shutdown via 'ovsdb-shutdown' in the CLI and the HSC service is
      # still enabled, the service should stop but the data should be preserved.
      # When deleting vtep.db, we always wait until Hsc finishes its conditional
      # cleanup of Ovsdb contents.
      clusterStatus = self.clusterStatusDir.status[ 'default' ]
      cleanupDatabase = not self.status.running and \
            ( not self.cvxConfig.enabled or
               ( not self.cliConfig.enableAgent and
                  clusterStatus.isStandaloneOrLeader ) )
      t1( "self.restart: %s, cvxConfig.enabled: %s, cliconfig.enableAgent: %s, "
            "clusterStatus.isStandaloneOrLeader: %s, cleanupDatabase: %s, "
            "status.running: %s"
            % ( self.restart, self.cvxConfig.enabled, self.cliConfig.enableAgent,
               clusterStatus.isStandaloneOrLeader, cleanupDatabase,
               self.status.running ) )
      qt1( "Hsc:: cvxConfig.enabled", qv( self.cvxConfig.enabled ),
            "cliConfig.enableAgent", qv( self.cliConfig.enableAgent ),
            "status.running", qv( self.status.running ),
            "clusterStatus.isStandaloneOrLeader",
            qv( clusterStatus.isStandaloneOrLeader ) )
      qt1( "Hsc:: self.restart", qv( self.restart ), "cleanupDatabase",
            qv( cleanupDatabase ) )

      if not self.restart and cleanupDatabase:
         if self.controllerFileReplicationSm:
            # As we are going to delete vtep.db, we are stopping replication here by
            # disabling file replication requester. Later, we will call
            # deleteImmediate if vtep.db needs to be deleted on the followers too.
            t1( "Disabling file replication requester" )
            self.controllerFileReplicationSm.requesterEnabledIs(
                  OvsdbConfigNotifieeBase.FILE_REP_REQ_NAME, False )

         self.cleanupVtepDb()

         if self.controllerFileReplicationSm and self.cvxConfig.enabled:
            # When Cvx is disabled on a Master, a new Master is elected from the rest
            # of the active Cvx nodes. We delete vtep.db from the old Master, but
            # keep it in its followers. This was done to avoid a race between two
            # Hsc SuperServer plugins running on the old and new Master: one is
            # creating vtep.db and the other is trying to delete it. So, we call
            # deleteImmediate only when cvx is enabled on this node.
            t1( "Force deleting the vtep.db file from the followers" )
            qt1( "Hsc:: Force deleting the vtep.db file from the followers" )
            self.controllerFileReplicationSm.deleteImmediate(
                  OvsdbConfigNotifieeBase.FILE_REP_REQ_NAME )

      if not clusterStatus.isStandalone:
         # handleFileReplication needs to be called only for a cluster deployment
         self.handleFileReplication()

class ClusterStatusNotifiee( Tac.Notifiee ):
   notifierTypeName = 'ControllerCluster::ClusterStatus'

   def __init__( self, clusterStatus, ovsdbConfigNotifiee ):
      t1( 'ClusterStatusNotifiee' )
      Tac.Notifiee.__init__( self, clusterStatus )
      self.ovsdbConfigNotifiee_ = ovsdbConfigNotifiee
      self.clusterStatus_ = clusterStatus
      self.handleIsStandaloneOrLeader()

   @Tac.handler( 'isStandaloneOrLeader' )
   def handleIsStandaloneOrLeader( self ):
      t1( 'ControllerCluster::ClusterStatus isStandaloneOrLeader is changed to %d' %
            self.clusterStatus_.isStandaloneOrLeader )
      qt1( 'Hsc:: isStandaloneOrLeader',
            qv( self.clusterStatus_.isStandaloneOrLeader ) )
      self.ovsdbConfigNotifiee_.handleVtepDb()
      self.ovsdbConfigNotifiee_.sync()

class ClusterStatusDirNotifiee( Tac.Notifiee ):
   notifierTypeName = 'ControllerCluster::ClusterStatusDir'

   def __init__( self, clusterStatusDir, ovsdbConfigNotifiee ):
      t1( 'ClusterStatusDirNotifiee' )
      Tac.Notifiee.__init__( self, clusterStatusDir )
      self.ovsdbConfigNotifiee_ = ovsdbConfigNotifiee
      self.clusterStatusNotifiee_ = {}
      self.clusterStatusDir_ = clusterStatusDir
      for clusterName in clusterStatusDir.status:
         self.handleStatus( clusterName )

   @Tac.handler( 'status' )
   def handleStatus( self, clusterName ):
      t1( 'controller/cluster/statusDir/status changed' )
      qt1( 'Hsc:: controller/cluster/statusDir/status changed' )
      if clusterName not in self.clusterStatusNotifiee_:
         self.clusterStatusNotifiee_[ clusterName] = \
               ClusterStatusNotifiee( self.clusterStatusDir_.status[ clusterName ],
                                      self.ovsdbConfigNotifiee_ )

class ControllerdbConfigNotifiee( Tac.Notifiee ):
   notifierTypeName = 'Controllerdb::Config'

   def __init__( self, controllerdbConfig, ovsdbConfigNotifiee ):
      t1( 'ControllerdbConfigNotifiee' )
      Tac.Notifiee.__init__( self, controllerdbConfig )
      self.ovsdbConfigNotifiee_ = ovsdbConfigNotifiee

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      t1( 'controller/config/enabled changed' )
      qt1( 'Hsc:: controller/config/enabled changed' )
      self.ovsdbConfigNotifiee_.handleVtepDb()

class ControllerdbStatusNotifiee( Tac.Notifiee ):
   notifierTypeName = 'Controllerdb::Status'

   def __init__( self, controllerdbStatus, ovsdbConfigNotifiee ):
      t1( 'ControllerdbStatusNotifiee' )
      Tac.Notifiee.__init__( self, controllerdbStatus )
      self.ovsdbConfigNotifiee_ = ovsdbConfigNotifiee

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      t1( 'controller/status/enabled changed' )
      qt1( 'Hsc:: controller/status/enabled changed' )
      self.ovsdbConfigNotifiee_.sync()

class HscStatusNotifiee( Tac.Notifiee ):
   notifierTypeName = 'Hsc::Status'

   def __init__( self, hscStatus, ovsdbConfigNotifiee ):
      t1( 'HscStatusNotifiee' )
      Tac.Notifiee.__init__( self, hscStatus )
      self.ovsdbConfigNotifiee_ = ovsdbConfigNotifiee

   @Tac.handler( 'running' )
   def handleRunning( self ):
      t1( 'hsc/status/running changed' )
      qt1( 'Hsc:: hsc/status/running changed' )
      self.ovsdbConfigNotifiee_.sync()

class OvsdbSslReactor( MgmtSecuritySslStatusSm.SslStatusSm ):
   __supportedFeatures__ = [ SslFeature.sslFeatureCertKey,
                             SslFeature.sslFeatureTrustedCert,
                             SslFeature.sslFeatureTls,
                             SslFeature.sslFeatureFips,
                             SslFeature.sslFeatureCipher ]

   def __init__( self, config, status, profileName, ovsdbCliConfig,
                 containingAgent ):
      self.constants_ = MGMT_SECURITY_SSL_CONSTANTS
      self.ovsdbCliConfig_ = ovsdbCliConfig
      self.containingAgent_ = containingAgent
      super( OvsdbSslReactor, self ).__init__( status, profileName, 'Hsc' )

   def handleProfileState( self ):
      if self.ovsdbCliConfig_.enableOvsdb:
         self.containingAgent_.sync()

   def handleProfileDelete( self ):
      if self.ovsdbCliConfig_.enableOvsdb:
         self.containingAgent_.sync()

class OvsdbServerBase( SuperServer.SuperServerAgent ):
   ''' This agent reacts to the Hsc objects and configures ovsdbserver
   appropriately. '''

   def __init__( self, entityManager, haFiles=None ):
      t1( 'OvsdbServerBase' )

      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()

      hscCliConfig = mg.mount( 'hsc/cliconfig', 'Hsc::CliConfig', 'r' )
      hscConfig = mg.mount( 'hsc/config', 'Hsc::Config', 'w' )
      hscStatus = mg.mount( 'hsc/status', 'Hsc::Status', 'ri' )
      sslConfig = mg.mount( 'mgmt/security/ssl/config',
                            'Mgmt::Security::Ssl::Config', 'r' )
      sslStatus = mg.mount( 'mgmt/security/ssl/status',
                            'Mgmt::Security::Ssl::Status', 'r' )
      cvxConfig = mg.mount( 'controller/config', 'Controllerdb::Config', 'r' )
      cvxStatus = mg.mount( 'controller/status', 'Controllerdb::Status', 'r' )
      clusterStatusDir = mg.mount( 'controller/cluster/statusDir',
            'ControllerCluster::ClusterStatusDir', 'r' )
      fileRepStatus = mg.mount(
         'controller/cluster/fileReplication/status/Hsc',
         'FileReplication::Status', 'w' )

      self.ovsdbConfigNotifiee_ = None
      self.hscStatusNotifiee_ = None
      self.controllerdbConfigNotifiee_ = None
      self.controllerdbStatusNotifiee_ = None
      self.clusterStatusDirNotifiee_ = None
      self.haFiles = haFiles if haFiles is not None else [ 'db', 'privKey',
                                                           'cert', 'caCert' ]

      def _finished():
         self.ovsdbConfigNotifiee_ = self.createOvsdbConfigNotifiee( hscConfig,
               hscCliConfig, hscStatus, sslConfig, sslStatus,
               cvxConfig, cvxStatus, fileRepStatus, clusterStatusDir, self.haFiles )

         self.hscStatusNotifiee_ = HscStatusNotifiee( hscStatus,
               self.ovsdbConfigNotifiee_ )

         self.controllerdbConfigNotifiee_ = ControllerdbConfigNotifiee( cvxConfig,
               self.ovsdbConfigNotifiee_ )

         self.controllerdbStatusNotifiee_ = ControllerdbStatusNotifiee( cvxStatus,
               self.ovsdbConfigNotifiee_ )

         self.clusterStatusDirNotifiee_ = ClusterStatusDirNotifiee( clusterStatusDir,
               self.ovsdbConfigNotifiee_ )

         t1( "All notifiees are created" )
      mg.close( _finished )

   def createOvsdbConfigNotifiee( self, hscConfig, hscCliConfig, hscStatus,
                                  sslConfig, sslStatus,
                                  cvxConfig, cvxStatus, fileRepStatus,
                                  clusterStatusDir, haFiles ):
      raise NotImplementedError

   def warm( self ):
      if self.ovsdbConfigNotifiee_:
         return self.ovsdbConfigNotifiee_.warm()
      else:
         return False

   def onSwitchover( self, protocol ):
      self.ovsdbConfigNotifiee_.onSwitchover( self.active() )
