# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import subprocess

ABOOT_LABEL = "ABOOT"
ONIE_LABEL = "ONIE-BOOT"

class Mount( object ):
   """ This class is used to mount a source path to a desired temporary
   mountpoint. This should preferably be used using the 'with' keyword. """

   def __init__( self, source, mountpoint ):
      self._source = source
      self._mountpoint = mountpoint

   def __enter__( self ):
      self.doMount()
      return self

   def __exit__( self, exceptionType, value, traceback ):
      self.doUnmount()
      mountPath = self._selectMountPoint()
      # Delete the mountpoint
      os.removedirs( mountPath )

   def _selectSource( self ):
      if self._source:
         if not os.path.exists( self._source ):
            raise IOError( "Path does not exist: %s" % self._source )
         return self._source
      else:
         raise ValueError( "No source path specified" )

   def _selectMountPoint( self ):
      if self._mountpoint:
         return self._mountpoint
      else:
         raise ValueError( "No Mountpoint path specified" )

   def doMount( self ):
      sourcePath = self._selectSource()
      mountPath = self._selectMountPoint()
      # Create the mountpoint directory
      if not os.path.exists( mountPath ):
         os.makedirs( mountPath )
      # Mount the partition
      subprocess.call( [ "/bin/mount", sourcePath, mountPath ] )

   def doUnmount( self ):
      mountPath = self._selectMountPoint()
      if not os.path.exists( mountPath ):
         raise IOError( "Path does not exist: %s" % mountPath )
      # Unmount the partition
      subprocess.call( [ "/bin/umount", mountPath ] )

def findPartByLabel( label ):
   if not label:
      raise ValueError( "No label specified" )
   # list the partitions
   partList = subprocess.check_output( "/sbin/blkid" ).splitlines()
   for part in partList:
      if label in part:
         # This will find the correct /dev/sda* path
         return part.split( ':' )[ 0 ]
   # no partition found
   raise LookupError( "No '%s' partition found!" % label )
