# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" FactoryReset is a library that provides an uninstallOnReboot method
that changes the next default reboot to ONIE install, rescue, or uninstall.
A mount utility is provided to assist in changing the grubenv variables. """

from GrubHelper import EditEnv
import OnieMountLib
import os.path

class InvalidPlatform( Exception ):
   pass

def checkOniePlatform():
   """ This method checks that your are on an ONIE platform. """
   try:
      OnieMountLib.findPartByLabel( OnieMountLib.ONIE_LABEL )
      # if "ONIE-BOOT" is not found a LookupError Exception is raised
   except:
      raise InvalidPlatform( "This platform is not supported." )

def setGrubEnvVar( partitionLabel, key, value ):
   # In order to change the grubenv we need to mount the partition
   part = OnieMountLib.findPartByLabel( partitionLabel )
   tempMountPoint = "/mnt/factoryReset" + partitionLabel
   with OnieMountLib.Mount( part, tempMountPoint ):
      grubEnvPath = os.path.join( tempMountPoint, "grub/grubenv" )
      ee = EditEnv( grubEnvPath )
      ee.read()
      ee.set( key, value )
      ee.write()

def setBootMode( mode ):
   """ This sets the boot_mode grub environment variable to
   one of install, uninstall, or rescue mode, otherwise clearing it.
   This is done by going to the ONIE menue and setting the onie_mode."""
   # Ensure you are on an ONIE platform
   checkOniePlatform()
   assert mode in [ '', 'install', 'uninstall', 'rescue' ], \
         'factory-reset does not support this mode'

   setGrubEnvVar( OnieMountLib.ABOOT_LABEL, 'boot_mode', 'install' )
   setGrubEnvVar( OnieMountLib.ONIE_LABEL, 'onie_mode', mode )

def uninstallOnReboot():
   """ This causes the next default boot to be ONIE: Uninstall. """
   setBootMode( "uninstall" )

def installOnReboot():
   """ This causes the next default boot to be ONIE: Install. """
   setBootMode( "install" )

def rescueOnReboot():
   """ This causes the next default boot to be ONIE: Rescue. """
   setBootMode( "rescue" )

def abortUninstallOnReboot():
   """ This causes the next default boot to be the default ABOOT. """
   setBootMode( "" )
