# Copyright (c) 2005-2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import sys

import CliMatcher
import CliParserCommon
import Tac
import Tracing

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2
t3 = th.trace3
t8 = th.trace8

class Generator( object ):
   '''We use this class to produce interfaces as a generator rather than a list.
   '''
   __slots__ = ( 'prefix', 'begin', 'end', '_endpoints' )

   def __init__( self, prefix, begin, end, endpoints ):
      self.begin = begin
      self.end = end
      self.prefix = prefix
      self._endpoints = endpoints
      assert isinstance( prefix, str )
      assert isinstance( begin, int )
      assert isinstance( end, int )

   def __cmp__( self, other ):
      if isinstance( other, Generator ):
         return cmp( self.prefix, other.prefix ) or cmp( self.begin, other.begin )
      return NotImplemented

   def __iter__( self ):
      # TODO perf and rewrite
      return ( self.prefix + str( i ) for i in xrange( self.begin, self.end + 1 ) )

   def __len__( self ):
      return self.end - self.begin + 1

   def __str__( self ):
      # TODO: Perf and rewrite this.
      result = [ self.prefix, str( self.begin ), '', '' ]
      if self.begin != self.end:
         result[ 2 ] = '-'
         result[ 3 ] = str( self.end )
      return ''.join( result )

def mergeRanges( ranges ):
   '''Given ranges like 1-2,3-4, result in a single range 1-4.
   '''
   ranges.sort()
   begin = ranges[ 0 ].begin
   end = ranges[ 0 ].end
   for idx, rng in enumerate( ranges ):
      try:
         nxt = ranges[ idx + 1 ]
         if rng.prefix != nxt.prefix or rng.end + 1 < nxt.begin: # Discontinuous.
            yield rng.prefix, begin, rng.end
            begin = nxt.begin
         end = end if end > nxt.end else nxt.end
      except IndexError: # We're at the end of the list.
         yield rng.prefix, begin, end if end > rng.end else rng.end

def numToDottedStr( s ):
   '''
   This function converts a U32 into <U16>.<U16> string format.
   It is used in Bgp Cli code.
   Example: 4294967295 in U32 is represented as 65535.65535.
   '''
   return "%d.%d" % ( ( s >> 16 ), ( s & 0x0000ffff ) )

class Endpoint( object ):
   def __init__( self, vmin=None, vmax=None, subMin=None, subMax=None,
                 terminal=False, invalid=False ):
      self.vmin = vmin
      self.vmax = vmax
      self.subMin = subMin
      self.subMax = subMax
      self.terminal = terminal
      self.invalid = invalid
      if self.invalid:
         assert ( not self.terminal and vmin is None and vmax is None and
                  subMin is None and subMax is None )
      else:
         assert self.terminal or not ( vmin is None or vmax is None )

   def __str__( self ):
      if self.invalid:
         return "INVALID_PREFIX"
      elif self.vmin is None:
         assert self.terminal
      s = "(%s,%s)" % ( self.vmin, self.vmax )
      if self.subMin:
         s += " sub(%s,%s)" % ( self.subMin, self.subMax )
      return "TERMINAL" + s if self.terminal else s

   def isTerminal( self, isSub ):
      return self == TERMINAL if isSub else self.terminal

   def inRange( self, val ):
      return self.vmin <= val <= self.vmax

   def inSubRange( self, val ):
      return self.subMin <= val <= self.subMax

   def hasNext( self ):
      return self.vmin is not None

# These are used only for generating Endpoint not for checking;
# Instead of checking if an endpoint is equal to TERMINAL or INVALID_PREFIX,
# check its terminal or invalid flag instead.
TERMINAL = Endpoint( terminal=True )
INVALID_PREFIX = Endpoint( invalid=True )

class IrPath( object ):

   def __init__( self, endpointFunc, sequence, hasSub=False, canSubComp=False ):
      self._endpoints = endpointFunc
      self.sequence_ = tuple( sequence )
      self.hasSub_ = hasSub # Whether this path contains a sub number in itself.
      self.canSubComp_ = canSubComp # Whether this path can lead to a sub completion.

   def copy( self ):
      '''Create a copy (another IrPath instance that is equal).'''
      return IrPath( self._endpoints, self.sequence_, self.hasSub_ )

   def sequence( self ):
      '''Extract a sequence of numbers corresponding to this path.'''
      return list( self.sequence_ )

   def endpoints( self ):
      '''Return endpoints of span that starts with my sequence.
      For example, if I am "3/2", and port 3/2 is connected to a
      24-port port expander, then this returns Endpoint(1,24).
      If 3/2 is an ordinary port, then this returns TERMINAL.

      Occasionally, an endpoint can have both a range AND the terminal
      flag set to true. This can happen if both "Et3" and "Et3/1" exist
      in the configuration (but not in status).
      '''
      return self._endpoints( self.sequence(), self.hasSub_ )

   def __str__( self ):
      if self.hasSub_:
         return ( "/".join( str( n ) for n in self.sequence_[ : -1 ] ) + "."
                  + str( self.sequence_[ -1 ] ) )
      else:
         return "/".join( str( n ) for n in self.sequence_ )

   def __repr__( self ):
      return self.__str__()

   def __cmp__( self, other ):
      return cmp( self.sequence_, other.sequence_ )

   def __hash__( self ):
      # needed so that a set of IrPath's correctly eliminates duplicates.
      return hash( self.sequence_ )

   def successor( self ):
      '''Find the IrPath of the next valid port.  Skip over any holes (things that
      were within the parent\'s range, but came back as INVALID_PREFIX).
      Return the next IrPath after this one, and the number of holes skipped.'''
      x = list( self.sequence_ )
      holes = 0
      isSub = self.hasSub_

      # Subinterface successor optimization.
      if self.hasSub_ and x[ -1 ] < CliParserCommon.subUpperBound:
         x[ -1 ] += 1
         return IrPath( self._endpoints, x, self.hasSub_ ), holes

      endpoints = self._endpoints( x )
      if endpoints.vmin is not None:
         # This interface has a next level (e.g., both Et3 and Et3/1
         # exist - this is probably a misconfig, but we handle it anyway.
         t2( x, "has next level", endpoints.vmin )
         x.append( endpoints.vmin )
         return IrPath( self._endpoints, x, self.hasSub_ ), holes

      while True:
         # This loop returns when we reach an actual port.  It
         # handles backing up when we reach the end of a span
         # (i.e., we get to 3/7/48) and need to increment a
         # higher-order component.  It internally descends (e.g.,
         # if we hit a port expander in the middle of a span).
         endpoints = self._endpoints( x[ : -1 ] )
         if isSub:
            assert endpoints.inSubRange( x[ -1 ] )
         else:
            assert endpoints.inRange( x[ -1 ] )
         if x[ -1 ] >= ( endpoints.subMax if isSub else endpoints.vmax ):
            # Hit end of span.  Back up.
            x = x[ : -1 ]
            assert x, "hit end of port space incrementing %r" % ( self.sequence_, )
            isSub = False
            continue
         x[ -1 ] = x[ -1 ] + 1
         while True:
            # This loop returns when it hits a real port, descending as needed
            # into newly discovered sub-ranges, and breaking (and continuing the
            # outer loop) if it hits an invaid port (that we need to skip over).
            leafEp = self._endpoints( x, isSub )
            if leafEp.invalid:
               # Nothing to do in this case, we'll just skip it.
               holes += 1
               break
            if leafEp.isTerminal( self.hasSub_ ):
               # Found an actual interface!  This is the result.
               return IrPath( self._endpoints, x, self.hasSub_ ), holes
            # Descend, and return the first real port in the
            # newly-discovered sub-span.  i.e., if we were iterating from
            # 3/10 to 3/20, and it turned out that 3/12 is connected to a
            # port expander, then we are now on 3/12/1.
            if self.hasSub_ and leafEp.subMin is not None:
               x.append( leafEp.subMin if self.hasSub_ else leafEp.vmin )
               isSub = True
            else:
               x.append( leafEp.vmin )

class GenericRangeList( object ):
   def __init__( self, typ, irRangeList, merge=True ):
      '''A set of integers.

           irRangeList:  list of integer ranges, like [ (1,3), (5,5) ]
           merge: sorts and merges the output ranges'''
      self.tagShort_ = typ.tagShort if typ else ''
      if merge:
         self.irRangeList_ = self._merge( irRangeList )
      else:
         self.irRangeList_ = irRangeList

   def _merge( self, rangeList ):
      # merge overlapping ranges
      rlist = sorted( rangeList )
      final = []
      prev = rlist[ 0 ]
      for r in xrange( 1, len( rlist ) ):
         cur = rlist[ r ]
         if prev[ 1 ] + 1 < cur[ 0 ]:
            final.append( prev )
            prev = cur
         elif prev[ 1 ] <= cur[ 1 ]:
            prev = ( prev[ 0 ], cur[ 1 ] )
      final.append( prev )
      return final

   @Tac.memoize
   def __str__( self ):
      '''Pretty representation like "1-3,5".'''
      s = ''
      for r in self.irRangeList_:
         if s:
            s += ','
         if r[ 0 ] == r[ 1 ]:
            s += str( r[ 0 ] )
         else:
            s += "%d-%d" % ( r[ 0 ], r[ 1 ] )
      return self.tagShort_ + s

   def aaaToken( self ):
      return self.__str__()

   def ranges( self ):
      '''Returns the range list such as [ (1,4), (6,6) ]'''
      return self.irRangeList_

   def values( self ):
      '''Returns a list of all numeric values in the range list, such
      as [ 1, 2, 3, 4, 6 ].

      Note this could be expensive if the range is large.'''
      s = []
      for r in self.irRangeList_:
         s.extend( range( r[ 0 ], r[ 1 ] + 1 ) )
      return s

class IntfList( object ):
   def __init__( self, typ, irPathSet, noHoleRange=False ):
      '''A set of intfs.

           typ:  an IntfRange.IntfType (see above), e.g., "Ethernet".
                  All intfs must be the same type.

           irPathSet:  set of paths (3/2/1, 3/2/2, 3/2/3, etc.)
           noHoleRange:  assume there is no hole b/w paths'''
      self.type_ = typ
      self.irPathList_ = sorted( irPathSet )
      self.noHoleRange_ = noHoleRange

   def type( self ):
      return self.type_

   @Tac.memoize
   def _seqStr( self ):
      '''Pretty representation like "4-7,13,19-26".'''
      if isinstance( self.irPathList_[ 0 ], Generator ):
         result = ','.join( prefix + ( str( lo )
                                       if lo == hi
                                       else str( lo ) + '-' + str( hi ) )
                            for prefix, lo, hi in mergeRanges( self.irPathList_ ) )
         return result

      irPaths = self.irPathList_
      ranges = []
      i = 0
      while i < len( irPaths ):
         startPath = irPaths[ i ]
         nextPath = startPath.copy()
         if self.noHoleRange_:
            i = len( irPaths )
         else:
            while True:
               i += 1
               # Test for end of list before incrementing "nextPath". This
               # way, we avoid incrementing off the end of the port list
               # when the last member of irPaths is the last port.
               if i == len( irPaths ):
                  break
               nextPath, holes = nextPath.successor()
               if holes or irPaths[ i ] != nextPath:
                  break
         endPath = irPaths[ i - 1 ]
         startSeq = startPath.sequence()
         endSeq = endPath.sequence()
         if startPath == endPath:
            ranges.append( str( startPath ) )
         elif startSeq[ : -1 ] == endSeq[ : -1 ]:
            # This is the 3/2/7-22 case.
            ranges.append( "%s-%s" % ( startPath, endSeq[ -1 ] ) )
         else:
            # This is the 3/2/7-3/3/12 case.
            ranges.append( "%s-%s" % ( startPath, endPath ) )
      return ",".join( ranges )

   def __str__( self ):
      return self.type_.tagShort + self._seqStr()

   def strWithLongTag( self ):
      return self.type_.tagLong + self._seqStr()

   def aaaToken( self ):
      return self.strWithLongTag()

   def __len__( self ):
      if isinstance( self.irPathList_[ 0 ], Generator ):
         return sum( hi - lo + 1 for _, lo, hi in mergeRanges( self.irPathList_ ) )
      return len( self.irPathList_ )

   def intfIds( self ):
      '''List (actually, generator) of interface names, like
      [ 4,5,6,8 ] '''
      intfIds = []
      try:
         intfIds = ( int( str( i ) ) for i in self.irPathList_ )
      except:
         raise CliParserCommon.ParseError( "interface Ids must be numerical" )
      return intfIds

   def intfNames( self, shortName=False ):
      '''List (actually, generator) of interface names, like
      "Ethernet4","Ethernet5",..., sorted.'''
      prefix = self.type_.tagShort if shortName else self.type_.tagLong
      if isinstance( self.irPathList_[ 0 ], Generator ):
         seen = set()
         for rng in self.irPathList_:
            for intf in rng:
               intf = str( intf )
               if intf not in seen:
                  seen.add( intf )
                  yield prefix + intf
      else:
         for rng in self.irPathList_:
            yield prefix + str( rng )

   def newIntfList( self, intfs ):
      # This function returns a new IntfList from a list of interfaces. It's
      # basically a reverse function of intfNames().
      # intfs is a list of interface names (string) or interfaces that implement
      # the name() function.
      assert self.irPathList_, "intfList is empty"
      # pylint: disable-msg=protected-access
      endpoint = self.irPathList_[ 0 ]._endpoints
      # pylint: enable-msg=protected-access
      irPathSet = []
      for intf in intfs:
         intfName = str( intf )
         if intfName.startswith( self.type_.tagLong ):
            intfNumber = intfName[ len( self.type_.tagLong ) : ]
         else:
            assert intfName.startswith( self.type_.tagShort )
            intfNumber = intfName[ len( self.type_.tagShort ) : ]
         irPath = IrPath( endpoint, [ int( x ) for x in
                                      re.split( r'/|\.', intfNumber ) ],
                          '.' in intfNumber )
         irPathSet.append( irPath )
      return IntfList( self.type_, irPathSet )

class IrPathSet( object ):
   def __init__( self, mergeRange ):
      self.mergeRange_ = mergeRange
      if mergeRange:
         self.data_ = set()
      else:
         self.data_ = list()

   def add( self, val ):
      if self.mergeRange_:
         self.data_.add( val )
      else:
         self.data_.append( val )

   def __iter__( self ):
      return self.data_.__iter__()

   def __len__( self ):
      return self.data_.__len__()


class DoParse( object ):
   '''This class behaves like a subroutine that parses interface ranges.
   It recognizes strings like the following (see also AID433):

     Ethernet 1/1, 1/8-10,3/1-3/3  # Et 1/1, 1/8, 1/9, 1/10, 3/1, 3/2, and 3/3
     Eth1/1- $/$                   # All Ethernet interfaces
     Vl 1, 2-3 ,5                  # Vlan 1, 2, 3, and 5
     Et1 , 13 - 15, 11             # Et 1, 11, 13, 14, and 15
     Et1/45-2/3                    # Et 1/45, 1/46, 1/47, 1/48, 2/1, 2/2, and 2/3

   The key (internal) function parse() attempts to parse what has been seen
   so far, and returns an actual range or error string (if we are at the end
   of the range) and a list of completions (to guide more being entered).

   This class behaves much more like a subroutine than like a normal
   class.  Once the constructor is done, all that is left are these

       buff        string that was parsed
       intfList      resulting set of interfaces (None if input not fully valid)
                     (type is IntfList; call intfList.intfNames() to get names).
       errorMsg      reason why intfList is None (None if input is fully valid)
       completions   set of CliParser.Completion that could follow (None if there
                     is no way that the input could ever be a valid intf range).
   '''
   typeRe_ = re.compile( r"[A-Za-z-]*" )
   nonEmptyTypeRe_ = re.compile( r"[A-Za-z-]+" )
   subRe_ = re.compile( r"/|\." )
   numberDottedRe_ = re.compile( r"([0-9]+\.)?[0-9]*" )
   numberRe_ = re.compile( r"[0-9]*" )
   dashRe_ = re.compile( r"-" )
   commaRe_ = re.compile( r"," )
   endRe_ = re.compile( r"[$]" )

   def __init__( self, mode, buff, noSingletons, explicitIntfTypes,
                 genericRange=False, dottedRange=False, maxRanges=0,
                 noHoleRange=False, dollarCompHelpText=None,
                 mergeRange=True ):
      '''Parse an interface range specification in "buff".
      Pass "noSingletons=True" to prevent me from accepting a range that
      contains only one interface.  This is needed when IntfRangeRule is
      in a disjunction with PhysicalIntfRule, to avoid ambiguity.

      If genericRange is True, it returns a list of ranges instead of individual
      items. This only works for integers (GenericRangeType).

      dottedRange will allow numbers in a range to be accepted in the "dotted"
      notation. This dotted notation is introduced to allow writing U32s in a
      "<U16>.<U16>" representation for a BGP specific feature today (asdot
      notation support).
      Examples: 65536      can be represented as 1.0
                4294967295 can be represented as 65535.65535
                100        can be represented as 0.100.

      dottedRange can be in the form 0.100-0.200 or 1.1-1.100.
      Note: dottedRange implemeted here has maximum range [0.0,65535.65535] which
      is [0, 4294967295] in decimal.

      If noHoleRange is True, it returns a range string assuming that interfaces
      are consecutive.

      if mergeRange is True, it returns the list in merged and sorted
      order. If it is not true, the ranges will be outputted in the order
      provided by the command line
      '''
      t2( "DoParse(", buff, ")" )
      self.mode_ = mode
      self.buffer = buff
      self.str_ = buff
      # With introduction of dottedRange feature in to MultiRangeRule,
      # input buff can have dot in the string. The logic to set the hasSub_
      # by checking '.' in the buff will be done only if the input is
      # not a dottedRange.
      if dottedRange:
         self.hasSub_ = False
      else:
         self.hasSub_ = '.' in buff # Whether this interface is a sub interface.
      self.noSingletons_ = noSingletons
      self.completions = None
      self.intfTypeNoTag_ = None
      self.type_ = None
      self.slashes = 0
      self.genericRange_ = genericRange
      self.dottedRange_ = dottedRange
      self.maxRanges_ = maxRanges
      self.noHoleRange_ = noHoleRange
      self.rangeStartHasNext_ = None
      self.rangeEndHasNext_ = None
      self.rangeStart_ = None
      self.rangeEnd_ = None
      self.mergeRange_ = mergeRange
      if dollarCompHelpText is None:
         self.dollarCompHelpText_ = "list end"
      else:
         self.dollarCompHelpText_ = dollarCompHelpText
      if not isinstance( explicitIntfTypes, dict ):
         # transform a list into a dictionary keyed by tagLong.
         # special care is taken for types without tag (such as
         # GenericRangeType), so _needType() can handle them
         # correctly.
         self.intfTypes_ = {}
         for t in explicitIntfTypes:
            if t.tagLong:
               self.intfTypes_[ t.tagLong ] = t
            else:
               assert len( explicitIntfTypes ) == 1
               self.intfTypeNoTag_ = t
               break
      else:
         # already a dictionary (such as Arnet.allIntfTypes_)
         self.intfTypes_ = explicitIntfTypes

      try:
         self._parse()
         self.errorMsg = None
      except CliParserCommon.ParseError, e:
         self.intfList = None
         self.errorMsg = str( e )
      t2( "Parsed %r: (%r,%r)" % ( buff, self.intfList, self.errorMsg ) )

   def _slashCompletions( self ):
      return CliParserCommon.Completion( "/", "slash to specify %s" %
            self._helpDescForNumber( 1 ) )

   def _commaCompletions( self ):
      return CliParserCommon.Completion( ",", "extend list" )

   def isSingleton( self ):
      if len( self.irPathSet_ ) > 1:
         return False
      if not self.genericRange_:
         return True
      # Check the range
      for r in self.irPathSet_:
         return r[ 0 ] == r[ 1 ]
      # we shouldn't be here
      return False

   def _moreRangesAllowed( self, havePendingRange=False ):
      if not self.maxRanges_:
         return True
      numRanges = len( self.irPathSet_ )
      if havePendingRange:
         numRanges += 1
      return numRanges < self.maxRanges_

   def _parse( self ):
      self._needType()
      if self.hasSub_ and not getattr( self.type_, 'subSupported_', False ):
         raise CliParserCommon.ParseError(
               "this rule does not support sub part number" )
      self.irPathSet_ = IrPathSet( self.mergeRange_ )
      # At the beginning the list is empty, so we can have sub
      # completion if supported.
      listCanSubComp = True
      while True:
         self._needRangeStart()
         self._eatSpace()
         completions = [ CliParserCommon.Completion( "-", "specify range" ) ]
         if self._moreRangesAllowed( havePendingRange=True ):
            completions.append( self._commaCompletions() )
         if self.rangeStartHasNext_:
            completions.append( self._slashCompletions() )
         if self.rangeStart_.canSubComp_ and listCanSubComp:
            # 'interface ethernet 1, 2 ?' should not have '.' as one completion.
            completions.append( CliParserCommon.Completion( ".", "subinterface" ) )
         self._completionsAre( completions )
         if self._takeIfLookingAt( self.dashRe_ ):
            self._skipRepeatType()
            self._needRangeEnd()
         else:
            self.rangeEnd_ = self.rangeStart_
            self.rangeEndHasNext_ = False
         self.addRangeToSet()
         if ( not self._takeIfLookingAt( self.commaRe_ ) or
              not self._moreRangesAllowed() ):
            # That's all I can take.
            self._eatSpace()
            if self._moreRangesAllowed():
               completions = [ self._commaCompletions() ]
            else:
               completions = []
            if self.rangeEndHasNext_:
               completions.append( self._slashCompletions() )
            self._completionsAre( completions )
            if not self._atEnd():
               raise CliParserCommon.ParseError(
                     "extra characters after range: %r" % self.str_ )
            break
         self._skipRepeatType()
         listCanSubComp = self.rangeStart_.hasSub_
      if self.noSingletons_ and self.isSingleton():
         # Not ready to go yet.
         if self._moreRangesAllowed():
            completions = [ self._commaCompletions() ]
         else:
            completions = []
         if self.rangeEndHasNext_:
            completions.append( self._slashCompletions() )
         self._completionsAre( completions )
         raise CliParserCommon.ParseError(
               "range must include more than one interface" )
      if self.genericRange_:
         self.intfList = GenericRangeList( self.type_, self.irPathSet_,
                                           merge=self.mergeRange_ )
      else:
         self.intfList = IntfList( self.type_, self.irPathSet_,
                                   self.noHoleRange_ )

   @staticmethod
   def matchingTypes( intfTypes, typeStr ):
      typeStr = typeStr.lower()
      if isinstance( intfTypes, dict ):
         types = intfTypes.itervalues()
      else:
         types = intfTypes
      found = []
      for a in types:
         if ( a.tagLong.lower().startswith( typeStr ) or
              a.tagShort.lower().startswith( typeStr ) ):
            found.append( a )
      return found

   def _needType( self ):
      if self.intfTypeNoTag_ is not None:
         self.type_ = self.intfTypeNoTag_
         return
      typeStr = self._need( self.typeRe_ )
      found = self.matchingTypes( self.intfTypes_, typeStr )
      if len( found ) == 1:
         comps = [ CliParserCommon.Completion( a.tagLong, a.helpText ) for a in found
                   if a.tagLong ]
         self._completionsAre( comps )
         if typeStr == '':
            # This case arises if exactly one interface type is registered.
            raise CliParserCommon.ParseError( 'Missing interface type' )
         self.type_ = found[ 0 ]
         return
      if not found:
         raise CliParserCommon.ParseError(
               "%r is not a recognized interface type" % typeStr )
      raise CliParserCommon.ParseError(
            "%r is ambiguous as an interface type" % typeStr )

   def _needRangeStart( self ):
      self.rangeStart_ = None
      self.rangeStart_, self.rangeStartHasNext_ = \
          self._needIrPath( startingPoint=None )
      t2( "_needRangeStart:", self.rangeStart_, self.rangeStartHasNext_ )

   def _needRangeEnd( self ):
      self.rangeEnd_, self.rangeEndHasNext_ = \
          self._needIrPath( startingPoint=self.rangeStart_ )
      t2( "_needRangeEnd:", self.rangeEnd_, self.rangeEndHasNext_ )

   def _helpDescForNumber( self, diff=0 ):
      index = self.slashes + diff
      if index >= len( self.type_.helpDesc ):
         index = -1
      helpdesc = self.type_.helpDesc[ index ]
      return helpdesc

   def _skipRepeatType( self ):
      '''Skip a repeat of the Intf type, so we can accept Eth3-Eth5.'''
      typ = self._takeIfLookingAt( self.nonEmptyTypeRe_ )
      if not typ:
         return
      typ = typ.group( 0 )
      if self.type_.tagLong.lower().startswith( typ.lower() ):
         return
      if self.type_.tagShort.lower().startswith( typ.lower() ):
         return
      raise CliParserCommon.ParseError( "cannot mix types in range (%s, %s)" % (
         self.type_.tagLong, typ ) )

   def addRangeToSet( self ):
      # if not self.rangeStart_:
      #   return
      # Iterate from rangeStart_ to rangeEnd_, adding everything in there to the set.
      rangeStart = str( self.rangeStart_ )
      rangeEnd = str( self.rangeEnd_ )
      if self.genericRange_:
         if not self.mergeRange_:
            # To conserve memory, we limit the size of the list to 4094 VNIs.
            # To accomplish this, we sums the size of all the the current ranges
            # and makes sure that adding the new range does not exceed the limit
            sizePathSet = sum( hi - lo + 1 for ( lo, hi )
                               in self.irPathSet_ )
            if sizePathSet + ( int( rangeEnd ) - int( rangeStart ) + 1 ) > 4094:
               raise CliParserCommon.ParseError( "VNI range too large" )

         self.irPathSet_.add( ( int( rangeStart ), int( rangeEnd ) ) )
         return

      # Use generators for subinterfaces; othewrise a million is a bit much.
      if '.' in rangeStart and self.hasSub_:
         split = rangeStart.rsplit( '.', 1 )
         begin = int( split[ 1 ] )
         end = int( rangeEnd.rsplit( '.', 1 )[ -1 ] )
         prefix = split[ 0 ] + '.'

         # pylint: disable-msg=protected-access
         self.irPathSet_.add( Generator( prefix, begin, end,
                                         self.rangeStart_._endpoints ) )
         # pylint: enable-msg=protected-access
         return

      # Use generators for virtual interfaces; othewrise a million is a bit much.
      virtualIntfs = { 'Fabric', 'Port-Channel', 'Loopback',
                       'Vxlan', 'Tunnel', } # TODO: Add Vlan and Test.
      if any( self.type_ == self.intfTypes_.get( t ) for t in virtualIntfs ):
         # pylint: disable-msg=protected-access
         self.irPathSet_.add( Generator( '', int( rangeStart ),
                                     int( rangeEnd ),
                                     self.rangeStart_._endpoints ) )
         # pylint: enable-msg=protected-access
         return

      x = self.rangeStart_
      while True:
         if not x.endpoints().isTerminal( self.hasSub_ ):
            raise CliParserCommon.ParseError(
                  "extra characters after range: %r" % self.str_ )
         self.irPathSet_.add( x.copy() )
         if x == self.rangeEnd_:
            break
         x, _ = x.successor()

   def _morePathComponents( self ):
      # It's the end of a single range (e.g, 3/1-3); followed by nothing
      # or a comma/dash (not more interface numbers)
      self._eatSpace()
      return self.str_.startswith( ( "/", "." ) )

   def _pathComplete( self, endpoints ):
      '''This tests if the endpoints should be interpreted as a complete
      path. First of all, the endpoints have to be a valid interface
      (temrinal == True ), but it's not enough. To avoid ambiguity,
      one of the following also has to be False:

      1. The endpoints can be further expanded.
      2. There is a following slash.
      '''
      return endpoints.isTerminal( self.hasSub_ ) and \
          not ( endpoints.hasNext() and self._morePathComponents() )

   def _needIrPath( self, startingPoint ):
      '''Need something like 3/7/48.  If we are at the start of a range,
      then must be a complete path.  If we are at the end of a range, then
      startingPoint is specified, and we need either a complete path
      (3/7/1-3/7/48) or a single component (3/7/1-48).  We do not allow
      arbitrary partial paths e.g. 3/7/1-7/48 because it is just too confusing,
      and it can be ambiguous if 7/48 is a full or partial path (is that port 48
      of module 7, or port 48 of a port expander hanging off of 3/7?)  It is
      technically possible that 3/7/1-48 could be ambiguous, if there is a
      port 3/7/1 and a port 48.  We resolve the ambiguity in favor of a partial
      path e.g. 3/7/1-48 means 3/7/1-3/7/48 even if "48" is is a port by itself.'''

      # Currently we don't support non-scalar range end due to
      # performance issue with iterating through large number of
      # sub interfaces space.
      nonScalarDisalbed = ( True if startingPoint and startingPoint.hasSub_
                            else False )
      topEndpoints = self._endpoints( [] )
      t2( "_needIrPath", startingPoint, "topEndpoints", topEndpoints )
      assert not topEndpoints.terminal, \
             "Interface type claims that tag by itself is an interface!"
      if topEndpoints.invalid:
         # There *are* no interfaces of this type, at all.
         if not self.completions:
            self.completions = []
         raise CliParserCommon.ParseError( "there are no interfaces of type %s" % (
            self.type_.tagLong ) )
      if startingPoint:
         startingPoint = startingPoint.sequence() # break open the IrPath.
         # Say startingPoint is 7/1.  Then if we are starting a full
         # path, we must start at 7 or later, e.g. 7/2-7/13,
         # i.e. 7/2-4/13 is not legal.  However suppose our linecard
         # has four ports.  Then 7/2-4 is legal, but 7/2-5 and 7/2-1
         # are not.  So we are in a bizarre case where the next number
         # can be in the range (2,4) (closing a span) or (7,8)
         # (starting a new top-level port number), but can't be 1, 5
         # or 6.  We call (7,8) the "top endpoints" because these are
         # valid if we are starting the top of a new fully qualified
         # port number.  We call (2,4) the "span endpoints" because
         # they'd be closing a span.
         topEndpoints = Endpoint( startingPoint[ 0 ],
                                  # The range (7,8) in example above.
                                  topEndpoints.vmax )
         spanEndpoints = self._endpoints( startingPoint[ : -1 ] )
         spanEndpoints = Endpoint( ( startingPoint[ -1 ]
                                     if spanEndpoints.vmin is not None else None ),
                                   # The range (2,4) in example above.
                                   spanEndpoints.vmax,
                                   startingPoint[ -1 ] if \
                                      spanEndpoints.subMin else None,
                                   spanEndpoints.subMax,
                                   terminal=spanEndpoints.terminal )
      else:
         spanEndpoints = None

      if getattr( self, 'rangeStart_', False ) and self.rangeStart_.hasSub_:
         preComma = self.str_.split( ',' )[ 0 ]
         if nonScalarDisalbed and '.' in preComma or '/' in preComma:
            # Currently we don't support non-scalar range end due to
            # performance issue with iterating through large number of
            # sub interfaces space.
            raise CliParserCommon.ParseError(
                  "non-scalar range not supported for sub interface" )
         firstNumber = self._needNumber( spanEndpoints, isSub=True )
         # This line is for the support of non-scalar range for sub
         # interface. If in the future this is to be supported, use
         # this commented line instead.
         #firstNumber = self._needNumber( topEndpoints, spanEndpoints, isSub=True )
      else:
         firstNumber = self._needNumber( topEndpoints, spanEndpoints )
      topResult = [ topEndpoints.vmax if firstNumber == "$" else firstNumber ]
      topEndpoints = self._endpoints( topResult )
      if startingPoint:
         if topResult[ 0 ] < startingPoint[ 0 ]:
            # This is the case of "7/2-4...".  We better be closing a span,
            # because "4" is less than "7".
            topResult = None
            topEndpoints = INVALID_PREFIX
         if firstNumber < startingPoint[ -1 ]:
            # This is the case of "3/9-5...".  We better be starting a new path,
            # because "5" is less than "9".
            spanResult = None
            spanEndpoints = INVALID_PREFIX
         else:
            endMax = spanEndpoints.subMax if self.hasSub_ else spanEndpoints.vmax
            spanResult = self._irPath(
               ( startingPoint[ : -1 ] + [ endMax if firstNumber == '$'
                                        else firstNumber ] ),
               self.hasSub_ )
            spanEndpoints = spanResult.endpoints()
         if spanEndpoints.invalid:
            # This is the case of "7/2-7..." when module 7 has only four ports,
            # or "3/9-7...".  Either way, the "7" must be starting a new top path.
            pass
         elif spanEndpoints.isTerminal( self.hasSub_ ):
            # This is the case of "7/2-4..." or "3/2-4...".  We have
            # closed a legal span, and we may or may not be starting a
            # new top-level port.
            if ( ( self.hasSub_ and topEndpoints.isTerminal( self.hasSub_ ) or
                   topEndpoints.invalid or
                   self._pathComplete( topEndpoints ) )
                 or not self.hasSub_ and ( topEndpoints.invalid or
                                           self._pathComplete( topEndpoints ) ) ):
               # It is not possible to accept anything more.
               # This is a "3/2-4" case.  On a 1U box it is possible that
               # port 3 connects to a port expander, so the user might
               # mean 3/2,3/3,3/4,3/5,...,3/48,4, where port 4 is not
               # connected to a port expander.  If so --- too bad.  There
               # is no way to enter that range.  We avoid the ambiguity
               # by assuming a span if possible.
               return spanResult, False
            else:
               # This is the "3/2-4..." case on a modular box.  We need to
               # go on and consider the possibility of "3/2-4/10".
               pass
         else:
            # This is the case where the switch has hybrid interfaces.
            # For example, on Silverado we have et1-48, et49/1-8 and et50/1-8.
            # We come here if the user types 'et1-$'. We need to continue
            # to accept inputs.
            pass
      else:
         spanResult = None
         spanEndpoints = INVALID_PREFIX

      while True:
         if topEndpoints.invalid:
            # We got to a point where the user has descended into something that
            # does not exist.
            if not self.completions:
               self.completions = []
            raise CliParserCommon.ParseError( "no such interface" )

         if ( self._pathComplete( topEndpoints ) and
              ( not startingPoint or startingPoint <= topResult ) ):
            # This is the case of "7" or "3-7" on a 1U box with no port expander.
            # Nothing more can follow.
            assert spanEndpoints.invalid, 'we should have returned the span'
            return ( self._irPath( topResult, self.hasSub_,
                                   topEndpoints.subMin is not None ),
                     topEndpoints.hasNext() )

         self._eatSpace()
         completions = []
         # Currently we don't support non-scalar range end due to
         # performance issue with iterating through large number of
         # sub interfaces space.
         if topEndpoints.vmin is not None and not nonScalarDisalbed:
            completions.append( self._slashCompletions() )
         if topEndpoints.subMin is not None and not nonScalarDisalbed:
            completions.append( CliParserCommon.Completion( ".",
               "dot to specify subinterface number" ) )
         self._completionsAre( completions )
         if ( not self._morePathComponents() and
              spanEndpoints.isTerminal( self.hasSub_ ) ):
            # This is the "3/2-8" case.
            return spanResult, False

         self._eatSpace()

         if self.dottedRange_:
            isSub = False
         else:
            isSub = self._need( self.subRe_ ) == '.'
         self.slashes += 1
         spanEndpoints = INVALID_PREFIX # Span is no longer possible.
         if startingPoint:
            # Restrict the range of allowed continuations --- may not go
            # backwards, i.e., 6/10-6/4 is not legal.
            idx = len( topResult ) - 1
            assert idx >= 0
            assert topResult[ 0 ] >= startingPoint[ 0 ]
            if len( topResult ) <= len( startingPoint ) and \
                   topResult[ idx ] == startingPoint[ idx ]:
               newIdx = idx + 1 if idx + 1 < len( startingPoint ) else -1
               if isSub:
                  topEndpoints = Endpoint( topEndpoints.vmin,
                                           topEndpoints.vmax,
                                           max( topEndpoints.subMin,
                                                startingPoint[ newIdx ] ),
                                           topEndpoints.subMax,
                                           terminal=topEndpoints.terminal )
               else:
                  topEndpoints = Endpoint( max( topEndpoints.vmin,
                                                startingPoint[ newIdx ] ),
                                           topEndpoints.vmax )
         num = self._needNumber( topEndpoints, isSub=isSub )
         if num == '$':
            num = topEndpoints.subMax if isSub else topEndpoints.vmax
         topResult.append( num )
         topEndpoints = self._endpoints( topResult, isSub )

   def matchNumber( self ):
      '''Specifies the criteria/regex to match for a number. This method
         can be subclassed to match to different number formats'''
      if self.dottedRange_:
         return self._need( self.numberDottedRe_ )
      else:
         return self._need( self.numberRe_ )

   def convertToNumber( self, soFar ):
      '''Specifies the criteria/regex to convert the matched string
         to an integer value. This method can be subclassed to convert
         different number formats to an integer'''
      if self.dottedRange_:
         if '.' not in soFar:
            raise CliParserCommon.ParseError( "Invalid input" )
         soFarSpl = soFar.split( '.' )
         highU16 = CliParserCommon.safeInt( soFarSpl[ 0 ] )
         lowU16 = CliParserCommon.safeInt( soFarSpl[ 1 ] )
         if highU16 is None or lowU16 is None:
            raise CliParserCommon.ParseError( "Invalid input" )
         if highU16 < 0 or highU16 > 65535 or \
               lowU16 < 0 or lowU16 > 65535:
            raise CliParserCommon.ParseError( "Invalid input" )
         return highU16 << 16 | lowU16
      else:
         return CliParserCommon.safeInt( soFar )

   def _needNumber( self, range1, range2=None, isSub=False ):
      ranges = [ range1 ]
      if range2:
         if ( range2.vmin >= range1.vmin and range2.vmax <= range1.vmax and
              range2.subMin >= range1.subMin and range2.subMax <= range1.subMax ):
            # First range subsumes the second.  This is a common case,
            # say when the string is "4-" on a 1U system.  In this case,
            # there is really no difference between closing a span,
            # and providing a new top-level port, because there is only
            # one level!
            pass
         else:
            ranges.append( range2 )
      dollarComp = CliParserCommon.Completion( "$", self.dollarCompHelpText_,
            literal=False )
      if self._takeIfLookingAt( self.endRe_ ):
         # I do not know which range to terminate yet!
         # consider "3/10-$" versus "3/10-$/48".  "$" might mean "48" in the
         # first case, and "8" in second case.  So we need to convert the $
         # to an actual number later.
         self._completionsAre( [ dollarComp ] )
         return '$'
      soFar = self.matchNumber()
      comp = []
      for r in ranges:
         if r.subMin is not None:
            comp.append( CliParserCommon.Completion( "<%d-%d>" %
               ( r.subMin, r.subMax ), self.type_.subHelpDesc, literal=False ) )
         if r.vmin is not None:
            if self.dottedRange_:
               comp.append( CliParserCommon.Completion( "<%s-%s>" %
                  ( numToDottedStr( r.vmin ), numToDottedStr( r.vmax ) ),
                  self._helpDescForNumber(), literal=False ) )
            else:
               comp.append( CliParserCommon.Completion(
                  "<%d-%d>" % ( r.vmin, r.vmax ), self._helpDescForNumber(),
                  literal=False ) )

      if not soFar:
         comp.append( dollarComp )
      if soFar:
         soFar = self.convertToNumber( soFar )
         for r in ranges:
            vmin, vmax = ( r.subMin, r.subMax ) if isSub else ( r.vmin, r.vmax )
            if soFar <= vmax:
               # We are not too big, so adding more digits can help.
               self._completionsAre( comp )
            if vmin <= soFar <= vmax:
               return soFar
         raise CliParserCommon.ParseError( "%r is out of range" % soFar )
      else:
         self._completionsAre( comp )
         raise CliParserCommon.ParseError( "number expected" )

   def _need( self, regEx, errorMsg=None ):
      m = self._takeIfLookingAt( regEx )
      if not m:
         raise CliParserCommon.ParseError( errorMsg or ( "%r expected" % regEx ) )
      return m.group( 0 )

   def _eatSpace( self ):
      self.str_ = self.str_.lstrip()

   def _takeIfLookingAt( self, regexp ):
      s = self.str_.lstrip()
      m = regexp.match( s )
      if m:
         self.str_ = s[ m.end() : ]
      return m

   def _completionsAre( self, completions ):
      # This is tricky.  This function is called when we have just
      # absorbed as much as we can (possibly none) of a
      # multi-character item, such as interface type or an interface
      # number.  Critically, we have not swallowed any final space.
      # So we record the completions only if we know there is no final
      # space.
      if self.str_:
         return

      # Further, if more than one such thing tries to provide
      # completions, we know to believe the first, because it may already
      # be complete and we tried to do the second and then found 0 characters
      # of the second.  This case arises when the input is just "Ethe".  That's
      # enough to know the type, so we call this function once, and then we
      # try to find a number and call the function a second time.  In this
      # case, we want the completion of "Ethe".  That is very different from
      # "Ethe ".  If there's a space at the end, the first completion won't
      # stick (because of the test above), but looking for "0-9" will eat the
      # space, so the second completion "<1-48>" sticks.
      if self.completions is not None:
         return

      self.completions = completions

   def _atEnd( self ):
      # The behavior of completions is very sensitive to whitespace.
      # Therefore, if there is more whitespace, we are not at the end yet.
      return self.str_ == ''

   def _endpoints( self, partialPath, hasSub=False ):
      if getattr( self.type_, "subIntfGuard_", False ) and self.mode_:
         result = self.type_.subIntfGuard_[ 0 ]( self.mode_, None )
         self.type_.subSupported_ = result is None
      if self.type_.dynamicRange_:
         return self.type_.endpoints( partialPath, hasSub, self.mode_ )
      return self.type_.endpoints( partialPath, hasSub )

   def _irPath( self, partialPath, hasSub=False, canSubComp=False ):
      return IrPath( self._endpoints, partialPath, hasSub, canSubComp )

class IrrState( object ):
   '''Private class holding state while the IntfRangeRule picks up tokens.
   If genericRange is True, return a list of ranges instead of individual
   items.
   '''

   def __init__( self, noSingletons, explicitIntfTypes,
                 genericRange=False, dottedRange=False, maxRanges=0,
                 dollarCompHelpText=None,
                 DoParseClass=DoParse ):
      self.buffer_ = ''
      self.noSingletons_ = noSingletons
      self.explicitIntfTypes_ = explicitIntfTypes
      self.tokenIndices_ = []
      self.genericRange_ = genericRange
      self.dottedRange_ = dottedRange
      self.maxRanges_ = maxRanges
      self.dollarCompHelpText_ = dollarCompHelpText
      self.type_ = None
      # cache to improve performance
      self.parsedBuffer_ = None
      self.parsedResult_ = None
      self.DoParseClass_ = DoParseClass

   def match( self, mode, token ):
      # for the new parser
      r = self.parse( mode, extra=token )
      if not r.completions:
         return CliParserCommon.noMatch
      self.buffer_ = r.buffer
      # if we return partialMatch, the caller might want to know
      # the type (to call guard function), so we save it here.
      self.type_ = r.type_
      if r.errorMsg:
         return CliMatcher.partialMatch
      return CliMatcher.MatchResult( r.intfList,
                                     r.intfList.aaaToken(),
                                     True )

   def append( self, mode, token ):
      r = self.parse( mode, extra=token.text )
      if not r.completions:
         return CliParserCommon.noMatch
      self.buffer_ = r.buffer
      self.tokenIndices_.append( token.index )
      return r

   def _completions( self, mode, token ):
      r = self.parse( mode, extra=token )
      return r.completions or []

   def completions( self, mode, token ):
      # We need to handle partial token matching multiple interfaces better.
      # The idea is that if we see things like "V1", we should expand to
      # "Vlan1" and "Vxlan1" respectively and let autocomplete return
      # AmbiguousCommandError.
      completions = self._completions( mode, token )
      if ( not self.buffer_ and not self.genericRange_ and
           not any( c for c in completions if c.literal ) ):
         m = DoParse.typeRe_.match( token )
         if m:
            typeStr = m.group()
            if not typeStr and token:
               # no interface type, but has numbers, just return
               return completions
            found = DoParse.matchingTypes( self.explicitIntfTypes_, typeStr )
            rest = token[ len( typeStr ) : ]
            completions = []
            literals = []
            for intfType in found:
               newToken = intfType.tagLong + rest
               # these completions should be non-literal
               compl = self._completions( mode, newToken )
               if compl:
                  completions += compl
                  c = CliParserCommon.Completion( newToken,
                                                  intfType.helpText,
                                                  literal=True,
                                                  partial=bool( rest ) )
                  literals.append( c )
            if len( literals ) > 1:
               # If we have multiple literal completions, throw away the
               # non-literal ones. This way 'v1' would expand to 'Vlan1'
               # and 'Vxlan1', without '<1-4094>' and '<1-1>' which is confusing.
               completions = literals
            else:
               completions += literals
      return completions

   def parse( self, mode, extra=None ):
      b = self.buffer_
      if b:
         b += ' '  # to terminate the last token.
      if extra:
         b += extra
         # Do not add space; do not terminate this token.  We want
         # completions starting from this exact point (after extra,
         # but with a space at the end if extra is None or ''.)
      if self.parsedBuffer_ is None or self.parsedBuffer_ != b:
         r = self.DoParseClass_( mode, b, self.noSingletons_,
                      self.explicitIntfTypes_,
                      genericRange=self.genericRange_,
                      dottedRange=self.dottedRange_,
                      maxRanges=self.maxRanges_,
                      dollarCompHelpText=self.dollarCompHelpText_ )
         self.parsedBuffer_ = b
         self.parsedResult_ = r
      return self.parsedResult_

def genericEndpoints( partialPath, rangeFn, hasSub=False, rangeArgs=() ):
   # partialPath should either be empty or have one element
   minimum, maximum = rangeFn( *rangeArgs )
   if not partialPath:
      return Endpoint( minimum, maximum )
   elif len( partialPath ) == 1:
      try:
         v = int( partialPath[ 0 ] )
         if minimum <= v <= maximum:
            return TERMINAL
      except ValueError:
         pass

   return INVALID_PREFIX

class GenericRangeType( object ):
   tagShort = ""

   def __init__( self, rangeFn, helpdesc, tagLong="", dynamicRange=False ):
      self.rangeFn = rangeFn
      self.tagLong = tagLong
      self.helpText = helpdesc
      self.helpDesc = [ helpdesc ]
      self.dynamicRange_ = dynamicRange

   def invalidate( self ):
      pass

   def endpoints( self, partialPath, hasSub=False ):
      return genericEndpoints( partialPath, self.rangeFn, hasSub )

class MultiRangeMatcher( CliMatcher.Matcher ):
   # new parser
   def __init__( self, rangeFn, noSingletons,
                 helpdesc, dottedRange=False, tagLong='', maxRanges=0,
                 DoParseClass=DoParse,
                 **kwargs ):
      self.noSingletons_ = noSingletons
      CliMatcher.Matcher.__init__( self, **kwargs )
      self.helpdesc_ = helpdesc
      self.rangeFn_ = rangeFn
      self.tagLong_ = tagLong
      self.dottedRange_ = dottedRange
      self.maxRanges_ = maxRanges
      self.DoParseClass_ = DoParseClass
      self.intfTypes_ = [ GenericRangeType( self.rangeFn_,
         self.helpdesc_, self.tagLong_ ) ]

   def __str__( self ):
      return "<range>"

   def _initState( self, context ):
      return IrrState( noSingletons=self.noSingletons_,
                       explicitIntfTypes=self.intfTypes_,
                       genericRange=True,
                       dottedRange=self.dottedRange_,
                       maxRanges=self.maxRanges_,
                       DoParseClass=self.DoParseClass_ )

   def match( self, mode, context, token ):
      s = context.state or self._initState( context )
      assert s.__class__ is IrrState
      r = s.match( mode, token )
      if r is not CliParserCommon.noMatch:
         context.state = s
      return r

   def completions( self, mode, context, token ):
      s = context.state or self._initState( context )
      assert s.__class__ is IrrState
      return s.completions( mode, token )

def _multiRangeToList( multiRange ):
   if not multiRange:
      return []
   maxInt = 0xFFFFFFFF
   gr = GenericRangeType( lambda: ( 0, maxInt ), "" )

   def endpointFunc( partial, hasSub=False ):
      return gr.endpoints( partial, hasSub )
   irPathSet = [ IrPath( endpointFunc, ( r, ) ) for r in multiRange ]
   intfList = IntfList( GenericRangeType, irPathSet )
   return str( intfList ).split( ',' )

def multiRangeToCanonicalString( multiRange ):
   """Generates a string from the range of numbers passed in.
   For example,  if multiRange = [1,2,3,4,8], the output string
   will be "1-4,8" """
   return ','.join( _multiRangeToList( multiRange ) )

def multiRangeToCanonicalStringGen( multiRange, strLimit=sys.maxint, delimiter=',' ):
   """Similar to multiRangeToCanonicalString, but returns a generator
   which guarantees each string does not exceed the specified size
   limit. strLimit defaults to a ridiculously high value so that we
   default to returning a single string."""
   rangeStrings = _multiRangeToList( multiRange )
   # Here's the generator part.  When the result string gets too long,
   # we 'yield' it as a result and then keep going.  We calculate the
   # current string length as we go rather than building the string and
   # checking its length. This avoids the quadratic cost of appending to
   # a string over and over again (realloc more memory, copy entire string
   # over, do it again for each append.
   resultStartIndex = 0
   resultLen = -1 # compensate for no prepended comma on the first string
   for i, r in enumerate( rangeStrings ):
      if resultLen + len( r ) + 1 > strLimit:
         assert i > resultStartIndex
         rangesThatFit = rangeStrings[ resultStartIndex : i ]
         yield delimiter.join( rangesThatFit )
         resultStartIndex = i
         resultLen = len( r )
      else:
         resultLen += ( len( r ) + 1 ) # plus 1 for the prepended comma
   rangesThatFit = rangeStrings[ resultStartIndex : ]
   if rangesThatFit:
      # Only yield non-empty results.  Otherwise, the code iterating over the
      # output has to check for empty string results, which seems wrong.
      yield delimiter.join( rangesThatFit )

def multiRangeFromCanonicalString( rangeSetString ):
   """From a canonical string such as "1-4,8" return a set of
   integers set( [ 1, 2, 3, 4, 8 ] ) """
   resultSet = set()
   if not rangeSetString:
      return resultSet
   rangeStrings = rangeSetString.split( ',' )
   for rangeString in rangeStrings:
      parts = [ int( p ) for p in rangeString.split( '-' ) ]
      if len( parts ) == 1:
         [ lbound ] = parts
         ubound = lbound
      else:
         [ lbound, ubound ] = parts
         assert ubound > lbound
      resultSet.update( range( lbound, ubound + 1 ) )
   return resultSet

# Helper methods to go with RangeSet management
def packRanges( tupleRangeList ):
   '''
   It packs a list of overlapping ranges into a list of non-overapping ranges.
   For example, when tupleRangeList is [ ( 2, 7), (8, 9), ( 20, 50 ), ( 45, 90 ) ]
   it returns [ ( 2, 9 ), ( 20, 90 ) ]. Note that the adjacent integers are
   also coalesced into a range
   '''
   newList = []
   if not tupleRangeList:
      return newList

   for aTuple in tupleRangeList:
      if map( type, aTuple ) != [ int, int ]:
         raise ValueError( "Wrong tuple type" )
      if aTuple[ 0 ] > aTuple[ 1 ]:
         raise ValueError( "Start of a range is higher than the end" )

   tupleRangeList = sorted( tupleRangeList )
   i = 0
   size = len( tupleRangeList )

   lowerBound, upperBound = tupleRangeList[ 0 ]
   while i < size - 1:
      i += 1
      if tupleRangeList[ i ][ 0 ] <= upperBound + 1:
         upperBound = max( tupleRangeList[ i ][ 1 ], upperBound )
      else:
         newList.append( ( lowerBound, upperBound ) )
         lowerBound, upperBound = tupleRangeList[ i ]

   newList.append( ( lowerBound, upperBound ) )
   return newList

def decodeStrRanges( ranges ):
   '''
   Takes a set of integer ranges encoded as a string and returns a non-overlapping
   ranges represented as a list of tuples. The input can conatin a sequence of
   integers or integer ranges separated by ','. A range can be represented as a
   hyphenated string, e.g., 3-10.
   For example:
      a) '' -> []
      b) '1' -> [ ( 1, 1 ) ]
      c) '1,2,3,5' -> [ ( 1, 3 ), ( 5, 5 ) ]
      d) '1,2-9,50,45-90' -> ( [ ( 1, 9 ), ( 45, 90 ) ] )
   '''
   initialList = []
   if not ranges:
      return initialList
   for aRange in ranges.split( ',' ):
      parts = aRange.split( '-' )
      if len( parts ) == 1 and not parts[ 0 ]:
         # empty string
         continue
      if len( parts ) == 1:
         initialList.append( ( int( parts[ 0 ] ), int( parts[ 0 ] ) ) )
      elif len( parts ) == 2 and int( parts[ 0 ] ) <= int( parts[ 1 ] ):
         initialList.append( ( int( parts[ 0 ] ), int( parts[ 1 ] ) ) )
      else:
         raise ValueError( "Not a valid string format" )

   return packRanges( initialList )

# Factory methods
def rangeSetFromInteger( value ):
   if not isinstance( value, int ):
      raise ValueError( "Not a valid integer" )
   return RangeSet( [ ( value, value ) ] )

def rangeSetFromPair( lower, upper ):
   if not all( isinstance( i, int ) for i in [ lower, upper ] ):
      raise ValueError( "Not a valid integer range" )
   if lower > upper:
      raise ValueError( "Lower bound: %d is higher than upper bound: %d"
            % ( lower, upper ) )
   return RangeSet( [ ( lower, upper ) ] )

def rangeSetFromString( ranges ):
   return RangeSet( decodeStrRanges( ranges ) )

class RangeBuilder( object ):
   def __init__( self, entries ):
      self.entries = entries
      self.pos = 0

   def start( self, offset=0 ):
      return self.entries[ self.pos + offset ][ 0 ]

   def end( self, offset=0 ):
      return self.entries[ self.pos + offset ][ 1 ]

   def atEnd( self ):
      return self.pos >= len( self.entries )

   def remainder( self ):
      return self.entries[ self.pos : ]

   def getUpper( self, other ):
      '''
      This method returns the upper bound of the range at self.pos (say, pivot range)
      by coalescing all overlapping ranges with it. As all ranges of one of the lists
      are non-overlapping and sorted, initially we need to consider only the ranges
      from the other one. Now, if we are able to coalesce one or more ranges from the
      other list and this extends the upper bound of our pivot range, we need to
      again check whether we can extend this range even further by considering the
      last range (that we coalesced) as the new pivot. In this case, we need to
      compare ranges after self.pos with the new pivot. The recursive call to
      getUpper achieves this.
      '''
      initialPos = other.pos

      while not other.atEnd() and other.start() <= ( self.end() + 1 ):
         other.pos += 1

      self.pos += 1
      if other.pos != initialPos and other.end( -1 ) > self.end( -1 ):
         other.pos -= 1
         return other.getUpper( self )
      else:
         return self.end( -1 )

   def nextRange( self, other ):
      '''
      This method returns a range by coalescing the ranges from two lists that
      overlap with the range at self.pos.
      '''
      return ( self.start(), self.getUpper( other ) )

class RangeSet( object ):
   '''
   RangeSet provides set union and set difference on a set of integer ranges. Each
   element of the set can either be an integer or a range where start of a range is
   less than the end.
   Both union and difference operations are of O(n), where n is the number of ranges.
   '''
   def __init__( self, ranges=None ):
      '''
      Argement 'ranges' can be in the following forms
         a) None
         b) A valid list of tuples. Each tuple represents a range where both lower
         and upper bounds are interger. Module level helper methods are provided to
         reformat/decode the supported input for RangeSet.
      '''
      self.entries = [] if not ranges else packRanges( ranges )

   def union( self, right ):
      '''
      Returns a new RangeSet as an union of two instances of RangeSet.
      '''
      rLeft = RangeBuilder( self.entries )
      rRight = RangeBuilder( right.entries )

      rUnion = []
      while not ( rLeft.atEnd() or rRight.atEnd() ):
         rUnion.append( rLeft.nextRange( rRight ) if rLeft.start() < rRight.start()
              else rRight.nextRange( rLeft ) )

      return RangeSet( rUnion + rLeft.remainder() + rRight.remainder() )

   def difference( self, right ):
      '''
      Returns a new RangeSet which is a difference of self and the right.
      '''
      def recDifference( j, current ):
         if j >= sizeY or not current:
            return current
         leftRange = leftStrip( current[ 0 ], right.entries[ j ] )
         rightRange = rightStrip( current[ 0 ], right.entries[ j ] )
         return ( leftRange +
               ( recDifference( j + 1, rightRange + current[ 1 : ] ) if rightRange
               else recDifference( j, current[ 1 : ] ) ) )

      def valid( rangeA ):
         return rangeA[ 0 ] <= rangeA[ 1 ]

      def leftStrip( rangeX, rangeY ):
         strip = ( rangeX[ 0 ], min( rangeX[ 1 ], rangeY[ 0 ] - 1 ) )
         return [ strip ] if valid( strip ) else []

      def rightStrip( rangeX, rangeY ):
         strip = ( max( rangeX[ 0 ], rangeY[ 1 ] + 1 ), rangeX[ 1 ] )
         return [ strip ] if valid( strip ) else []

      sizeY = len( right.entries )
      return RangeSet( recDifference( 0, self.entries ) )

   def __str__( self ):
      '''
      Encodes the list of ranges into a string.
         a) [] -> ''
         b) [ ( 1, 1 ), ( 20, 30 ) ] -> '1,20-30'
      '''
      rangeEncoding = ( lambda entry: str( entry[ 0 ] )
            if entry[ 0 ] == entry[ 1 ] else '%d-%d' % entry )
      output = ( rangeEncoding( entry ) for entry in self.entries )
      return ','.join( output )

   __or__ = union
   __sub__ = difference
