#!/usr/bin/env python2.7
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import threading
import traceback
import Tac
import MgmtSecuritySslStatusSm
from MssPolicyMonitor import Lib
from MssPolicyMonitor.Lib import b0, b1, b2, b3, b4, v, VALID, ACTIVE
from MssPolicyMonitor.Lib import UNCLASSIFIED_ZONE, NON_INTERCEPT_ZONE, ZONE_A, t2
from MssPolicyMonitor.Lib import INTERCEPT_ZONE, ZONE_B, DRY_RUN, SUSPEND, SHUTDOWN
from MssPolicyMonitor.Lib import __defaultTraceHandle__  #pylint: disable-msg=W0611


NullEthAddr = Tac.Value('Arnet::EthAddr')
activityLockHolder = None


def genServicePoliciesAsNeeded( devicePolicy, sysdbMgr ):
   ''' Trigger MSS ServicePolicy generation when DevicePolicy is complete,
       valid and if there are intercepts in a zone.  If policy has hosts
       in both zones then destination (ZoneB) has precedence and source
       (ZoneA) hosts will be ignored and not intercepted or redirected.
   '''
   if not devicePolicyComplete( devicePolicy ):
      return None

   if Lib.isMssL3EnabledAndL3Policy( devicePolicy ):
      return None  # no MssSvcPolicies to create if MssL3 and L3 zones

   zoneAIntercepts = devicePolicy.zoneA.intercept
   zoneBIntercepts = devicePolicy.zoneB.intercept
   if devicePolicy.zoneB.zoneClass == INTERCEPT_ZONE and zoneBIntercepts:
      servicePolicy = sysdbMgr.getMssServicePolicy( devicePolicy, ZONE_B )
      sysdbMgr.populateSvcPolicyIntercepts( zoneBIntercepts, servicePolicy )
   elif devicePolicy.zoneA.zoneClass == INTERCEPT_ZONE and zoneAIntercepts:
      servicePolicy = sysdbMgr.getMssServicePolicy( devicePolicy, ZONE_A )
      sysdbMgr.populateSvcPolicyIntercepts( zoneAIntercepts, servicePolicy )
   else:
      return None
   return servicePolicy


def threadName():
   return threading.currentThread().name


def getDeviceSetsUsingLock( mpmConfig ):
   with ActivityLock():
      return mpmConfig.deviceSet.values()


def exception( function ):
   ''' decorator to log error message on exceptions
   '''
   def logErrors( *args, **kwargs ):
      try:
         function( *args, **kwargs )
      except Exception:  # pylint: disable-msg=W0703
         traceback.print_exc()
   return logErrors


def deviceConfigComplete( serviceDevice, deviceSet ):
   ''' Verify that all necessary attributes have been set.
   '''
   complete = bool(
      serviceDevice.username and
      serviceDevice.encryptedPassword and
      serviceDevice.protocol and
      serviceDevice.protocolPortNum and

      # both isAggregationMgr and a group must be set if either is set
      ( ( deviceSet.isAggregationMgr and serviceDevice.group ) or
        ( not deviceSet.isAggregationMgr and not serviceDevice.group ) ) and

      # if Fortinet plugin then virtualDomain must be configured
      ( deviceSet.policySourceType != Lib.FORTIMGR_PLUGIN or
        deviceSet.virtualDomain and deviceSet.adminDomain )
   )
   b3( v( serviceDevice.ipAddrOrDnsName ), 'DSet', v( deviceSet.name ),
       'complete', v( complete ) )
   return complete


def devicePolicyComplete( devicePolicy ):
   ''' Verify that all attributes necessary to generate MSS
       ServicePolicy and ServiceInterfaces have been set.
   '''
   complete = bool(
      devicePolicy.zoneA and devicePolicy.zoneA.link and
      devicePolicy.zoneB and devicePolicy.zoneB.link and
      validServiceDeviceLinks( devicePolicy ) and
      validZoneInterceptClasses( devicePolicy ) and
      ( devicePolicy.zoneA.intercept or
        devicePolicy.zoneB.intercept ) )
   b3( v( devicePolicy.serviceDeviceId ), 'policy', v( devicePolicy.policyName ),
       'complete', v( complete ) )
   return complete


def validZoneInterceptClasses( devicePolicy ):
   ''' DevicePolicy entities have a zoneA and a zoneB.
      The only valid ZoneInterceptClass state is when one zone is
      ZoneInterceptClass.intercept and the other is
      ZoneInterceptClass.nonIntercept.
   '''
   if ( Lib.isMssL3EnabledAndL3Policy( devicePolicy ) and
        devicePolicy.zoneA.zoneClass == INTERCEPT_ZONE and
        devicePolicy.zoneB.zoneClass == INTERCEPT_ZONE ):
      return True
   else:
      return (
         ( devicePolicy.zoneA.zoneClass == NON_INTERCEPT_ZONE and
           devicePolicy.zoneB.zoneClass == INTERCEPT_ZONE ) or
         ( devicePolicy.zoneA.zoneClass == INTERCEPT_ZONE and
           devicePolicy.zoneB.zoneClass == NON_INTERCEPT_ZONE ) )


def validServiceDeviceLinks( devicePolicy ):
   ''' For MssL2 all attributes of ServiceDeviceLink must have a value and
       vlan range must be the same for both service device links.  For
       MssL3 switchId, switchIntf and allowedVlan will not be set.
       Note: may be a transient state where change has not propagated
             to both links
   '''
   l2Policy = not Lib.isMssL3EnabledAndL3Policy( devicePolicy )
   allowedVlanSets = []
   for sdLinks in [ devicePolicy.zoneA.link.values(),
                    devicePolicy.zoneB.link.values() ]:
      for sdLink in sdLinks:
         if ( not sdLink.linkName or
              not sdLink.serviceDeviceIntf or
              l2Policy and (
                 sdLink.switchId == NullEthAddr.stringValue or
                 not sdLink.switchIntf or
                 not sdLink.allowedVlan ) ):
            b2('not a valid link:', v( sdLink ) )
            return False
         allowedVlanSets.append( set( sdLink.allowedVlan.keys() ) )

   if l2Policy:
      # ensure allowedVlan range equal in all vwire links
      baseVlanSet = allowedVlanSets[ 0 ]
      for vlanSet in allowedVlanSets[ 1: ]:
         if vlanSet ^ baseVlanSet:  # check symmetric_difference between sets
            b1( 'Notice: DevicePolicy', v( devicePolicy.policyName ),
                'current links have different vlans:', v( vlanSet ), '!=',
                v( baseVlanSet ) )
            return False
   return True

####################################################################################
class TopologyConvergenceSM( Tac.Notifiee ):
   notifierTypeName = 'NetworkTopologyAggregatorV3::Status'

   def __init__( self, cdbTopology, mpmAgent ):
      Tac.Notifiee.__init__( self, cdbTopology )
      b2( 'initializing' )
      self.mpmAgent = mpmAgent
      self.topoConvergenceCompleteSm = None
      if cdbTopology.convergenceStatus:  # check before exiting SM constructor
         self.topoConvergenceCompleteSm = TopologyConvergenceCompleteSM(
            cdbTopology.convergenceStatus, mpmAgent )

   @Tac.handler('convergenceStatus')
   def handleConvergenceStatus( self ):
      cdbTopology = self.notifier_
      if cdbTopology.convergenceStatus and not self.topoConvergenceCompleteSm:
         self.topoConvergenceCompleteSm = TopologyConvergenceCompleteSM(
            cdbTopology.convergenceStatus, self.mpmAgent )

####################################################################################
class TopologyConvergenceCompleteSM( Tac.Notifiee ):
   notifierTypeName = 'NetworkTopologyAggregatorV3::ConvergenceStatus'

   def __init__( self, topoConvergenceStatus, mpmAgent ):
      Tac.Notifiee.__init__( self, topoConvergenceStatus )
      self.mpmAgent = mpmAgent
      self.agentInitialized = False
      self.maybeFinishAgentInit( topoConvergenceStatus )

   @Tac.handler('convergenceInProgress')
   def handleConvergenceInProgress( self ):
      self.maybeFinishAgentInit( self.notifier_ )

   def maybeFinishAgentInit( self, topoConvergenceStatus ):
      if not self.agentInitialized:
         if topoConvergenceStatus.convergenceInProgress:
            b0( 'waiting for network topology service convergence...' )
         else:
            self.mpmAgent.finishAgentInit()
            self.agentInitialized = True

####################################################################################
class ServiceDeviceZoneSM( Tac.Notifiee ):
   ''' This state machine handles changes to ServiceDeviceZone name,
       switch-service device link and intercepts.
   '''
   notifierTypeName = 'MssPolicyMonitor::ServiceDeviceZone'

   def __init__( self, entity, sysdbMgr ):
      Tac.Notifiee.__init__( self, entity )
      self.sysdbMgr = sysdbMgr
      serviceDeviceZone = self.notifier_
      b4( 'zone:', v( serviceDeviceZone.zoneId ), 'policy:', v( self.policyName() ) )
      self.shadowLinkZone = None
      self.shadowZoneName = ''
      self.shadowZoneInterceptClass = UNCLASSIFIED_ZONE

   @Tac.handler('zoneName')
   @exception
   def handleZoneName( self ):
      svcDeviceZone = self.notifier_
      if Lib.isMssL3EnabledAndL3Zone( svcDeviceZone ):
         return  # no processing to do here for MssL3
      zoneName = svcDeviceZone.zoneName
      oldZoneName = self.shadowZoneName
      if self.deviceStateActive():
         b2( v( self.policyName() ), v( svcDeviceZone.zoneId ), 'oldZoneName:',
             v( oldZoneName ), 'newZoneName:', v( zoneName ), 'for links:',
             v( svcDeviceZone.link.keys() ) )
         if svcDeviceZone.link and oldZoneName:
            self.sysdbMgr.updateZoneName( oldZoneName, zoneName, svcDeviceZone.link )
      self.shadowZoneName = zoneName

   @Tac.handler('link')
   @exception
   def handleLink( self, linkName ):
      serviceDeviceZone = self.notifier_
      if Lib.isMssL3EnabledAndL3Zone( serviceDeviceZone ):
         return  # no processing to do here for MssL3
      oldLinks = self.shadowLinkZone
      newLinks = serviceDeviceZone.link
      zoneId = serviceDeviceZone.zoneId
      zoneName = serviceDeviceZone.zoneName
      isIceptZone = ( serviceDeviceZone.zoneClass == INTERCEPT_ZONE )
      if self.deviceStateActive():
         #b3( 'link update', v( zoneId ), v( zoneName ),
         #    'OLD:', v( oldLinks.keys() if oldLinks else 'None' ),
         #    'NEW:', v( newLinks.keys() if newLinks else 'None' ) )
         if not newLinks:
            b2( v( self.policyName() ), v( zoneId ), 'link removed',
                v( oldLinks.keys() if oldLinks else '??' ) )
            if self.sysdbMgr.isSharingChildServicePolicy( self.devicePolicy() ):
               self.handleSharedServicePolicy( self.devicePolicy(),
                                               serviceDeviceZone )
            else:
               self.sysdbMgr.updateServiceDeviceLinks(
                  None, oldLinks, zoneName, isIceptZone, self.devicePolicy() )
         elif not oldLinks:
            b2( v( self.policyName() ), v( zoneId ), 'link added',
                v( newLinks.keys() if newLinks else '??' ) )
            genServicePoliciesAsNeeded( self.devicePolicy(), self.sysdbMgr )
         else:
            b2( v( self.policyName() ), v( zoneId ), 'link changed OLD:',
                v( oldLinks.keys() if oldLinks else 'None' ), 'NEW:',
                v( newLinks.keys() if newLinks else 'None' ) )

            if self.sysdbMgr.isSharingChildServicePolicy( self.devicePolicy() ):
               self.handleSharedServicePolicy( self.devicePolicy(),
                                               serviceDeviceZone )
            else:
               newServicePolicy = genServicePoliciesAsNeeded( self.devicePolicy(),
                                                              self.sysdbMgr )
               if newServicePolicy:
                  obsoleteServicePolicies = (
                     set( self.devicePolicy().childServicePolicy.keys() ) -
                     set( [ newServicePolicy.name ] ) )
                  for servicePolicyName in obsoleteServicePolicies:
                     del self.devicePolicy().childServicePolicy[ servicePolicyName ]
                     self.sysdbMgr.deleteServicePolicyNamed( servicePolicyName )
         self.shadowLinkZone = dict( newLinks )

   @Tac.handler('intercept')
   @exception
   def handleIntercept( self, ipAddr ):
      serviceDeviceZone = self.notifier_
      if Lib.isMssL3EnabledAndL3Zone( serviceDeviceZone ):
         return  # no processing to do here for MssL3
      zoneId = serviceDeviceZone.zoneId
      numIntercepts = len( serviceDeviceZone.intercept )
      if self.deviceStateActive():
         if ipAddr not in serviceDeviceZone.intercept:
            self.sysdbMgr.deleteServicePolicyIntercept(
               ipAddr, zoneId, self.devicePolicy(), numIntercepts )
         else:
            if serviceDeviceZone.zoneClass == INTERCEPT_ZONE:
               servicePolicy = self.sysdbMgr.getMssServicePolicy(
                  self.devicePolicy(), zoneId )
               self.sysdbMgr.addServicePolicyIntercept( ipAddr, servicePolicy )

   @Tac.handler('zoneClass')
   @exception
   def handleZoneClass( self ):
      svcDeviceZone = self.notifier_
      if Lib.isMssL3EnabledAndL3Zone( svcDeviceZone ):
         return  # no processing to do here for MssL3
      newZoneInterceptClass = svcDeviceZone.zoneClass
      oldZoneInterceptClass = self.shadowZoneInterceptClass
      if self.deviceStateActive():
         #b3( v( self.policyName() ), v( svcDeviceZone.zoneName ),
         #    'ZoneIceptClass', 'change from', v( oldZoneInterceptClass ),
         #    'to', v( newZoneInterceptClass ) )
         if ( self.isZoneInterceptClassSwap( oldZoneInterceptClass,
                                             newZoneInterceptClass ) and
              validZoneInterceptClasses( self.devicePolicy() ) ):
            self.updateSvcPoliciesOnZoneInterceptClassSwap()
      self.shadowZoneInterceptClass = newZoneInterceptClass

   def isZoneInterceptClassSwap( self, oldZoneInterceptClass,
                                 newZoneInterceptClass ):
      return ( ( oldZoneInterceptClass == NON_INTERCEPT_ZONE and
                 newZoneInterceptClass == INTERCEPT_ZONE ) or
               ( oldZoneInterceptClass == INTERCEPT_ZONE and
                 newZoneInterceptClass == NON_INTERCEPT_ZONE ) )

   def updateSvcPoliciesOnZoneInterceptClassSwap( self ):
      ''' First remove ServicePolicies for old intercept zone then generate
          new ServicePolicies.
      '''
      b2( 'updating ServicePolicies for ZoneInterceptClass swap' )
      # determine zoneId for zone that changed from intercept to nonIntercept
      if self.devicePolicy().zoneA.zoneClass == NON_INTERCEPT_ZONE:
         serviceDeviceZone = self.devicePolicy().zoneA
      else:
         serviceDeviceZone = self.devicePolicy().zoneB

      oldInterceptZoneId = serviceDeviceZone.zoneId
      numIntercepts = len( serviceDeviceZone.intercept )
      if numIntercepts > 0:
         for ipAddr in serviceDeviceZone.intercept:
            numIntercepts -= 1
            self.sysdbMgr.deleteServicePolicyIntercept(
               ipAddr, oldInterceptZoneId, self.devicePolicy(), numIntercepts )

      genServicePoliciesAsNeeded( self.devicePolicy(), self.sysdbMgr )

   def handleSharedServicePolicy( self, devicePolicy, serviceDeviceZone ):
      ''' When a zone link changes in a DevicePolicy that is sharing a
          ServicePolicy, we must generate a new ServicePolicy and remove the
          DevicePolicy name and intercepts from the previous ServicePolicy.
      '''
      b2( v( devicePolicy.policyName ), 'shares a ServicePolicy' )
      intercepts = serviceDeviceZone.intercept.keys()
      if intercepts:
         self.sysdbMgr.removeInterceptsFromChildServicePolicies( devicePolicy,
                                                                 intercepts )
      self.sysdbMgr.removePolicyNameFromChildServicePolicies( devicePolicy )
      devicePolicy.childServicePolicy.clear()
      genServicePoliciesAsNeeded( devicePolicy, self.sysdbMgr )

   def devicePolicy( self ):
      return self.notifier_.parent

   def policyName( self ):
      return self.devicePolicy().policyName

   def deviceStateActive( self ):
      ''' Verify that service device state is still active.
          (cli user may have just disabled deviceSet, or agent shutting down, etc.)
      '''
      return self.sysdbMgr.isDeviceStateActive( self.devicePolicy().serviceDeviceId )

####################################################################################
class DevicePolicySM( Tac.Notifiee ):
   ''' This state machine handles policies from service devices and handles
       population of Mss::ServicePolicyConfig and Mss::ServiceIntfConfig
       sysdb objects that in turn drive network provisioning to change the
       network logical topology to insert service devices into the path
       of traffic.
   '''
   notifierTypeName = 'MssPolicyMonitor::DevicePolicy'

   def __init__( self, entity, sysdbMgr ):
      Tac.Notifiee.__init__( self, entity )
      devicePolicy = self.notifier_
      self.sysdbMgr = sysdbMgr
      self.zoneASM = None
      self.zoneBSM = None
      b4( 'created for DevicePolicy:', v( devicePolicy.policyName ) )
      # regenerate ServicePolicies on agent restart
      genServicePoliciesAsNeeded( devicePolicy, sysdbMgr )

   @Tac.handler('zoneA')
   @exception
   def handleZoneA( self ):
      devicePolicy = self.notifier_
      zone = devicePolicy.zoneA
      if devicePolicy.zoneA:
         b3( v( devicePolicy.policyName ), 'new', v( zone.zoneId ),
             v( zone.zoneName ) )
         self.zoneASM = ServiceDeviceZoneSM( zone, self.sysdbMgr )

   @Tac.handler('zoneB')
   @exception
   def handleZoneB( self ):
      devicePolicy = self.notifier_
      zone = devicePolicy.zoneB
      if devicePolicy.zoneB:
         b3( v( devicePolicy.policyName ), 'new', v( zone.zoneId ),
             v( zone.zoneName ) )
         self.zoneBSM = ServiceDeviceZoneSM( zone, self.sysdbMgr )

####################################################################################
class DevicePolicyListSM( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::DevicePolicyList'

   def __init__( self, entity, sysdbMgr ):
      Tac.Notifiee.__init__( self, entity )
      devicePolicyList = self.notifier_
      b1( 'initializing', v( devicePolicyList.serviceDeviceId ) )
      self.sysdbMgr = sysdbMgr
      # create a state machine for each new DevicePolicy
      with ActivityLock():
         self.devicePolicyReactor = Tac.collectionChangeReactor(
            devicePolicyList.devicePolicy, DevicePolicySM,
            reactorArgs=( sysdbMgr, ) )

   @Tac.handler( 'policyConverge' )
   @Tac.handler( 'serviceDeviceConverge' )
   @exception
   def handleConverge( self ):
      device = self.notifier_
      t2( 'device: ', device.serviceDeviceId,
          ' policyConverge=', device.policyConverge,
          ' serviceDeviceConverge=', device.serviceDeviceConverge )
      if ( device.policyConverge and device.serviceDeviceConverge and
           not self.sysdbMgr.mssSvcPolicySrcState.convergenceComplete ):
         self.sysdbMgr.handleInitialReadNotify()

####################################################################################
class ModifierTagSM( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::ModifierTag'

   def __init__( self, entity, sysdbMgr ):
      self.sysdbMgr = sysdbMgr
      Tac.Notifiee.__init__( self, entity )
      self.handleSeqNo()

   @Tac.handler('seqNo')
   @exception
   def handleSeqNo( self ):
      modifier = self.notifier_
      t2( 'deviceSet: ', modifier.parent.name, ' type: ', modifier.type, 
          'tags :', modifier.tag.keys() )
      self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( modifier.parent )

####################################################################################
class VirtualInstanceSM( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::VirtualInstance'

   def __init__( self, entity, sysdbMgr ):
      Tac.Notifiee.__init__( self, entity )
      vinst = self.notifier_

      b4( 'initializing, deviceSet:', v( vinst.parent.parent.name ),
          'device:', v( vinst.parent.ipAddrOrDnsName ),
          'virtual instance:', v( vinst.name ) )
      self.sysdbMgr = sysdbMgr

   @Tac.handler('firewallVrf')
   @exception
   def firewallVrf( self, fwVrfName ):
      vinst = self.notifier_
      if fwVrfName in vinst.firewallVrf:
         b2( 'deviceSet:', v( vinst.parent.parent.name ),
             'device:', v( vinst.parent.ipAddrOrDnsName ),
             'virtual instance:', v( vinst.name ), 
             'added network vrf: ', v( fwVrfName ),
             ' -> ', v( vinst.firewallVrf[ fwVrfName ] ) )
      else:
         b2( 'deviceSet:', v( vinst.parent.parent.name ),
             'device:', v( vinst.parent.ipAddrOrDnsName ),
             'virtual instance:', v( vinst.name ), 
             'deleted network vrf: ', v( fwVrfName ) )

      self.sysdbMgr.restartMonitoringDeviceAsNeeded( vinst.parent,
                                                     vinst.parent.parent )

####################################################################################
class ServiceDeviceSM( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::ServiceDevice'

   def __init__( self, entity, sysdbMgr ):
      Tac.Notifiee.__init__( self, entity )
      device = self.notifier_
      b4( 'initializing, deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ) )
      self.sysdbMgr = sysdbMgr
      self.shadowGroup = device.group
      self.sslProfileStateReactor = None

      self.virtualInstanceTacCollectionReactor = Tac.collectionChangeReactor(
            device.virtualInstance, VirtualInstanceSM,
            reactorArgs=( sysdbMgr, ) )

   @Tac.handler('intfMap')
   @exception
   def handleIntfMap( self, serviceDeviceIntf ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new intfMap intf:', v( serviceDeviceIntf ) )

   @Tac.handler('group')
   @exception
   def handlegroup( self ):
      device = self.notifier_
      deviceSet = device.parent
      b2( 'deviceSet:', v( deviceSet.name ), 'device:', v( device.ipAddrOrDnsName ),
          'new group:', v( device.group ), 'previous group:', v( self.shadowGroup ) )
      # if active must cleanup devStatus, policies and restart deviceSet monitoring
      if deviceConfigComplete( device, deviceSet ):
         b1( 'group name change, restart if deviceSet active:', v( deviceSet.name ) )
         self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( deviceSet, cleanup=True )
      self.shadowGroup = device.group

   @Tac.handler('username')
   @exception
   def handleUsername( self ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new username' )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler('encryptedPassword')
   @exception
   def handleEncryptedPassword( self ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new encryptedPasswd' )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler('protocol')
   @exception
   def handleProtocol( self ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new protocol:', v( device.protocol ) )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler('protocolPortNum')
   @exception
   def handleProtocolPortNum( self ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new protocolPortNum:',
          v( device.protocolPortNum ) )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler('sslProfileName')
   @exception
   def handleSslProfileName( self ):
      device = self.notifier_
      self.sslProfileStateReactor = None
      if device.sslProfileName:
         b2( 'deviceSet:', v( device.parent.name ), 'device:',
             v( device.ipAddrOrDnsName ), 'new sslProfileName:',
             v( device.sslProfileName ) )
         self.sslProfileStateReactor = SslProfileStateReactor(
            self.sysdbMgr, device )
         b2( 'Initialize profile state reactor' )
      else:
         self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler( 'virtualInstance' )
   @exception
   def handleVInst( self, vInstName ):
      device = self.notifier_
      if vInstName in device.virtualInstance:
         b2( 'deviceSet:', v( device.parent.name ),
             'device:', v( device.ipAddrOrDnsName ),
             'added virtual system:', v( vInstName ) )
      else:
         b2( 'deviceSet:', v( device.parent.name ),
             'device:', v( device.ipAddrOrDnsName ),
             'deleted virtual system:', v( vInstName ) )
         self.sysdbMgr.cleanupForDeletedVInst( device, vInstName, device.parent )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

   @Tac.handler('vrf')
   @exception
   def handleVrf( self ):
      device = self.notifier_
      b2( 'deviceSet:', v( device.parent.name ), 'device:',
          v( device.ipAddrOrDnsName ), 'new vrf:', v( device.vrf ) )
      self.sysdbMgr.restartMonitoringDeviceAsNeeded( device, device.parent )

####################################################################################
class SslProfileStateReactor( MgmtSecuritySslStatusSm.SslStatusSm ):
   __supportedFeatures__ = [ Lib.SslFeature.sslFeatureTrustedCert ]

   def __init__( self, sysdbMgr, device ):
      self.device = device
      self.sslProfileName = device.sslProfileName
      self.ipAddrOrDnsName = device.ipAddrOrDnsName
      self.sysdbMgr = sysdbMgr
      self.sslStatus = sysdbMgr.sslStatus
      super( SslProfileStateReactor, self ).__init__(
         self.sslStatus, self.sslProfileName,
         'MssPolicyMonitorAgent', callBackNow=False )

   # Overide the function handleProfileState in SslStatusSm to get profile
   # state change notification.
   def handleProfileState( self ):
      self._handleProfile()

   # Overide the function handleProfileDelete in SslStatusSm to get profile
   # delete notification.
   def handleProfileDelete( self ):
      self._handleProfile()

   # React only when user set a new valid certificate; otherwise deal with the
   # profile status change in the polling cycle.
   def _handleProfile( self ):
      if ( self.profileStatus_ and self.profileStatus_.state == VALID ):
         b2( 'deviceSet:', v( self.device.parent), 'device:',
             v( self.device.ipAddrOrDnsName ), 'new sslProfileName:',
             v( self.sslProfileName ), 'new trustedCertsPath:',
             v( self.profileStatus_.trustedCertsPath ) )
         self.sysdbMgr.restartMonitoringDeviceAsNeeded( self.device,
                                                        self.device.parent )

####################################################################################
class DeviceSetReactor( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::DeviceSet'

   def __init__( self, entity, sysdbMgr, policyMonitorInstances ):
      Tac.Notifiee.__init__( self, entity )
      b0( 'initializing:', v( self.deviceSet().name ) )
      policyMonitorInstances[ self.deviceSet().name ] = {}
      self.mpmStatus = sysdbMgr.mpmStatus  # MssPolicyMonitor agent status
      self.sysdbMgr = sysdbMgr

      # create config reactor for every new ServiceDevice entity
      self.serviceDeviceCollectionReactor = Tac.collectionChangeReactor(
         self.deviceSet().serviceDevice, ServiceDeviceSM,
         reactorArgs=( sysdbMgr, ) )

      self.modifierTagCollectionReactor = Tac.collectionChangeReactor(
         self.deviceSet().modifierTag, ModifierTagSM,
         reactorArgs=( sysdbMgr, ) )
      # for agent restarts
      self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( self.deviceSet() )

   @Tac.handler('serviceDevice')
   @exception
   def handleServiceDevice( self, ipAddrOrDns ):
      ''' A ServiceDevice has been added or removed.
      '''
      if ipAddrOrDns in self.deviceSet().serviceDevice:
         b1( 'deviceSet:', v( self.deviceSet().name ), 'device added:',
             v( ipAddrOrDns ) )
         # Note: start monitoring on adding a new device is handled by
         # ServiceDeviceSM when all necessary attribs present
      else:
         b1( 'deviceSet:', v( self.deviceSet().name ), 'device deleted:',
             v( ipAddrOrDns ) )
         # Ignore deletions of 'map-only' devices in aggMgr device-sets. Since
         # can't check device entry that was deleted, must infer by existence of the
         # one allowed aggMgr device in the device-set w/ isAccessedViaAggrMgr=False
         if ( self.deviceSet().isAggregationMgr and
              any( [ not dev.isAccessedViaAggrMgr
                     for dev in self.deviceSet().serviceDevice.values() ] ) ):
            b2( v( ipAddrOrDns ), 'was a map-only device entry' )
            return

         self.sysdbMgr.stopMonitoringServiceDevice( ipAddrOrDns, self.deviceSet() )
         self.sysdbMgr.deletePoliciesForDevice(
            ipAddrOrDns, aggMgr=self.deviceSet().isAggregationMgr )
         self.sysdbMgr.deleteDeviceStatus( ipAddrOrDns )

   @Tac.handler('state')
   @exception
   def handleState( self ):
      b1( v( self.deviceSet().name ), 'devices:',
          v( ",".join( self.deviceSet().serviceDevice.keys() ) ), 'thread:',
          v( threadName() ), 'NEW STATE=', v( self.deviceSet().state ) )
      if ( self.deviceSet().state == ACTIVE or self.deviceSet().state == DRY_RUN ):
         self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( self.deviceSet() )

      elif self.deviceSet().state == SUSPEND:
         self.sysdbMgr.stopMonitoringDeviceSet( self.deviceSet() )

      elif self.deviceSet().state == SHUTDOWN:
         self.sysdbMgr.stopMonitoringDeviceSet( self.deviceSet() )
         self.sysdbMgr.deleteAllPoliciesForDeviceSet( self.deviceSet() )
         self.sysdbMgr.deleteDeviceStatusForDeviceSet( self.deviceSet().name )
      else:
         b2( 'nothing to do deviceSet state:' , v( self.deviceSet().state ) )

   @Tac.handler('exceptionHandling')
   @exception
   def handleExceptionHandling( self ):
      self.restartAsNeeded( 'exceptionHandling', self.deviceSet().exceptionHandling )

   @Tac.handler('queryInterval')
   @exception
   def handleQueryInterval( self ):
      self.restartAsNeeded( 'queryInterval', self.deviceSet().queryInterval )

   @Tac.handler('timeout')
   @exception
   def handleTimeout( self ):
      self.restartAsNeeded( 'timeout', self.deviceSet().timeout )

   @Tac.handler('retries')
   @exception
   def handleRetries( self ):
      self.restartAsNeeded( 'retries', self.deviceSet().retries )

   @Tac.handler('policyTag')
   @exception
   def handlePolicyTag( self, tag ):
      self.restartAsNeeded( 'policyTag', self.deviceSet().policyTag.keys() )

   @Tac.handler('offloadTag')
   @exception
   def handleOffloadTag( self, tag ):
      self.restartAsNeeded( 'offloadTag', self.deviceSet().offloadTag.keys() )

   @Tac.handler('verifyCertificate')
   @exception
   def handleVerifyCertificate( self ):
      self.restartAsNeeded( 'verifyCertificate', self.deviceSet().verifyCertificate )

   @Tac.handler('policySourceType')
   @exception
   def handlePolicySourceType( self ):
      self.restartAsNeeded( 'policySourceType', self.deviceSet().policySourceType )

   @Tac.handler('isAggregationMgr')
   @exception
   def handleIsAggregationMgr( self ):
      self.restartAsNeeded( 'isAggregationMgr', self.deviceSet().isAggregationMgr )

   @Tac.handler('adminDomain')
   @exception
   def handleAdminDomain( self ):
      self.restartAsNeeded( 'adminDomain', self.deviceSet().adminDomain )

   @Tac.handler('virtualDomain')
   @exception
   def handleVirtualDomain( self ):
      self.restartAsNeeded( 'virtualDomain', self.deviceSet().virtualDomain )

   def restartAsNeeded( self, msg, attrib ):
      t2( 'deviceSet:', self.deviceSet().name, 'new', msg, 'value:', attrib, 'on:',
          threadName() )  # use Trace as BothTrace sometimes fails for unknown reason
      self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( self.deviceSet() )

   def deviceSet( self ):
      return self.notifier_

####################################################################################
class RunnabilityHandler( object ):
   def __init__( self, mssStatus, mssL3Status, sysdbMgr ):
      self.mssStatus = mssStatus
      self.mssL3Status = mssL3Status
      self.sysdbMgr = sysdbMgr
      self.mpmConfig = sysdbMgr.mpmConfig
      self.mpmStatus = sysdbMgr.mpmStatus
      self.shuttingDown = False

      with ActivityLock():
         if mssStatus.running or mssL3Status.running:
            b0( 'MSS or MssL3 agent is running, setting mpmStatus.running=True' )
            self.mpmStatus.running = True

   def processRunnability( self ):
      if not self.mssStatus.running and not self.mssL3Status.running:
         b0( 'MssPolicyMonitor agent shutting down' )
         self.shuttingDown = True
         with ActivityLock():
            for deviceId, devStatus in self.mpmStatus.serviceDeviceStatus.items():
               b1( 'setting deviceStatus.state=shutdown for device:', v( deviceId ) )
               devStatus.state = SHUTDOWN
         for deviceSet in getDeviceSetsUsingLock( self.mpmConfig ):
            self.sysdbMgr.stopMonitoringDeviceSet( deviceSet )
         with ActivityLock():
            self.sysdbMgr.doShutdownCleanup()
            b0( 'shutdown cleanup complete, setting mpmStatus.running=False' )
            self.mpmStatus.running = False  # do last, Launcher will now kill agent
      elif self.shuttingDown:
         # This case can happen when ProcMgr is not able to kill Mss PM agent before
         # its runnability condition again changes (during its shutting down
         # sequence). A quick shut/no shut of Mss service can create such scenario.
         # When this happens, we reinitialize the necessary states again
         self.shuttingDown = False
         b0( 'handleMssRunning MSS agent running, setting mpmStatus.running=True' )
         with ActivityLock():
            self.mpmStatus.running = True
         for deviceSet in getDeviceSetsUsingLock( self.mpmConfig ):
            b1( 'handleMssRunning call restartMonDevSet:', v( deviceSet.name ) )
            self.sysdbMgr.restartMonitoringDeviceSetAsNeeded( deviceSet )
      else:
         b0( 'MssPolicyMonitor is already running. NoOp' )

####################################################################################
class MssStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Mss::Status'

   def __init__( self, mssStatus, runnabilityHandler ):
      Tac.Notifiee.__init__( self, mssStatus )
      self.runnabilityHandler = runnabilityHandler

   @Tac.handler('running')
   @exception
   def handleMssRunning( self ):
      mssStatus = self.notifier_
      b0( 'MSS agent status.running is now:', v( mssStatus.running ) )
      self.runnabilityHandler.processRunnability()

####################################################################################
class MssL3StatusReactor( Tac.Notifiee ):
   notifierTypeName = 'MssL3::Status'

   def __init__( self, mssL3Status, runnabilityHandler ):
      Tac.Notifiee.__init__( self, mssL3Status )
      self.runnabilityHandler = runnabilityHandler

   @Tac.handler('running')
   @exception
   def handleMssRunning( self ):
      mssL3Status = self.notifier_
      b0( 'MSS L3 agent status.running is now:', v( mssL3Status.running ) )
      self.runnabilityHandler.processRunnability()

####################################################################################
class MPMConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'MssPolicyMonitor::Config'

   def __init__( self, sysdbMgr, policyMonitorInstances ):
      Tac.Notifiee.__init__( self, sysdbMgr.mpmConfig )
      self.sysdbMgr = sysdbMgr
      self.monitorInstances = policyMonitorInstances

   @Tac.handler('deviceSet')
   @exception
   def handleDeviceSet( self, deviceSetName ):
      mpmConfig = self.notifier_
      with ActivityLock():
         if deviceSetName in mpmConfig.deviceSet:
            return
      b2( 'DeviceSet', v( deviceSetName ), 'deleted by:', v( threadName() ) )
      self.sysdbMgr.cleanupForDeletedDeviceSetName( deviceSetName )
      if deviceSetName in self.monitorInstances:
         self.monitorInstances.pop( deviceSetName, None )  # delete

####################################################################################
class ActivityLock( Tac.ActivityLockHolder ):
   ''' Used to detect incorrect lock/mutex acquisition order to prevent
       deadlocks.  If a thread requires both a deviceStateMutex and the
       Tac activity lock, it must acquire them in that order.
   '''
   def __enter__( self, *args, **kwargs ):  # pylint: disable=W0221
      lock = super( ActivityLock, self ).__enter__( *args, **kwargs )
      global activityLockHolder
      activityLockHolder = threading.currentThread().name
      return lock

   def __exit__( self, *args, **kwargs ):  # pylint: disable=W0221
      global activityLockHolder
      activityLockHolder = None
      return super( ActivityLock, self ).__exit__( *args, **kwargs )
