#!/usr/bin/python2.7
#  /usr/local/bin/python2.7 is 2.7.9 that checks SSL/TLS certs
#
# Copyright (c) 2015-2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable-msg=E0202, E1121, R1702, C0325, C1801

''' Access a Palo Alto Netwoks firewall or Panorama via the PAN-OS XML API
    See PANW XML-API-<ver> Usage Guide
    Also see:  https://<pan-firewall-ip>/api
               https://<panorama-ip>/api
'''

import re
import time
from copy import deepcopy
import xml.etree.ElementTree as xmlParser
import requests
from requests.adapters import HTTPAdapter
from Arnet import IpAddrCompiledRe
from MssPolicyMonitor import Lib
from MssPolicyMonitor.Error import ServiceDeviceError, FirewallAPIError
from MssPolicyMonitor.Lib import t0, t2, t4, t6
from MssPolicyMonitor.Logger import MssFirewallLogger
from MssPolicyMonitor.PluginLib import ( ServiceDeviceHAState, ServiceDevicePolicy,
                                         NetworkInterface,
                                         ServiceDeviceRoutingTables )
from Toggles import MssToggleLib

# TODO: suppress warning message until we support TLS/SSL certificate validation
import urllib3
urllib3.disable_warnings( urllib3.exceptions.InsecureRequestWarning )

t4( 'imported python requests version:', requests.__version__ )


# Resource identifiers
MGT_CFG = '/config/mgt-config'
LOG_CFG = '/config/shared/log-settings'
DEVICES_CFG = '/config/devices'

LAG_INTF_PREFIX = 'ae'
INTF_TYPE_FILTER = [ 'ethernet', 'ae' ]  # ignore other intf types on firewall
STATIC = 'static'
DYNAMIC = 'dynamic'
VWIRE_TAG_ALLOWED_DEFAULT = ''  # '0' PAN default is untagged frames only
DEFAULT_VLAN_RANGE = '0-4094'
INTERCEPT_SPEC_PREFIX = 'MSS_Intercept_Spec:'
RESP_STATUS_SUCCESS = 'success'
RESP_STATUS_ERROR = 'error'
CHILD_NOT_CONNECTED = 'not connected'
FILE_MISSING_MALFORMED = 'is missing/malformed'
NO_SUCH_NODE = 'No such node'
EMPTY_RESPONSE = "<result></result>"
PANOS_HA_STRING_MAP = {
   'Active-Passive': Lib.HA_ACTIVE_PASSIVE,
   'Active-Active': Lib.HA_ACTIVE_ACTIVE,
   'active': Lib.HA_ACTIVE,
   'passive': Lib.HA_PASSIVE,
   'active-primary': Lib.HA_ACTIVE_PRIMARY,
   'active-secondary': Lib.HA_ACTIVE_SECONDARY,
   'suspended': Lib.HA_DEVICE_SUSPENDED,
}

class Response( object ):
   def __init__( self, text ):
      self.text = text
      self.status_code = requests.codes.ok # pylint: disable=no-member

####################################################################################
class PanDevice( object ):
   ''' Use for direct PAN-OS XML API access to a Palo Alto Networks firewall
       or to a Panorama server.  Use class FirewallViaPanorama for access to
       a firewall via a Panorama server.
   '''
   def __init__( self, config, apiAuthKey=None ):
      t4('PanDevice init args:', Lib.hidePassword( config ) )
      self.config = config
      self.deviceSet = config[ 'deviceSet' ]
      self.ipAddr = config[ 'ipAddress' ]
      self.timeout = config[ 'timeout' ]
      self.retries = config[ 'retries' ]
      self.apiAuthKey = apiAuthKey if apiAuthKey else ''
      self.panApi = self.getApiConnection()
      self.deviceInfo = None
      # XXX This should be removed once address group API permission issue is fixed.
      # See BUG514522 and BUG515380.
      self.addrFailed = False
      self.addrGroupFailed = False
      self.serviceFailed = False
      self.firewallLogger = MssFirewallLogger()

      self.baseUrl = '%s://%s:%s/api' % (
         config[ 'protocol' ], self.ipAddr, config[ 'protocolPortNum' ] )
      self.apiAuthUrl = '%s/?type=keygen&user=%s&password=%s' % (
         self.baseUrl, config[ 'username' ], config[ 'password' ] )
      self.sslProfileName = config[ 'sslProfileName' ]
      self.trustedCertsPath = config.get( 'trustedCertsPath', '' )
      t4( 'PanDevice baseUrl:', self.baseUrl )
      if 'extAttr' in config:
         if 'useApiTestStub' in config[ 'extAttr' ]:
            t2( '$$ initializing PanDevice using ApiTestStub' )
            self.respDict = Lib.loadTestData( Lib.TEST_DATA_ENV_VAR )
            self._getUrl = self.mockGetUrl
         elif 'mockGetUrl' in config[ 'extAttr' ]:
            self._getUrl = config[ 'extAttr' ][ 'mockGetUrl' ]

      if MssToggleLib.toggleMssL3V2Enabled():
         # If no 'virtual instance' CLI config is provided, we pull data only from
         # the default vsys named 'vsys1' and the default virtual-router.
         self.vsys = config[ 'virtualInstance' ] or [ 'vsys1' ]
         self.vrouters = config[ 'vrouters' ] or [ 'default' ]
      else:
         self.vsys = [ self.getSystemTargetVsys() ]
         self.vrouters = [ 'default' ]


   def mockGetUrl( self, url, *args, **kwargs ):
      t4( 'mockGetUrl:', url, 'args:', args, 'kwargs:', kwargs )
      resp = ''
      if url.find( 'api/?' ) > -1:
         req = url.partition( 'api/?' )[ 2 ]
         resp = self.respDict.get( req,
               '<response status="success"><result></result></response>' )
         t4( 'mockGetUrl req:', req, 'resp:\n', resp )
      return Response( resp )

   def getApiConnection( self ):
      t4( 'generating new API connection' )
      panApi = requests.session()  # use HTTP 1.1 persistent (TCP) connection
      panApi.mount( 'http://', HTTPAdapter( max_retries=self.retries ) )
      panApi.mount( 'https://', HTTPAdapter( max_retries=self.retries ) )
      panApi.verify = self.config[ 'verifyCertificate' ]
      return panApi

   def closeConnection( self ):
      t4( 'close API connection to PAN device:', self.ipAddr )
      self.apiAuthKey = ''
      if self.panApi:
         self.panApi.close()
         self.panApi = None

   def loginAsNeededAndConfirm( self ):
      """ Get API authorization key from Palo Alto Networks firewall or Panorama.
          Return True on success.
      """
      if self.apiAuthKey:
         return True
      t6( 'get PAN API authorization key' )
      resp = self._getUrl( self.apiAuthUrl, loginRequest=True )
      root = self._parseXmlString( resp.text )
      elem = root.findall("./result/key") if root is not None else None
      self.apiAuthKey = elem[0].text if elem is not None and len( elem ) > 0 else ''
      if self.apiAuthKey:
         t4( 'get PAN API authorization key was successful for:', self.ipAddr )
         return True
      else:
         raise ServiceDeviceError( 'login failed' )

   def _isErrorResponse( self, respElem ):
      if respElem is None:
         t4( '_isErrorResponse: response is None' )
         return True
      status = respElem.attrib.get( 'status' )
      if not status:
         t4( '_isErrorResponse status is None' )
         return True
      t4( '_isErrorResponse response status=%s' % status )
      msgLine = respElem.find( 'msg/line' )
      if msgLine is not None:
         t4( '_isErrorResponse msgLine=%s' % msgLine.text )
      return status != RESP_STATUS_SUCCESS

   def _isNoSuchNodeError( self, respElem ):
      if respElem is None:
         return False
      status = respElem.attrib.get( 'status' )
      msgLine = respElem.find( 'msg/line' )
      return ( status == RESP_STATUS_ERROR and
               msgLine is not None and
               msgLine.text == NO_SUCH_NODE )

   def _getUrl( self, url, childId='', loginRequest=False ):
      if loginRequest:
         t6( 'PAN API LOGIN' )
      else:
         t6( 'PAN API REQ URL:', url )

      if not self.panApi:
         self.panApi = self.getApiConnection()
      if not loginRequest and not self.loginAsNeededAndConfirm():
         return None  # login failed
      if self.apiAuthKey:
         url = '%s&key=%s' % ( url, self.apiAuthKey )
      targetDevice = childId + ':via:' + self.ipAddr if childId else self.ipAddr
      if self.sslProfileName and not self.trustedCertsPath:
         raise ServiceDeviceError( Lib.SSL_ERROR_MSG )

      connectionFailed = False
      for attempt in range( 1, self.retries + 2 ):
         resp = None
         try:
            resp = self.panApi.get( url,
               verify=( self.trustedCertsPath if self.sslProfileName else False ),
               timeout=self.timeout )
            connectionFailed = False

            if resp.status_code == requests.codes.ok:  # pylint: disable-msg=E1101
               break
         except requests.exceptions.SSLError:
            self.closeConnection()
            raise ServiceDeviceError( Lib.SSL_ERROR_MSG )
         except Exception as ex:  # pylint: disable-msg=W0703
            if loginRequest:
               t4( '%s PAN XML API login access attempt %s, %s' % (
                    targetDevice, attempt, type( ex ) ) )
            else:
               t4( '%s PAN XML API access attempt %s, %s' % (
                    targetDevice, attempt, ex ) )
            connectionFailed = True

      if connectionFailed:
         # Connection failed after max retries
         raise ServiceDeviceError( 'Connection error' )
      if resp.status_code != requests.codes.ok: # pylint: disable-msg=no-member
         self.closeConnection()
      t6('PAN API RESP:', Lib.httpStatus( resp.status_code ), 'text:\n', resp.text )
      return resp

   # FIXME _getApiRespXmlString and _getApiRespXmlElement should be merged
   # all API function should manage XML element instead of XML strings.
   def _getApiRespXmlString( self, xpath='', reqType='config', action='', cmd='',
                             category='' ):
      ''' Access PAN-OS REST XML-API.
          Actions: show, get, set, edit, delete, rename, clone, complete, override
      '''
      #t6('PAN-API call to:', self.ipAddr, 'type=', reqType, 'action=', action,
      #   'cmd=', cmd, 'xpath=', xpath )
      url = '%s/?type=%s%s%s%s&xpath=%s' % (
         self.baseUrl, reqType, '&action=%s' % action if action else '',
         '&cmd=%s' % cmd if cmd else '',
         '&category=%s' % category if category else '', xpath )
      resp = None
      respElem = None
      for _ in range( 1, self.retries + 2 ):  
         resp = self._getUrl( url ) 
         respElem = self._parseXmlString( resp.text, raiseException=False )
         if self._isNoSuchNodeError( respElem ):
            # Some APIs may reply with a "No such node" error, especially after
            # sanitizing the firewall configuration, when expexted reply should
            # be an empty response. This is a workaround of this firewall bug.
            t4( 'No such node error: handle as an empty response' )
            return EMPTY_RESPONSE
         if self._isErrorResponse( respElem ):
            continue
         return resp.text
      raise FirewallAPIError( resp.status_code,
                              respElem.attrib.get( 'code', 0 )
                              if respElem is not None else '' )

   def _getApiRespXmlElement( self, *args, **kwargs ):
      xmlString = self._getApiRespXmlString( *args, **kwargs )
      return self._parseXmlString( xmlString )

   # FIXME XML parsing should be done only at one place.
   def _parseXmlString( self, xmlString, raiseException=True ):
      try:
         return xmlParser.fromstring( xmlString )
      except Exception:  # pylint: disable-msg=W0703
         t4( 'XML parse error' )
         if raiseException:
            raise FirewallAPIError( requests.codes.ok, # pylint: disable=no-member
                                    '' )
         return None

   def firewallIp( self ):
      ''' override in subclasses when accessing a firewall via Panorama
      '''
      return self.ipAddr

   def getGroupMembers( self, name ):
      ''' Panorama only
         Returns serialNumbers of firewalls in the device group
      '''
      xpath = '/config/devices/entry/device-group/entry[@name="%s"]/devices' % name
      membersElem = self._getApiRespXmlElement( xpath, action='show' )
      if membersElem is None:
         return []
      serialNums = []
      for device in membersElem.findall( 'result/devices/entry' ):
         serialNums.append( device.attrib[ 'name' ] )
      return serialNums

   def showSystemResources( self ):
      cmd = '<show><system><resources></resources></system></show>'
      sysResourcesElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if sysResourcesElement is None:
         return {}
      sysRes = getSubElementText( sysResourcesElement, 'result' )
      sysResSummaryMatch = re.match( r'(.*)\s{4,}PID', sysRes, re.DOTALL )
      if sysResSummaryMatch:
         sysResSummary = sysResSummaryMatch.group( 1 )
      else:
         sysResSummary = 'No System Summary'
      info = sysResSummary + '\n'

      cmd = '<show><system><disk-space></disk-space></system></show>'
      disksElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if disksElement is not None:
         info += '\n' + getSubElementText( disksElement, 'result' )
      return { 'resourceInfo' : info }

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      routingTables = ServiceDeviceRoutingTables()
      cmd = '<show><routing><route><type>static</type></route></routing></show>'
      routingElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if routingElement is None:
         t0( 'Error: Cannot fetch the routing table:' )
         return routingTables
      for entry in routingElement.findall( 'result/entry' ):
         # Skip virtual routers that are not configured
         vrfName = getSubElementText( entry, 'virtual-router' )
         if vrfName not in self.vrouters:
            continue

         destination = getSubElementText( entry, 'destination' )
         interface = getSubElementText( entry, 'interface' )
         flags = getSubElementText( entry, 'flags' )

         # Add only routes which nexthop is an IPv4 address
         nexthop = getSubElementText( entry, 'nexthop' )
         if "A" not in flags:
            t0( 'Skipping inactive route:' +
                str( ( vrfName, destination, interface, nexthop ) ) )
         elif IpAddrCompiledRe.match( nexthop ):
            routingTables.addRoute( vrfName, destination, interface, nexthop )

      routingTables.featureSupported = True
      return routingTables

   def getHighAvailabilityState( self ):
      haState = ServiceDeviceHAState()
      cmd = '<show><high-availability><all></all></high-availability></show>'
      haElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if haElement is None:
         return haState
      enabled = getSubElementText( haElement, 'result/enabled' )
      mode = getSubElementText( haElement, 'result/group/mode' )
      state = getSubElementText( haElement, 'result/group/local-info/state' )
      peerState = getSubElementText( haElement, 'result/group/peer-info/state' )
      mgmtIp = getSubElementText( haElement, 'result/group/local-info/mgmt-ip' )
      peerMgmtIp = getSubElementText( haElement, 'result/group/peer-info/mgmt-ip' )
      t4('HA raw: enabled', enabled, 'mode', mode, 'state', state, 'mgmtIp', mgmtIp )
      haState.enabled = ( enabled == 'yes' )
      haState.mode = PANOS_HA_STRING_MAP.get( mode, '' )
      haState.state = PANOS_HA_STRING_MAP.get( state, '' )
      haState.peerDeviceState = PANOS_HA_STRING_MAP.get( peerState, '' )
      haState.mgmtIp = mgmtIp
      haState.peerMgmtIp = peerMgmtIp
      return haState

   def showSecurityPolicies( self ):
      out = 'Firewall security policies:\n'
      cmd = '<show><running><security-policy></security-policy></running></show>'
      policiesElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if policiesElement is None:
         return out
      policies = policiesElement.find( 'result/member' )
      out += policies.text if policies is not None else ''
      return out

   def getInterfaceNeighbors( self, intf='all' ):
      ''' Get info on directly attached neighbors via LLDP or other means
      '''
      #startTime = time.time()
      cmd = '<show><lldp><neighbors>%s</neighbors></lldp></show>' % intf
      lldpElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if lldpElement is None:
         return {}
      neighbors = {}
      for entry in lldpElement.findall( 'result/entry' ):
         fwIntf = entry.attrib[ 'name' ]
         if entry.find( 'neighbors/entry' ) is None:
            t4('interface', fwIntf, 'has no LLDP neighbor')
            continue
         mgmtAddr = entry.find( 'neighbors/entry/management-address/entry' )
         nborMgmtIp = mgmtAddr.attrib[ 'name' ] if mgmtAddr is not None else ''
         portId = entry.find( 'neighbors/entry/port-id' )
         switchIntf = portId.text if portId is not None else ''
         sysName = entry.find( 'neighbors/entry/system-name' )
         nborSysName = sysName.text if sysName is not None else ''
         chassis = entry.find( 'neighbors/entry/chassis-id' )
         switchChassisId = chassis.text if chassis is not None else ''
         sysDesc = entry.find( 'neighbors/entry/system-description' )
         nborDesc = sysDesc.text if sysDesc is not None else ''
         nborDesc = compressNeighborDescription( nborDesc )
         #t6('fwIp', self.firewallIp(), 'intf', fwIntf, 'switchIntf', switchIntf,
         #  'mgmtIp', nborMgmtIp, 'sysName', nborSysName, 'nborId', switchChassisId )
         neighbors[ fwIntf ] = { 'nborMgmtIp': nborMgmtIp,
                                 'switchIntf': switchIntf,
                                 'nborSysName': nborSysName,
                                 'switchChassisId': switchChassisId,
                                 'nborDesc': nborDesc }
      #t8('getIntfNeighbors total time:', ( time.time() - startTime ) * 1000, 'ms')
      #t6('dump lldpNeighbors:\n', Lib.dumpDict( neighbors ) )
      return neighbors

   def showSystemInfo( self ):
      cmd = '<show><system><info></info></system></show>'
      return self._getApiRespXmlElement( reqType='op', cmd=cmd )

   def getSystemTargetVsys( self ):
      cmd = '<show><system><setting><target-vsys>' \
            '</target-vsys></setting></system></show>'
      targetVsysElement = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if targetVsysElement is None:
         return 'vsys1'
      vsys = targetVsysElement.find( 'result' )
      if vsys.text not in [ None, 'None', 'none' ]:
         targetVsys = vsys.text
      else:
         targetVsys = 'vsys1'
      t4( 'Target vsys:', targetVsys )
      return targetVsys

   def getDeviceInfo( self, cachedOk=True ):
      if cachedOk and self.deviceInfo:
         t4('get device info, cached')
         return self.deviceInfo
      t4('get device info, live')
      devInfo = {}
      sysInfoElement = self.showSystemInfo()
      if sysInfoElement is None:
         return devInfo
      name = sysInfoElement.find( 'result/system/devicename' )
      model = sysInfoElement.find( 'result/system/model' )
      mac = sysInfoElement.find( 'result/system/mac-address' )
      ipAddr = sysInfoElement.find( 'result/system/ip-address' )
      devInfo[ 'name' ] = name.text if name is not None else ''
      devInfo[ 'model' ] = model.text if model is not None else ''
      devInfo[ 'mac' ] = mac.text if mac is not None else ''
      devInfo[ 'ipAddr' ] = ipAddr.text if ipAddr is not None else ''
      self.deviceInfo = devInfo
      return devInfo

   def getInterfacesInfo( self, intfs='all' ):
      ''' Get all necessary interface information for service devices.
          Supports 4 types of interface: ethernet, ethernet-subinterface,
          aggregateEthernet and aggregateEthernet-subinterface.
          Returns a list of NetworkInterface objects
      '''
      cmd = '<show><interface>%s</interface></show>' % intfs
      allIntfsElem = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      # xmlParser.dump( allIntfsElem )  # useful for debugging
      interfaces = { vsys : {} for vsys in self.vsys }
      t6( 'Virtual Systems: ', self.vsys )
      intfVsysMap = {} # reverse map
      # get all interfaces, zones and for sub-intfs populate vlans
      if allIntfsElem is None:
         intfElems = []
      else:
         intfElems = allIntfsElem.findall( 'result/ifnet/entry' )
      for intfElem in intfElems:
         intfName = getSubElementText( intfElem, 'name' )
         if not Lib.filterOnStartsWith( intfName, INTF_TYPE_FILTER ):
            continue

         # In this API call, vsys is presented as an ID to which we must append the
         # 'vsys' prefix to get the matching entry name from vsys config API call.
         vsys = 'vsys' + getSubElementText( intfElem, 'vsys' )
         if vsys not in self.vsys:
            continue

         intf = NetworkInterface( intfName )
         intf.zone = getSubElementText( intfElem, 'zone' )
         fwd = getSubElementText( intfElem, 'fwd' )
         if fwd != 'N/A':
            intf.attribs[ 'mode' ] = fwd
            if fwd.startswith( 'vr:' ):
               intf.vrf = fwd.replace( 'vr:', '' )
            t6('service device intf mode:', fwd )
         ip = intfElem.find( 'ip' )
         if ip is not None and ip.text != 'N/A':
            intf.ipAddr = ip.text
         if intfName.startswith( LAG_INTF_PREFIX ):
            intf.isLag = True
         if '.' in intfName:  # for subinterfaces get vlan tag(s) here
            intf.isSubIntf = True
            tag = intfElem.find( 'tag' )
            intf.vlans = tag.text.split( ',' ) if tag is not None else []
         interfaces[ vsys ][ intfName ] = intf
         intfVsysMap[ intfName ] = vsys

      macMap = {}
      lagMemberIntfs = {}
      # extract intf state and physical Ethernet members of aggEth logical intfs
      if allIntfsElem is None:
         intfElems = []
      else:
         intfElems = allIntfsElem.findall( 'result/hw/entry' )
      # Set state for each interface and collect lag members interfaces
      for hwIntfElem in intfElems:
         hwIntfName = getSubElementText( hwIntfElem, 'name' )
         mac = hwIntfElem.find( 'mac' )
         macMap[ hwIntfName ] = mac.text if mac is not None else ''
         panState = getSubElementText( hwIntfElem, 'state' )
         state = self.translateIntfState( panState, hwIntfName )
         if Lib.filterOnStartsWith( hwIntfName, INTF_TYPE_FILTER ):
            if hwIntfName in intfVsysMap:
               vsys = intfVsysMap[ hwIntfName ]
               interfaces[ vsys ][ hwIntfName ].state = state
               if not interfaces[ vsys ][ hwIntfName ].isLag:
                  interfaces[ vsys ][ hwIntfName ].addPhysicalIntf( hwIntfName,
                                                                    state )
            else:  # hwIntf is a lag member
               lagMemberIntfs[ hwIntfName ] = NetworkInterface( hwIntfName,
                                                                state=state )

      # Associate lag members interfaces to their parent and add them to `interfaces`
      for hwIntfElem in intfElems:
         hwIntfName = getSubElementText( hwIntfElem, 'name' )
         if not Lib.filterOnStartsWith( hwIntfName, INTF_TYPE_FILTER ):
            continue
         if hwIntfName in intfVsysMap:
            vsys = intfVsysMap[ hwIntfName ]
            if interfaces[ vsys ][ hwIntfName ].isLag:
               for lagMemberIntfElem in hwIntfElem.findall( 'ae_member/member' ):
                  interfaces[ vsys ][ hwIntfName ].addPhysicalIntf(
                        lagMemberIntfElem.text )
                  intf = lagMemberIntfs[ lagMemberIntfElem.text ]
                  interfaces[ vsys ][ lagMemberIntfElem.text ] = intf

      # set state in aggEth physical intfs
      for intfList in interfaces.values():
         for intf in intfList.values():
            if intf.isLag and not intf.isSubIntf:
               for lagMemberIntf in intf.physicalIntfs:
                  if lagMemberIntf.name in intfList:
                     lagMemberIntf.state = intfList[ lagMemberIntf.name ].state

      # set state and physical intfs for subinterfaces
      for intfList in interfaces.values():
         for intf in intfList.values():
            if intf.isSubIntf:
               parentIntf = intfList.get( intf.name.split( '.' )[ 0 ] )
               intf.state = parentIntf.state
               intf.physicalIntfs = parentIntf.physicalIntfs

      # set mac addresses for all physical intfs
      for intfList in interfaces.values():
         for intf in intfList.values():
            for eth in intf.physicalIntfs:
               eth.macAddr = macMap.get( eth.name, '' )

      self.populateVlansAndVwire( interfaces )
      #t8('getInterfacesInfo total time:', ( time.time() - startTime ) * 1000, 'ms')
      return interfaces

   def populateVlansAndVwire( self, interfaces ):
      ''' Populates vlans and vwire name for dict of NetworkInterface
          objects.
      '''
      vlanMap = self.getVlansForAllInterfaces()
      for intfList in interfaces.values():
         for intf in intfList.values():
            if intf.name in vlanMap:
               if 'vwire' in vlanMap[ intf.name ]:
                  intf.attribs[ 'vwire' ] = vlanMap[ intf.name ][ 'vwire' ]
               if not intf.isSubIntf and 'vlans' in vlanMap[ intf.name ]:
                  intf.vlans = vlanMap[ intf.name ][ 'vlans' ]

   def getVlansForAllInterfaces( self ):
      ''' Returns a dict key is firewall interface name
          value is a dict with interface type, vlans, vwireName
      '''
      intfVlanMap = {}
      netCfgElem = self._getApiRespXmlElement( xpathCfgNetwork(), action="show" )
      # xmlParser.dump( netCfgElem )  # debug
      # get vlans for layer2 interfaces
      if netCfgElem is None:
         intfsElems = []
      else:
         intfsElems = netCfgElem.findall( 'result/network/interface/ethernet/entry' )
      for intf in intfsElems:
         for l2Intf in intf.findall( 'layer2/units/entry' ):
            if l2Intf is not None:
               l2IntfName = l2Intf.attrib[ 'name' ]
               tag = l2Intf.find( 'tag' )
               vlans = [ tag.text ] if tag is not None else []
               intfVlanMap[ l2IntfName ] = { 'vlans': vlans, 'type': Lib.LAYER2 }

      # get vlans for remaining vwire interfaces
      if netCfgElem is None:
         vwireElems = []
      else:
         vwireElems = netCfgElem.findall( 'result/network/virtual-wire/entry' )
      for vwire in vwireElems:
         vwireName = vwire.attrib[ 'name' ]
         intf1 = getSubElementText( vwire, 'interface1' )
         intf2 = getSubElementText( vwire, 'interface2' )
         tags = vwire.find( 'tag-allowed' )
         vlans = tags.text.split( ',' ) if tags is not None \
                                        else [ DEFAULT_VLAN_RANGE ]
         for intf in intf1, intf2:
            if intf not in intfVlanMap:
               intfVlanMap[ intf ] = { 'type': Lib.VIRTUAL_WIRE }
               if '.' not in intf:  # skip sub intfs, already have correct vlans
                  intfVlanMap[ intf ][ 'vlans' ] = vlans
            intfVlanMap[ intf ][ 'vwire' ] = vwireName

      # t0( 'intfVlanMap:\n', Lib.dumpDict( intfVlanMap ) )
      return intfVlanMap

   def getInterfaceState( self, intf ):
      ''' Returns interface connection state
      '''
      cmd = '<show><interface>%s</interface></show>' % intf
      elem = self._getApiRespXmlElement( reqType='op', cmd=cmd )
      if elem is None:
         return Lib.LINK_STATE_UNKNOWN
      state = elem.find( 'result/hw/state' )
      return self.translateIntfState( state, intf )

   def translateIntfState( self, panState, intf ):
      ''' Translate PAN intf state to MSS intf state
      '''
      if panState is None or panState.upper() == 'UKN':
         return Lib.LINK_STATE_UNKNOWN
      elif panState.upper() == 'UP':
         return Lib.LINK_STATE_UP
      elif panState.upper() == 'DOWN':
         return Lib.LINK_STATE_DOWN
      else:
         t0( 'intf:', intf, 'unrecognized link state value:', panState )
         return Lib.LINK_STATE_UNKNOWN

   def getInterfaceComment( self, intf ):
      elem = self._getApiRespXmlElement( 
         xpathCfgNetwork() + '/interface/ethernet/entry[@name="%s"]' % intf )
      if not elem:
         return ''
      commentElem = elem.find( './result/entry/comment' )
      comment = commentElem.text if commentElem is not None else None
      return comment

   def getVirtualWiresXml( self ):
      return self._getApiRespXmlString( xpathCfgNetwork() + '/virtual-wire' )

   def getVirtualWire( self, name='' ):
      elem = self._getApiRespXmlElement(
         xpathCfgNetwork() + '/virtual-wire/entry[@name="%s"]' % name )
      if elem is None:
         return None
      vwire = elem.find( './result/entry' )
      if vwire is not None:
         panVwire = PanVirtualWire( name )
         intf1 = vwire.find( 'interface1' )
         panVwire.intf1 = intf1.text if intf1 is not None else ''
         intf2 = vwire.find( 'interface2' )
         panVwire.intf2 = intf2.text if intf2 is not None else ''
         tags = vwire.find( 'tag-allowed' )
         panVwire.allowedVlans = ( tags.text if tags is not None 
                                   else VWIRE_TAG_ALLOWED_DEFAULT )
         return panVwire
      return None

   def getZones( self, vsys ):
      zones = {}
      elem = self._getApiRespXmlElement( xpathCfgVsys( vsys=vsys ) + '/zone',
                                         action='show' )
      if elem is None:
         return zones
      for zone in elem.findall( 'result/zone/entry' ):
         name = zone.attrib[ 'name' ]
         zoneType = '?'
         interfaces = []
         if zone.find( 'network/virtual-wire' ) is not None:
            zoneType = Lib.VIRTUAL_WIRE
            for intf in zone.findall( 'network/virtual-wire/member' ):
               interfaces.append( intf.text )
         elif zone.find( 'network/layer2' ) is not None:
            zoneType = Lib.LAYER2
            for intf in zone.findall( 'network/layer2/member' ):
               interfaces.append( intf.text )
         elif zone.find( 'network/layer3' ) is not None:
            zoneType = Lib.LAYER3
            for intf in zone.findall( 'network/layer3/member' ):
               interfaces.append( intf.text )
         zones[ name ] = PanZone( name, zoneType, interfaces )
      return zones

   def getDynamicAddrGroup( self, name='all' ):
      if name == 'all':
         nameTag = '<all></all>'  # or just '</all>' 
      else:
         nameTag = '<name>%s</name>' % name
      showDagsCmd = ( '<show><object><dynamic-address-group>%s'
                      '</dynamic-address-group></object></show>' % nameTag )
      return self._getApiRespXmlString( '', reqType='op', cmd=showDagsCmd )

   def getAddressObjectsXml( self, vsys ):
      ''' Return all address objects.
      '''
      try:
         addresses = self._getApiRespXmlString( xpathCfgVsys( vsys=vsys )
                                                + '/address', action='show' )
      except FirewallAPIError as fwe:
         t4( "address API failed" )
         # Prevent the error to be logged multiple times across polling cycles
         self.firewallLogger.log( self.ipAddr, fwe )
         # Provide an empty response so parsing doesn't fail
         addresses = EMPTY_RESPONSE

      panoramaAddresses = self._getApiRespXmlString(
         xpathCfgPanorama() + '/address', action='get' )
      return [ addresses, panoramaAddresses ]

   def getAddrGroupsXml( self, vsys ):
      ''' return static and dynamic address groups
      '''
      try:
         addrGroups = self._getApiRespXmlString( xpathCfgVsys( vsys=vsys )
                                                 + '/address-group', action='show' )
      except FirewallAPIError as fwe:
         t4( "address-group API failed" )
         # Prevent the error to be logged multiple times across polling cycles
         self.firewallLogger.log( self.ipAddr, fwe )
         # Provide an empty response so parsing doesn't fail
         addrGroups = EMPTY_RESPONSE

      addrGroupsPanorama = self._getApiRespXmlString(
         xpathCfgPanorama() + '/address-group', action='get' )
      return [ addrGroups, addrGroupsPanorama ]

   # Policies
   def getSecurityPoliciesXml( self ):
      ''' Get all security policies on firewall.
          Returns a list where each item is a dictionary of XML doc for a "chunk" of
          policy rules indexed per vsys.
      '''
      preRules = { vsys : self._getApiRespXmlString(
         xpathCfgPanorama( vsys ) + '/pre-rulebase/security', action='get' )
         for vsys in self.vsys }

      deviceRules = { vsys : self._getApiRespXmlString( xpathCfgVsys( vsys=vsys ) +
                                                        '/rulebase/security',
                                                        action='show' )
                      for vsys in self.vsys }

      postRules = { vsys : self._getApiRespXmlString(
         xpathCfgPanorama( vsys ) + '/post-rulebase/security', action='get' )
         for vsys in self.vsys }

      t6( 'policy pre-rulebase:', preRules, '\npolicy rulebase:', deviceRules,
          '\npolicy post-rulebase:', postRules )

      return [ preRules, deviceRules, postRules ]

   def getPolicies( self, mssTags=None, ):
      num = 1
      policies = { vsys : [] for vsys in self.vsys }
      if not mssTags:
         return policies
      for deviceRuleChunk in self.getSecurityPoliciesXml():
         for vsys, policyRuleChunk in deviceRuleChunk.iteritems():
            elem = self._parseXmlString( policyRuleChunk )
            if elem is not None:
               for policyXml in elem.findall( 'result/security/rules/entry' ):
                  disabled = policyXml.find( 'disabled' )
                  if disabled is not None and disabled.text.lower() == 'yes':
                     continue
                  policyTags = [ tag.text
                                 for tag in policyXml.findall( 'tag/member' ) ]
                  if not ( set( policyTags ) & set( mssTags ) ): # check intersection
                     continue  # skip policies that don't satisfy tagFitler
                  policies[ vsys ].append( PanPolicy( policyXml, self.firewallIp(),
                                                      vsys, num ) )
                  num += 1
      self.resolveL4Ports( policies )
      self.resolvePolicyAddrNames( policies )
      self.resolveZoneIntfs( policies )
      return policies

   def resolveZoneIntfs( self, vsysPolicyMap ):
      ''' Populate all interfaces and vlans for each zone in policies
          For efficiency do all policies at once (instead of within
          PanPolicy class by querying specific zones with API).
      '''
      interfaces = self.getInterfacesInfo()
      for vsys, policies in vsysPolicyMap.iteritems():
         zones = self.getZones( vsys )
         vsysIntfs = interfaces[ vsys ]
         for policy in policies:
            #t4('get zone interface info for policy:', policy.name, 'srcZone:',
            #   policy.srcZoneName, 'dstZone:', policy.dstZoneName )
            if policy.srcZoneName != 'any' and policy.srcZoneName in zones:
               srcZoneObj = zones[ policy.srcZoneName ]
               policy.srcZoneType = srcZoneObj.zoneType
               for intfName in srcZoneObj.interfaces:
                  if intfName in vsysIntfs:
                     policy.srcZoneInterfaces.append( vsysIntfs[ intfName ] )
            if policy.dstZoneName != 'any' and policy.dstZoneName in zones:
               dstZoneObj = zones[ policy.dstZoneName ]
               policy.dstZoneType = dstZoneObj.zoneType
               for intfName in dstZoneObj.interfaces:
                  if intfName in vsysIntfs:
                     policy.dstZoneInterfaces.append( vsysIntfs[ intfName ] )

   def resolvePolicyAddrNames( self, vsysPolicyMap ):
      ''' For efficiency do all policies at once here (instead of within
          PanPolicy class by querying specific address and group names)
          Each value in a policy address field may be a string that
          represents an IP address, an IP address range, an IP subnet,
          an IP address object name or an IP address group name. IP
          address groups may be static or dynamic.
      '''
      for vsys, policies in vsysPolicyMap.iteritems():
         addrNameBindings = self.getAddressNameBindings( vsys )
         for policy in policies:
            policy.srcIpAddrList = self.resolveAddressEntries( policy.srcAddrEntries,
                                                               addrNameBindings )
            policy.dstIpAddrList = self.resolveAddressEntries( policy.dstAddrEntries,
                                                               addrNameBindings )
      return vsysPolicyMap

   def resolveAddressEntries( self, addrEntries, addrNameBindings ):
      addresses = []
      for addrOrName in addrEntries:
         if addrOrName == 'any':  # ignore
            continue
         if addrOrName in addrNameBindings:
            addresses.extend( addrNameBindings[ addrOrName ] )  # entry was a name
         else:
            addresses.append( addrOrName )
      return addresses

   def getAddressNameBindings( self, vsys ):
      ''' Returns a dict where:
            key is an IP address object name or an IP address group name
            value is a list of individual: IP address(es), IP address
            range(s) and/or IP subnet(s)
      '''
      bindings = self.getAddressObjects( vsys )  # add individual address names
      for addrGroupsXml in self.getAddrGroupsXml( vsys ):
         groupsElem = self._parseXmlString( addrGroupsXml )
         # add static group names and members
         if groupsElem is not None:
            for addrGroup in groupsElem.findall( 'result/address-group/entry' ):
               groupName = addrGroup.attrib[ 'name' ]
               if addrGroup.find( 'static' ) is not None:
                  members = []
                  for member in addrGroup.findall( 'static/member' ):
                     if member.text in bindings:
                        members.extend( bindings[ member.text ] )
                  bindings[ groupName ] = members

      dynamicGroups = self._parseXmlString( self.getDynamicAddrGroup( name='all' ) )
      if dynamicGroups is not None:
         for groupEntry in dynamicGroups.findall( 'result/dyn-addr-grp/entry' ):
            groupName = getSubElementText( groupEntry, 'group-name' )
            members = []
            for memberEntry in groupEntry.findall( 'member-list/entry' ):
               members.append( memberEntry.attrib[ 'name' ] )
            bindings[ groupName ] = members
      #t5('dump final addrNameBindings:', Lib.dumpDict( bindings ) )
      return bindings

   def getAddressObjects( self, vsys ):
      addressObjects = {}
      for addrObj in self.getAddressObjectsXml( vsys ):
         elem = self._parseXmlString( addrObj )
         if elem is not None:
            for entry in elem.findall( 'result/address/entry' ):
               name = entry.attrib[ 'name' ]
               if entry.find( 'ip-netmask' ) is not None:  # ip or subnet
                  addressObjects[ name ] = [ entry.find( 'ip-netmask' ).text ]
               elif entry.find( 'ip-range' ) is not None:
                  addressObjects[ name ] = [ entry.find( 'ip-range' ).text ]
               else:
                  t0('Warning: unable to resolve addr obj name', name )
      #t5('dump addressObjects:', Lib.dumpDict( addressObjects ) )
      return addressObjects

   def resolveL4Ports( self, vsysPolicyMap ):
      ''' if a policy has both an application and service with the same name
          then the protocol and ports defined for the service take precedence.
          Note that a given app can have both TCP and UDP ports (e.g. DNS)
      '''
      for vsys, policies in vsysPolicyMap.iteritems():
         serviceProtocolPortMap = {}  # all pre and user defined service on firewall
         appProtocolPortMapCache = {}  # cache for app names already looked up

         for policy in policies:
            l4Services = {}
            for service in policy.services:
               if service == 'application-default':
                  continue
               # lazy init
               serviceProtocolPortMap = self.getServiceProtocolPortMap( vsys )
               if service in serviceProtocolPortMap:
                  t6( vsys, ': serviceProtocolPortMap:', serviceProtocolPortMap )
                  l4Services[ service ] = deepcopy(
                        serviceProtocolPortMap[ service ] )
            t6( 'l4Services after processing policy.services:', l4Services )

            for app in policy.applications:
               if app in l4Services or app == 'any':
                  t6( 'ignoring app:', app, 'overridden by service or any')
                  continue
               if app in appProtocolPortMapCache:
                  l4Services[ app ] = deepcopy( appProtocolPortMapCache[ app ] )
               elif app.upper() == 'ICMP':  # special case
                  l4Services[ app ] = { 'ICMP' : [] }
               else:
                  panApp = self._getApiRespXmlElement(
                     '/config/predefined/application/entry[@name="%s"]' % app,
                     action='get' )
                  if panApp is None:
                     panApp = []
                  newApp = False
                  for l4Info in panApp.findall( 'result/entry/default/port/member' ):
                     newApp = True
                     protocol, portStr = l4Info.text.split( '/' )
                     protocol = protocol.upper()
                     ports = [ p for p in portStr.split(',') ]
                     if app not in l4Services:
                        l4Services[ app ] = { protocol : ports }
                     elif protocol not in l4Services[ app ]:
                        l4Services[ app ].update( { protocol : ports } )
                     else:
                        l4Services[ app ][ protocol ].extend( ports )
                  if newApp:
                     appProtocolPortMapCache[ app ] = deepcopy( l4Services[ app ] )
            t4( 'appProtocolPortMapCache:', appProtocolPortMapCache )
            t4( 'l4Services after processing policy.applications:', l4Services )

            # aggregate TCP ports, UDP ports write policy.dstL4Services:
            for protocolPorts in l4Services.values():
               for protocol, ports in protocolPorts.items():
                  if protocol not in policy.dstL4Services:
                     policy.dstL4Services[ protocol ] = ports
                  else:
                     policy.dstL4Services[ protocol ].extend( ports )
            t4( 'policy:', policy.name, 'apps:', policy.applications, 'services:',
                policy.services, 'L4Ports:', policy.dstL4Services )

   def getServiceProtocolPortMap( self, vsys ):
      predefinedServices = '/config/predefined/service'
      userDefinedServices = xpathCfgVsys( vsys=vsys ) + '/service'
      svcMap = {}
      for url in [ predefinedServices, userDefinedServices ]:
         try:
            action = 'get' if 'predefined' in url else 'show'
            serviceElement = self._getApiRespXmlElement( url, action=action )
         except FirewallAPIError as fwe:
            self.firewallLogger.log( self.ipAddr, fwe )
            continue

         if serviceElement is None:
            return {}
         services = serviceElement.find( 'result/service' )
         if services is not None:
            for svcEntry in services:
               service = svcEntry.attrib[ 'name' ]
               protocol = svcEntry.find( 'protocol' )
               if protocol is not None and len( protocol ) > 0:
                  l4ProtoElem = protocol[ 0 ]
                  l4Proto = l4ProtoElem.tag
                  portElem = l4ProtoElem.find( 'port' )
                  if portElem is not None:
                     ports = [ p for p in portElem.text.split( ',' ) ]
                     svcMap[ service ] = { l4Proto.upper() : ports }
      t4( 'L4 services map:', svcMap )
      return svcMap

   def _dumpXml( self, xmlString ):
      # alt:  return xml.dom.minidom.parseString( xmlString ).toprettyxml()
      return xmlParser.dump( self._parseXmlString( xmlString ) )

   #################################################################################
   # Currently unused API calls, left here for potential future use
   # def getAllPolicies( self ):
   #    return self._getApiRespXmlString( xpathCfgVsys() + '/rulebase' )
   #
   # def getDosPolicies( self ):
   #    return self._getApiRespXmlString( xpathCfgVsys() + '/rulebase/dos' )
   #
   # def getDosObjects( self ):
   #    return self._getApiRespXmlString(
   #       xpathCfgVsys() + '/profiles/dos-protection' )

   # def getConfigExport( self ):
   #    return self._getApiRespXmlString( reqType='export',
   #                                      category='configuration' )

   # def getDeviceModel( self ):
   #    ''' if getting multiple fields from system info use getDeviceInfo
   #    '''
   #    sysInfoElement = self.showSystemInfo()
   #    model = sysInfoElement.find( 'result/system/model' )
   #    return model.text
   #
   # def getDeviceMacAddr( self ):
   #    ''' if getting multiple fields from system info use getDeviceInfo
   #    '''
   #    sysInfoElement = self.showSystemInfo()
   #    mac = sysInfoElement.find( 'result/system/mac-address' )
   #    return mac.text
   #
   # def getDeviceIpAddr( self ):
   #    ''' if getting multiple fields from system info use getDeviceInfo
   #    '''
   #    sysInfoElement = self.showSystemInfo()
   #    return sysInfoElement.find( 'result/system/ipAddress' ).text

   # def getDeviceName( self ):
   #    ''' if getting multiple fields from system info use getDeviceInfo
   #    '''
   #    sysInfoElement = self.showSystemInfo()
   #    return sysInfoElement.find( 'result/system/devicename' ).text

   # def getAllConfig( self ):
   #    return self._getApiRespXmlString( '/config' )
   #
   # def getLogSettings( self ):
   #    return self._getApiRespXmlString( LOG_CFG )
   #
   # def getTags( self ):
   #    return self._getApiRespXmlString( xpathCfgVsys() + '/tag' )

   # def getInterfaceNames( self ):
   #    intfsElem = self._getApiRespXmlElement( xpathCfgNetwork() + '/interface' )
   #    intfNames = []
   #    for intf in intfsElem.findall( 'result/interface/ethernet/entry' ):
   #       intfNames.append( intf.attrib[ 'name' ] )
   #    return intfNames

   # def getIntfForMacAddr( self, macAddr ):
   #    intfName = '?'
   #    cmd = '<show><interface>hardware</interface></show>'
   #    intfInfoElem = self._getApiRespXmlElement( reqType='op', cmd=cmd )
   #    for intf in intfInfoElem.findall( 'result/hw/entry' ):
   #       if intf.find( 'mac' ).text == macAddr:
   #          intfName = intf.find( 'name' ).text
   #    return intfName
   #
   # def registerVm(self, xml):
   #    ''' register or unregister an IP address with tags
   #       XML determines operation: <payload><register> or <payload><unregister>
   #    '''
   #    return self._getApiRespXmlString( xpathCfgVsys() + "/user-id-agent",
   #                                      reqType='user-id', action='set', cmd=xml )


####################################################################################
class FirewallViaPanorama( PanDevice ):
   ''' Use this class when Panorama is being used to access a specific firewall
       by serial number.
   '''
   def __init__( self, deviceConfig, serialNum, apiAuthKey=None ):
      self.serialNum = serialNum # assign serial before any API call
      super( FirewallViaPanorama, self ).__init__( deviceConfig, apiAuthKey )
      self.fwIp = self.getDeviceInfo()[ 'ipAddr' ]
      t4('FwViaPanorama', self.ipAddr, 'fwIp', self.fwIp, 'baseUrl', self.baseUrl )

   def firewallIp( self ):
      ''' override here since using panorama to access firewall
      '''
      return self.fwIp

   def _getApiRespXmlString( self, xpath='', reqType='config', action='', cmd='',
                             category='' ):
      ''' Access PAN-OS REST XML-API.
          Valid actions: show, get, set, edit, delete, rename, clone,
                         complete, override
          Overrides super class method in order to use self.serialNum in url
      '''
      t6( 'PAN-API call: FVP dev=', self.ipAddr, 'sn=', self.serialNum,
          'type=', reqType, 'action=', action, 'cmd=', cmd, 'xpath=', xpath )
      url = '%s/?type=%s%s%s%s%s&xpath=%s' % (
         self.baseUrl, reqType,
         '&target=%s' % self.serialNum,
         '&action=%s' % action if action else '',
         '&cmd=%s' % cmd if cmd else '',
         '&category=%s' % category if category else '',
         xpath )
      resp = None
      respElem = None
      for attempt in range( 1, self.retries + 2 ):  # retries for panorama children
         resp = self._getUrl( url, self.serialNum )  # access child via panorama
         respElem = self._parseXmlString( resp.text, raiseException=False )
         if self._isNoSuchNodeError( respElem ):
            # Some APIs may reply with a "No such node" error, especially after
            # sanitizing the firewall configuration, when expexted reply should
            # be an empty response. This is a workaround of this firewall bug.
            t4( 'No such node error: handle as an empty response' )
            return EMPTY_RESPONSE
         if self._isErrorResponse( respElem ):
            t4( '%s API access via %s attempt %d failed' % (
               self.serialNum, self.ipAddr, attempt ) )
            continue
         return resp.text
      raise FirewallAPIError( resp.status_code,
                              respElem.attrib.get( 'code', 0 )
                              if respElem is not None else '' )

####################################################################################
class PanPolicy( ServiceDevicePolicy ):
   ''' Represents a Policy on a Palo Alto Networks Firewall
       Policy rules always specify source and destination zones of the same type.
   '''
   def __init__( self, policyElement, firewallIp, vsys, number ):
      t6('PanPolicy XML:\n', xmlParser.tostring( policyElement ) )
      super( PanPolicy, self ).__init__( policyElement.attrib[ 'name' ],
                                         firewallIp, vsys, number )
      self.policyElement = policyElement
      self.srcZoneName = getSubElementText( policyElement, 'from/member',
                                            default='any' )
      self.dstZoneName = getSubElementText( policyElement, 'to/member',
                                            default='any' )
      self.srcAddrEntries = self.getAddrMembers( 'source/member' )
      self.dstAddrEntries = self.getAddrMembers( 'destination/member' )
      self.services = [ m.text for m in policyElement.findall( 'service/member') ]
      self.applications = [ m.text
                            for m in policyElement.findall( 'application/member') ]
      self.tags = self.loadTags()
      self.action = getSubElementText( policyElement, 'action',
                                       default='allow' ).upper()

   def getAddrMembers( self, xpath ):
      addrNames = []
      for member in self.policyElement.findall( xpath ):
         addrNames.append( member.text )
      return addrNames

   def loadTags( self ):
      tags = []
      for tag in self.policyElement.findall( 'tag/member' ):
         tags.append( tag.text )
      return tags


####################################################################################
class PanZone( object ):

   def __init__( self, name, zoneType, interfaces ):
      self.name = name
      self.zoneType = zoneType
      self.interfaces = interfaces

   def __str__( self ):
      return 'Zone: %s  type: %s  intfs: %s' % (
         self.name, self.zoneType, self.interfaces )


####################################################################################
class PanVirtualWire( object ):

   def __init__( self, name ):
      self.name = name
      self.intf1 = None
      self.intf2 = None
      self.allowedVlans = None

   def __str__( self ):
      return 'vwire: %s intf1: %s  intf2: %s  allowedVlans: %s' % (
         self.name, self.intf1, self.intf2, self.allowedVlans )


####################################################################################
# API utility functions

def getSubElementText( elementObj, xpath, default='' ):
   try:
      subElem = elementObj.find( xpath )
   except AttributeError:
      subElem = None
   if subElem is None:
      t0( 'Error: No sub element \"%s\"' % xpath )
      return default
   return subElem.text

def xpathCfg( host='' ):
   ''' for Panorama config
   '''
   return "/config/devices/entry%s" % ("[@name='%s']" % host if host else '' )


def xpathCfgPanorama( vsys='vsys1' ):
   return "/config/panorama/vsys/entry%s" % ("[@name='%s']" % vsys if vsys else '' )


def xpathCfgVsys( host='', vsys='' ):
   ''' for direct firewall access
      vsys = virtual firewall instances (virtual systems) within a firewall
   '''
   return "/config/devices/entry%s/vsys/entry%s" % (
             "[@name='%s']" % host if host else '',
             "[@name='%s']" % vsys if vsys else '' )


def xpathCfgNetwork( host='' ):
   return ( "/config/devices/entry%s/network"
            % ( "[@name='%s']" % host if host else '' ) )


def getPanDevice( device ):
   ''' device is a dict with 'ipAddress', 'username' and 'password' for 
       an individual PAN firewall or a Panorama server
   '''
   return PanDevice( device[ 'ipAddress' ], device[ 'username' ],
                     device[ 'password' ] )


def getFwViaPanorama( panorama, fwSerialNum ):
   return FirewallViaPanorama( panorama[ 'ipAddress' ], panorama[ 'username' ],
                               panorama[ 'password' ], fwSerialNum )


def getPanDevicesFrom( deviceList ):
   ''' Each device in deviceList is a dict with 'ipAddress', 'username'
       and 'password' for either an individual PAN firewall or a
       Panorama server. If device is Panorama it must include the key 
       'panorama-device-groups' with value as a list of device groups defined
       on the Panorama server.  This function expands the each Panorama device
       group into a list of serial numbers for each firewall in the device group.
       Returns a list where each item is a PanDevice or FirewallViaPanorama.
   '''
   panDevices = []
   for device in deviceList:
      if 'panorama-device-groups' not in device:
         panDev = getPanDevice( device )
         panDevices.append( panDev )  # a standalone firewall
      else:  # device is a Panorama server
         panorama = getPanDevice( device )
         for groupName in device[ 'panorama-device-groups' ]:
            for fwSerialNum in panorama.getGroupMembers( groupName ):
               panDevices.append( getFwViaPanorama( device, fwSerialNum ) )
   return panDevices


def compressNeighborDescription( desc ):
   if desc:
      desc = desc.replace( 'Networks EOS version', 'EOS' )
      desc = desc.replace( 'running on an Arista Networks', 'on' )
      return desc
   return ''

####################################################################################
# Module testing functions
def getConfig( device, user, password ):
   return {
      'ipAddress': device, 'username': user, 'password': password,
      'deviceSet': 'test_dummy', 'serviceDeviceType': 'PAN_test',
      'queryInterval': 10, 'timeout': 5, 'retries': 1, 'protocol': 'https',
      'protocolPortNum': 443, 'verifyCertificate': False, 'group': '',
      'sslProfileName': '', 'isAggregationMgr': False,
      'policyTags': [ 'mss1', 'mss2', 'mss_offload',
                      'offload', 'verbatim', 'redirect' ],
      'virtualInstance' : [ 'vsys3' ],
      'vrouters' : [ 'default' ], }

def test1( device, user, password, sleep ):
   print 'Running api tests...'
   cfg = getConfig( device, user, password )
   fw = PanDevice( cfg )
   print 'getDeviceInfo:', fw.getDeviceInfo()
   for j in range( 100 * 1000 ):
      print '\nTest iteration #%s ' % j, device
      for vsys, intfs in fw.getInterfacesInfo().iteritems():
         print 'VSYS', vsys
         print 'IntfInfo:', [ i.name for i in intfs ]
         #print ' HA passive:', fw.isInHAModeAndNotActiveOrPrimary()
         time.sleep( sleep )


def policyTest( device, user, password ):
   import Tracing
   Tracing.traceSettingIs( 'MssPolicyMonitor/0-4' )
   cfg = getConfig( device, user, password )
   fw = PanDevice( cfg )
   print 'policies:'
   for vsys, policyList in fw.getPolicies( cfg[ 'policyTags' ] ).iteritems():
      print vsys
      for p in policyList:
         print str( p )
         print 'srcIpAddrList:', p.srcIpAddrList, '\ndstIpAddrList:', p.dstIpAddrList
         print 'dstL4Services:', p.dstL4Services
         #print 'policyElement:', xmlParser.tostring( p.policyElement )
         for intfs in [ p.srcZoneInterfaces, p.dstZoneInterfaces ]:
            for intf in intfs:
               print str( intf )


def intftest( device, user, password ):
   print 'Running api tests...'
   cfg = getConfig( device, user, password )
   fw = PanDevice( cfg )
   print 'getDeviceInfo:', fw.getDeviceInfo()
   print 'IntfInfo:'
   for vsys, intfs in fw.getInterfacesInfo().iteritems():
      print 'VSYS', vsys
      for intf in intfs:
         print intf
         for physIntf in intf.physicalIntfs:
            print "  ", physIntf


def haTest( device, user, password ):
   cfg = getConfig( device, user, password )
   fw = PanDevice( cfg )
   haState = fw.getHighAvailabilityState()
   print '\n', haState
   print '\nHA passiveOrSecondary:', haState.isHaPassiveOrSecondary()


def testPanoramaApi( cfg ):
   try:
      pano =  PanDevice( cfg )
      info = pano.getDeviceInfo()
      print 'PANORAMA DeviceInfo:', info
   except Exception as ex:  # pylint: disable-msg=W0703
      print 'Error:', ex, '\n'


def testPanFwApi( cfg ):
   try:
      fw = PanDevice( cfg )
      info = fw.getDeviceInfo()
      print 'PAN FW DeviceInfo:', info
   except Exception as ex:  # pylint: disable-msg=W0703
      print 'Error:', ex, '\n'


def testFwViaPanoramaApi( cfg, fwSerialNum ):
   try:
      fwViaPano = FirewallViaPanorama( cfg, fwSerialNum )
      info = fwViaPano.getDeviceInfo()
      print 'PAN FW VIA PANORAMA DeviceInfo:', info
   except Exception as ex:  # pylint: disable-msg=W0703
      print 'Error:', ex, '\n'


def testTimeoutAndRetries():
   print '\n\nTEST BAD IP ADDR'
   testPanFwApi( getConfig( 'BAD_IP', 'mss_readonly', 'arista' ) )
   testPanoramaApi( getConfig( 'BAD_IP', 'admin', 'arista' ) )
   testFwViaPanoramaApi( getConfig( 'BAD_IP', 'admin', 'arista' ), '001801014981' )
   print '\nTEST BAD FW SERIAL NUMBER'
   testFwViaPanoramaApi( getConfig( 'bizdev-panorama', 'admin', 'arista' ), '000' )

   print '\n\nTEST BAD PASSWORD'
   testPanFwApi( getConfig( 'bizdev-pan3060', 'mss_readonly', 'BAD_PASSWORD' ) )
   testPanoramaApi( getConfig( 'bizdev-panorama', 'admin', 'BAD_PASSWORD' ) )
   testFwViaPanoramaApi( getConfig( 'bizdev-panorama', 'admin', 'BAD_PASSWD' ),
                         '001801014981' )

   print '\n\nTEST TIMEOUT'
   cfg = getConfig( 'bizdev-pan3060', 'mss_readonly', 'arista' )
   cfg[ 'timeout' ] = 0.001
   testPanFwApi( cfg )
   cfg = getConfig( 'bizdev-panorama', 'admin', 'arista' )
   cfg[ 'timeout' ] = 0.001
   testPanoramaApi( cfg )

   print '\nTest Palo Alto API Timeouts and Retries, ALL GOOD'
   testPanFwApi( getConfig( 'bizdev-pan3060', 'mss_readonly', 'arista' ) )
   testPanoramaApi( getConfig( 'bizdev-panorama', 'admin', 'arista' ) )
   testFwViaPanoramaApi( getConfig( 'bizdev-panorama', 'admin', 'arista' ),
                         '001801014981' )

def dumpPolicies( polPerVsys ):
   for vsys, pols in polPerVsys.iteritems():
      print "Virtual System: %s" % vsys
      for pol in pols:
         print pol

def dumpIntfs( intfsPerVsys ):
   for vsys, intfs in intfsPerVsys.iteritems():
      print "Virtual System: %s" % vsys
      for intf in intfs:
         print intf

if __name__ == '__main__':
   panFw = PanDevice( getConfig( 'fwpan101', 'admin', 'arastra' ) )
   panFw.showSystemResources()
   print panFw.getDeviceRoutingTables()
   dumpIntfs( panFw.getInterfacesInfo() )
   dumpPolicies(
         panFw.getPolicies( mssTags=[ 'Arista_MSS', 'Arista_MSS_offload',
                                      'redirect', 'offload', 'verbatim' ] ) )
   print panFw.getAddressNameBindings( 'vsys3' )
   print panFw.getServiceProtocolPortMap( 'vsys3' )
