#!/usr/bin/env python2.7
#
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

''' API access to a Fortinet FortiManager Centralized Security Management platform
    Initial JSON-RPC Request API developed and tested with FortiManager 5.6.2

'''
# pylint: disable=E1101

import re
import json
import time
import requests
from requests.adapters import HTTPAdapter
from copy import deepcopy
from MssPolicyMonitor import Lib
from MssPolicyMonitor.Error import ServiceDeviceError, FirewallAPIError
from MssPolicyMonitor.Lib import ( t0, t2, t4, t6,
                                   LINK_STATE_UNKNOWN, LINK_STATE_UP,
                                   LINK_STATE_DOWN, HA_ACTIVE, HA_PASSIVE,
                                   HA_ACTIVE_PASSIVE, HA_ACTIVE_ACTIVE )
from MssPolicyMonitor.PluginLib import ( ServiceDeviceHAState, ServiceDevicePolicy,
                                         NetworkInterface,
                                         ServiceDeviceRoutingTables )

# TODO: suppress "Unverified HTTPS request is being made" warning message
# until we add support for TLS/SSL certificate validation
import urllib3
urllib3.disable_warnings( urllib3.exceptions.InsecureRequestWarning )

t4( 'imported python requests version:', requests.__version__ )

POLICY_TAGS_REGEX = re.compile( r'tags[[(](.+?)[])]', re.IGNORECASE | re.DOTALL )
DEFAULT_VLAN_RANGE = '1-4094'
HA_STRING_MAP = {
   'a-p': HA_ACTIVE_PASSIVE,
   'a-a': HA_ACTIVE_ACTIVE,
   # 'active': HA_ACTIVE,
   # 'passive': HA_PASSIVE,
   # 'active-primary': HA_ACTIVE_PRIMARY,
   # 'active-secondary': HA_ACTIVE_SECONDARY,
}

####################################################################################
class FortiManager( object ):

   def __init__( self, config ):
      self.config = config
      self.ipAddr = config[ 'ipAddress' ]
      protocol = config[ 'protocol' ]
      portNum = config[ 'protocolPortNum' ]
      self.username = config[ 'username' ]
      self.password = config[ 'password' ]
      self.timeout = config[ 'timeout' ]
      self.retries = config[ 'retries' ]
      self.adom = config[ 'adminDomain' ]
      self.vdom = config[ 'virtualDomain' ]
      # The opmode for a vdom can be 'transparent' (L2) or 'nat' (L3)
      self.opmodeDict = {}
      self.baseUrl = '%s://%s:%s/jsonrpc' % ( protocol, self.ipAddr, portNum )
      t2('FortiManager plugin initializing, API baseUrl:', self.baseUrl,
         'ADOM:', self.adom )
      self.requestHeaders = { 'content-Type': 'application/json',
                              'accept':       'application/json' }
      self.deviceInfo = {}
      self.fmgrApi = None
      self.sessionId = ''
      self.sslProfileName = config[ 'sslProfileName' ]
      self.trustedCertsPath = config.get( 'trustedCertsPath', '' )
      self.setupSession()

   def setupSession( self ):
      self.fmgrApi = requests.session()  # use HTTP 1.1 persistent TCP connection
      self.fmgrApi.mount( 'http://', HTTPAdapter( max_retries=self.retries ) )
      self.fmgrApi.mount( 'https://', HTTPAdapter( max_retries=self.retries ) )
      self.fmgrApi.verify = self.config[ 'verifyCertificate' ]  # also see getUrl

   def closeApiConnection( self ):
      if self.sessionId:
         t4('closing API connection for sessionId:', self.sessionId )
         self.getUrl( 'exec', '/sys/logout' )
         self.fmgrApi.close()
         self.sessionId = ''

   def resetApiConnection( self ):
      t4('resetting API connection to FortiMgr:', self.ipAddr )
      self.closeApiConnection()
      time.sleep( 1 )
      self.setupSession()
      self.loginAsNeededAndConfirm()

   def loginAsNeededAndConfirm( self ):
      """ Login to FortiManager Server.  Return True on success.
      """
      if self.sessionId:
         return True
      credentials = { 'user': self.username, 'passwd': self.password }
      #t2('login to FortiManager with:', credentials )
      resp = self.getUrl( 'exec', '/sys/login/user', loginRequest=True,
                          addParams={ 'data': credentials } )
      if not resp or 'session' not in resp:
         t4( '%s FortiManager API login failed, check device address, '
              'username, password and network availability' % self.ipAddr )
         return False
      else:
         self.sessionId = resp[ 'session' ]
         t4('login successful, sessionId:', self.sessionId )
         return True

   def getUrl( self, method, urlParam, addParams=None, loginRequest=False ):
      """ Make REST API calls to a FortiManager
          method: get, add, set, update, delete, move, clone, replace, and execute
      """
      if not loginRequest and not self.loginAsNeededAndConfirm():
         raise ServiceDeviceError( 'login failed' )
      params = { 'url': urlParam }
      if addParams:
         params.update( addParams )
      requestBody = { 'method': method, 'params': [ params ], 'id': 1, 'skip' : 0,
                      'session': self.sessionId, 'verbose': 1, }  #'jsonrpc': '2.0' }
      url = self.baseUrl
      if loginRequest:
         t4( 'API REQ LOGIN' )
      else:
         t4('API REQ URL', url, 'BODY:', requestBody )
      if self.sslProfileName and not self.trustedCertsPath:
         raise ServiceDeviceError( Lib.SSL_ERROR_MSG )

      connectionFailed = False
      for attempt in range( 1, self.retries + 2 ):
         resp = None
         try:
            resp = self.fmgrApi.post(
               url, data=json.dumps( requestBody ), headers=self.requestHeaders,
               verify=( self.trustedCertsPath if self.sslProfileName else False ),
               timeout=self.timeout )
            connectionFailed = False
            if resp.status_code == requests.codes.ok:  # pylint: disable=E1101
               break
         except requests.exceptions.SSLError:
            raise ServiceDeviceError( Lib.SSL_ERROR_MSG )
         except Exception as ex:  # pylint: disable=W0703
            if loginRequest:
               t4( '%s FortiManager API login access attempt %s, %s' % (
                    self.ipAddr, attempt, type( ex ) ) )
            else:
               t4( '%s FortiManager API access attempt %s, %s' % (
                    self.ipAddr, attempt, ex ) )
            connectionFailed = True

      if connectionFailed:
         # Connection failed after max retries
         raise ServiceDeviceError( 'Connection error' )

      respJson = None
      try:
         respJson = resp.json()
      except ValueError as err:
         t4( 'API RESP: json decode error ', err )
         raise FirewallAPIError( resp.status_code, '' )

      t4('API RESP', resp.status_code, json.dumps( respJson, indent=3, sort_keys=1 ))
      if ( resp.status_code != requests.codes.ok  # pylint: disable=E1101
           or respJson[ 'result' ][ 0 ][ 'status' ][ 'code' ] != 0 ):
         if not loginRequest:
            raise FirewallAPIError( resp.status_code,
                                    respJson[ 'result' ][ 0 ][ 'status' ][ 'code' ] )
      return respJson if loginRequest else respJson[ 'result' ][ 0 ]

   def getDeviceInfo( self, cachedOk=True ):
      if cachedOk and self.deviceInfo:
         t4('get device info, cached')
         return self.deviceInfo

      resp = self.getUrl( 'get', '/sys/status' )
      if not resp:
         return {}

      devInfo = {}
      info = resp[ 'data' ]
      devInfo[ 'model' ] = str( info[ 'Platform Full Name' ] )
      devInfo[ 'name' ] = str( info[ 'Hostname' ] )
      t2( info[ 'Hostname' ], info[ 'Platform Full Name' ], 'version:',
          info[ 'Version' ] )

      resp = self.getUrl( 'get', '/cli/global/system/interface' )
      if not resp:
         devInfo[ 'ipAddr' ] = devInfo[ 'name' ]  # best guess dns name
         return devInfo

      devInfo[ 'ipAddr' ] = str( resp[ 'data' ][ 0 ][ 'ip' ][ 0 ] )
      self.deviceInfo = devInfo  # update cache
      return devInfo

   def getGroupMembers( self, groupName ):
      resp = self.getUrl(
         'get', '/dvmdb/adom/%s/group/%s' % ( self.adom, groupName ),
         { 'expand member': [
            { 'url': '/device', 'fields': [ 'name', 'ip' ] } ] } )
      if not resp or 'data' not in resp or 'expand member' not in resp[ 'data' ]:
         return []

      members = []
      for device in resp[ 'data' ][ 'expand member' ][ '/device' ]:
         members.append( str( device[ 'name' ] ) )
         if 'vdom' in device:
            for vdom in device[ 'vdom' ]:
               self.opmodeDict[ vdom[ 'name' ] ] = vdom[ 'opmode' ]
      return members

   def getInterfacesVirtualWires( self ):
      resp = self.getUrl(
         'get', '/pm/config/adom/%s/obj/system/virtual-wire-pair' % self.adom )
      if not resp:
         return {}

      intfs = {}
      for vw in resp[ 'data' ]:
         vwireName = str( vw[ 'name' ] )
         vlans = [ DEFAULT_VLAN_RANGE if vw[ 'wildcard-vlan' ] == 'enable' else '0' ]
         for intf in vw[ 'member' ]:
            intfs[ str( intf ) ] = { 'vwire': vwireName, 'allowedVlans': vlans }
      return intfs

   def getVirtualWires( self ):
      resp = self.getUrl(
         'get', '/pm/config/adom/%s/obj/system/virtual-wire-pair' % self.adom )
      if not resp:
         return {}

      vwires = {}
      for vw in resp[ 'data' ]:
         name = str( vw[ 'name' ] )
         intfs = [ str( intf ) for intf in vw[ 'member' ] ]
         vlans = [ DEFAULT_VLAN_RANGE if vw[ 'wildcard-vlan' ] == 'enable' else '0' ]
         vwires[ name ] = { 'intfs': intfs, 'allowedVlans': vlans }
      return vwires


####################################################################################
class FortiGate( object ):
   ''' Represents a FortiGate firewall
       Uses parent FortiManager proxy API to access firewall objects
   '''

   def __init__( self, config, deviceName ):
      self.config = config
      self.deviceName = deviceName
      self.ipAddr = '?'
      self.adom = config[ 'adminDomain' ]
      # If no 'virtual instance' CLI config was provided, we will pull data only
      # from the default VDOM configured from 'virtualDomain' in legacy code.
      self.vdom = config[ 'virtualInstance' ] or [ config[ 'virtualDomain' ] ]
      self.vrouters = config.get( 'vrouters', [ '0' ] )
      self.mgmtIntfVdom = config[ 'mgmtIntfVdom' ]
      self.deviceInfo = {}
      self.parentUrl = '/sys/proxy/json'
      t2('FortiGate:', deviceName, 'ADOM:', self.adom, 'VDOM:', self.vdom,
         'mgmtIntfVdom:', self.mgmtIntfVdom, 'initializing parent FMgr proxy API' )
      self.parentFortiManager = FortiManager( config )
      self.parentFortiManager.getGroupMembers( config[ 'group' ] )
      # We assume all VDOM configured are either L3 or L2 but never both. Thus,
      # we check the device mode only on the first VDOM.
      if self.vdom[ 0 ] in self.parentFortiManager.opmodeDict and \
         self.parentFortiManager.opmodeDict[ self.vdom[ 0 ] ] == 'nat' :
         self.deviceMode = Lib.LAYER3
      else:
         self.deviceMode = Lib.VIRTUAL_WIRE

   def closeApiConnection( self ):
      self.parentFortiManager.closeApiConnection()

   def resetApiConnection( self ):
      t2('reset API connection deviceInfo:', self.deviceInfo )
      self.parentFortiManager.resetApiConnection()
      self.getDeviceInfo( cachedOk=False )
      t2('done resetting conn. deviceInfo:', self.deviceInfo )

   def getUrl( self, resource, resultsOnly=True ):
      params = {
         'data': {
            'action': 'get',
            'resource': resource,
            'target': [ 'adom/%s/device/%s' % ( self.adom, self.deviceName ) ] } }
      resp = self.parentFortiManager.getUrl( 'exec', self.parentUrl, params )
      if resp[ 'data' ][ 0 ][ 'status' ][ 'code' ] != 0:
         warnMsg = '%s FortiGate API url %s returned status %s target %s' % \
                   ( self.deviceName, resource, resp[ 'data' ][ 0 ][ 'status' ],
                     params[ 'data' ][ 'target' ] )
         t0( warnMsg )
         raise FirewallAPIError( requests.codes.ok,
                                 resp[ 'data' ][ 0 ][ 'status' ][ 'code' ] )
      if resultsOnly:
         return resp[ 'data' ][ 0 ][ 'response' ][ 'results' ]
      else:
         return resp[ 'data' ][ 0 ][ 'response' ]

   def getDeviceInfo( self, cachedOk=True ):
      if cachedOk and self.deviceInfo:
         t4('get device info, cached')
         return self.deviceInfo
      resp = self.getUrl( '/api/v2/monitor/web-ui/state', resultsOnly=False )
      if not resp or 'results' not in resp:
         return {}
      devInfo = {
         'model': '%s %s' % ( str( resp[ 'results' ][ 'model_name' ] ),
                              str( resp[ 'results' ][ 'model_number' ] ) ),
         'name': str( resp[ 'results' ][ 'hostname' ] ),
         'serialNum': str( resp[ 'serial' ] ),
         'ipAddr': '' }
      t0( resp[ 'results' ][ 'hostname' ], resp[ 'results' ][ 'model_name' ],
          resp[ 'results' ][ 'model_number' ], 'version:', resp[ 'version' ] )
      self.deviceInfo = devInfo
      resp = self.getUrl( '/api/v2/monitor/system/interface?vdom=%s' %
                          self.mgmtIntfVdom )
      if not resp:
         return devInfo
      for mgmtIntf in [ 'mgmt', 'mgmt1', 'mgmt2' ]:  # search intfs in this order
         if ( mgmtIntf in resp and 'ip' in resp[ mgmtIntf ] and
              resp[ mgmtIntf ][ 'ip' ] and resp[ mgmtIntf ][ 'link' ] ):
            ip = str( resp[ mgmtIntf ][ 'ip' ] )
            devInfo[ 'ipAddr' ] = ip
            self.ipAddr = ip
      return devInfo

   def getHighAvailabilityState( self ):
      haState = ServiceDeviceHAState()
      haState.mgmtIp = self.ipAddr
      resp = self.getUrl( '/api/v2/cmdb/system/ha' )
      if not resp:
         return haState
      mode = resp[ 'mode' ]
      haState.enabled = False if mode == 'standalone' else True
      haState.mode = HA_STRING_MAP[ mode ] if mode in HA_STRING_MAP else ''
      if not haState.enabled:
         return haState

      resp = self.getUrl( '/api/v2/monitor/system/ha-checksums', resultsOnly=False )
      if not resp or 'results' not in resp:
         return haState
      for haDevice in resp[ 'results' ]:
         state = HA_ACTIVE if haDevice[ 'is_root_master' ] else HA_PASSIVE
         serialNum = haDevice[ 'serial_no' ]
         t2('FortiGate serialNumber:', serialNum, 'haState:', state )
         if serialNum == resp[ 'serial' ]:
            haState.state = state

      if ( 'serialNum' in self.deviceInfo and
           self.deviceInfo[ 'serialNum' ] != resp[ 'serial' ] ):
         t2('possible HA master change, updating deviceInfo, old:', self.deviceInfo )
         self.getDeviceInfo( cachedOk=False )
         t2('new deviceInfo:', self.deviceInfo )
      return haState

   def getServiceProtocolPortMap( self ):
      svcMap = { vdom : {} for vdom in self.vdom }
      for vdom in self.vdom:
         resp = self.getUrl(
            '/api/v2/cmdb/firewall.service/custom?vdom=%s' % vdom )
         if not resp:
            continue
         for serviceJson in resp:
            service = serviceJson[ 'name' ]
            svcMap[ vdom ][ str( service ) ] = {}
            protocols = serviceJson[ 'protocol' ]
            for protocol in protocols.split( '/' ) :
               ports = ''
               if protocol == 'TCP':
                  ports = serviceJson[ 'tcp-portrange' ]
               elif protocol == 'UDP':
                  ports = serviceJson[ 'udp-portrange' ]
               elif protocol == 'SCTP':
                  ports = serviceJson[ 'sctp-portrange' ]
               elif protocol == 'IP':
                  ports = serviceJson[ 'protocol-number' ]
               elif protocol == 'ICMP' or protocol == 'ICMP6':
                  ports = serviceJson[ 'icmptype' ]
               ports = str( ports )
               svcMap[ vdom ][ service ][ str( protocol ) ] = ports.split()
      t4( 'L4 services map:', svcMap )
      return svcMap

   def resolveL4Ports( self, policies ):
      serviceProtocolPortMap = self.getServiceProtocolPortMap()

      for vdom, policyList in policies.iteritems():
         for policy in policyList:
            l4Services = {}
            for service in policy.services:
               if service == 'application-default':
                  continue
               if service in serviceProtocolPortMap[ vdom ]:
                  t6( 'serviceProtocolPortMap:', serviceProtocolPortMap[ vdom ] )
                  l4Services[ service ] = deepcopy(
                        serviceProtocolPortMap[ vdom ][ service ] )
            t6( 'l4Services after processing policy.services:', l4Services )

            # aggregate TCP ports, UDP ports write policy.dstL4Services:
            for protocolPorts in l4Services.values():
               for protocol, ports in protocolPorts.items():
                  if protocol == 'IP':
                     protocol = 'IPv4'
                  if protocol not in Lib.IP_PROTOCOL_TAC_VALUE:
                     t0( 'Unknown protocol: %s' % protocol )
                     continue
                  if protocol not in policy.dstL4Services:
                     policy.dstL4Services[ protocol ] = ports
                  else:
                     policy.dstL4Services[ protocol ].extend( ports )
            t4( 'policy:', policy.name, 'apps:', policy.applications, 'services:',
                policy.services, 'L4Ports:', policy.dstL4Services )

   def getPolicies( self, mssTags=None ):
      # alt: /api/v2/monitor/firewall/policy?vdom=
      mssTags = set( mssTags )
      intfInfo = self.getInterfacesInfo()
      policies = { vdom : [] for vdom in self.vdom }

      for vdom in self.vdom:
         addressObjects = self.getAddressObjects( vdom )
         resp = self.getUrl(
            '/api/v2/cmdb/firewall/policy?vdom=%s&filter=status==enable'
            '&filter=comments=@tags(,comments=@tags['
            '&format=name|comments|srcaddr|dstaddr|srcintf|dstintf|action|status'
            '|service'
            % vdom )
         if not resp:
            continue

         policyOrder = 1
         for polJson in resp:
            match = POLICY_TAGS_REGEX.search( str( polJson[ 'comments' ] ) )
            if not match:
               continue
            tags = [ t.strip() for t in match.group( 1 ).split( ',' ) ]
            tags = set( [ t for t in tags if t ] ) # filter empty strings, make set
            t4( polJson[ 'name' ], 'extracted policy tags:', tags )
            if tags.isdisjoint( mssTags ):
               continue
            policy = ServiceDevicePolicy( str( polJson[ 'name' ] ),
                                          managementIp=self.ipAddr,
                                          number=policyOrder )
            t4( 'Processed Policy:', str( polJson[ 'name' ] ),
                'policyOrder:', policyOrder )
            policyOrder += 1
            policy.tags = list( tags )
            policy.action = str( polJson[ 'action' ] )
            policy.srcZoneName, policy.srcZoneInterfaces, \
               policy.srcIpAddrList, policy.srcZoneType = \
               getZoneInfo( polJson, intfInfo[ vdom ], addressObjects,
                            'srcintf', 'srcaddr', self.deviceMode )
            policy.dstZoneName, policy.dstZoneInterfaces, \
               policy.dstIpAddrList, policy.dstZoneType = \
               getZoneInfo( polJson, intfInfo[ vdom ], addressObjects,
                            'dstintf', 'dstaddr', self.deviceMode )
            policy.services = [ service[ 'name' ]
                                for service in polJson[ 'service' ] ]
            policy.applications = []
            policies[ vdom ].append( policy )
      self.resolveL4Ports( policies )
      return policies

   def getInterfacesInfo( self ):
      ''' Get all necessary interface information for service devices.
          Returns a list of NetworkInterface objects.
      '''
      intfVwires = self.parentFortiManager.getInterfacesVirtualWires()
      interfaces = { vdom : {} for vdom in self.vdom }

      for vdom in self.vdom:
         resp = self.getUrl( '/api/v2/monitor/system/available-interfaces?vdom=%s' %
                             vdom )
         if not resp:
            continue

         for intf in resp:
            if 'name' not in intf or 'link' not in intf:
               continue
            intfName = str( intf[ 'name' ] )
            linkState = translateIntfState( intf[ 'link' ], intfName )
            if intf[ 'type' ] == 'physical':
               netIntf = NetworkInterface( intfName, state=linkState,
                                           isEthernet=True )
            elif intf[ 'type' ] == 'vlan':
               vlan = str( intf[ 'vlan_id' ] )
               netIntf = NetworkInterface( intfName, state=linkState, isSubIntf=True,
                                           vlans=[ vlan ] )
               physIntfName = str( intf[ 'vlan_interface' ] )
               netIntf.addPhysicalIntf( physIntfName, linkState )
            elif intf[ 'type' ] == 'aggregate':
               netIntf = NetworkInterface( intfName, state=linkState, isLag=True )
               for physIntfName in intf[ 'members' ]:
                  netIntf.addPhysicalIntf( str( physIntfName ) )
            else:
               continue  # unsupported intf type

            interfaces[ vdom ][ intfName ] = netIntf
            if 'virtual_wire_pair' in intf:
               vwireName = str( intf[ 'virtual_wire_pair' ] )
               netIntf.attribs[ 'vwire' ] = vwireName
               netIntf.zone = '%s_%s' % ( vwireName, intfName )
               if intfName not in intfVwires:
                  t2('intf:', intfName, 'not found in:', intfVwires )
                  continue
               if 'allowedVlans' in intfVwires[ intfName ]:
                  netIntf.vlans = intfVwires[ intfName ][ 'allowedVlans' ]
               if ( 'vwire' in intfVwires[ intfName ] and
                     intfVwires[ intfName ][ 'vwire' ] != vwireName ):
                  t2('vwire names do not match:', vwireName,
                     intfVwires[ intfName ][ 'vwire' ] )
            elif 'is_routable' in intf and intf[ 'is_routable' ] and \
                 'ipv4_addresses' in intf:
               netIntf.zone = 'L3_%s' % ( intfName, )
               netIntf.ipAddr = str( intf[ 'ipv4_addresses' ][ 0 ][ 'ip' ] )
               # Fortinet supports only VRF 0 at this time
               netIntf.vrf = '0'

      # now set link state for LAG member intfs
      for vdom, intfs in interfaces.iteritems():
         for netIntf in intfs.values():
            if netIntf.isLag:
               for lagMemberIntf in netIntf.physicalIntfs:
                  if lagMemberIntf.name in intfs:
                     lagMemberIntf.state = intfs[ lagMemberIntf.name ].state

      return interfaces

   def getInterfaceNeighbors( self ):
      neighbors = {}
      try:
         resp = self.getUrl( '/api/v2/monitor/network/lldprx/neighbors?vdom=%s'
                             % self.vdom[ 0 ] )
         if not resp:
            return {}
         intfIndexMap = self.getInterfaceIndexMap()
         for nbr in resp:
            fwIntfIndex = nbr[ 'port' ]
            if fwIntfIndex in intfIndexMap:
               chassis = str( nbr[ 'chassis_id' ] ) if nbr[ 'chassis_id' ] else ''
               intf = str( nbr[ 'port_id' ] ) if nbr[ 'port_id' ] else ''
               desc = str( nbr[ 'system_desc' ] ) if nbr[ 'system_desc' ] else ''
               sys = str( nbr[ 'system_name' ] ) if nbr[ 'system_name' ] else ''
               neighbors[ intfIndexMap[ fwIntfIndex ] ] = {
                  'switchChassisId': chassis, 'switchIntf': intf, 'nborDesc': desc,
                  'nborSysName': sys }
      except Exception:  # pylint: disable=W0703
         t2('unable to get LLDP data, some FortiOS versions do not support LLDP' )
      t4('LLDP neighbors:', neighbors )
      return neighbors

   def getInterfaceIndexMap( self ):
      intfIndexMap = {}
      resp = self.getUrl(
         '/api/v2/cmdb/system/interface?format=name|devindex&vdom=%s' %
         self.vdom[ 0 ] )
      if not resp:
         return {}
      for intf in resp:
         intfIndexMap[ intf[ 'devindex' ] ] = str( intf[ 'name' ] )
      return intfIndexMap

   def getDeviceResources( self ):
      # URL below is used by FortiGate webUI for system resources view
      # The first cpu entry is average utilization over last 5 minutes (per Fortinet)
      resp = self.getUrl(
         '/api/v2/monitor/system/resource/usage?scope=global&interval=1-min' )
      if not resp:
         return {}
      pfmt = '%-20s %3s%%\n'
      fmt =  '%-20s %3s\n'
      info =  pfmt % ( 'CPU Utilization:', resp[ 'cpu' ][ 0 ][ 'current' ] )
      info += pfmt % ( 'Memory Utilization:', resp[ 'mem' ][ 0 ][ 'current' ] )
      info += pfmt % ( 'Disk Utilization: ', resp[ 'disk' ][ 0 ][ 'current' ] )
      info += fmt % ( 'Number of Sessions:', resp[ 'session' ][ 0 ][ 'current' ] )
      info += '\n'
      return { 'resourceInfo' : info }

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      routingTables = ServiceDeviceRoutingTables()
      for vdom in self.vdom:
         resp = self.getUrl( '/api/v2/monitor/router/ipv4?vdom=%s&type=3' % vdom )
         for route in resp:
            # Fortinet supports only VRF 0 at this time
            vrfName = '0'
            destination = str( route[ 'ip_mask' ].replace( "\\", "" ) )
            interface = str( route[ 'interface' ] )
            nexthop = str( route[ 'gateway' ] )
            routingTables.addRoute( vrfName, destination, interface, nexthop )
      routingTables.featureSupported = True
      return routingTables

   def getAddressObjects( self, vdom ):
      resp = self.getUrl( '/api/v2/cmdb/firewall/address?vdom=%s'
                          '&format=name|type|ipmask|start-ip|end-ip|subnet' % vdom )
      if not resp:
         return {}

      objs = {}
      for addr in resp:
         if 'start-ip' in addr and addr[ 'start-ip' ] == '0.0.0.0':
            continue
         if addr[ 'type' ] == 'ipmask':
            if 'subnet' in addr:
               # SW version >= 6.2.0
               ip, mask = str( addr[ 'subnet' ] ).split()
               if ip == '0.0.0.0':
                  continue
               if mask == '255.255.255.255':
                  objs[ str( addr[ 'name' ] ) ] = ip
               else:
                  objs[ str( addr[ 'name' ] ) ] = '%s|%s' % ( ip, mask )
            elif 'end-ip' in addr and 'start-ip' in addr:
               # SW version < 6.2.0
               if addr[ 'end-ip' ] == '255.255.255.255':
                  objs[ str( addr[ 'name' ] ) ] = str( addr[ 'start-ip' ] )
               else:
                  objs[ str( addr[ 'name' ] ) ] = '%s|%s' % (
                     str( addr[ 'start-ip' ] ), str( addr[ 'end-ip' ] ) )
            else:
               t0( 'Invalid address reply: %s' % str( addr ) )
         elif addr[ 'type' ] == 'iprange':
            objs[ str( addr[ 'name' ] ) ] = '%s-%s' % ( str( addr[ 'start-ip' ] ),
                                                        str( addr[ 'end-ip' ] ) )
      resp = self.getUrl( '/api/v2/cmdb/firewall/addrgrp?vdom=%s' % vdom )
      if not resp:
         return objs

      for agp in resp:
         groupName = str( agp[ 'name' ] )
         addrList = []
         for member in agp[ 'member' ]:
            memberName = str( member[ 'name' ] )
            if memberName in objs:
               addrList.append( objs[ memberName ] )
         objs[ groupName ] = addrList
      return objs

# helper functions
def translateIntfState( state, intfName ):
   ''' Translate FortiOS intf state to MSS intf state
   '''
   if state.lower() == 'up':
      return LINK_STATE_UP
   elif state.lower() == 'down':
      return LINK_STATE_DOWN
   else:
      t4('intf:', intfName, 'unrecognized link state value:', state )
      return LINK_STATE_UNKNOWN

def getZoneInfo( polJson, intfInfo, addressObjects, intfKey, addrKey, opmode ):
   zoneName = ''
   zoneType = opmode
   intfs = []
   addrs = []
   for intfName in [ str( i[ 'name' ] ) for i in polJson[ intfKey ] ]:
      if intfName in intfInfo:
         intfs.append( intfInfo[ intfName ] )
         zoneName = intfInfo[ intfName ].zone
         if zoneName.startswith( 'L3' ):
            zoneType = Lib.LAYER3

   for addr in [ str( a[ 'name' ] ) for a in polJson[ addrKey ] ]:
      if addr in addressObjects:
         if addr == 'all': # Not supported
            continue
         Lib.appendOrExtend( addrs, addressObjects[ addr ] )
   return zoneName, intfs, addrs, zoneType


####################################################################################
# test cases
def testRetiresAndTimeouts( deviceDict ):
   print '\n\nTEST BAD FMGR ADDR'
   orig = deviceDict[ 'ipAddress' ]
   deviceDict[ 'ipAddress' ] = 'BAD_FMGR_ADDR'
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'ipAddress' ] = orig

   print '\n\nTEST BAD FMGR PASSWORD'
   orig = deviceDict[ 'password' ]
   deviceDict[ 'password' ] = 'BAD_PASSWORD'
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'password' ] = orig

   print '\n\nTEST BAD FGATE IP'
   testFGateApi( deviceDict, 'BAD_FGATE_IP' )

   print '\n\nTEST TIMEOUT'
   orig = deviceDict[ 'timeout' ]
   deviceDict[ 'timeout' ] = 0.001
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )
   deviceDict[ 'timeout' ] = orig

   print '\n\nTEST ALL GOOD'
   testFMgrApi( deviceDict )
   testFGateApi( deviceDict, 'bizdev-fnet' )

def testFMgrApi( config ):
   print '\nTest FortiManager API'
   fm = FortiManager( config )

   info = fm.getDeviceInfo()
   print 'DeviceInfo:', info

   for group in [ 'MssFortinet', 'MssFortinetHA' ]:
      gm = fm.getGroupMembers( groupName=group )
      print 'GroupMembers:', group, '=', gm

   # vw = fm.getVirtualWires()
   # print '\nVirtual Wires: %s \n' % vw

   # intfVw = fm.getInterfacesVirtualWires()
   # print '\nIntf Vwires: %s \n' % intfVw

   fm.closeApiConnection()


def testFGateApi( config, deviceName ):
   print '\nTest FortiGate API access via FortiManager'
   fg = FortiGate( config, deviceName )

   info = fg.getDeviceInfo()
   print deviceName, 'DeviceInfo:', info, '\n'

   res = fg.getDeviceResources()
   print '\nRESOURCES:\n', res[ 'resourceInfo' ] if res else ''

   ha = fg.getHighAvailabilityState()
   print 'HA State:', ha

   intfs = fg.getInterfacesInfo()  # getPolicies also calls getInterfacesInfo
   print '\nIntfInfo:'
   for vdom, intfList in intfs.iteritems():
      print "%s: %s" % ( vdom, intfList )

   nbors = fg.getInterfaceNeighbors()
   print 'LLDP Neighbors:'
   for p, n in nbors.items():
      print p, n

   pols = fg.getPolicies( mssTags=[ 'Arista_MSS', 'mss1', 'mss2' ] )
   print '\nPolicies:'
   for vdom, p in pols.iteritems():
      print '%s: %s' % ( vdom, p ), '\n'

   # ado = fg.getAddressObjects()
   # print 'AddressObjects:', ado

   fg.closeApiConnection()


if __name__ == "__main__":
   cfg = {
      'ipAddress': 'mss-fortimgr-v56', 'username': 'admin',
      'password': 'Arastra123!', 'protocol': 'https', 'protocolPortNum': 443,
      'method': 'tls', 'verifyCertificate': False, 'timeout': 5, 'retries': 1,
      'exceptionMode': 'bypass', 'group': 'MssFortinet', 'mgmtIntfVdom': 'root',
      'adminDomain': 'root', 'virtualDomain': 'L2_Firewall', 'sslProfileName' : '',
      'interfaceMap': {
         'port17': {
            'switchIntf': 'Port-Channel70', 'switchChassisId': '001c.737e.2811' },
         'port18': {
            'switchIntf': 'Port-Channel70', 'switchChassisId': '001c.737e.2811' },
         'port19': {
            'switchIntf': 'Port-Channel75', 'switchChassisId': '001c.737e.2811' },
         'port20': {
            'switchIntf': 'Port-Channel75', 'switchChassisId': '001c.737e.2811' },
         'port29': {
            'switchIntf': 'Ethernet39', 'switchChassisId': '001c.737e.2811' },
         'port30' : {
            'switchIntf' : 'Ethernet40', 'switchChassisId' : '001c.737e.2811' },
         'port31' : {
            'switchIntf' : 'Ethernet41', 'switchChassisId' : '001c.737e.2811' },
         'port32' : {
            'switchIntf' : 'Ethernet42', 'switchChassisId' : '001c.737e.2811' }, },
      'virtualInstance' : [ 'fwfortvd6', 'fwfortvd7', 'fwfortvd8' ] }

   testFMgrApi( cfg )
   testFGateApi( cfg, 'fwfort102' )
