# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import abc

from MssPolicyMonitor.Logger import CVX_MSS_FIREWALL_ACCESS_ERROR
from MssPolicyMonitor.Logger import CVX_MSS_FIREWALL_ACCESS_RECOVERED
from MssPolicyMonitor.Logger import CVX_MSS_FIREWALL_API_ERROR
from MssPolicyMonitor.Logger import CVX_MSS_FIREWALL_API_RECOVERED

import Logging

class MssPolicyMonitorError( Exception ):
   ''' Base class for MssPolicyMonitor exceptions '''
   pass

class FirewallError( MssPolicyMonitorError ):
   ''' Base class for Firewall exceptions '''
   __metaclass__ = abc.ABCMeta

   @abc.abstractmethod
   def log( self, deviceId ):
      ''' Add syslog reporting the firewall error '''
      pass

   @abc.abstractmethod
   def recover( self, deviceId ):
      ''' Add syslog reporting firewall recovery '''
      pass

class ServiceDeviceError( FirewallError ):
   ''' Exception for error when accessing a ServiceDevice '''

   def __init__( self, msg ):
      super( ServiceDeviceError, self ).__init__()
      self.msg = msg

   def __eq__( self, otherError ):
      return ( isinstance( otherError, ServiceDeviceError ) and
               self.msg == otherError.msg )

   def __ne__( self, otherError ):
      return not self == otherError

   def log( self, deviceId ):
      Logging.log( CVX_MSS_FIREWALL_ACCESS_ERROR, deviceId, self.msg )

   def recover( self, deviceId ):
      Logging.log( CVX_MSS_FIREWALL_ACCESS_RECOVERED, deviceId )

   def __str__( self ):
      return self.msg

   def __reduce__( self ):
      return ( ServiceDeviceError, ( self.msg ) )

class FirewallAPIError( FirewallError ):
   ''' Exception for error in firewall API request '''
   def __init__( self, httpStatusCode, errorCode ):
      self.httpStatusCode = httpStatusCode
      self.errorCode = errorCode or ''

   def __eq__( self, otherError ):
      return ( isinstance( otherError, FirewallAPIError ) and
               self.httpStatusCode == otherError.httpStatusCode and
               self.errorCode == otherError.errorCode )

   def __ne__( self, otherError ):
      return not self == otherError

   def log( self, deviceId ):
      Logging.log( CVX_MSS_FIREWALL_API_ERROR, deviceId, self.httpStatusCode,
                   self.errorCode )

   def recover( self, deviceId ):
      Logging.log( CVX_MSS_FIREWALL_API_RECOVERED, deviceId )

