# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac

TEXT_MODE = 1
JSON_MODE = 2

# For policies configured via CLI, we add the following prefix to the policy name
# and used it as the key for servicePolicy collection. This will help us to identify
# the policy in the global policy space.
cliPrefix = 'CLI+'
separator = '+'

# L3 static policy config
MssL3PolicyAction = Tac.Type( 'MssL3::MssAction' )
MssL3V2PolicyAction = Tac.Type( 'MssL3V2::MssAction' )

MssL3PolicyModifier = Tac.Type( 'MssL3::MssPolicyModifier' )
mssL3ModifierVerbatim = MssL3PolicyModifier.verbatim

MssL3V2PolicyModifier = Tac.Type( 'MssL3V2::MssPolicyModifier' )
mssL3V2ModifierVerbatim = MssL3V2PolicyModifier.verbatim

PolicyState = Tac.Type( 'MssL3V2::MssPolicyState' )
LocalData = Tac.Type( 'MssL3V2::LocalData' )

ipProtocol = {
   1: 'ICMP', 6: 'TCP', 17: 'UDP'
}

l3Protocol = {
   1: 'ICMP'
}

l4Protocol = {
   6: 'TCP', 17: 'UDP'
}

stateMap = {
   'Internal Redirect': 'IR',
   'dryrunPending': 'AllocationPending',
   'dryrunActive': 'AllocationSuccess',
   'dryrunError': 'AllocationError',
   'pending': 'Pending',
   'active': 'Active',
   'error': 'Error'
}

abbreviatedStateMap = {
   'IR': 'IR',
   'AllocationPending': 'AP',
   'AllocationSuccess': 'AS',
   'AllocationError': 'AE',
   'Pending': 'P',
   'Active': 'A',
   'Error': 'E'
}

def externalState( state ):
   strState = str( state )
   if strState.startswith( 'ip' ):
      strState = strState[ 2: ]
   elif strState.startswith( 'rule' ):
      strState = strState[ 4: ]
   strState = stateMap.get( strState, strState )
   return strState

def abbreviateState( state ):
   return abbreviatedStateMap.get( state, state )

# Common token definitions (used in Mss, MssClient CLI)
#------------------------------------------------------------------------------
# MssDeviceTokens
#------------------------------------------------------------------------------
matcherMssDevice = CliMatcher.KeywordMatcher( 'device',
      helpdesc='Set service device properties' )

#------------------------------------------------------------------------------
# Show policy keywords
#------------------------------------------------------------------------------
matcherMssStaticPolicySource = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Policies configured via CLI' )

def getHostname( topologyStatus, chassisId ):
   def getName():
      for hostId, host in topologyStatus.host.iteritems():
         if hostId.lower() == chassisId:
            return host.hostname
      return ''

   return '' if not topologyStatus else getName()

class PolicyFilter( object ):
   def __init__( self, policy="", source="", device="", vsys="", vrf="",
                 switch="", ipv4=None, active=False, group=False ):
      self.policy = policy
      self.source = source
      self.device = device
      self.vsys = vsys
      self.vrf = vrf
      self.switch = switch
      self.ipv4 = ipv4
      self.active = active
      self.group = group

   def isAllowed( self, policy, source, device, vsys='', vrf='', ips=None,
                  switches=None, state=None ):
      allowed = (
         lambda actual, filterValue: actual == filterValue if filterValue else True )
      ipIncluded = bool( self.ipv4 & ips ) if self.ipv4 else True
      switchIncluded = self.switch in switches if self.switch else True
      activePolicy = state == PolicyState.active if self.active else True
      groupPolicy = policy.startswith(
         LocalData.internalRedirectPolicyName ) if self.group else True

      return allowed( policy, self.policy ) and allowed( source, self.source ) \
         and allowed( device, self.device ) and allowed( vsys, self.vsys ) \
         and allowed( vrf, self.vrf ) and switchIncluded and ipIncluded \
         and activePolicy and groupPolicy

   def areIpsNeeded( self ):
      return self.ipv4

   def areSwitchesNeeded( self ):
      return self.switch

   def isPolicyGroupAllowed( self, policies, source, device ):
      return any( self.isAllowed( i, source, device ) for i in policies )

#------------------------------------------------------------------------------------
# Command class for syntax of traffic inspection config. Used by both Mss and MssPM
#           [ no|default ] traffic inspection {local|local outbound}
#------------------------------------------------------------------------------------
class TrafficInspectionCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic inspection TYPES'
   noOrDefaultSyntax = 'traffic inspection ...'
   data = {
      'traffic' : 'Traffic type for policy enforcement',
      'inspection': 'Traffic type for policy enforcement',
      'TYPES' : CliCommand.setCliExpression( {
         'local' : 'Consider East-West traffic',
         'outbound' : 'Consider Northbound traffic' } )
   }

   @staticmethod
   def handler( mode, args ):
      local = "local" in args
      outbound = "outbound" in args
      if outbound and not local:
         mode.addError( "Only Northbound traffic inspection is not supported" )
         return
      mode.trafficInspectionIs( local, outbound )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noTrafficInspection()
