# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import collections
from IpLibConsts import DEFAULT_VRF
from ClientCore import traceroute
from ClientCommonLib import (
   AddressFamily,
   getNhgId, 
   labelStackToList,
   resolveHierarchical,
   isIpv6Addr, 
   resolveNexthop,
   getTunnelNhgName,
   _resolveSrTePolicyTunnels,
   generateMplsEntropyLabel,
   getStaticFec,
   isNexthopGroupVia, 
   getNexthopGroupId,
   getNhgIdToName,
   isNexthopGroupTunnelVia,
   DynTunnelIntfId,
   resolveNhgTunnelFibEntry,
   getRsvpFec,
   getDsMappingInfo,
   getIntfPrimaryIpAddr,
   getLdpFec,
   getProtocolIpFec, 
   IPV4, IPV6,
   MplsLabel, MldpInfo,
   getBgpLuTunnelFibEntry,
   getNhAndLabelFromTunnelFibVia,
   validateBgpLuResolvedPushVia,
   LspPingTypeBgpLu,
)
from ClientState import getGlobalState
from MplsPingClientLib import lspTracerouteLabelDistStatisticsRender

from ForwardingHelper import getNhgSize

import sys, random, Tac
from TypeFuture import TacLazyType

# ---------------------------------------------------------
#                   Local Utils 
# ---------------------------------------------------------

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
NexthopGroupType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )

ELI = Tac.Type( 'Arnet::MplsLabel' ).entropyLabelIndicator

def getL3IntfMtu( l3Intf, mount ):
   return mount.allIntfStatusDir[ l3Intf ].mtu

# ---------------------------------------------------------
#                   LspTraceroute BGP LU
# ---------------------------------------------------------

def handleLspTracerouteBgpLu( prefix, mount, src=None, dst=None, smac=None,
                              dmac=None, vrf=None, interface=None, interval=1,
                              count=None, label=None, verbose=False, entry=None,
                              tc=None, nexthop=None, standard=None, size=None,
                              padReply=False, egressValidateAddress=None,
                              multiLabel=1, tos=None,
                              **kwargs ):
   state = getGlobalState()
   viaInfo, clientIdToVias = [], {}
   resolvedPushVia = collections.defaultdict( list )
   unresolvedPushVia = []
   ipv = IPV6 if isIpv6Addr( str( prefix ) ) else IPV4

   tunnelFibEntry, err = getBgpLuTunnelFibEntry( mount, prefix )
   if err:
      print err
      return errno.EINVAL

   for tunnelVia in tunnelFibEntry.tunnelVia.itervalues():
      nextHopAndLabel, err = getNhAndLabelFromTunnelFibVia( mount, tunnelVia )
      if err:
         print err
         return errno.EINVAL
      nexthopIp = nextHopAndLabel.nextHopIp
      labels = nextHopAndLabel.label
      intfId = nextHopAndLabel.intfId

      # If nexthop and/or label stack are specified, use only the via that has
      # that nexthop/label stack.
      if nexthop and IpGenAddr( nexthop ) != nexthopIp:
         continue
      if label and label != labels:
         continue

      nexthopAddr = nexthopIp.v4Addr if ipv == IPV4 else nexthopIp.v6Addr
      # Map L3 nexthop to L2 nexthop
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthopAddr,
                                                    intf=intfId )
      if not ( nexthopIntf and nexthopEthAddr ):
         unresolvedPushVia.append( ( nexthopIp, labels ) )
         continue
      else:
         key = ( nexthopIntf, tuple( labels ), nexthopEthAddr )
         resolvedPushVia[ key ].append( ( nexthopIp, labels, nexthopIntf ) )
         # Use only first tunnelVia for traceroute
         break

   if nexthop:
      err = validateBgpLuResolvedPushVia( resolvedPushVia, nexthop, label )
      if err:
         print err
         return errno.EINVAL

   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVias[ idx ] = val
      viaInfo.append( key )

   if not viaInfo:
      # No resolved interface
      print 'Failed to find a valid output interface'
      return errno.EINVAL

   setFecValidateFlag = bool( egressValidateAddress )
   ( nexthopIntf, labelStack, nexthopEthAddr ) = viaInfo[ 0 ]
   nexthopIp = clientIdToVias[ 0 ][ 0 ][ 0 ]
   print '  via {}, label stack: {}'.format( nexthopIp, list( labelStack ) )
   return traceroute( mount, state, nexthopIntf, labelStack,
                      src, dst, smac, nexthopEthAddr, count=1, interval=1,
                      protocol=LspPingTypeBgpLu, prefix=prefix,
                      ipv=ipv, tc=tc, standard=standard, size=size,
                      padReply=padReply, setFecValidateFlag=setFecValidateFlag,
                      tos=tos, egressValidateAddress=egressValidateAddress,
                      lookupLabelCount=multiLabel ).retVal

# ---------------------------------------------------------
#                   LspTraceroute nexthop-group
# ---------------------------------------------------------

def handleLspTracerouteNhg( nhgName, mount, entry, src=None, dst=None, smac=None,
                            dmac=None, count=None, interval=1, verbose=False,
                            vrf=None, tc=None, nhgTunnel=None, standard=None,
                            size=None, padReply=False, egressValidateAddress=None,
                            multiLabel=1, tos=None, **kwargs ):
   nhgId = getNhgId( nhgName, mount )
   if nhgId is None:
      print 'Nexthop-group %s does not exist.' % nhgName
      return errno.EINVAL
   
   # we are expecting a programmed MPLS nexthop-group
   nhgConfig = mount.routingNhgConfig.nexthopGroup.get( nhgName )
   if not nhgConfig:
      print 'Nexthop-group {} cannot be found'.format( nhgId )
      return errno.EINVAL
   if nhgConfig.type != NexthopGroupType.mpls:
      print 'Nexthop-group %s is not in MPLS type.' % nhgName
      return errno.EINVAL

   entrySize = getNhgSize( nhgConfig )

   if entry is None:
      entry = random.randint( 0, entrySize - 1 )
      if nhgTunnel is not None:
         print ( "Traceroute over nexthop-group tunnel index %s,"
                 " nexthop-group %s Entry %d" ) % ( nhgTunnel, nhgName, entry )
      else:
         print "Traceroute over nexthop-group %s Entry %d" % ( nhgName, entry )
   
   # validate entry
   if entry < 0 or entry >= entrySize:
      print 'Invalid nexthop-group entry: %d' % entry
      return errno.EINVAL

   # entry resolution
   vrf = vrf or DEFAULT_VRF
   nhgMplsLabelStack = nhgConfig.mplsLabelStack    
   hwVrfStatus = mount.routingHwNexthopGroupStatus.vrfStatus.get( vrf )
   if not hwVrfStatus:
      print 'Cannot find next-hop group status.'
      return errno.EINVAL
   if nhgId not in hwVrfStatus.nexthopGroupAdjacency:
      print 'Nexthop-group %s is not programmed.' % nhgName
      return errno.EINVAL
      
   nhgAdjVia = hwVrfStatus.nexthopGroupAdjacency.get( nhgId ).via
   if not nhgAdjVia:
      print 'Via for Nexthop-group {} cannot be resolved'.format( nhgName )
      return errno.EINVAL
   via = nhgAdjVia.get( entry )
   viaInfoList = []
   if via and via.l2Via.vlanId != 0:
      # resolved tunnel

      # XXX We are not supporting a nexthop-group having both V4 & V6 tunnel dests.
      ipv = IPV6 if isIpv6Addr( via.hop.stringValue ) else IPV4

      labelStack = labelStackToList( nhgMplsLabelStack[ entry ] )
      nhMac = via.l2Via.macAddr
      nhIntf = via.l3Intf
      viaInfoTuple = ( nhIntf, labelStack, nhMac )
      viaInfoList.append( viaInfoTuple )
   elif via and via.nextLevelFecIndex:
      labelStack = labelStackToList( nhgMplsLabelStack[ entry ] )
      nhList, _ = resolveHierarchical( mount, fecId=via.nextLevelFecIndex )
      ipv = IPV6 if isIpv6Addr( via.route.stringValue ) else IPV4
      state = getGlobalState()
      if nhList:
         for nexthop in nhList:
            nexthop = nexthop.v4Addr if nexthop.af == AddressFamily.ipv4 else \
                      nexthop.v6Addr
            nhIntf, nhMac = resolveNexthop( mount, state, nexthop )
            if not nhIntf or not nhMac:
               continue
            viaInfoTuple = ( nhIntf, labelStack, nhMac )
            viaInfoList.append( viaInfoTuple )
   if not viaInfoList:
      print 'Nexthop-group %s: entry %d not configured or resolved' % (
                                                               nhgName, entry )
      return errno.EINVAL
   print "Entry {}".format( entry )

   setFecValidateFlag = bool( egressValidateAddress )
   for viaInfoTuple in viaInfoList:
      state = getGlobalState()
      retVal = traceroute( mount, state, viaInfoTuple[ 0 ], viaInfoTuple[ 1 ], src,
                           dst, smac, viaInfoTuple[ 2 ], 1, 1, ipv=ipv, tc=tc,
                           standard=standard, size=size, padReply=padReply,
                           setFecValidateFlag=setFecValidateFlag, tos=tos,
                           egressValidateAddress=egressValidateAddress,
                           lookupLabelCount=multiLabel ).retVal
      if retVal < 0:
         return retVal
      print '\n'
   return 0

# ---------------------------------------------------------
#           LspTraceroute nexthop-group tunnel
# ---------------------------------------------------------

def handleLspTracerouteNhgTunnel( endpoint, mount, src=None, dst=None, smac=None,
                                  dmac=None, count=None, interval=1, verbose=False,
                                  vrf=None, tc=None, entry=None, standard=None,
                                  size=None, padReply=False, tos=None,
                                  egressValidateAddress=None, multiLabel=1,
                                  **kwargs ):
   ( nhgName, tunnelIndex, err ) = getTunnelNhgName( mount, endpoint )
   if err is not None:
      print err
      return errno.EINVAL

   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint

   return handleLspTracerouteNhg( nhgName, mount, src=src, dst=dst, smac=smac,
                                  dmac=dmac, vrf=vrf, interval=interval,
                                  count=count, verbose=verbose, entry=entry,
                                  tc=tc, nhgTunnel=tunnelIndex, standard=standard,
                                  size=size, padReply=padReply, tos=tos,
                                  egressValidateAddress=egressValidateAddress,
                                  multiLabel=multiLabel )

# ---------------------------------------------------------
#                   LspTraceroute SR-TE
# ---------------------------------------------------------

def handleLspTracerouteSrTe( endpoint, mount, color, trafficAf, dst=None, smac=None,
                             dmac=None, count=None, interval=1, verbose=False,
                             tc=None, standard=None, size=None, padReply=False,
                             egressValidateAddress=None, tos=None, **kwargs ):

   ipv = IPV6 if isIpv6Addr( str( endpoint ) ) else IPV4
   if trafficAf:
      ipv = IPV6 if trafficAf == 'v6' else IPV4
   tunnelToAdjacencies = _resolveSrTePolicyTunnels( mount, endpoint, color,
                                                    trafficAf=trafficAf )
   if tunnelToAdjacencies == errno.EINVAL:
      return errno.EINVAL

   # If the user does not provide any egress address, we consider the endpoint
   # as the egressValidateAddress by default.
   setFecValidateFlag = bool( egressValidateAddress )
   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint


   for tunnel in sorted( tunnelToAdjacencies.keys() ):
      state = getGlobalState()
      labelTup, nhMac, l3Intf = tunnelToAdjacencies[ tunnel ]
      labelStack = list( labelTup )

      # Add the entropy labels if EL push is enabled
      if mount.mplsTunnelConfig.entropyLabelPush:
         udpPam = state.lspPingClientRootUdpPam
         el = generateMplsEntropyLabel(
               srcIp=getIntfPrimaryIpAddr( mount, l3Intf, ipv ),
               dstIp=endpoint,
               udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
         labelStack.extend( [ ELI, el ] )

      print 'Segment list label stack: {}'.format( labelStack )
      retVal = traceroute( mount, state, l3Intf, tuple( labelStack ), None,
                           dst, smac, nhMac, 1, 1, ipv=ipv, tc=tc,
                           standard=standard, size=size, padReply=padReply,
                           setFecValidateFlag=setFecValidateFlag, tos=tos,
                           egressValidateAddress=egressValidateAddress ).retVal
      if retVal < 0 :
         return retVal
      print '\n'

   return 0

# ---------------------------------------------------------
#                   LspTraceroute static
# ---------------------------------------------------------

def handleLspTracerouteStatic( prefix, mount, label, nexthop, src, dst, vrf,
                               smac, dmac, count, interval, verbose, tc,
                               standard=None, size=None, padReply=False,
                               egressValidateAddress=None, multiLabel=1,
                               tos=None, **kwargs ):
   fec, err = getStaticFec( mount, prefix )
   if fec is None:
      return err
   
   if not fec.via:
      print 'No adjacency found for prefix %s' % prefix
      return errno.EINVAL

   state = getGlobalState()
   unresolvedPushVias = []
   resolvedPushVias = {}
   nhgNames = []
   nhgTunIdexToNhgName = {}
   for via in fec.via.values():
      if isNexthopGroupVia( via ):
         nexthopGroupId = getNexthopGroupId( via )
         nhgName = getNhgIdToName( nexthopGroupId, mount )
         if not nhgName:
            print 'No nexthop-group with id %d' % nexthopGroupId
            return errno.EINVAL
         if nexthop and label is not None:
            print ( '%s is a nexthop-group route. '
                    'Please use traceroute mpls nexthop-group.' ) % prefix
            return errno.EINVAL
         print '%s: nexthop-group route (nexthop-group name: %s)' % ( prefix,
                                                                      nhgName )
         nhgNames.append( nhgName )
         continue
      if isNexthopGroupTunnelVia( via ):
         tunnelId = DynTunnelIntfId.tunnelId( via.intfId )
         ( nhgName, tunnelIndex, err ) = resolveNhgTunnelFibEntry( mount, tunnelId )
         if err is not None:
            print err
            return errno.EINVAL
         nhgTunIdexToNhgName[ tunnelIndex ] = nhgName
         nhgNames.append( nhgName )
         continue
      if not MplsLabel( via.mplsLabel ).isValid():
         continue
      l3Intf, nhMac = resolveNexthop( mount, state, via.hop )
      if hasattr( via.hop, 'stringValue' ):
         viaNexthop = via.hop.stringValue
      else:
         assert isinstance( via.hop, str )
         viaNexthop = via.hop
      if not ( l3Intf and nhMac ):
         unresolvedPushVias.append( ( viaNexthop, via.mplsLabel ) )
         continue
      resolvedPushVias[ ( viaNexthop, via.mplsLabel ) ] = ( l3Intf, nhMac )

   if nhgNames:
      # We are only choosing one of the NHGs because we want to keep consistent with
      # how traceroute static only picks one of the vias.
      selectedNhg = random.choice( nhgNames )
      selectedNhgTunnel = None
      if nhgTunIdexToNhgName:
         selectedNhgTunnel = random.choice( nhgTunIdexToNhgName.keys() )
         selectedNhg = nhgTunIdexToNhgName[ selectedNhgTunnel ]
      return handleLspTracerouteNhg( selectedNhg, mount, src=src, dst=dst, smac=smac,
                                     dmac=dmac, vrf=vrf, interval=interval,
                                     count=count, verbose=verbose, entry=None,
                                     tc=tc, nhgTunnel=selectedNhgTunnel,
                                     standard=standard, size=size, tos=tos,
                                     padReply=padReply, multiLabel=multiLabel,
                                     egressValidateAddress=egressValidateAddress )
   if label and nexthop:
      selectedPushVia = ( nexthop, label[ 0 ] )
   else:
      selectedPushVia = random.choice( resolvedPushVias.keys() ) \
                                     if resolvedPushVias else None
   labelStack = []
   nextHop = ''
   viaStr = '  '
   if selectedPushVia:
      assert isinstance( selectedPushVia[ 1 ], int )
      nextHop = selectedPushVia[ 0 ]
      labelStack = [ selectedPushVia[ 1 ] ]
      viaStr += 'via %s, label stack: %s' % ( nextHop, labelStack )
   print viaStr
   if not selectedPushVia or selectedPushVia in unresolvedPushVias:
      print 'via not resolved'
      return errno.EINVAL
   if selectedPushVia not in resolvedPushVias:
      print 'via not found'
      return errno.EINVAL

   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4
   l3Intf, nhMac = resolvedPushVias[ selectedPushVia ]
   setFecValidateFlag = bool( egressValidateAddress )

   # Send dsmap if no NHG is being used
   dsMappingInfo = getDsMappingInfo( nextHop, labelStack,
                                     getL3IntfMtu( l3Intf, mount ),
                                     multipath=False, baseip='127.0.0.0',
                                     numMultipathBits=0 )

   retVal = traceroute( mount, state, l3Intf, labelStack, src, dst,
                        smac, nhMac, 1, 1, ipv=ipv, tc=tc, standard=standard,
                        size=size, padReply=padReply, tos=tos,
                        dsMappingInfo=dsMappingInfo,
                        setFecValidateFlag=setFecValidateFlag,
                        egressValidateAddress=egressValidateAddress,
                        lookupLabelCount=multiLabel ).retVal
   return retVal

# ---------------------------------------------------------
#                   LspTraceroute RSVP
# ---------------------------------------------------------
def handleLspTracerouteRsvp( prefix, mount, label, nexthop, src, dst, vrf,
                             smac, dmac, count, interval, verbose, tc, session,
                             multipath, multipathbase, multipathcount, lsp=None,
                             standard=None, size=None, padReply=False,
                             tos=None, **kwargs ):
   nextHopsAndLabels, err, rsvpSpIds, rsvpSenderAddr, _ = \
      getRsvpFec( mount, session, lsp )
   if nextHopsAndLabels is None or rsvpSpIds is None:
      return err
   # The CLI/script enforces that either a specific LSP is specified (session + LSP
   # IDs) or that a name is specified. In the latter case, it is possible that we
   # find more than on matching LSP, in this case we reject the input and will later
   # display the potential matches to the user (BUG286407)
   if len( rsvpSpIds ) > 1:
      print 'More than one match for session "%s", use arguments `id` and `lsp`' \
            % ( session )
      return errno.EINVAL
   # For traceroute, it can be only one rsvpSpId. The CLI wouldn't allow to get
   # here with more than one LSP.
   rsvpSpId = rsvpSpIds[ 0 ]
   nexthop, label, _ = nextHopsAndLabels[ 0 ]
   dstIp = rsvpSpId.sessionId.dstIp
   state = getGlobalState()
   # Map L3 nexthop to L2 nexthop
   nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthop )
   if not ( nexthopIntf and nexthopEthAddr ):
      print 'via not found or not resolved'
      return errno.EINVAL

   labelStack = label if isinstance( label, list ) else [ label ]
   # Add the entropy labels if EL push is enabled
   ipv = IPV6 if isIpv6Addr( dstIp ) else IPV4
   if mount.mplsTunnelConfig.entropyLabelPush:
      udpPam = state.lspPingClientRootUdpPam
      el = generateMplsEntropyLabel(
            srcIp=( src or getIntfPrimaryIpAddr( mount, nexthopIntf, ipv ) ),
            dstIp=dstIp,
            udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
      labelStack.extend( [ ELI, el ] )

   print '  via {}, label stack: {}'.format( nexthop, labelStack )
   l3Intf, nhMac = nexthopIntf, nexthopEthAddr

   # The label list could potentially contain the implicit null label. We leave
   # it in the list above when we pass it to the clientIdToVias because it is
   # nice and consistent to see it in the ping output, however we remove the
   # label before passing it to viaInfo so that the rest of the processing does
   # not have to special case it (the label isn't actually in the stack).
   # However, if the list ONLY contains the implicit null, we leave it in,
   # since the client config needs to have at least one label.
   if any( label != 3 for label in labelStack ):
      labelStack = [ l for l in labelStack if l != 3 ]

   dsMappingInfo = getDsMappingInfo( nexthop, labelStack,
                                     getL3IntfMtu( l3Intf, mount ),
                                     multipath, multipathbase, multipathcount )
   retVal = traceroute( mount, state, l3Intf, labelStack, src, dst,
                        smac, nhMac, 1, interval=5, ipv=ipv, rsvpSpId=rsvpSpId,
                        rsvpSenderAddr=rsvpSenderAddr,
                        dsMappingInfo=dsMappingInfo, tc=tc, standard=standard,
                        size=size, padReply=padReply, tos=tos ).retVal
   return retVal



# ---------------------------------------------------------
#                   LspTraceroute LDP
# ---------------------------------------------------------
def handleLspTracerouteLdp( prefix, mount, label, nexthop, src, dst, vrf,
                            smac, dmac, count, interval, verbose, tc,
                            multipath, multipathbase, multipathcount,
                            maxTtl=None, standard=None, size=None, padReply=False,
                            dstype=None, tos=None, **kwargs ):
   if label is not None:
      print 'Non-empty label arg provided'
      return errno.EINVAL

   protocol = kwargs[ 'type' ]
   # nextHopsAndLabels is a dictionary of NextHopAndLabel objects
   # keyed by a str generated using the getNextHopAndLabelKey function.
   nextHopsAndLabels, err = getLdpFec( mount, prefix )
   if not nextHopsAndLabels:
      return err

   res = 0
   # If we are unable to resolve any vias, we want to print an error at the end
   resolvedAnyVia = False

   # this loop will run multiple times only for multipath LDP/SR traceroute
   # with ECMP at origin
   for nhlEntry in nextHopsAndLabels:
      state = getGlobalState()
      ipv = IPV6 if isIpv6Addr( prefix ) else IPV4
      nextHopIp = nhlEntry.nextHopIp
      intfId = nhlEntry.intfId
      labelStack = ( nhlEntry.label if isinstance( nhlEntry.label, list ) else
                     [ nhlEntry.label ] )

      nextHopIntf, nextHopEthAddr = resolveNexthop( mount, state, nextHopIp,
                                                    intf=intfId )
      if not ( nextHopIntf and nextHopEthAddr ):
         # If it's multipath, we'd print the error and try next path. If it's regular
         # traceroute, we'd only print the error at end if we can't resolve any via
         if multipath:
            errOutput = 'via not found or not resolved for next-hop IP {}'.format(
                                                                          nextHopIp )
            print errOutput
         continue
      resolvedAnyVia = True

      # Generate the entropy label stack (will be empty if push is disabled) and add
      # it to the given label stack
      if mount.mplsTunnelConfig.entropyLabelPush:
         udpPam = state.lspPingClientRootUdpPam
         genPrefix = IpGenPrefix( str( prefix ) )
         el = generateMplsEntropyLabel(
               srcIp=( src or getIntfPrimaryIpAddr( mount, nextHopIntf, ipv ) ),
               dstIp=( genPrefix.v4Addr if ipv == IPV4 else genPrefix.v6Addr ),
               udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
         labelStack.extend( [ ELI, el ] )

      output = '  via {}, label stack: {}'.format( nextHopIp, labelStack )
      print output

      multipathcount = multipathcount if multipath else 0
      dsMappingInfo = getDsMappingInfo( str( nextHopIp ), labelStack,
                                        getL3IntfMtu( nextHopIntf, mount ),
                                        multipath, multipathbase, multipathcount )
      sys.stdout.flush()

      # Strip the implicit nulls from the label stack that will be going on the wire
      # if there are other labels
      if any( label != 3 for label in labelStack ):
         labelStack = [ l for l in labelStack if l != 3 ]

      res, _, _ = traceroute( mount, state, nextHopIntf, labelStack,
                              src, dst, smac, nextHopEthAddr, 1,
                              interval=interval, hops=maxTtl,
                              prefix=prefix, ipv=ipv, protocol=protocol,
                              dsMappingInfo=dsMappingInfo, tc=tc,
                              multipath=multipath, nextHopIp=nextHopIp,
                              standard=standard, size=size, tos=tos,
                              padReply=padReply, dstype=dstype )

      if not multipath:
         return res

   if not multipath and not resolvedAnyVia:
      print 'via not found or not resolved'
      return errno.EINVAL
   return res

# ---------------------------------------------------------
#                   LspTraceroute MLDP and SR
# ---------------------------------------------------------
def handleLspTracerouteMldpSr( prefix, mount, label, nexthop, src, dst, vrf,
                               smac, dmac, count, interval, verbose, tc,
                               genOpqVal=None, sourceAddrOpqVal=None,
                               groupAddrOpqVal=None, jitter=None,
                               responderAddr=None, maxTtl=None,
                               standard=None, size=None, padReply=False,
                               dstype=None, tos=None, **kwargs ):
   if label is not None:
      print 'Non-empty label arg provided'
      return errno.EINVAL

   protocol = kwargs[ 'type' ]
   # nextHopsAndLabels is a dictionary of NextHopAndLabel objects
   # keyed by a str generated using the getNextHopAndLabelKey function.
   nextHopsAndLabels, err = getProtocolIpFec( mount, prefix, kwargs.get( 'type' ),
                                              genOpqVal, sourceAddrOpqVal,
                                              groupAddrOpqVal )
   if not nextHopsAndLabels:
      return err

   res = 0
   # If we are unable to resolve any vias, we want to print an error at the end
   resolvedAnyVia = False

   # this loop will run multiple times only for multipath LDP/SR traceroute
   # with ECMP at origin
   for nhlEntry in nextHopsAndLabels:
      state = getGlobalState()
      ipv = IPV6 if isIpv6Addr( prefix ) else IPV4
      nextHopIp = nhlEntry.nextHopIp
      intfId = nhlEntry.intfId
      labelStack = ( nhlEntry.label if isinstance( nhlEntry.label, list ) else
                     [ nhlEntry.label ] )

      nextHopIntf, nextHopEthAddr = resolveNexthop( mount, state, nextHopIp,
                                                    intf=intfId )
      if not ( nextHopIntf and nextHopEthAddr ):
         # For regular traceroute, we'd only print the error
         # at end if we can't resolve any via
         continue
      resolvedAnyVia = True

      # Generate the entropy label stack (will be empty if push is disabled) and add
      # it to the given label stack
      if mount.mplsTunnelConfig.entropyLabelPush:
         udpPam = state.lspPingClientRootUdpPam
         genPrefix = IpGenPrefix( str( prefix ) )
         el = generateMplsEntropyLabel(
               srcIp=( src or getIntfPrimaryIpAddr( mount, nextHopIntf, ipv ) ),
               dstIp=( genPrefix.v4Addr if ipv == IPV4 else genPrefix.v6Addr ),
               udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
         labelStack.extend( [ ELI, el ] )

      output = '  via {}, label stack: {}'.format( nextHopIp, labelStack )
      print output

      dsMappingInfo = getDsMappingInfo( str( nextHopIp ), labelStack,
                                        getL3IntfMtu( nextHopIntf, mount ),
                                        multipath=False, baseip='127.0.0.0',
                                        numMultipathBits=0 )
      mldpInfo = MldpInfo( genOpqVal, sourceAddrOpqVal, groupAddrOpqVal,
                           jitter, responderAddr )
      sys.stdout.flush()

      # Strip the implicit nulls from the label stack that will be going on the wire
      # if there are other labels
      if any( label != 3 for label in labelStack ):
         labelStack = [ l for l in labelStack if l != 3 ]

      startTime = Tac.now()
      res, txPkts, replyHostRtts = traceroute( mount, state, nextHopIntf, labelStack,
                                               src, dst, smac, nextHopEthAddr, 1,
                                               interval=interval, hops=maxTtl,
                                               prefix=prefix, ipv=ipv,
                                               protocol=protocol,
                                               dsMappingInfo=dsMappingInfo, tc=tc,
                                               nextHopIp=nextHopIp,
                                               mldpInfo=mldpInfo, standard=standard,
                                               size=size, tos=tos,
                                               padReply=padReply, dstype=dstype )
      time = int( ( Tac.now() - startTime ) * 1000 ) # microsecond

      if protocol == 'mldp':
         # traceroute statistics support only for mldp for now.
         lspTracerouteLabelDistStatisticsRender( time, txPkts, replyHostRtts,
                                                 protocol, prefix,
                                                 [ ( nextHopIp, labelStack,
                                                     nextHopIntf ) ] )
         return res

   if not resolvedAnyVia:
      print 'via not found or not resolved'
      return errno.EINVAL
   return res

# ---------------------------------------------------------
#                   LspTraceroute raw
# ---------------------------------------------------------

def handleLspTracerouteRaw( prefix, mount, label, src, dst, smac, dmac,
                            interface, count, interval, verbose,
                            nexthop=None, tc=None, standard=None, size=None,
                            padReply=False, tos=None, **kwargs ):
   state = getGlobalState()
   if dmac is None or interface is None:
      if nexthop:
         interface, dmac = resolveNexthop( mount, state, nexthop )
      if interface is None or dmac is None:
         print "Failed to find a valid output interface"
         return errno.EINVAL 

   # Translation from default Ipv4 to default Ipv6 if not given
   ipv = IPV4
   if ( prefix and isIpv6Addr( prefix ) or
        nexthop and isIpv6Addr( nexthop ) or
        src and isIpv6Addr( src ) ):
      ipv = IPV6

   retVal = traceroute( mount, state, interface, label, src, dst, smac, dmac,
                        count, interval, verbose, ipv=ipv, tc=tc,
                        standard=standard, size=size, padReply=padReply,
                        tos=tos ).retVal
   return retVal
