# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
from IpLibConsts import DEFAULT_VRF
from ClientCore import ping
from ClientCommonLib import (
   AddressFamily,
   resolveNexthop,
   getNhgId, 
   labelStackToList,
   resolveHierarchical,
   isIpv6Addr, 
   getTunnelNhgName,
   _resolveSrTePolicyTunnels,
   generateMplsEntropyLabel,
   getStaticFec,
   isNexthopGroupVia, 
   getNexthopGroupId,
   getNhgIdToName,
   isNexthopGroupTunnelVia,
   DynTunnelIntfId,
   resolveNhgTunnelFibEntry,
   getIntfPrimaryIpAddr,
   getRsvpFec,
   getProtocolIpFec,
   getPwLdpFec,
   IPV4, IPV6, MldpInfo,
   getBgpLuTunnelFibEntry,
   getNhAndLabelFromTunnelFibVia,
   validateBgpLuResolvedPushVia,
   LspPingTypeBgpLu,
)

from MplsPingClientLib import (
   lspPingBgpLuReplyRender,
   lspPingBgpLuStatisticsRender,
   lspPingStaticReplyRender,
   lspPingStaticStatisticsRender,
   lspPingNhgReplyRender,
   lspPingNhgStatisticsRender,
   lspPingPwLdpReplyRender,
   lspPingPwLdpStatisticsRender,
   lspPingSrTeReplyRender,
   lspPingSrTeReplyRenderWithoutCode,
   lspPingSrTeStatisticsRender,
   lspPingRawReplyRender,
   lspPingRawStatisticsRender,
   lspPingRsvpReplyRender,
   lspPingRsvpStatisticsRender,
   lspPingLabelDistReplyRender,
   lspPingLabelDistStatisticsRender,
   SrTePingRenderArgs, 
   LspPingNhgModeAllEntries, 
   LspPingNhgModeOneEntry
)

from ClientState import getGlobalState
from ForwardingHelper import getNhgSize

from PseudowireLib import (
   PwPingCcType,
   PwPingCvType,
   pwPingCcEnumToVal,
   pwPingCvEnumToVal,
   pwPingCcMsg,
   pwPingCvMsg,
)

import collections
import Toggles.PseudowireToggleLib
from TypeFuture import TacLazyType

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
NexthopGroupType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )

ELI = TacLazyType( 'Arnet::MplsLabel' ).entropyLabelIndicator

# ---------------------------------------------------------
#                   LspPing BGP LU
# ---------------------------------------------------------

def handleLspPingBgpLu( prefix, mount, src=None, dst=None, smac=None, dmac=None,
                        vrf=None, interface=None, interval=1, count=None,
                        label=None, verbose=False, entry=None, tc=None,
                        nexthop=None, standard=None, size=None, padReply=False,
                        egressValidateAddress=None, multiLabel=1, tos=None,
                        **kwargs ):
   state = getGlobalState()
   viaInfo, clientIdToVias = [], {}
   resolvedPushVia = collections.defaultdict( list )
   unresolvedPushVia = []
   ipv = IPV6 if isIpv6Addr( str( prefix ) ) else IPV4

   tunnelFibEntry, err = getBgpLuTunnelFibEntry( mount, prefix )
   if err:
      print err
      return errno.EINVAL

   for tunnelVia in tunnelFibEntry.tunnelVia.itervalues():
      nextHopAndLabel, err = getNhAndLabelFromTunnelFibVia( mount, tunnelVia )
      if err:
         print err
         return errno.EINVAL
      nexthopIp = nextHopAndLabel.nextHopIp
      labels = nextHopAndLabel.label
      intfId = nextHopAndLabel.intfId

      # If nexthop and/or label stack are specified, use only the via that has
      # that nexthop/label stack.
      if nexthop and IpGenAddr( nexthop ) != nexthopIp:
         continue
      if label and label != labels:
         continue

      nexthopAddr = nexthopIp.v4Addr if ipv == IPV4 else nexthopIp.v6Addr
      # Map L3 nexthop to L2 nexthop
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthopAddr,
                                                    intf=intfId )
      if not ( nexthopIntf and nexthopEthAddr ):
         unresolvedPushVia.append( ( nexthopIp, labels ) )
         continue
      else:
         key = ( nexthopIntf, tuple( labels ), nexthopEthAddr )
         resolvedPushVia[ key ].append( ( nexthopIp, labels, nexthopIntf ) )

   if nexthop:
      err = validateBgpLuResolvedPushVia( resolvedPushVia, nexthop, label )
      if err:
         print err
         return errno.EINVAL

   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVias[ idx ] = val
      viaInfo.append( key )

   if not viaInfo:
      print 'Failed to find a valid output interface'
      return errno.EINVAL

   timeout = 5
   print '   timeout is {}ms, interval is {}ms'.format( timeout * 1000,
                                                        interval * 1000 )

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingBgpLuReplyRender, lspPingBgpLuStatisticsRender,
                renderArg=( clientIdToVias, prefix, 'BGP labeled unicast',
                            unresolvedPushVia ),
                ipv=ipv, tc=tc, standard=standard, size=size,
                protocol=LspPingTypeBgpLu, padReply=padReply, tos=tos )

# ---------------------------------------------------------
#                   LspPing nexthop-group
# ---------------------------------------------------------

def handleLspPingNhg( nhgNames, mount, prefix=None, src=None, dst=None, 
                      smac=None, dmac=None, vrf=None, interface=None, interval=1, 
                      count=None, label=None, verbose=False, entry=None, tc=None,
                      nhgNameToNhgTunnelIdx=None, standard=None,
                      size=None, padReply=False, egressValidateAddress=None,
                      multiLabel=1, tos=None, **kwargs ):
   state = getGlobalState()
   idxBase = 0
   viaInfo, clientIdToTunnels, clientIdBaseToNhgName = [], {}, {}
   nhgNameToClientIds, nhgNameToUnresolvedTunnels = {}, {}
   IPv4 = AddressFamily.ipv4
   if not isinstance( nhgNames, list ):
      nhgNames = [ nhgNames ]

   #pylint: disable-msg=too-many-nested-blocks
   for nhgName in nhgNames:
      nhgId = getNhgId( nhgName, mount )
      if nhgId is None:
         print 'Nexthop-group %s does not exist.' % nhgName
         return errno.EINVAL
      # we are expecting a programmed MPLS nexthop-group
      nhgConfig = mount.routingNhgConfig.nexthopGroup[ nhgName ]
      if nhgConfig.type != NexthopGroupType.mpls:
         print 'Nexthop-group %s is not in MPLS type.' % nhgName
         return errno.EINVAL
      # mapping from adj (meaning labelStack + l2Adj) to tunnels
      adjToTunnels = {}
      unresolvedTunnels = []
      nhgMplsLabelStack = nhgConfig.mplsLabelStack
      # XXX VRF
      hwVrfStatus = mount.routingHwNexthopGroupStatus.vrfStatus.get( DEFAULT_VRF )
      if not hwVrfStatus:
         print 'Cannot find next-hop group status.'
         return errno.EINVAL
      if nhgId not in hwVrfStatus.nexthopGroupAdjacency:
         print 'Nexthop-group %s is not programmed.' % nhgName
         return errno.EINVAL

      nhgAdjVia = hwVrfStatus.nexthopGroupAdjacency.get( nhgId ).via
      if not nhgAdjVia:
         print 'Via for Nexthop-group {} cannot be resolved'.format( nhgName )
         return errno.EINVAL

      entrySize = getNhgSize( nhgConfig )

      if entry is None or entry < 0:
         mode = LspPingNhgModeAllEntries
         entries = range( entrySize )
      else:
         mode = LspPingNhgModeOneEntry
         entries = [ entry ]

      TunnInfo = collections.namedtuple( 'tunnelInfo', 'entryIndex nexthop' )
      for tunEntry in entries:
         entryResolved = False
         via = nhgAdjVia.get( tunEntry )
         if via and via.l2Via.vlanId != 0:
            # resolved tunnel
            ipv = IPV6 if isIpv6Addr( via.hop.stringValue ) else IPV4
            key = ( nhgMplsLabelStack[ tunEntry ], via.l2Via.macAddr, via.l3Intf )
            tunnInfo = TunnInfo( tunEntry, str( via.hop ) )
            adjToTunnels.setdefault( key, [] ).append( tunnInfo )
            entryResolved = True
         elif via and via.nextLevelFecIndex:
            ipv = IPV6 if isIpv6Addr( via.route.stringValue ) else IPV4
            nextLevelFecId = via.nextLevelFecIndex
            nextHops, _ = resolveHierarchical( mount, fecId=nextLevelFecId )
            if nextHops:
               for nexthop in nextHops:
                  nexthop = nexthop.v4Addr if nexthop.af == IPv4 else \
                            nexthop.v6Addr
                  nhIntf, nhMac = resolveNexthop( mount, state, nexthop )
                  if not nhIntf or not nhMac:
                     continue
                  key = ( nhgMplsLabelStack[ tunEntry ], nhMac, nhIntf )
                  tunnInfo = TunnInfo( tunEntry, str( nexthop ) )
                  adjToTunnels.setdefault( key, [] ).append( tunnInfo )
                  entryResolved = True
         if not entryResolved:
            unresolvedTunnels.append( tunEntry )

      nhgNameToUnresolvedTunnels[ nhgName ] = unresolvedTunnels

      if not adjToTunnels:
         if len( entries ) == entrySize:
            errMsg = 'no entry resolved'
         else:
            errMsg = 'entry %d not resolved' % entries[ 0 ]
         print 'nexthop-group %s: %s' % ( nhgName, errMsg )
         return errno.EINVAL

      tunnelsToAdj = { tuple( v ) : k for k, v in adjToTunnels.items() }
      sortedKeys = sorted( tunnelsToAdj.keys(),
                           key=lambda entries: entries[ 0 ].entryIndex )
      nhgNameToClientIds[ nhgName ] = []
      clientIdBaseToNhgName[ idxBase ] = nhgName
      for idx, entries in enumerate( sortedKeys ):
         clientIdToTunnels[ idxBase + idx ] = entries
         labelStack, nhMacAddr, l3Intf = tunnelsToAdj[ entries ]
         viaInfo.append( ( l3Intf, labelStackToList( labelStack ), nhMacAddr ) )
         nhgNameToClientIds[ nhgName ].append( idxBase + idx )
      idxBase += len( sortedKeys )

   #pylint: enable-msg=too-many-nested-blocks
   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingNhgReplyRender, lspPingNhgStatisticsRender,
                ( clientIdToTunnels, nhgNames, nhgNameToClientIds,
                  nhgNameToUnresolvedTunnels, clientIdBaseToNhgName,
                  prefix, mode, nhgNameToNhgTunnelIdx ),
                ipv=ipv, tc=tc, standard=standard, size=size, padReply=padReply,
                egressValidateAddress=egressValidateAddress, tos=tos,
                lookupLabelCount=multiLabel )

# ---------------------------------------------------------
#                LspPing Nexthop Group Tunnel
# ---------------------------------------------------------

def handleLspPingNhgTunnel( endpoint, mount, src=None, dst=None, smac=None,
                            dmac=None, vrf=None, interface=None, interval=1,
                            count=None, label=None, verbose=False, entry=None,
                            tc=None, standard=None, size=None, padReply=False,
                            egressValidateAddress=None, multiLabel=1, tos=None,
                            **kwargs ):
   ( nhgName, tunnelIndex, err ) = getTunnelNhgName( mount, endpoint )
   if err is not None:
      print err
      return errno.EINVAL

   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint

   return handleLspPingNhg( [ nhgName ], mount, prefix=endpoint, src=src, 
                            dst=dst, smac=smac, dmac=dmac, vrf=vrf, entry=entry, 
                            tc=tc, interval=interval, count=count, verbose=verbose, 
                            nhgNameToNhgTunnelIdx={ nhgName : tunnelIndex },
                            standard=standard, size=size, padReply=padReply,
                            egressValidateAddress=egressValidateAddress,
                            multiLabel=multiLabel, tos=tos )

# ---------------------------------------------------------
#                   LspPing static
# ---------------------------------------------------------

def handleLspPingStatic( prefix, mount, src=None, dst=None, smac=None, 
                         dmac=None, vrf=None, interval=1, count=None, verbose=False,
                         label=None, interface=None, entry=None, tc=None,
                         standard=None, size=None, padReply=False,
                         egressValidateAddress=None, multiLabel=1, tos=None,
                         **kwargs ):
   fec, err = getStaticFec( mount, prefix )
   if fec is None:
      return err

   if not fec.via:
      print 'No adj found for prefix'
      return errno.EINVAL

   state = getGlobalState()
   resolvedPushVia = {}
   unresolvedPushVia = []
   nhgNames = []
   nhgNameToNhgTunnelIdx = {}

   for idx in range( len( fec.via ) ):
      via = fec.via[ idx ]
      if isNexthopGroupVia( via ):
         nexthopGroupId = getNexthopGroupId( via )
         nhgName = getNhgIdToName( nexthopGroupId, mount )
         if not nhgName:
            print 'No nexthop-group with id %d' % nexthopGroupId
            return errno.EINVAL
         nhgNames.append( nhgName )
         continue
      if isNexthopGroupTunnelVia( via ):
         tunnelId = DynTunnelIntfId.tunnelId( via.intfId )
         ( nhgName, tunnelIndex, err ) = resolveNhgTunnelFibEntry( mount, tunnelId )
         if err is not None:
            print err
            return errno.EINVAL
         nhgNameToNhgTunnelIdx[ nhgName ] = tunnelIndex
         nhgNames.append( nhgName )
         continue
      if not MplsLabel( via.mplsLabel ).isValid():
         continue
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, via.hop )
      if not ( nexthopIntf and nexthopEthAddr ):
         unresolvedPushVia.append( ( via.hop, via.mplsLabel ) )
         continue
      key = ( nexthopIntf, via.mplsLabel, nexthopEthAddr )
      resolvedPushVia.setdefault( key, [] ).append( ( via.hop, via.mplsLabel,
                                                      nexthopIntf ) )

   if nhgNames:
      return handleLspPingNhg( nhgNames, mount, prefix=prefix, src=src, dst=dst, 
                               smac=smac, dmac=dmac, vrf=vrf, interval=interval, 
                               count=count, verbose=verbose, tc=tc,
                               nhgNameToNhgTunnelIdx=nhgNameToNhgTunnelIdx,
                               standard=standard, size=size, padReply=padReply,
                               egressValidateAddress=egressValidateAddress,
                               multiLabel=multiLabel, tos=tos )
   if not resolvedPushVia:
      print 'No via resolved'
      return errno.EINVAL

   viaInfo, clientIdToVia = [], {}
   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVia[ idx ] = val
      viaInfo.append( key )
   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingStaticReplyRender, lspPingStaticStatisticsRender, 
                ( clientIdToVia, prefix, unresolvedPushVia ), ipv=ipv, tc=tc,
                standard=standard, size=size, padReply=padReply, tos=tos,
                egressValidateAddress=egressValidateAddress,
                lookupLabelCount=multiLabel )

# ---------------------------------------------------------
#                   LspPing Ldp Pseudowire
# ---------------------------------------------------------

def handleLspPingPwLdp( pwLdpName, mount, prefix=None, src=None, dst=None, 
                        smac=None, dmac=None, vrf=None, interface=None, interval=1, 
                        count=None, label=None, verbose=False, entry=None, tc=None,
                        standard=None, size=None, padReply=False, tos=None,
                        **kwargs ):

   pwLdpInfo, nextHopAndLabel, err = getPwLdpFec( mount, pwLdpName )
   if pwLdpInfo is not None:
      infoMsg = 'Neighbor {}, pseudowire id {}'.format( pwLdpInfo.remoteRouterId,
                                                        pwLdpInfo.pwId )
      if pwLdpInfo.vcLabel:
         infoMsg += ', pseudowire label {}'.format( pwLdpInfo.vcLabel )
      print infoMsg
   if err is not None:
      print err
      return errno.EINVAL
   nexthop = nextHopAndLabel.nextHopIp
   labels = nextHopAndLabel.label
   intfId = nextHopAndLabel.intfId

   # Currently, we only support LSP ping as CV type
   if not pwLdpInfo.cvTypes & pwPingCvEnumToVal[ PwPingCvType.lspPing ]:
      msg = ( 'Neighbor does not support using {} as a connectivity verification'
              ' mechanism.' ).format( pwPingCvMsg[ PwPingCvType.lspPing ] )
      print msg
      return errno.EOPNOTSUPP

   state = getGlobalState()
   # Print the CC Type being used and update pwLdpInfo to identify active CC type.
   # Currently, we support Control Word or RA label as CC Types. For TTL Expiry,
   # our hardware does not support pipe mode at ingress. We need pipe mode at both
   # ingress and PHP node to correctly process the VC label.
   if ( pwLdpInfo.controlWord and
        Toggles.PseudowireToggleLib.togglePseudowirePingCwEnabled() ):
      if pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.cw ]:
         print '   ' + 'CC type: {}'.format( pwPingCcMsg[ PwPingCcType.cw ] )
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.cw ] )
      elif pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.ra ]:
         # Add router alert label to the stack
         labels.append( MplsLabel.routerAlert )
         print '   ' + 'CC type: {}'.format( pwPingCcMsg[ PwPingCcType.ra ] )
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.ra ] )
      else:
         msg = ( 'Neighbor does not support using either {} or {} as connectivity'
                 ' check mechanisms.' ).format( pwPingCcMsg[ PwPingCcType.cw ],
                                                pwPingCcMsg[ PwPingCcType.ra ] )
         print msg
         return errno.EOPNOTSUPP
   else:
      if pwLdpInfo.ccTypes & pwPingCcEnumToVal[ PwPingCcType.ra ]:
         # Add router alert label to the stack
         labels.append( MplsLabel.routerAlert )
         print '   ' + 'CC type: {}'.format( pwPingCcMsg[ PwPingCcType.ra ] )
         pwLdpInfo = pwLdpInfo._replace(
                        ccTypes=pwPingCcEnumToVal[ PwPingCcType.ra ] )
      else:
         msg = ( 'Neighbor does not support using {} as a connectivity'
                 ' check mechanism.' ).format( pwPingCcMsg[ PwPingCcType.ra ] )
         print msg
         return errno.EOPNOTSUPP

   # Add VC label to the stack and remove Implicit Null if it's there
   labels.append( pwLdpInfo.vcLabel )
   if labels[ 0 ] == MplsLabel.implicitNull:
      labels = labels[ 1 : ]

   # Map L3 nexthop to L2 nexthop
   resolvedPushVia = {}
   unresolvedPushVia = []
   nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthop.v4Addr, 
                                                 intf=intfId )
   if not ( nexthopIntf and nexthopEthAddr ):
      unresolvedPushVia.append( ( nexthop, labels ) )
   else:
      key = ( nexthopIntf, tuple( labels ), nexthopEthAddr )
      resolvedPushVia.setdefault( key, [] ).append(
         ( nexthop, labels, nexthopIntf ) )

   viaInfo, clientIdToVias = [], {}
   for idx, ( key, val ) in enumerate( resolvedPushVia.items() ):
      clientIdToVias[ idx ] = val
      viaInfo.append( key )

   if not viaInfo:      # No resolved interface
      print "Failed to find a valid output interface"
      return errno.EINVAL

   ipv = IPV6 if isIpv6Addr( pwLdpInfo.remoteRouterId ) else IPV4
   timeout = 5
   print '   timeout is {}ms, interval is {}ms'.format( timeout * 1000,
                                                        interval * 1000 )

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingPwLdpReplyRender, lspPingPwLdpStatisticsRender,
                ( clientIdToVias, pwLdpName, pwLdpInfo ),
                ipv=ipv, pwLdpInfo=pwLdpInfo, tc=tc, standard=standard, size=size,
                padReply=padReply, tos=tos )

# ---------------------------------------------------------
#                   LspPing Sr-Te Tunnels
# ---------------------------------------------------------

def handleLspPingSrTe( endpoint, mount, color, trafficAf=None, src=None, 
                       dst=None, smac=None, dmac=None, vrf=None, interface=None,
                       interval=1, count=None, label=None, verbose=False, tc=None,
                       standard=None, size=None, padReply=False, tos=None,
                       egressValidateAddress=None, **kwargs ):
   clientIdToTunnels = []
   viaInfo = []
   ipv = IPV6 if isIpv6Addr( str( endpoint ) ) else IPV4
   if trafficAf:
      ipv = IPV6 if trafficAf == 'v6' else IPV4
   tunnelToAdjacencies = _resolveSrTePolicyTunnels( mount, endpoint, color,
                                                    trafficAf=trafficAf )
   if tunnelToAdjacencies == errno.EINVAL:
      return errno.EINVAL

   state = getGlobalState()

   for tunnel in sorted( tunnelToAdjacencies.keys() ):
      # If MPLS EL push is enabled, we want to add the ELI and EL to the tunnel's
      # labelStack and save that to be used for rendering and for the ping request.
      labelStack, nhMac, l3Intf = tunnelToAdjacencies[ tunnel ]
      # Convert from a tuple to a list to operate on
      labelStack = list( labelStack )
      if mount.mplsTunnelConfig.entropyLabelPush:
         udpPam = state.lspPingClientRootUdpPam
         el = generateMplsEntropyLabel(
               srcIp=( src or getIntfPrimaryIpAddr( mount, l3Intf, ipv ) ),
               dstIp=endpoint,
               udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
         labelStack.extend( [ ELI, el ] )
      # Update the tunnel tuple  with the entropy label stack
      # Tunnel looks like tuple( tuple( nh, tuple( labelStack ) ) )
      tunnel = ( ( tunnel[ 0 ][ 0 ], tuple( labelStack ) ), )
      clientIdToTunnels.append( tunnel )
      viaInfo.append( ( l3Intf, labelStack, nhMac ) )

   renderArgs = SrTePingRenderArgs( clientIdToTunnels=clientIdToTunnels,
                                    endpoint=endpoint, color=color )

   replyRenderMethod = ( lspPingSrTeReplyRender if egressValidateAddress else
                         lspPingSrTeReplyRenderWithoutCode )
   # If the user does not provide any egress address, we consider the endpoint
   # as the egressValidateAddress by default.
   if egressValidateAddress == 'default':
      egressValidateAddress = endpoint

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                replyRenderMethod, lspPingSrTeStatisticsRender,
                renderArgs, ipv=ipv, tc=tc, standard=standard, size=size,
                padReply=padReply, egressValidateAddress=egressValidateAddress,
                tos=tos )

# ---------------------------------------------------------
#                   LspPing raw
# ---------------------------------------------------------

def handleLspPingRaw( prefix, mount, label, smac, dmac, interface, src, dst,
                      interval, count, verbose, nexthop, tc, standard=None,
                      size=None, padReply=False, tos=None, **kwargs ):
   state = getGlobalState()
   if dmac is None or interface is None:
      if nexthop:
         interface, dmac = resolveNexthop( mount, state, nexthop )
      if interface is None or dmac is None:
         print "Failed to find a valid output interface"
         return errno.EINVAL

   # Translation from default Ipv4 to default Ipv6 if not given
   ipv = IPV4
   if ( prefix and isIpv6Addr( prefix ) or
        nexthop and isIpv6Addr( nexthop ) or
        src and isIpv6Addr( src ) ):
      ipv = IPV6
   viaInfo = [ ( interface, label, dmac ) ]

   return ping( mount, state, src, dst, smac, interval, count, viaInfo,
                lspPingRawReplyRender, lspPingRawStatisticsRender, prefix, ipv=ipv,
                tc=tc, standard=standard, size=size, padReply=padReply, tos=tos )

# ---------------------------------------------------------
#                   LspPing Rsvp
# ---------------------------------------------------------

def handleLspPingRsvp( prefix, mount, src=None, dst=None, smac=None, 
                       dmac=None, vrf=None, interval=0, count=None, verbose=False,
                       label=None, interface=None, entry=None, tc=None, 
                       session=None, lsp=None, standard=None, size=None,
                       padReply=False, tos=None, **kwargs ):
   nextHopsAndLabels, err, rsvpSpIds, rsvpSenderAddr, lspIds = \
      getRsvpFec( mount, session, lsp )
   if nextHopsAndLabels is None or rsvpSpIds is None:
      return err

   state = getGlobalState()
   # All SPs have the same destination, so just take the first one.
   prefix = rsvpSpIds[ 0 ].sessionId.dstIp
   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4

   viaInfo, clientIdToVias, clientIdToLsp = [], {}, {}
   for idx in range( 0, len( rsvpSpIds ) ):
      # Map L3 nexthop to L2 nexthop
      nexthop, label, _ = nextHopsAndLabels[ idx ]
      # label can be a list or single label. Converting it to a list here makes the
      # rest of the logic simpler
      labelStack = [ label ] if not isinstance( label, list ) else label
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthop )
      if nexthopIntf and nexthopEthAddr:
         # Generate the EL by hashing over the relevant fields
         if mount.mplsTunnelConfig.entropyLabelPush:
            udpPam = state.lspPingClientRootUdpPam
            genPrefix = IpGenPrefix( str( prefix ) )
            el = generateMplsEntropyLabel(
                  srcIp=( src or getIntfPrimaryIpAddr( mount, nexthopIntf, ipv ) ),
                  dstIp=( genPrefix.v4Addr if ipv == IPV4 else genPrefix.v6Addr ),
                  udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
            labelStack.extend( [ ELI, el ] )
         clientIdToVias[ idx ] = [ ( nexthop, labelStack, nexthopIntf ) ]
         # The label list could potentially contain the implicit null label. We leave
         # it in the list above when we pass it to the clientIdToVias because it is
         # nice and consistent to see it in the ping output, however we remove the
         # label before passing it to viaInfo so that the rest of the processing does
         # not have to special case it (the label isn't actually in the stack).
         # However, if the list ONLY contains the implicit null, we leave it in,
         # since the client config needs to have at least one label.
         if any( label != 3 for label in labelStack ):
            labelStack = [ l for l in labelStack if l != 3 ]

         viaInfo.append( ( nexthopIntf, labelStack, nexthopEthAddr ) )
         # No need to save the lspIds if the lsp was specified in the ping command.
         if not lsp:
            clientIdToLsp[ idx ] = lspIds[ idx ]

   if not viaInfo:      # No resolved interface
      print "Failed to find a valid output interface"
      return errno.EINVAL

   timeout = 5
   print '   timeout is {}ms, interval is {}ms'.format( timeout*1000, 
                                                        interval*1000 )
   return ping( mount, state, src, dst, smac, interval, count, viaInfo, 
                lspPingRsvpReplyRender, lspPingRsvpStatisticsRender, 
                renderArg=( clientIdToVias, prefix, 'RSVP', clientIdToLsp ),
                ipv=ipv, rsvpSpIds=rsvpSpIds, rsvpSenderAddr=rsvpSenderAddr,
                timeout=timeout, tc=tc, standard=standard, size=size,
                padReply=padReply, tos=tos )

# --------------------------------------------------------------------
#                   LspPing LDP, MLDP and Segment Routing
# --------------------------------------------------------------------

def handleLspPingLdpMldpSr( prefix, mount, src=None, dst=None, smac=None, 
                            dmac=None, vrf=None, interval=0, count=None, 
                            verbose=False, label=None, interface=None, entry=None, 
                            tc=None, genOpqVal=None, sourceAddrOpqVal=None,
                            groupAddrOpqVal=None, jitter=None, responderAddr=None,
                            standard=None, size=None, padReply=False, tos=None,
                            **kwargs ):
   protocol = kwargs[ 'type' ]
   state = getGlobalState()
   nextHopsAndLabels, err = getProtocolIpFec( mount, prefix, protocol,
                                              genOpqVal, sourceAddrOpqVal,
                                              groupAddrOpqVal )

   ipv = IPV6 if isIpv6Addr( prefix ) else IPV4
   if not nextHopsAndLabels:
      return err

   # always use first via from given fecVia list
   nhlEntry = nextHopsAndLabels[ 0 ]
   nextHopIp = nhlEntry.nextHopIp
   intfId = nhlEntry.intfId
   labelStack = ( nhlEntry.label if isinstance( nhlEntry.label, list ) else
                  [ nhlEntry.label ] )

   # We don't need to keep the implicit null if there are other labels
   if any( label != 3 for label in labelStack ):
      labelStack = [ l for l in labelStack if l != 3 ]

   nextHopIntf, nextHopEthAddr = resolveNexthop( mount, state, nextHopIp,
                                                 intf=intfId )
   # Map L3 nexthop to L2 nexthop
   viaInfo, clientIdToVia = [], {}
   if ( nextHopIntf and nextHopEthAddr ):
      if mount.mplsTunnelConfig.entropyLabelPush:
         # Generate the EL by hashing over the relevant fields
         udpPam = state.lspPingClientRootUdpPam
         genPrefix = IpGenPrefix( str( prefix ) )
         el = generateMplsEntropyLabel(
               srcIp=( src or getIntfPrimaryIpAddr( mount, nextHopIntf, ipv ) ),
               dstIp=( genPrefix.v4Addr if ipv == IPV4 else genPrefix.v6Addr ),
               udpSrcPort=( 0 if not udpPam else udpPam.rxPort ) )
         labelStack.extend( [ ELI, el ] )
      labelTup = tuple( labelStack )
      info = ( nextHopIntf, labelTup, nextHopEthAddr )
      viaInfo.append( info )
      clientIdToVia[ 0 ] = [ ( nextHopIp, labelTup, nextHopIntf ) ]
   else:
      errOutput = "Failed to find a valid output interface for next-hop {}".format(
                                                                          nextHopIp )
      print errOutput
      return errno.EINVAL

   timeout = 5
   print '   timeout is {}ms, interval is {}ms'.format( timeout * 1000,
                                                        interval*1000 )
   protocolStr = None
   if protocol == 'ldp':
      protocolStr = 'LDP'
   elif protocol == 'mldp':
      protocolStr = 'MLDP'
   else:
      protocolStr = 'Segment-Routing'

   mldpInfo = MldpInfo( genOpqVal, sourceAddrOpqVal, groupAddrOpqVal,
                        jitter, responderAddr )

   return ping( mount, state, src, dst, smac, interval, count, viaInfo, 
                lspPingLabelDistReplyRender, lspPingLabelDistStatisticsRender, 
                ( clientIdToVia, prefix, protocolStr ), protocol=protocol,
                ipv=ipv, timeout=timeout, tc=tc, mldpInfo=mldpInfo,
                standard=standard, size=size, padReply=padReply, tos=tos )
