# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Cell
from EbraTestBridgePlugin.FwdingEtba import (
   forwardingBridgeInit,
   mountNexthopResolver,
)
from EbraTestBridgePlugin.MplsEtbaHardwareStatusSm import MplsHardwareStatusSm
from ForwardingHelper import RouteOrFecChangeReactor
import FibUtils
import Tac
import Tracing
import SharedMem
import Smash
from TypeFuture import TacLazyType

from Agent import Agent

th = Tracing.Handle( "MplsEtbaAgent" )
t0 = th.trace0

clientMountName = "mpls-etba"
TunnelProgrammingStatus = TacLazyType( "Tunnel::Hardware::TunnelProgrammingStatus" )

class MplsEtbaContext( object ):
   def __init__( self, entityManager ):
      self._entityManager = entityManager
      self._smashEntityManager = SharedMem.entityManager( entityManager.sysname(),
                                                          entityManager.local() )

   def em( self ):
      return self._entityManager

   def sEm( self ):
      return self._smashEntityManager

def mplsHardwareStatusSm( context ):
   em = context.em()
   sEm = context.sEm()
   return MplsHardwareStatusSm(
      routingHardwareStatus=em.entity( 'routing/hardware/status' ),
      mplsHwStatus=em.entity( 'routing/hardware/mpls/status' ),
      mplsHwTunnelStatus=em.entity( 'routing/hardware/mpls/tunnel-status' ),
      mplsBackupHwStatus=em.entity( 'routing/hardware/mpls/backup-status' ),
      mplsBackupHwTunnelStatus=em.entity( 
                               'routing/hardware/mpls/backup-tunnel-status' ),
      proactiveArpNexthopConfig=sEm.getTacEntity( 'arp/proactive/config/mpls' ),
      lfib=sEm.getTacEntity( 'mpls/transitLfib' ),
      arpSmash=sEm.getTacEntity( 'arp/status' ),
      bridgingConfig=em.entity( 'bridging/config' ),
      bridgingStatus=em.entity( 'bridging/status' ),
      fwdHelper=context.fwdHelper,
      agentRoot=context.agentRoot,
   )

def routeOrFecChangeReactor( context ):
   sEm = context.sEm()
   context.routingStatus_ = sEm.getTacEntity( 'routing/status' )
   context.routing6Status_ = sEm.getTacEntity( 'routing6/status' )
   if context.fecModeStatus.fecMode == 'fecModeUnified':
      context.forwardingStatus_ = sEm.getTacEntity( 'forwarding/unifiedStatus' )
      context.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/unifiedStatus' )
   else:
      context.forwardingStatus_ = sEm.getTacEntity( 'forwarding/status' )
      context.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/status' )
   context.trie = Tac.root.newEntity( "Routing::Trie", "trie" )
   assert context.trieBuilder, \
         "trieBuilder should have been created in forwardingBridgeInit"
   context.v6trie = Tac.root.newEntity( "Routing6::Trie", "v6trie" )
   assert context.v6trieBuilder, \
         "v6trieBuilder should have been created in forwardingBridgeInit"
   return RouteOrFecChangeReactor( context.trie,
                                   context.v6trie,
                                   context.forwardingStatus_,
                                   context.forwarding6Status_,
                                   context.mplsHardwareStatusSm )

class MplsEtba( Agent ):
   def mountsComplete( self ):
      t0( "call forwardingBridgeInit" )
      forwardingBridgeInit( self.context,
                            clientName=clientMountName )
      t0( "initialize mplsHardwareStatusSm" )
      self.context.mplsHardwareStatusSm = mplsHardwareStatusSm( self.context )
      self.context.routeOrFecChangeReactor_ = routeOrFecChangeReactor( self.context )
      t0( "init complete" )

   def doInit( self, entityManager ):
      # pylint: disable=attribute-defined-outside-init
      self.context = MplsEtbaContext( entityManager )
      self.context.agentRoot = self.createLocalEntity(
            "MplsEtbaAgentRoot", "MplsEtbaAgent::MplsEtbaAgentRoot", "agent/root" )
      mg = entityManager.mountGroup()
      mg.mount( "arp/monitorednexthop/input/status/mplsEtba",
                "Arp::MonitoredNexthop::Input::Status", "wc" )
      mg.mount( "bridging/config", "Bridging::Config", "r" )
      mg.mount( "bridging/status", "Bridging::Status", "r" )
      mg.mount( "l3/config", "L3::Config", "r" )
      mg.mount( "l3/intf/config", "L3::Intf::ConfigDir", "r" )
      mg.mount( "routing/hardware/mpls/backup-status", 
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/backup-tunnel-status", 
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/status", "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/mpls/tunnel-status",
                "Mpls::Hardware::Status", "wS" )
      mg.mount( "routing/hardware/nexthopgroup/status",
                "Routing::Hardware::NexthopGroupStatus", "r" )
      mg.mount( "routing/hardware/status", "Routing::Hardware::Status", "w" )
      mg.mount( "routing/mpls/status", "Mpls::Status", "rS" )
      mg.mount( "routing/vrf/routingInfo/status", "Tac::Dir", "r" )
      mg.mount( "routing6/vrf/routingInfo/status", "Tac::Dir", "r" )
      mg.mount( "interface/status/eth/intf", "Interface::EthIntfStatusDir", "r" )
      mg.mount( "bfd/status/peer", "Bfd::StatusPeer", "r" )
      mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                           'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

      mountNexthopResolver( mg, clientName=clientMountName )
      mg.mount( Cell.path( 'ip/vrf/status/local' ), 'Ip::AllVrfStatusLocal', 'r' )

      arpShadow = Smash.mountInfo( "shadow" )
      fwdInfo = FibUtils.forwardingStatusInfo( "keyshadow" )
      routeInfo = FibUtils.routeStatusInfo( "keyshadow" )
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      shmemEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                                         Smash.mountInfo( 'keyshadow' ) )
      shmemEm.doMount(
         'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
         Smash.mountInfo( 'keyshadow' ) )

      shmemEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                                      Smash.mountInfo( 'keyshadow' ) )
      shmemEm.doMount( "routing/status", "Smash::Fib::RouteStatus", routeInfo )
      shmemEm.doMount( "routing6/status", "Smash::Fib6::RouteStatus", routeInfo )
      shmemEm.doMount( "arp/status", "Arp::Table::Status", arpShadow )

      proactiveArpMountInfo = (
         Smash.mountInfo( 'writer',
                          collectionInfo=[ ( 'request', 1024 ) ] ) )
      shmemEm.doMount( 'arp/proactive/config/mpls',
                       'Arp::ProactiveArp::ClientConfig',
                       proactiveArpMountInfo )
      shmemEm.doMount( "forwarding/srte/status", "Smash::Fib::ForwardingStatus",
                   fwdInfo )
      shmemEm.doMount( "forwarding/status", "Smash::Fib::ForwardingStatus", fwdInfo )
      shmemEm.doMount( "forwarding/unifiedStatus", "Smash::Fib::ForwardingStatus",
                   fwdInfo )
      shmemEm.doMount( "forwarding6/status", "Smash::Fib6::ForwardingStatus",
                       fwdInfo )
      shmemEm.doMount( "forwarding6/unifiedStatus", "Smash::Fib6::ForwardingStatus",
                   fwdInfo )

      t0( "close mg" )
      mg.close( self.mountsComplete )

      # Create a CPU Usage quicktracer
      self._cpuUsageCollectorSm = Tac.newInstance(
            "CpuUsage::CpuUsageCollectorSm", "MplsEtba" )
