# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import sys

import MosCliPrinter
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IntfCli as IntfCli
import CliPlugin.MacAddr as MacAddr
import EbnfParser
import MultiRangeRule
import ShowCommand

COMMANDS_TO_ADD = []

# /mm/projects/python/cli/parser.py has the list of all of the possible
# reserved words. However for EOS we don't need to support all of them.
# to add a new recognized reserved word please add it to this list and then
# add it to the _getMatcher function in this file.
RESERVED_WORDS = ( 'NUMBER', 'COUNT', 'HEXNUMBER', 'NEGNUMBER', 'FLOAT',
                   'NUMRANGE', 'CIDR', 'ADDRESS', 'NETMASK', 'MACADDR', 'MACMASK',
                   'NAME', 'STRING', 'LINE', 'INTERFACE' )

class CommandInfo( object ):
   def __init__( self, docstring ):
      m = re.match( r'''^
            (?P<name>.*?)\s-\s(?P<desc>.*)\n
            (?:\s*Usage:\s(?P<usage>.*)\n?)?
            (?:\s*Group:\s(?P<group>.*)\n?)?
            (?:\s*Modes?:\s(?P<modes>.*)\n?)?
            (?:\s*Hidden?:\s(?P<hidden>.*)\n?)?
            (?:\s*Default?:\s(?P<default>.*)\n?)?
            (?P<text>(?:.*\n?)*)
        $''', docstring, re.VERBOSE )
      if not m:
         raise ValueError( 'Failed to parse docstring: \'%s\'' % docstring )

      self.name = m.group( 'name' )
      self.desc = m.group( 'desc' )
      self.usage = m.group( 'usage' ) or re.sub( r' \(.*', '', self.name )
      self.group = m.group( 'group' )
      modes = m.group( 'modes' )
      if modes:
         self.modes = modes.split()
      else:
         self.modes = [ 'exec' ]

      self.hidden = m.group( 'hidden' )
      self.default = m.group( 'default' )
      self.text = m.group( 'text' )

class Command( object ):
   def __init__( self, func ):
      self.func = func
      info = CommandInfo( func.__doc__ )
      self.name = info.name
      self.desc = info.desc
      self.usage = self._getUsage( info.usage )
      self.group = info.group
      self.modes = info.modes
      self.hidden = bool( info.hidden )
      self.default = info.default

   def __call__( self, *args, **kwargs ):
      return self.func( *args, **kwargs )

   def _getUsage( self, usage ):
      # don't escape any strings
      usage = re.sub( r'[\\]+', '', usage )

      # if we have repeated reserved words we need to replace them with unique values
      cnt = 0
      tokens = []
      for token in EbnfParser.tokenize( usage ):
         if token in RESERVED_WORDS:
            token += '_%s' % cnt
            cnt += 1
         tokens.append( token )
      return ' '.join( tokens )

class ShimContext( object ):
   def __init__( self, mode, app ):
      self.mode_ = mode
      self.session_ = mode.session
      self.debug = False
      self.json_api = False # TODO: set this from the mode object
      self.app_ = app
      self.mode_ctx = { 'app': app }

   @property
   def startupConfig( self ):
      return self.session_.startupConfig()

   @property
   def mode( self ):
      # use cases:
      # while ctx.mode != 'config-app-metafilter':
      # while ctx.mode != 'config-app-metamux':
      # while ctx.mode != 'config-app-metaprotect':
      # while ctx.mode != 'config-app-multiaccess':
      return self.mode_.name

   def enter_mode( self, mode, context=None ):
      raise Exception( 'enter_mode method not supported' )

   def leave_mode( self ):
      raise Exception( 'leave_mode method not supported' )

def _getHandler( cmd, mosApi ):
   combinedTermsGroups = {}
   mandatoryTerms = []
   currentTerms = None
   currGroupNum = 0
   for token in EbnfParser.tokenize( cmd.usage ):
      if token == '[' or token == '(':
         assert currentTerms is None, 'nested optional not supported'
         currentTerms = []
         currGroupNum += 1
         groupName = 'INTERNAL_GROUP_%s' % currGroupNum
         combinedTermsGroups[ groupName ] = currentTerms
         mandatoryTerms.append( groupName )
         continue
      elif token == ']' or token == ')':
         currentTerms = None
         continue

      if currentTerms is not None:
         currentTerms.append( token )
      else:
         result = re.search( '([A-Z_0-9]+)', token )
         if result and result.groups()[ 0 ] == token:
            mandatoryTerms.append( token )

   def handler( mode, args ):
      match = re.match( 'app-([A-Za-z0-9-_]*)', mode.name )
      appName = match.groups()[ 0 ] if match else None
      app = mosApi.apps.get( appName )
      ctx = ShimContext( mode, app )
      # TODO: need to now allow the command to run in config session mode for now
      funcArgs = []
      for i in mandatoryTerms:
         if i.startswith( 'INTERNAL_GROUP' ):
            termsList = []
            for j in combinedTermsGroups[ i ]:
               if j not in args:
                  continue
               termsList.append( str( args[ j ] ) )
            terms = ' '.join( termsList )
            funcArgs.append( terms )
         elif i in args:
            funcArgs.append( str( args[ i ] ) )
         else:
            funcArgs.append( '' )
      try:
         result = cmd.func( ctx, *funcArgs )
      except Exception as e: # pylint: disable-msg=broad-except
         mode.addError( e.message )
         return
      if result is not None:
         MosCliPrinter.printObject( sys.stdout, result )

   return handler

def _getMatcher( token, cmdDesc ):
   match = re.match( r'([A-Z]*)_[0-9]+', token )
   if not match:
      return token

   token = match.groups()[ 0 ]
   if token not in RESERVED_WORDS:
      return token

   if token == 'NUMBER' or token == 'COUNT' or token == 'HEXNUMBER':
      return CliMatcher.IntegerMatcher( 0, sys.maxint,
                                        helpdesc='positive integer value' )
   elif token == 'NEGNUMBER':
      return CliMatcher.IntegerMatcher( -sys.maxint - 1, sys.maxint,
                                        helpdesc='integer value' )
   elif token == 'FLOAT':
      return CliMatcher.FloatMatcher( sys.float_info.min, sys.float_info.max,
                                      helpdesc='Floating-point number',
                                      precisionString='%.2g' )
   elif token == 'NUMRANGE':
      def rangeFn():
         return ( -sys.maxint - 1, sys.maxint )
      return MultiRangeRule.MultiRangeMatcher( rangeFn=rangeFn,
                                               helpdesc='A number range',
                                               noSingletons=True,
                                               maxRanges=1 )
   elif token == 'CIDR':
      assert False # TODO
      return None
   elif token == 'ADDRESS':
      return IpAddrMatcher.IpAddrMatcher( 'IP address in dot-decimal notation' )
   elif token == 'NETMASK':
      return IpAddrMatcher.IpAddrMatcher( 'Netmask in dot-decimal notation' )
   elif token == 'MACADDR':
      return MacAddr.MacAddrMatcher( 'Ethernet MAC address as 3 groups of hex '
                                     'digits' )
   elif token == 'MACMASK':
      return MacAddr.MacAddrMatcher( 'Ethernet MAC wild card mask as 3 groups of '
                                     'hex digits' )
   elif token == 'NAME':
      return CliMatcher.PatternMatcher( r'\S+', helpdesc='String that is a name',
                                        helpname='NAME' )
   elif token == 'STRING':
      return CliMatcher.PatternMatcher( r'\S+', helpdesc='String value',
                                        helpname='STRING' )
   elif token == 'LINE':
      return CliMatcher.StringMatcher( helpname='LINE', helpdesc='LINE' )
   elif token == 'INTERFACE':
      return IntfCli.Intf.matcher
   else:
      assert 'Special token \'%s\' not defined' % token
      return None

def _getData( cmd, ignoreShow=True ):
   cmdData = {}
   for term in EbnfParser.getTerms( EbnfParser.tokenize( cmd.usage ) ):
      if not ignoreShow and term == 'show':
         continue
      if term in ( 'no', 'default' ):
         continue
      cmdData[ term ] = _getMatcher( term, cmd.desc )
   return cmdData

def _getMode( mode, mosApi ):
   if mode == 'priv':
      return BasicCliModes.EnableMode
   elif mode == 'exec':
      return BasicCliModes.UnprivMode
# TODO: not supported yet
#   elif mode == 'config':
#      return BasicCliModes.GlobalConfigMode
   else:
      match = re.match( 'config-app-([A-Za-z0-9-_]*)', mode )
      if match:
         appName = match.groups()[ 0 ]
         return mosApi.appCliModes.get( appName, None )
      else:
         return None

def _getModes( modes, mosApi ):
   return [ _getMode( mode, mosApi ) for mode in modes ]

def _registerShowCommand( cmd, mosApi ):
   class MosApiShowCommand( ShowCommand.ShowCliCommandClass ):
      syntax = cmd.usage
      data = _getData( cmd, ignoreShow=False )
      handler = _getHandler( cmd, mosApi )
      hidden = cmd.hidden

   for mode in _getModes( cmd.modes, mosApi ):
      if mode == BasicCliModes.UnprivMode:
         BasicCli.addShowCommandClass( MosApiShowCommand )
      else:
         mode.addShowCommandClass( MosApiShowCommand )

def _registerNonShowCommand( cmd, mosApi ):
   if cmd.usage.startswith( 'default ' ):
      class MosApiDefaultCommand( CliCommand.CliCommandClass ):
         defaultSyntax = cmd.usage[ 8 : ]
         data = _getData( cmd )
         defaultHandler = _getHandler( cmd, mosApi )
         hidden = cmd.hidden

      for mode in _getModes( cmd.modes, mosApi ):
         mode.addCommandClass( MosApiDefaultCommand )
   elif cmd.usage.startswith( 'no ' ):
      class MosApiNoCommand( CliCommand.CliCommandClass ):
         noSyntax = cmd.usage[ 3 : ]
         data = _getData( cmd )
         noHandler = _getHandler( cmd, mosApi )
         hidden = cmd.hidden

      for mode in _getModes( cmd.modes, mosApi ):
         mode.addCommandClass( MosApiNoCommand )
   else:
      class MosApiCommand( CliCommand.CliCommandClass ):
         syntax = cmd.usage
         data = _getData( cmd )
         handler = _getHandler( cmd, mosApi )
         hidden = cmd.hidden

      for mode in _getModes( cmd.modes, mosApi ):
         mode.addCommandClass( MosApiCommand )

def _registerCommand( cmd, mosApi ):
   for mode in cmd.modes:
      # check that all of the modes are valid
      if _getMode( mode, mosApi ) is None:
         print 'ERROR: Unknown Mode %s in command %r' % ( mode, cmd.usage )
         return

   if cmd.usage.startswith( 'show' ):
      _registerShowCommand( cmd, mosApi )
   else:
      _registerNonShowCommand( cmd, mosApi )

def registerCommands( mosApi ):
   for cmd in COMMANDS_TO_ADD:
      _registerCommand( cmd, mosApi )
