# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import fnmatch
import traceback

import LauncherDaemonConstants
import LauncherUtil

MOSAPI_INSTANCE = None

class _AppBase( object ):
   def _getConfig( self ):
      raise NotImplementedError

   def is_shutdown( self ):
      return not self._getConfig().enabled

   def get_config( self, key, section='user' ):
      assert section == 'user', 'Only user section allowed'
      config = self._getConfig()
      if config is None:
         return None
      return config.option.get( key, None )

   def iter_config( self, pattern='*', section='user' ):
      """Returns an iterator over the config for matching entries"""
      assert section == 'user', 'Only user section allowed'
      config = self._getConfig()
      for k, v in config.option.iteritems():
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

class AppDaemon( _AppBase ):
   name = None
   exe = None
   heartbeatPeriod = 0

   def __init__( self, application ):
      assert self.name, 'Name must be specified'
      self.application_ = application
      self.statusDir_ = MOSAPI_INSTANCE.statusDir
      self.configDir_ = MOSAPI_INSTANCE.configDir
      self.agentConfigCliDir_ = MOSAPI_INSTANCE.agentConfigCliDir

   def _getConfig( self ):
      return self.configDir_.get( self.name, None )

   def _getStatus( self ):
      return self.statusDir_.get( self.name, None )

   def get_status( self, key ):
      status = self._getStatus()
      if not status:
         return None
      return status.data.get( key, None )

   def _set_config( self, key, value='' ):
      config = self._getConfig()
      if config is None:
         return
      config.option[ key ] = str( value )

   def _remove_config( self, key=None ):
      config = self._getConfig()
      if config is None:
         return
      del config.option[ key ]

   def iter_status( self, pattern='*' ):
      """Returns an iterator over the status for matching entries"""
      status = self._getStatus()
      if not status:
         raise StopIteration
      for k, v in status.data.iteritems():
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

   def start_daemon( self, args=None ):
      ''' Programs into LauncherConfig '''
      if args is None:
         args = []

      if self.name in self.agentConfigCliDir_:
         return
      daemonConfig = self.agentConfigCliDir_.newAgent( self.name )
      daemonConfig.userDaemon = True
      daemonConfig.heartbeatPeriod = self.heartbeatPeriod
      daemonConfig.useEnvvarForSockId = True # True for EosSdk
      daemonConfig.oomScoreAdj = LauncherDaemonConstants.DEFAULT_OOM_SCORE_ADJ
      daemonConfig.exe = self.exe
      for i, x in enumerate( args ):
         daemonConfig.argv[ i ] = x

      # Set up the runnability criteria, so the agent only runs when
      # the genericAgentCfg.enabled is True on the application. Meaning only if the
      # application is enabled do we run the daemon
      # (even if we try to start it anyway)
      daemonConfig.runnability = ( 'runnability', )
      daemonConfig.runnability.qualPath = ( 'daemon/agent/runnability/%s' %
                                             self.application_.name )

      for redProto in LauncherUtil.allRedProtoSet:
         daemonConfig.criteria[ redProto ] = LauncherUtil.activeSupervisorRoleName
      daemonConfig.stable = True
      self._getConfig().enabled = True

   def stop_daemon( self ):
      self._getConfig().enabled = False
      del MOSAPI_INSTANCE.agentConfigCliDir[ self.name ]

def get_app_by_name( appName ):
   assert MOSAPI_INSTANCE, 'MOS not initialized'
   return MOSAPI_INSTANCE.apps.get( appName, None )

class App( _AppBase ):
   '''The base class for all applications.

    Each app must inherit from App and extend with specific requirements.

    If subtypes overload a method, please remember in general you must also
    call the super's implementation.
    '''

   name = None
   appDaemons = []

   '''This is a class-member string containing the name of the application.
     It is used to match against the 'application <name>' CLI command used
     to enter the app's mode.'''
   def __init__( self, **kwargs ):
      assert self.name, 'Name must be specified'
      assert MOSAPI_INSTANCE, 'MOS not initialized'
      self.daemons_ = self._createDaemons()
      self.configDir_ = MOSAPI_INSTANCE.configDir

   def _createDaemons( self ):
      daemons = []
      for appDaemon in self.appDaemons:
         if hasattr( self, appDaemon.name ):
            raise ValueError( 'Daemon name conflicts with %s' % appDaemon.name )

         daemon = appDaemon( self )
         daemons.append( daemon )
         setattr( self, daemon.name, daemon )

      return daemons

   def _getConfig( self ):
      return self.configDir_.get( self.name, None )

   def get_description( self ):
      return ''

   def shutdown( self, ctx=None ):
      self._getConfig().enabled = False

   def no_shutdown( self, ctx=None ):
      self._getConfig().enabled = True

   def start( self ):
      return self.no_shutdown()

   def no_start( self ):
      return self.shutdown()

   def set_config( self, key, value='', section='user' ):
      assert section == 'user', 'Only user section allowed'
      config = self._getConfig()
      if config is None:
         return
      config.option[ key ] = str( value )
      for daemon in self.daemons_:
         daemon._set_config( key, value ) # pylint: disable-msg=protected-access

   def remove_config( self, key=None, section='user' ):
      assert section == 'user', 'Only user section allowed'
      config = self._getConfig()
      if config is None:
         return
      del config.option[ key ]

      for daemon in self.daemons_:
         daemon._remove_config( key ) # pylint: disable-msg=protected-access

def loadApps( mosApi, mosAppCliLoader ):
   global MOSAPI_INSTANCE
   MOSAPI_INSTANCE = mosApi
   appClasses = App.__subclasses__()
   return loadAppsHelper( mosAppCliLoader, appClasses )

def loadAppsHelper( mosAppCliLoader, appClasses ):
   apps = {}
   daemons = {}
   for appClass in appClasses:
      try:
         assert appClass.name not in apps, '%s App is defined twice' % appClass.name
         appDaemons = []
         for appDaemon in appClass.appDaemons:
            if appDaemon.name in appDaemons:
               raise ValueError( 'Daemon %s defined in app \'%s\' is defined twice' %
                     ( appDaemon.name, appClass.name ) )
            if appDaemon.name in daemons:
               otherAppName = daemons[ appDaemon.name ].name
               raise ValueError( 'Daemon \'%s\' defined in app \'%s\' is already '
                                 'defined in app \'%s\'' %
                                 ( appDaemon.name, appClass.name, otherAppName ) )
            appDaemons.append( appDaemon.name )

         app = appClass()
         apps[ app.name ] = app
         for appDaemon in appClass.appDaemons:
            daemons[ appDaemon.name ] = app
      except Exception: # pylint: disable-msg=broad-except
         print( 'App %s failed to load, below is the backtrace' % appClass.name )
         traceback.print_exc()

   return apps
