# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys
import os
from contextlib import contextmanager
from importlib import import_module
from glob import glob

import BasicCli
import CliCommand
import CliMatcher
import CliMode.MosApiMode as MosApiMode
import CliParser
import ConfigMount
import LauncherLib
import LazyMount
import MainCli
import MosApp
import MosCli
from TypeFuture import TacLazyType

MOSAPI_INSTANCE = None

AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )
MOSAPI_APP_PATHS = [
   '/mnt/flash/apps',
   '/opt/apps'
   ]

class MosAppCliLoader( object ):
   def load( self, cls ):
      appName = cls.name
      cliMode = self._generateCliMode( appName )
      self._addGotoModeCmd( cliMode, appName )
      self._addDefaultCmds( cliMode, appName )
      return cliMode

   def _createAppConfig( self, appName ):
      assert appName in MOSAPI_INSTANCE.apps, 'App %s not found' % appName
      app = MOSAPI_INSTANCE.apps[ appName ]
      self._createConfig( app.name )
      for daemon in app.appDaemons:
         self._createConfig( daemon.name )

   def _createConfig( self, name ):
      if name in MOSAPI_INSTANCE.configDir:
         return
      genericAgentCfg = MOSAPI_INSTANCE.configDir.newEntity(
            'GenericAgent::Config', name )
      genericAgentCfg.agentType = AGENT_TYPE.mosApiShim
      MOSAPI_INSTANCE.aclConfigDir.newEntity( 'Acl::ServiceAclTypeVrfMap', name )

   def _clearAppConfig( self, appName ):
      assert appName in MOSAPI_INSTANCE.apps, 'App %s not found' % appName
      app = MOSAPI_INSTANCE.apps[ appName ]
      app.shutdown( None ) # make sure that the app is properly shutdown
      self._clearConfig( appName )
      for daemon in app.appDaemons:
         self._clearConfig( daemon.name )

   def _clearConfig( self, name ):
      MOSAPI_INSTANCE.configDir.deleteEntity( name )
      MOSAPI_INSTANCE.aclConfigDir.deleteEntity( name )

   def _generateCliMode( self, appName ):
      class AppCliMode( MosApiMode.MosApiAppMode, BasicCli.ConfigModeBase ):
         name = 'app-%s' % appName
         modeParseTree = CliParser.ModeParseTree()

         def __init__( self, parent, session ):
            MosApiMode.MosApiAppMode.__init__( self, appName )
            BasicCli.ConfigModeBase.__init__( self, parent, session )

      return AppCliMode

   def _addGotoModeCmd( self, cliMode, appName ):
      def appNameFunc( mode ):
         try:
            desc = MOSAPI_INSTANCE.apps[ appName ].get_description()
         except KeyError:
            desc = ''
         return { appName: desc }

      class GotoAppModeCmd( CliCommand.CliCommandClass ):
         syntax = 'application %s' % appName
         noOrDefaultSyntax = 'application %s' % appName
         data = {
                  'application': 'Configure application',
                  appName: CliMatcher.DynamicKeywordMatcher( appNameFunc )
                }

         @staticmethod
         def handler( mode, args ):
            if appName not in MOSAPI_INSTANCE.configDir:
               # pylint: disable-msg=protected-access
               self._createAppConfig( appName )

            genericAgentCfg = MOSAPI_INSTANCE.configDir[ appName ]
            # TODO: we need to protect a mos daemon also having the same
            # name as an DaemonCli
            if genericAgentCfg.agentType != AGENT_TYPE.mosApiShim:
               mode.addError( 'Unable to create application \'%s\' because daemon '
                              '\'%s\' already exists. Please delete the daemon'
                              % ( appName, appName ) )
               return
            childMode = mode.childMode( cliMode )
            mode.session_.gotoChildMode( childMode )

         @staticmethod
         def noOrDefaultHandler( mode, args ):
            genericAgentCfg = MOSAPI_INSTANCE.configDir.get( appName )
            if not genericAgentCfg:
               return

            # TODO: we should only clear it if it's a real app, not a daemon app
            if genericAgentCfg.agentType != AGENT_TYPE.mosApiShim:
               mode.addError( 'Unable to delete application \'%s\' because it is a '
                              'CLI Daemon' % appName )
               return
            self._clearAppConfig( appName ) # pylint: disable-msg=protected-access

      BasicCli.GlobalConfigMode.addCommandClass( GotoAppModeCmd )

   def _addDefaultCmds( self, cliMode, appName ):
      class ShutdownCmd( CliCommand.CliCommandClass ):
         syntax = 'shutdown'
         noOrDefaultSyntax = 'shutdown'
         data = { 'shutdown': 'Shutdown %s' % appName }

         @staticmethod
         def handler( mode, args ):
            if appName not in MOSAPI_INSTANCE.apps:
               return
            app = MOSAPI_INSTANCE.apps[ appName ]
            ctx = MosCli.ShimContext( mode, app )
            app.shutdown( ctx )

         @staticmethod
         def noOrDefaultHandler( mode, args ):
            if appName not in MOSAPI_INSTANCE.apps:
               return

            app = MOSAPI_INSTANCE.apps[ appName ]
            ctx = MosCli.ShimContext( mode, app )
            app.no_shutdown( ctx )

      cliMode.addCommandClass( ShutdownCmd )

class MosApi( object ):
   def __init__( self, entityManager, mount=True ):
      if mount: # used for testing
         self.configDir = ConfigMount.mount( entityManager,
               'daemon/agent/config', 'Tac::Dir', 'wi' )
         self.statusDir = LazyMount.mount( entityManager,
               'daemon/agent/status', 'Tac::Dir', 'ri' )
         self.aclConfigDir = ConfigMount.mount( entityManager,
               'daemon/acl/config', 'Tac::Dir', 'wi' )
         self.agentConfigCliDir = ConfigMount.mount( entityManager,
               LauncherLib.agentConfigCliDirPath, 'Launcher::AgentConfigDir', 'wi' )
      self.apps = {}
      self.appCliModes = {}
      self._importApps()
      self.mosAppCliLoader_ = MosAppCliLoader()
      for subclass in MosApp.App.__subclasses__():
         self.appCliModes[ subclass.name ] = self.mosAppCliLoader_.load( subclass )
      MosCli.registerCommands( self )
      MainCli.pluginsLoadedHook.addExtension( self._loadApps )

   def _loadApps( self, em ):
      self.apps = MosApp.loadApps( self, self.mosAppCliLoader_ )

   @contextmanager
   def _addAppPaths( self ):
      oldPath = sys.path[ : ]
      try:
         sys.path.extend( MOSAPI_APP_PATHS )
         yield
      finally:
         sys.path = oldPath

   def _appCandidates( self ):
      return sum( ( glob( p + '/*' ) for p in MOSAPI_APP_PATHS ), [] )

   def _importApps( self ):
      with self._addAppPaths():
         for appCandidate in self._appCandidates():
            name = os.path.basename( appCandidate )
            if name.endswith( '.pyc' ):
               name = name.replace( '.pyc', '' )
            elif name.endswith( '.py' ):
               name = name.replace( '.py', '' )
            elif os.path.exists( appCandidate + '/__init__.py' ) is False:
               return
            try:
               import_module( name )
            except Exception as e: # pylint: disable-msg=broad-except
               print( 'Error \'%s\' Loading module %r' % ( e, name ) )

def instantiateMosApi( entityManager ):
   global MOSAPI_INSTANCE
   if MOSAPI_INSTANCE:
      return
   MOSAPI_INSTANCE = MosApi( entityManager )
