# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Common tools for reading and parsing data about eMMC flash devices.

import Tac

CPUBOARDS_WITH_EMMC = [
   'BelvedereCpu',
   'Crow',
   'Grackle',
   'IslandCpu',
   'Magpie',
   'Mendocino',
   'NewportCpu',
   'Woodpecker',
]

CROW_PRODUCTS_WITHOUT_EMMC = (
   r'Caspar|'
   r'Cazadero|'
   r'Clearlake$|'
   r'Doran|'
   r'Forestville|'
   r'Guerneville|'
   r'Larkfield|'
   r'Yreka'
)

# Lookup table for flash vendor ID to vendor name
MANUFACTURER_ID_TABLE = {
   '0x11' : 'Toshiba',
   '0x13' : 'Micron',
   '0x15' : 'Samsung',
   '0xf6' : 'Smart Modular'
}

# Start/end bit positions of each field within the eMMC flash device CID
# register. Indexed starting at LSB. Offsets for each field in CID register are
# from: http://rere.qmqm.pl/~mirq/JESD84-A44.pdf, page 132
MANUFACTURER_ID_START = 128
MANUFACTURER_ID_END   = 120

PRODUCT_NAME_START    = 104
PRODUCT_NAME_END      = 56

FWREV_UPPER_START     = 56
FWREV_UPPER_END       = 52

FWREV_LOWER_START     = 52
FWREV_LOWER_END       = 48

SERIAL_NUM_START      = 48
SERIAL_NUM_END        = 16

# Read CID register for flash device. Contains information about
# manufacturer, serial number, etc. encoded in a 32-character hex string,
# indexed starting at LSB.
def readCidRegister( device ):
   try:
      cid = Tac.run( [ "/bin/cat", "/sys/block/" + device + "/device/cid" ],
                     stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return None

   return cid

# Returns the size of a storage device rounded up to the nearest GB.
def readDeviceSize( device ):
   BYTES_PER_GB = 1000 ** 3

   try:
      fdisk_output = Tac.run( [ "/sbin/fdisk", "-l", "/dev/" + device ], asRoot=True,
                              stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return None

   if "No such file" in fdisk_output:
      return None

   sizeBytes = fdisk_output.split('\n')[1].split(' ')[4]

   return int( round( float( sizeBytes ) / BYTES_PER_GB ) )

# Converts a bit position in a CID register to a character index within the CID
# string returned from the sysfs eMMC driver that contains that bit. The CID
# register is indexed starting at the LSB, but the hex string that represents
# the CID contents is indexed from the MSB.
def __cidBitPosToCharPos( bitPos ):
   CID_NUM_BITS = 128

   if bitPos < 0 or bitPos > CID_NUM_BITS:
      return -1

   return ( CID_NUM_BITS - bitPos ) / 4

def manufacturerId( cid ):
   return '0x' + cid[ __cidBitPosToCharPos( MANUFACTURER_ID_START ) :
                      __cidBitPosToCharPos( MANUFACTURER_ID_END ) ].lower()

def manufacturerName( cid ):
   manfId = manufacturerId( cid ).lower()
   if manfId in MANUFACTURER_ID_TABLE:
      return MANUFACTURER_ID_TABLE[ manfId ]

   return 'Unknown'

def productName( cid ):
   return cid[ __cidBitPosToCharPos( PRODUCT_NAME_START ) :
               __cidBitPosToCharPos( PRODUCT_NAME_END ) ].decode('hex')

def firmwareRevision( cid ):
   firmwareRevisionUpper = str( int( \
         cid[ __cidBitPosToCharPos( FWREV_UPPER_START ) :
              __cidBitPosToCharPos( FWREV_UPPER_END ) ], 16 ) )
   firmwareRevisionLower = str( int( \
         cid[ __cidBitPosToCharPos( FWREV_LOWER_START ) :
              __cidBitPosToCharPos( FWREV_LOWER_END ) ], 16 ) )

   return firmwareRevisionUpper + '.' + firmwareRevisionLower

def serialNumber( cid ):
   return cid[ __cidBitPosToCharPos( SERIAL_NUM_START ) :
               __cidBitPosToCharPos( SERIAL_NUM_END ) ]

# Code to parse lifetime data out of the EXT_CSD register on eMMC 5.0 or later
# devices
REV = 192
DEVICE_LIFE_TIME_EST_TYP_A = 268
DEVICE_LIFE_TIME_EST_TYP_B = 269
PRE_EOL_INFO = 267

LIFE_TIME_EST = {
   '01': 1.0, # 100% to 90% remaining
   '02': 0.9, # 90% to 80% remaining
   '03': 0.8, # 80% to 70% remaining
   '04': 0.7, # 70% to 60% remaining
   '05': 0.6, # 60% to 50% remaining
   '06': 0.5, # 50% to 40% remaining
   '07': 0.4, # 40% to 30% remaining
   '08': 0.3, # 30% to 20% remaining
   '09': 0.2, # 20% to 10% remaining
   '0a': 0.1, # 10% to 0% remaining
   '0b': 0.0, # Past design lifetime
}

EOL_INFO = {
   '01': 1.0, # Normal
   '02': 0.2, # 20% remaining
   '03': 0.1, # 10% remaining
}

def getbyte( ecsd, index ):
   return ecsd[ index*2:index*2 + 2 ]

SLC = 'SLC'
MLC = 'MLC'
RESERVES = 'Reserves'

def emmcLifetimes():
   '''Returns a dict of dicts like so:
   {
      'flash': {
                 'SLC': 0.9,
                 'MLC': 0.9,
                 'Reserves': 1.0,
              },
      ...
   }
   '''
   lifetimes = {}

   try:
      devices = Tac.run( [ '/bin/sh', '-c', 'ls /monitor/mmc*/mmc*/ext_csd' ],
                         stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, asRoot=True )
   except Tac.SystemCommandError:
      # The paths likely don't exist
      return lifetimes

   devices = devices[ : -1 ] # Remove final \n

   for drive in devices.splitlines():
      # We only ever expect one eMMC device and we expect it will be the flash
      # device. Cheat for now and don't correlate mmcX to /mnt/Y.
      driveName = 'flash'
      assert driveName not in lifetimes
      lifetimes[ driveName ] = {}

      ecsd = Tac.run( [ 'cat', drive ], stdout=Tac.CAPTURE, asRoot=True )

      mmcVersion = int( getbyte( ecsd, REV ), base=16 )
      if mmcVersion < 7:
         # Not available before version 8
         continue

      slc = getbyte( ecsd, DEVICE_LIFE_TIME_EST_TYP_A )
      if slc in LIFE_TIME_EST:
         lifetimes[ driveName ][ SLC ] = LIFE_TIME_EST[ slc ]

      mlc = getbyte( ecsd, DEVICE_LIFE_TIME_EST_TYP_B )
      if mlc in LIFE_TIME_EST:
         lifetimes[ driveName ][ MLC ] = LIFE_TIME_EST[ mlc ]

      reserves = getbyte( ecsd, PRE_EOL_INFO )
      if reserves in EOL_INFO:
         lifetimes[ driveName ][ RESERVES ] = EOL_INFO[ reserves ]

   return lifetimes
