# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Python helpers for read-mounting mlag/status

mlag/status mounts require certain interface paths to be explicitly mounted
as well.

  peerLinkIntf
    interface/status/eth/phy/slice (Tac::Dir)
  localInterface
    interface/status/eth/vlan
'''

import LazyMount
import Tracing

th = Tracing.defaultTraceHandle()
t0 = th.trace0

# Enumerate the common data used by mountMlagStatus and MlagStatusLazyMounter
_mlagStatusRequires = {
   'phy': ( 'interface/status/eth/phy/slice', 'Tac::Dir', 'ri' ),
   'vlan': ( 'interface/status/eth/vlan', 'Interface::VlanIntfStatusDir', 'r' ),
}

def mountMlagConfig( emOrMg ):
   # Simply mount as Mlag::Config no longer uses implicit mount
   t0( 'mountMlagConfig: mount mlag/config' )
   return emOrMg.mount( 'mlag/config', 'Mlag::Config', 'r' )

def mountMlagStatus( emOrMg ):
   '''
   First mount the required paths for mlag/status, and then return
   the result of
      emOrMg.mount( 'mlag/status', 'Mlag::Status', 'r' )
   '''
   for mountArgs in _mlagStatusRequires.values():
      emOrMg.mount( *mountArgs )
      t0( 'mountMlagStatus: mount', *mountArgs )
   t0( 'mountMlagStatus: mount mlag/status' )
   return emOrMg.mount( 'mlag/status', 'Mlag::Status', 'r' )

class __MlagLazyMounter( object ):
   def getMountArgs( self ):
      raise NotImplementedError

   def requires( self ):
      raise NotImplementedError

   def trace( self, func, *args ):
      t0( self.__class__.__name__ + '.' + func, *args )

   def __init__( self, entityManager ):
      self._forced = False
      self._prerequisite = {}
      path, tacType, flags = self.getMountArgs()
      for key, reqArgs in self.requires().iteritems():
         self.trace( '__init__', path, 'requires', reqArgs )
         self._prerequisite[ key ] = LazyMount.mount( entityManager, *reqArgs )
      self._mainObj = LazyMount.mount( entityManager, path, tacType, flags )

   def __maybeMountPrerequisites( self ):
      if self._forced:
         return
      for entityProxy in self._prerequisite.values():
         self.trace( '__maybeMountPrerequisites', 'force mount', entityProxy )
         entityProxy.force()
      self._forced = True

   _proxyAttrs = frozenset( [ 'peerLinkIntf', 'localInterface',
                              '_mount', 'force' ] )
   def __getattr__( self, name ):
      self.trace( '__getattr__', 'name:', name )
      if name in self._proxyAttrs:
         self.__maybeMountPrerequisites()
      return getattr( self._mainObj, name )

class MlagStatusLazyMounter( __MlagLazyMounter ):
   '''
   Helper class to LazyMount mlag/status and all of its dependent paths.

   To be used in place of LazyMount.mount( 'mlag/status' ) in CliPlugin code.
   The MlagStatusLazyMounter instance can be used transparently in place of a
   LazyMount proxy.

   The only difference is that LazyMount.force() cannot be used on an
   MlagStatusLazyMounter instance.  Instead, just call mlagStatus.force().

   Under the hood, it will create proxies for the required paths and then force
   mount those proxies when the main 'mlag/status' path is force mounted.
   '''
   def getMountArgs( self ):
      return ( 'mlag/status', 'Mlag::Status', 'r' )

   def requires( self ):
      return _mlagStatusRequires
