#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Pci
import sys
import os
import time
import array
import Cell
import Tac
import re
import FpgaUtil     # pylint: disable-msg=F0401
import PicassoInit  # pylint: disable-msg=F0401



def printToFile( msg ):
   with open( '/var/log/NorCalInit', 'a' ) as f:
      f.write( msg )

def sid():
   try:
      return re.search( r'sid=(\w+) ', open( "/proc/cmdline" ).read() ).group( 1 )
   # pylint: disable-msg=bare-except
   except:
      return ""

def fimwareFile():
   firmwareMap = { 'OtterLake':     'firmware-1.9.63.pmc',
                   'Redwood':       'firmware-1.9.63.pmc',
                   'HorseshoeLake': 'firmware-1.B.8C.pmc',
                   'Narwhal':       'firmware-1.B.8C.pmc',
                   'NarwhalClock':  'firmware-1.B.8C.pmc',
                   'Penguin':       'firmware-1.B.8C.pmc',
                   'PenguinClock':  'firmware-1.B.8C.pmc',
   }
   if sid() in firmwareMap:
      return firmwareMap[ sid() ]
   else:
      return ""

class MicrosemiConsts( object ):
   GAS_DONE = 2
   GAS_INPROGRESS = 1
   PMC_SUCCESS = 0
   MRPC_BG_STAT_IDLE = 0
   MRPC_BG_STAT_DONE = 2
   SWITCHTEC_MAX_PORTS = 48
   firmwareImage = "/usr/share/Microsemi/%s" % fimwareFile()
   microsemiVendorID = 0x11f8
   microsemiDeviceIDs = [ 0x8533, 0x8534, 0x8532 ]
   microsemiBinaryHeaderSize = 630
   downloadHeaderSize = 0x10
   def configImage( self ):
      return "/usr/share/Microsemi/%s-sup%d.pmc" % ( sid(), Cell.cellId() )

class MicrosemiGAS( object ):
   GAS_INPUT_DATA = 0
   GAS_OUTPUT_DATA = 0x400
   GAS_COMMAND = 0x800
   GAS_STATUS = 0x804
   GAS_RETURNVALUE = 0x808
   GAS_BLOCKSIZE = 0x400
   GAS_VendorTableRevision = 0x2010
   GAS_FirmwareAddress = 0x2204
   GAS_FirmwareVersion = 0x2208
   GAS_ConfigVersion = 0x2214
   GAS_VendorTableRevision = 0x2010
   GAS_InactiveFirmwareAddress = 0x221C
   GAS_InactiveFirmwareVersion = 0x2220
   GAS_InactiveVendorTableRevision = 0x2228
   GAS_InactiveConfigVersion = 0x22c

class MicrosemiDownloadSubcommand( object ):
   MRPC_DNLD_Download = 1
   MRPC_DNLD_ImageOffset = 4
   MRPC_DNLD_ImageLength = 8
   MRPC_DNLD_DataLength = 12
   MRPC_DNLDSTATUS_Check = 0
   MRPC_DNLDSTATUS_BackgroundStatus = 1
   MRPC_DNLDSTATUS_InProgress = 2

class MCRPC_P2PSubcommand( object ):
   MRPC_P2P_BIND = 0
   MRPC_P2P_UNBIND = 1
   MRPC_P2P_INFO = 3

class MRPC_LTSSMSubcommand( object ):
   MRPC_LTSSM_LogDump = 7

class MicrosemiMRPC( MicrosemiDownloadSubcommand,
                     MCRPC_P2PSubcommand,
                     MRPC_LTSSMSubcommand ):
   MRPC_DIAG_PMC_START = 0
   MRPC_TWI = 1
   MRPC_VGPIO = 2
   MRPC_PWM = 3
   MRPC_DIETEMP = 4
   MRPC_FWDNLD = 5
   MRPC_FWLOGRD = 6
   MRPC_PMON = 7
   MRPC_PORTLN = 8
   MRPC_PORTARB = 9
   MRPC_MCOVRLY = 10
   MRPC_STACKBIF = 11
   MRPC_PORTPARTP2P = 12
   MRPC_DIAG_TLP_INJECT = 13
   MRPC_DIAG_TLP_GEN = 14
   MRPC_DIAG_PORT_EYE = 15
   MRPC_DIAG_POT_VHIST = 16
   MRPC_DIAG_PORT_LTSSM_LOG = 17
   MRPC_DIAG_PORT_TLP_ANL = 18
   MRPC_DIAG_PORT_LN_ADPT = 19
   MRPC_NT_MCG_CAP = 23
   MRPC_TACHO = 24
   MRPC_SMBUS = 26
   MRPC_RESET = 27
   MRPC_LNKSTAT = 28
   MRPC_SES = 30
   MRPC_ECHO = 65

class MicrosemiGasOps( object ):
   # Region 6 at 16k (0x10000)
   # NTB Memoory Mapped Registers, page 174 in Microsemi Device Specifications
   # See 17.8, page 480 for format of the region
   MRPC_NTB_BASE = 0x10000
   MRPC_NTB_CONTROL_OFFSET = 0x4000
   MRPC_NTB_CONTROL_BLOCK_SIZE = 0x2000
   MRPC_NTB_REQUESTER_TABLE_OFFSET = 0x400

   def NTxBASE( self, x ):
      # Address of NTx Control Registers area
      # see 17.8 in Microsemi Device Specifications
      return self.MRPC_NTB_BASE + \
         self.MRPC_NTB_CONTROL_OFFSET + x * self.MRPC_NTB_CONTROL_BLOCK_SIZE

   def requesterTable( self, x ):
      # See 17.8.2 in Microsemi Device Specifications
      return self.NTxBASE( x ) + self.MRPC_NTB_REQUESTER_TABLE_OFFSET

   def requesterMask( self, origDevice, origFunction, destDevice, destFunction ):
      valid = 1
      # See page 489, Table 296, in Microsemi Device Specifications
      return ( ( origDevice << 3 | origFunction ) << 16 |
               ( destDevice << 3 | destFunction ) << 1 |
               valid )

   MRPC_NTOP_LOCK = 1
   MRPC_NTOP_CONF = 2
   MRPC_NTOP_RESET = 3

   def ntOp( self, x ):
      # See 17.8.2 in Microsemi Device Specifications
      return self.NTxBASE( x ) + 4

   MRPC_PARTITIONS = [ 0, 1 ]
   MRPC_NTBAR = 4             # Physical BAR4 used for NTB
   MRPC_NTBARSETUPSIZE = 0x10 # Page 482 of DevSpec
   MRPC_NTBARSETUP_VALID = 1
   MRPC_NT_TYPE_64BIT = 0b100
   MRPC_NT_LUT_WIN_ENABLE = 0b10000

   # The following two values mean that
   # window size is 2GB and least
   # 31 bits are offset within a window
   # See table 293 in device spec, offsets 0x4-0x7
   MRPC_NT_WIN_DIR_SIZE = 0x80000000L
   MRPC_NT_DIR_XLATE_POS = 31
   MRPC_NT_DIR_XLATE_BASE_ADDR_HIGH = 64 / 4

class MicrosemiBase( MicrosemiMRPC, MicrosemiGAS, MicrosemiGasOps, MicrosemiConsts ):
   microsemiBar = int( 0 )
   debug = False
   managementEndpoint_ = None
   dataEndpoint_ = None
   microsemiFunctionBar = None
   microsemiFunctionBar_ = None
   
   def __init__( self, verbose=True ):
      self.verbose = verbose
      self.size = 0
      self.bdf = 0
      self.prot = 0
      self.device = None
      self.dmamem = None

   def allMicrosemiDevices( self ):
      devices = []
      for deviceId in self.microsemiDeviceIDs:
         devices.extend( Pci.allDevicesById( Pci.Id( self.microsemiVendorID,
                                                     deviceId ) ) )
      return [ d for d in devices if d != None ]

   def dataEndpoint( self, ):
      if self.dataEndpoint_ is None:
         devices = self.allMicrosemiDevices()
         dataDevices = [ d for d in devices if ( d.devfn() == 1 and
                                                 d.address().bus == 0x43 ) ]
         if dataDevices == [] :
            dataDevices = [ d for d in devices if ( d.devfn() == 1 and
                                                    d.address().bus == 5 ) ]

         assert dataDevices[ 0 ] != None, "No microsemi device is present"
         self.dataEndpoint_ = dataDevices[ 0 ].address()
         if self.verbose:
            print "Data endpoint is ", self.dataEndpoint_
      return self.dataEndpoint_

   def managementEndpoint( self ):
      if self.managementEndpoint_ is None:
         devices = self.allMicrosemiDevices()
         devices = [ d for d in devices if ( d.devfn() == 1 and
                                             d.address().bus == 5 ) ]
         assert devices[ 0 ] != None, "No microsemi device is present"
         self.managementEndpoint_ = devices[ 0 ].address()
         if self.verbose:
            print "Management endpoint is ", self.managementEndpoint_
      return self.managementEndpoint_

   def managementEndpointBar( self ):
      if self.microsemiFunctionBar is None:
         self.microsemiFunctionBar = Pci.Device (
            self.managementEndpoint() ).resource( self.microsemiBar )
      return self.microsemiFunctionBar


   def resetManagementEndpoint( self, BDF ):
      def deviceByBDF( BDF ):
         for device in Pci.allDevices():
            addr = device.address().value()
            thisDeviceSpec = "%02x:%02x.%01x" % ( addr.bus,
                                                  addr.slot,
                                                  addr.function )
            if thisDeviceSpec == BDF:
               return device
         assert 0, "Device does not exists"
         return None
      self.managementEndpoint_ = deviceByBDF( BDF ).address()
      self.microsemiFunctionBar_ = Pci.Device (
         self.managementEndpoint() ).resource( self.microsemiBar )


   def read8( self, offset ):
      return self.managementEndpointBar().read8( offset )

   def read32( self, offset ):
      return self.managementEndpointBar().read32( offset )

   def write8( self, offset, value ):
      return self.managementEndpointBar().write8( offset, value )

   def write32( self, offset, value ):
      return self.managementEndpointBar().write32( offset, value )


   def hardwareIsPresent( self ):
      try:
         self.managementEndpoint()
      # pylint: disable-msg=bare-except
      except:
         return False
      else:
         return True


   def showRegs( self ):
      return "INPUT=%x, OUTPUT=%x, COMMAND=%x, STATUS=%x, RET_VALUE=%x" % (
         self.read32( self.GAS_INPUT_DATA ),
         self.read32( self.GAS_OUTPUT_DATA ),
         self.read32( self.GAS_COMMAND ),
         self.read32( self.GAS_STATUS ),
         self.read32( self.GAS_RETURNVALUE ) )

   def gasWait( self, timeout=600 ):
      def status():
         status = self.read32( self.GAS_STATUS )
         assert status == self.GAS_DONE or status == self.GAS_INPROGRESS, \
            "MRPC FAULT: %s" % self.showRegs()
         return status
      Tac.waitFor( lambda: status() == self.GAS_DONE,
                   timeout=timeout )
      return self.read32( self.GAS_RETURNVALUE )


   # PFX Device Specification, chapter 7.7.2:
   # The process for using the MRPC is as follows:
   # 1. Write the input data parameters as defined in the command data structure
   #    into the Input Data region.
   # 2. Write the command values into command registers. This will begin the command
   #    execution by the CPU. MRPC status will be set to 0x1 ( MRPC_STAT_INPROGRESS )
   #    and the firmware calls the related command handler to process the command.
   #    If an unsupported command is sent, MRPC status will be set to
   #    0xFF( MRPC_STAT_ERROR ).
   # 3. After the command handler done, firmware will set the MRPC status to 0x2
   # ( MRPC_STAT_DONE ) and send MRPC Command Complete event to the host. The host
   # should read the MRPC status value after receiving the MRPC Command Complete
   # event, until it reads 0x2 ( MRPC_STAT_DONE )

   def doGas( self,
              argument, cmd ):
      if self.debug:
         print "cmd = %x, arg =%x\n" % ( cmd, argument )
      self.write32( self.GAS_INPUT_DATA, argument )
      self.write32( self.GAS_COMMAND, cmd )
      return self.gasWait()

   def getImageInfo( self, fileName ):
      with open( fileName ) as f:
         magic = f.read( 3 )
         assert magic == "PMC", "Wrong image given - %s" % fileName
         f.read( 1 )
         values = array.array( 'i' )
         values.read( f, self.microsemiBinaryHeaderSize )
         length = values[ 0 ]
         imgType = values[ 1 ]
         loadAddr = values[ 2 ]
         version = values[ 3 ]
         vendor = values[ 16 ] >> 24
         revision = self.swap16(
            values[ 17 ] >> 16 ) << 16 | self.swap16( values[ 18 ] )
         return ( length, imgType, loadAddr, version, vendor, revision )


   def firmwareDownload( self, fileName, BDF=None ):
      imageLength = os.path.getsize( fileName )

      if BDF is not None:
         self.resetManagementEndpoint( BDF )
         
      def downloadChunk( f ):
         self.cleanBufs()

         filePos = f.tell()
         offset = self.GAS_INPUT_DATA + self.downloadHeaderSize
         block = f.read( self.GAS_BLOCKSIZE - self.downloadHeaderSize )
         for byte in block:
            self.write8( offset, ord( byte ) )
            offset = offset + 1

         self.write32( self.MRPC_DNLD_ImageOffset, filePos )
         self.write32( self.MRPC_DNLD_ImageLength, imageLength )
         self.write32( self.MRPC_DNLD_DataLength, block.__len__() )
         assert self.doGas( self.MRPC_DNLD_Download,
                            self.MRPC_FWDNLD ) == self.PMC_SUCCESS

         print "Wrote chunk at %d length %d of total %d " % (
            filePos, block.__len__(), imageLength )

      def waitCompletion():
         def backgroundStatus():
            assert self.doGas( self.MRPC_DNLDSTATUS_Check,
                               self.MRPC_FWDNLD ) == self.PMC_SUCCESS
            return self.read8( self.GAS_OUTPUT_DATA +
                               self.MRPC_DNLDSTATUS_BackgroundStatus )

         Tac.waitFor( lambda: backgroundStatus() ==
                      self.MRPC_DNLDSTATUS_InProgress )

      with open( fileName ) as f:
         while f.tell() < imageLength:
            downloadChunk( f )
            waitCompletion()
      FpgaUtil.printToConsole( "File %s downloaded." % fileName )

   def updateConfig( self, fileName=None, force=False ):
      if fileName is None:
         fileName = self.configImage()
      if self.hardwareIsPresent() and os.path.exists( fileName ):
         ( _, _, _, version, _, revision ) = self.getImageInfo( fileName )
         hwRevision = self.read32( self.GAS_VendorTableRevision )
         hwVersion = self.read32( self.GAS_ConfigVersion )
         print "Customer table revision in file/hardware %x/%x, versions %x/%x" % (
            revision, hwRevision, version, hwVersion )
         if revision != hwRevision or version != hwVersion or force:
            FpgaUtil.printToConsole(
               "Customer table revision in file/hardware %x/%x, versions %x/%x,"
               " updating...\n" % ( revision, hwRevision, version, hwVersion ) )
            print "Updating customer table from %s..." % fileName
            self.firmwareDownload( fileName )
            print "A reboot required"
            return True # ask NorCal to restart supervisor
         else:
            print "Customer table revision in file is up to date"
      return False # dont ask NorCal to restart supervisor

   def updateFirmware( self, force=False ):
      if self.hardwareIsPresent() and os.path.exists( self.firmwareImage ) :
         ( _, _, _,
           version, _, _ ) = self.getImageInfo( self.firmwareImage )
         hwVersion = self.read32( self.GAS_FirmwareVersion )
         if version != hwVersion or force:
            FpgaUtil.printToConsole(
               "Firmware version in file is %x and in hardware is %x,"
               " updating...\n" % ( version, hwVersion ) )
            print "Updating firmware from ... %s" % self.firmwareImage
            self.firmwareDownload( self.firmwareImage )
            print "A reboot required"
            return True # ask NorCal to restart supervisor
         else:
            print "Microsemi firmware is up to date"
      return False # dont ask NorCal to restart supervisor

   def readGroup4( self, offset ):
      value = self.read32( offset )
      return ( value & 0xff,
               ( value >> 4 ) & 0xf,
               ( value >> 8 ) & 0xf,
               ( value >> 12 ) & 0xf,
               ( value >> 16 ) & 0xf,
               ( value >> 10 ) & 0xf,
               ( value >> 24 ) & 0xf,
               ( value >> 28 ) & 0xf )

   def readGroup8( self, offset ):
      value = self.read32( offset )
      return ( value & 0xff,
               ( value >> 8 ) & 0xff,
               ( value >> 16 ) & 0xff,
               ( value >> 24 ) & 0xff )

   def readGroup16( self, offset ):
      value = self.read32( offset )
      return ( value & 0xffff,
               ( value >> 16 ) & 0xffff )

   def cleanBufs( self ):
      for offset in range( 0, self.GAS_OUTPUT_DATA - 4, 4 ):
         self.write32( offset, 0 )

   def getBits( self, value, fro, to ):
      return ( value >> to ) & ( ( 1<< ( fro - to + 1 ) ) - 1 )

   def allPorts( self,
                 partition=0 ):
      ports = dict()
      assert self.doGas( partition << 8 | self.MRPC_P2P_INFO,
                         self.MRPC_PORTPARTP2P ) == 0
      for offset in range( self.GAS_OUTPUT_DATA, self.GAS_OUTPUT_DATA + 0x10, 4 ):
         ( p0, s0, p1, s1 ) = self.readGroup8( offset )
         if ( p0 != 0 and p0 != 255 and s0 != 0 ):
            ports[ p0 ] = 1
         if ( p1 != 0 and p1 != 255 and s1 != 0 ):
            ports[ p1 ] = 1
      return ports.keys()

   def swap16( self, x ):
      return ( ( x >> 8 ) & 0xff ) | ( ( x << 8 ) & 0xff00 )

   def getVersion( self, offset ):
      rawValue = self.read32( offset )
      return "%x.%x.%x" % ( rawValue >> 24,
                            ( rawValue >> 16 ) & 0xff,
                            rawValue & 0xffff )

   def linksStates( self ):
      states = dict()
      self.cleanBufs()
      self.doGas( 0, self.MRPC_LNKSTAT )

      for offset in range(
            self.GAS_OUTPUT_DATA,
            self.GAS_OUTPUT_DATA + self.SWITCHTEC_MAX_PORTS * 12,
            12 ):
         ( physPortId, partId, logPortId, stkID ) = self.readGroup8( offset )
         ( cfgLinkWidth, negLinkWidth, usp, linkrate ) = self.readGroup8(
            offset + 4 )
         ( LTSSM, _ ) = self.readGroup16( offset + 8 )

         states[ physPortId ] = { 'partID': partId,
                                  'logPortId' : logPortId,
                                  'stkID': stkID,
                                  'cfgLinkWidth': cfgLinkWidth,
                                  'negLinkWidth': negLinkWidth,
                                  'usp': usp,
                                  'linkrate': linkrate,
                                  'LTSSM': LTSSM }
      return states

   def NTPartitionGetInfo( self, nt ):
      part = dict()

      ntBase = self.NTxBASE( nt )
      part[ 'ntBase' ] = ntBase

      ( part[ 'ntStatL' ], part[ 'ntStatH' ],
        part[ 'lockedId' ], part[ 'partID' ] ) = self.readGroup8( ntBase )
      ( part[ 'ntOpc' ], _ ) = self.readGroup16( ntBase + 4 )
      part[ 'ntStat' ] = part[ 'ntStatH' ] << 8 | part[ 'ntStatL' ]
      part[ 'ntControl' ] = self.read32( ntBase + 8 )
      ( part[ 'barNum' ], part[ 'barOffset' ] ) = self.readGroup16(
         ntBase + 0xc )
      ( part[ 'ntErrorIndex' ], part[ 'ntError' ] ) = self.readGroup16(
         ntBase + 0x10 )
      part[ 'ntTableError' ] = self.read32( ntBase + 0x18 )
      part[ 'ntRequesterError' ] = self.read32( ntBase + 0x20 )
      for n in self.MRPC_PARTITIONS:
         value = self.read32( self.requesterTable( nt ) + 4 * n )
         part[ 'Requester%d' % n ] = {
            'value': value,
            'enabled': value & 1 == 1,
            'NTProxy': value >> 1 & 0xf,
            'RequesterID': value >> 16
         }
      return part

   def NTBBarGetInfo( self, nt, ntBarNo ):
      bar = dict()
      dw = range( 4 )
      for idx in range( 0, 4 ):
         ( barNum, barOffset ) = self.readGroup16( self.NTxBASE( nt ) + 0xc )
         dw[ idx ] = self.read32( self.NTxBASE( nt ) + barOffset +
                                  ntBarNo * 0x10 + idx * 4 )

         if dw[ 0 ] & 1:
            bar[ 'valid' ] = True

            if dw[ 0 ] & 0b110 == 0:
               bar[ 'mode' ] = "32-bit "
            if dw[ 0 ] & 0b110 == 0b100:
               bar[ 'mode' ] = "64-bit "

            bar[ 'prefetch' ] = bool( dw[ 0 ] & 0b1000 )

            if dw[ 0 ] & 0b10000:
               bar[ 'mappingType' ] = "NT-Direct "
               bar[ 'NtTranslationPosition' ] = 1 << ( dw[ 1 ] & 0x3F )
               bar[ 'NtTranslationSize' ] = ( dw[ 1 ] >> 12 ) * 4 * 1024
               bar[ 'NtDestinationPartition' ] = dw[ 2 ] & 0x3f
               bar[ 'BaseAddr' ] = dw[ 3 ] << 32 | ( ( dw[ 2 ] >> 12 ) << 12 )
            else:
               if dw[ 0 ] & 0b100000:
                  bar[ 'mappingType' ] = "NT-LUT "
               else:
                  del bar[ 'valid' ]

         # This is for debugging
         bar[ 'dw' ] = dw
         bar[ 'barNum' ] = barNum
         bar[ 'barOffset' ] = barOffset
      return bar

   def bind( self,
             port,
             DSP=1, # DSPs 1,2,3 exists on eval board
             partition = 0 ):
      status = self.doGas( self.MRPC_P2P_BIND | int( partition ) << 8 |
                           int( DSP ) << 16 | int( port ) << 24,
                           self.MRPC_PORTPARTP2P )
      return status

   def ntReset( self ):
      for nt in self.MRPC_PARTITIONS:
         def inReset( x ):
            self.write32( self.ntOp( x ), self.MRPC_NTOP_RESET )
            return self.NTPartitionGetInfo( x )[ 'ntStat' ] == 5
         Tac.waitFor( lambda x=nt: inReset( x ), timeout=60 )

   def ntEnable( self ):
      def setNTBar( nt ):
         dw = [ self.MRPC_NT_LUT_WIN_ENABLE | self.MRPC_NT_TYPE_64BIT |
                self.MRPC_NTBARSETUP_VALID,
                self.MRPC_NT_WIN_DIR_SIZE | self.MRPC_NT_DIR_XLATE_POS,
                None,
                self.MRPC_NT_DIR_XLATE_BASE_ADDR_HIGH ]
         # Set target partition
         dw[ 2 ] = 1 - nt
         barOffset = self.MRPC_NTBAR * self.MRPC_NTBARSETUPSIZE
         for idx in range( 0, 4 ):
            self.write32( self.NTxBASE( nt ) + 0xc, barOffset )
            self.write32( self.NTxBASE( nt ) + barOffset +
                          self.MRPC_NTBAR * self.MRPC_NTBARSETUPSIZE + idx * 4,
                          dw[ idx ] )

      def ntIsSet( nt ):
         NTBInfo = self.NTBBarGetInfo( nt, self.MRPC_NTBAR )
         partInfo = self.NTPartitionGetInfo( nt )
         return ( 'valid' in NTBInfo and
                  partInfo[ 'Requester0' ][ 'enabled' ] and
                  partInfo[ 'Requester1' ][ 'enabled' ] )

      def ntSetup():
         for nt in self.MRPC_PARTITIONS:
            if not ntIsSet( nt ):
               self.write32( self.ntOp( nt ), self.MRPC_NTOP_LOCK )
               setNTBar( nt )
               # Translate BDF 0:0.0 -> x:0.1 (Root Complex)
               self.write32( self.requesterTable( nt ) + 0,
                             self.requesterMask( 0, 0, 1, 0 ) )
               # Translate BDF 0:3.0 -> x:1.1 (Host Bridge)
               self.write32( self.requesterTable( nt ) + 4,
                             self.requesterMask( 3, 0, 1, 1 ) )
               self.write32( self.ntOp( nt ), self.MRPC_NTOP_CONF )
         return ntIsSet( 0 ) and ntIsSet( 1 )

      # Setup NTBs to access between sup1 and sup2
      try:
         Tac.waitFor( ntSetup, timeout=5 * 60 )
         assert ntIsSet( 0 ) and ntIsSet( 1 )
      # pylint: disable-msg=bare-except
      except:
         for nt in self.MRPC_PARTITIONS:
            if not ntIsSet( nt ):
               printToFile(
                  "Microsemi: NTB%d is not set, peer is missing" % nt )

   def initPicasso( self ):
      if Cell.cellId() == 1:
         busNo = 0x5
      else:
         busNo = 0x43
      PicassoInit.picassoSetNtBus( busNo, ntFunctionNo=1 )

   def startup( self ):
      if self.hardwareIsPresent():
         if not os.path.exists( "/mnt/flash/skipEepromUpgrade" ):
            print >> sys.stderr, "Updating microsemi eeprom if neccessary"
            configIsUpdated = self.updateConfig()
            firmwareIsUpdated = self.updateFirmware()
            if ( configIsUpdated or firmwareIsUpdated ):
               FpgaUtil.printToConsole( "Microsemi EEPROM Upgraded" )
               Tac.run( [ 'reboot' ] )
               # We don't want NorCalInit to continue execution before the switch
               # is rebooted. As such, we do an infinite loop here to prevent the
               # script from going forward.
               while True:
                  time.sleep( 1 )

         self.initPicasso()

         Tac.run( [ "/sbin/modprobe", "microsemi",
                    ( "microsemi_debug=%d" % 0 ) ] )
