# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Logging

traceHandle = Tracing.Handle( 'MgmtSecuritySslStatusSm' )
error = traceHandle.trace0
warn  = traceHandle.trace1
info  = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
SslFeature = Tac.Type( "Mgmt::Security::Ssl::SslFeature" )
SslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

SSL_FEATURES = {
   SslFeature.sslFeatureCertKey: ( 'certificate', 'certKeyPath', '' ),
   SslFeature.sslFeatureChainedCert: ( 'chain certificate', 'chainedCerts',
                                       False ),
   SslFeature.sslFeatureTrustedCert: ( 'trust certificate', 'trustedCertsPath',
                                       '' ),
   SslFeature.sslFeatureCrl: ( 'crl', 'crlsPath', '' ),
   SslFeature.sslFeatureTls: ( 'tls versions', 'tlsVersion',
                               SslConstants.allTlsVersion ),
   SslFeature.sslFeatureFips: ( 'fips-mode', 'fipsMode', False ),
   SslFeature.sslFeatureCipher: ( 'cipher-list', 'cipherSuite',
                                  SslConstants.defaultCipherSuite() )
}

Logging.logD( id="SECURITY_SSL_PROFILE_FEATURE_UNSUPPORTED",
              severity=Logging.logWarning,
              format=( "SSL profile '%s' has '%s' configured which is not "
                       "supported by the '%s' agent." ),
              explanation=( "The agent is configured to use an SSL profile "
                            "whose feature-set it does not fully support." ),
              recommendedAction=( "Reset the unsupported feature in the SSL "
                                  "profile or create a new profile without "
                                  "the feature for use by the agent." ) )

class SslProfileStatusSm( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::ProfileStatus'

   def __init__( self, profileStatus, statusSm, callBackNow ):
      trace( "SslProfileStatusSm.__init__ start" )
      self.profileStatus_ = profileStatus
      self.statusSm_ = statusSm
      Tac.Notifiee.__init__( self, self.profileStatus_ )
      # Notify initial state
      info( "Initial state is", self.profileStatus_.state )
      if callBackNow:
         self.statusSm_.handleProfileState()
      trace( "SslProfileStatusSm.__init__ end" )

   @Tac.handler( 'state' )
   def handleState( self ):
      trace( "SslProfileStatusSm.handleState start. State is",
             self.profileStatus_.state )
      if self.profileStatus_.state == ProfileState.valid:
         for feature, ( cmd, attrName, defaultValue ) in SSL_FEATURES.iteritems():
            if feature in self.statusSm_.__supportedFeatures__:
               continue
            if getattr( self.profileStatus_, attrName ) == defaultValue:
               continue
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_PROFILE_FEATURE_UNSUPPORTED,
                         self.statusSm_.profileName_,
                         cmd,
                         self.statusSm_.agentName_ )
      # Ignore the transitionary state
      if self.profileStatus_.state != ProfileState.updating:
         self.statusSm_.handleProfileState()
      trace( "SslProfileStatusSm.handleState end" )

   def close( self ):
      if not self.deleted_:
         Tac.Notifiee.close( self )

class SslStatusSm( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::Status'
   __supportedFeatures__ = []

   def __init__( self, sslStatus, profileName, agentName, callBackNow=True ):
      trace( "SslStatusSm.__init__ start" )
      self.sslStatus_ = sslStatus
      self.profileName_ = profileName
      self.agentName_ = agentName
      self.profileStatusSm_ = None 
      self.profileStatus_ = None
      
      Tac.Notifiee.__init__( self, self.sslStatus_ )
      for profileName in self.sslStatus_.profileStatus:
         self.handleProfileStatus( profileName, callBackNow=callBackNow )
      trace( "SslStatusSm.__init__ end" )
   
   @Tac.handler( 'profileStatus' )
   def handleProfileStatus( self, profileName, callBackNow=True ):
      trace( "SslStatusSm.handleProfileStatus start for: ", profileName )
      
      if profileName != self.profileName_:
         return
      
      if profileName in self.sslStatus_.profileStatus:
         info( "profile status added:", profileName )
         self.profileStatus_ = self.sslStatus_.profileStatus[ profileName ]
         self.profileStatusSm_ = SslProfileStatusSm( self.profileStatus_, 
                                                     self, callBackNow=callBackNow )
      else:
         info( "profile status deleted:", profileName )
         if self.profileStatusSm_:
            self.profileStatusSm_.close()
            self.profileStatusSm_ = None
         self.profileStatus_ = None
         self.handleProfileDelete()
      trace( "SslStatusSm.handleProfileStatus end for: ", profileName )

   @Tac.handler( 'dhparamsResetProcessed' )
   def handleDhparamsResetProcessed( self ):
      # The decorator function is not overridable
      # hence the indirection
      self.handleDhparamsReset()
   
   def handleDhparamsReset( self ):
      # The derived class should override this to
      # get dhparams reset notification
      pass

   def handleProfileState( self ):
      # The derived class should override this to
      # get profile state change notification
      raise NotImplementedError()

   def handleProfileDelete( self ):
      # The derived class should override this to
      # get profile delete notification
      raise NotImplementedError()
   
   def close( self ):
      if self.profileStatusSm_:
         self.profileStatusSm_.close()
         self.profileStatusSm_ = None
      Tac.Notifiee.close( self )
