# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import random
import BothTrace
import McsApiModels
import Tac
import Tracing

info = BothTrace.tracef0
warn = BothTrace.tracef1
error = BothTrace.tracef2
log = BothTrace.tracef3
__defaultTraceHandle__ = Tracing.Handle( 'McsStateHAReplicate' )

class UpdatingApiConfig( object ):
   def __init__( self, status, sync=False ):
      self.status = status
      self._sync = sync

   def __enter__( self ):
      if self._sync:
         info( "Setting syncInProgress to True" )
         self.status.syncInProgress = True
      else:
         info( "Setting updateInProgress to True" )
         self.status.updateInProgress = True

   def __exit__( self, exc, excVal, tb ):
      if self._sync:
         info( "Setting syncInProgress to False" )
         self.status.syncInProgress = False
      else:
         info( "Setting updateInProgress to False" )
         self.status.updateInProgress = False
      if tb:
         raise exc, excVal, tb

   def __call__( self, func ):
      def wrapper( *args, **kwargs ):
         with self:
            return func( *args, **kwargs )
      return wrapper

class McsStateReplicate( object ):
   def __init__( self, apiConfig, clusterStatus, apiStatus, cliStatus, mcsStatusHA ):
      self.apiConfig = apiConfig
      self.clusterStatus = clusterStatus
      self.apiStatus = apiStatus
      self.cliStatus = cliStatus
      self.mcsStatusHA = mcsStatusHA

   def compareObjects( self, obj1, obj2 ):
      defaultAttrs = [ 'name', 'fullName', 'parent', 'parentAttrName', 'entity',
                       'hash' ]
      if Tac.isCollection( obj1 ) and Tac.isCollection( obj2 ):
         if len( obj1 ) != len( obj2 ):
            return False

         for key, value in obj1.iteritems():
            res = self.compareObjects( value, obj2[ key ] )
            if not res:
               return False
         return True

      try:
         objAttrs = obj1.attributes
         attrs = set( objAttrs ) - set( defaultAttrs )
         for attr in attrs:
            res = self.compareObjects( getattr( obj1, attr ), getattr( obj2, attr ) )
            if not res:
               return False
         return True
      except AttributeError:
         return obj1 == obj2

   def replicate( self ):
      return UpdatingApiConfig( self.apiStatus )

   def sync( self ):
      return UpdatingApiConfig( self.apiStatus, sync=True )

   def syncApiConfig( self, sync=True ):
      log( "mountDone: %s" % self.cliStatus.mountDone )
      if not self.cliStatus.mountDone:
         info( "Mount is not done. Not starting API Config sync process" )
         return

      info( "Start API Config sync" )

      @self.sync()
      def doSync():
         # Run the code if the cluster is leader.
         if not self.clusterStatus.status[ 'default' ].isStandaloneOrLeader:
            info( 'CVX is not leader. Do not update API config' )
            return
         info( 'Updating ApiConfig after CVX leadership change' )

         # Sort client based on the update count
         updateCount = sorted( self.mcsStatusHA.items(),
                               key=lambda x: x[ 1 ].updateCount )
         # Find client that are in sync
         clientsInSync = [ client for client, entity in self.mcsStatusHA.items()
                           if entity.clientInSync ]
         info( "clientsInSync: %s" % clientsInSync )
         info( "updateCount: %s" % updateCount )

         # Now select client from clientsInSync or updateCount
         client = random.choice( clientsInSync ) if clientsInSync else (
            updateCount[ -1 ][ 0 ] if updateCount else [] )
         info( "Selecting client %s as a source to sync API config" % client )
         if not ( client and self.mcsStatusHA[ client ].updateCount ):
            warn( "There is no data available in the client for sync" )
            return

         newApiConfig = self.mcsStatusHA[ client ]

         # Do clean up before updating the apiConfig collection.
         # 1. Remove entries from old apiConfig that do not exist in new apiConfig,
         # 2. Check the common members of collection in old and new config are the
         #    same, and remove them from apiConfig if they are not the same,
         # 3. In the last step, only update apiConfig with missing entries.

         # Delete OUIs that do not exist in recent config
         expectedVendors = newApiConfig.vendor.keys()
         existedVendors = self.apiConfig.vendor.keys()
         toBeDeleted = set( existedVendors ) - set( expectedVendors )
         for oui in toBeDeleted:
            log( "Deleting OUI: %s" % oui )
            del self.apiConfig.vendor[ oui ]
         # Check the common members with the same key have the same content.
         # If the values are not the same delete them from collection.
         commonMembers = set( existedVendors ) & set( expectedVendors )
         for oui in commonMembers:
            if not self.compareObjects( newApiConfig.vendor[ oui ],
                                        self.apiConfig.vendor[ oui ] ):
               log( "OUI: %s IS THE NOT THE SAME. DELETING" % oui )
               # Delete the entry
               del self.apiConfig.vendor[ oui ]

         # Delete McastSenders that do not exist in recent config
         expectedMcastSenders = newApiConfig.mcastSender.keys()
         existedMcastSenders = self.apiConfig.mcastSender.keys()
         toBeDeleted = set( existedMcastSenders ) - set( expectedMcastSenders )
         for sender in toBeDeleted:
            log( "Deleting sender: %s" % sender )
            del self.apiConfig.mcastSender[ sender ]

         # Check common members for mcastSenders are the same
         commonMembers = set( existedMcastSenders ) & set( expectedMcastSenders )
         for sender in commonMembers:
            if not self.compareObjects( newApiConfig.mcastSender[ sender ],
                                    self.apiConfig.mcastSender[ sender ] ):
               log( "SENDER: %s IS THE NOT THE SAME. DELETING" % sender )
               del self.apiConfig.mcastSender[ sender ]

         # Delete McastReceivers that do not exist in recent config
         expectedMcastReceivers = newApiConfig.mcastReceiver.keys()
         existedMcastSenders = self.apiConfig.mcastReceiver.keys()
         toBeDeleted = set( existedMcastSenders ) - set( expectedMcastReceivers )
         for receiver in toBeDeleted:
            log( "Deleting receiver: %s" % receiver )
            del self.apiConfig.mcastReceiver[ receiver ]

         # Check common members in mcastReceiver are the same
         commonMembers = set( existedMcastSenders ) & set( expectedMcastReceivers )
         for receiver in commonMembers:
            if not self.compareObjects( newApiConfig.mcastReceiver[ receiver ],
                                        self.apiConfig.mcastReceiver[ receiver ] ):
               log( "Deleting receiver: %s" % receiver )
               del self.apiConfig.mcastReceiver[ receiver ]

         # Delete ReservationPercentage that do not exist in recent config
         expectedRp = newApiConfig.reservationPercentage.keys()
         existedRp = self.apiConfig.reservationPercentage.keys()
         toBeDeleted = set( existedRp ) - set( expectedRp )
         for rp in toBeDeleted:
            log( "Deleting ReservationPercentage: %s" % rp )
            del self.apiConfig.reservationPercentage[ rp ]
         # Check the common members with the same key have the same content.
         # If the values are not the same delete them from collection.
         commonMembers = set( existedRp ) & set( expectedRp )
         for rp in commonMembers:
            if not self.compareObjects( newApiConfig.reservationPercentage[ rp ],
                                        self.apiConfig.reservationPercentage[ rp ] ):
               # Delete the entry
               del self.apiConfig.reservationPercentage[ rp ]

         # Update only missing OUI
         for oui in ( set( newApiConfig.vendor.keys() ) -
                      set( self.apiConfig.vendor.keys() ) ):
            ouiData = { 'oui': oui,
                        'vendorName': newApiConfig.vendor[ oui ].vendorName }
            log( "Updating OUI with: %s" % ouiData )
            model = McsApiModels.OuiModel()
            model.populateModelFromJson( ouiData )
            model.toSysdb( self.apiConfig )

         # Update only missing McastSender
         for mckey in ( set( newApiConfig.mcastSender.keys() ) -
                        set( self.apiConfig.mcastSender.keys() ) ):
            mcs = newApiConfig.mcastSender[ mckey ]
            if not mcs.senderId:
               continue
            mcsData = { 'flow-action': 'addSenders',
                        'transactionID': mcs.transactionId,
                        'trackingID': mcs.trackingId,
                        'data': [
                           { 'destinationIP': mckey.group,
                             'sourceIP': mckey.source,
                          'bandwidth': int( mcs.bw.value ),
                          'bwType': 'k',
                          'inIntfID': "%s-%s" % ( mcs.senderId.device,
                                                   mcs.senderId.intfId )
                        } ]
                   }
            log( "Updating McastSender with: %s" % mcsData )
            model = McsApiModels.McastSenderModel()
            model.populateModelFromJson( mcsData )
            model.toSysdb( self.apiConfig )

         # Update only missing McastReceiver
         for mckey in ( set( newApiConfig.mcastReceiver.keys() ) -
                        set( self.apiConfig.mcastReceiver.keys() ) ):
            mcr = newApiConfig.mcastReceiver[ mckey ]
            for rkey, receiver in mcr.receivers.items():
               for intfKey, intfs in receiver.recvIntfs.items():
                  mcrData = { 'flow-action': 'addReceivers',
                              'transactionID': intfs.transactionId,
                              'trackingID': intfs.trackingId,
                              'data': [
                                 { 'destinationIP': mckey.group,
                                   'sourceIP': mckey.source,
                                   'outIntfID': [ "%s-%s" % ( rkey, intfKey ) ]
                                 } ]
                            }
                  log( "Updating McastReceiver with: %s" % mcrData )
                  model = McsApiModels.McastReceiverModel()
                  model.populateModelFromJson( mcrData )
                  model.toSysdb( self.apiConfig )

         # Update ReservationPercentage entity when api model is finalized
         for rp in ( set( newApiConfig.reservationPercentage.keys() ) -
                      set( self.apiConfig.reservationPercentage.keys() ) ):
            rpData = { 'chassis-id': rp.device,
                       'interface-name': rp.intfId,
                       'reservation-percent': str( newApiConfig
                                                    .reservationPercentage[ rp ]
                                                    .percentValue ) }
            log( "Updating ReservationPercentage with: %s" % rpData )
            model = McsApiModels.RPModel()
            model.populateModelFromJson( rpData )
            model.toSysdb( self.apiConfig )

      # Start the sync
      doSync()
