# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ApiBaseModels import ModelJsonSerializer
from McsHttpErrorCodes import errorCodes
from GenericReactor import GenericReactor
from UwsgiRequestContext import ( HttpNotFound,
                                 UwsgiRequestContext )
from ControllerdbEntityManager import Controllerdb
from CliPlugin.IntfModel import intfOperStatusToEnum
from CliPlugin.IntfCli import _forwardingModelMap
from CliPlugin.LldpStatusCli import formatAddr
from httplib import OK
from httplib import NO_CONTENT
from httplib import responses

import Agent
import Tac
import Tracing
import BothTrace
import UwsgiConstants
import UrlMap
import McsApiModels
import simplejson
import Arnet
import os
import McsHttpAgentLib
import McsStateHAReplicate

info = BothTrace.tracef0
warn = BothTrace.tracef1
error = BothTrace.tracef2
log = BothTrace.tracef3



bv = BothTrace.Var
__defaultTraceHandle__ = Tracing.Handle( 'McsHttpAgent' )

class McsControllerMounts( object ):
   def __init__( self, cdbMg ):

      self.cdbTopology = cdbMg.mount( 'topology/version3/global/status',
                                      'NetworkTopologyAggregatorV3::Status', 'r' )
      self.intfStatusDir = cdbMg.mount( 'mcs/v1/fromSwitch/intfStatus',
                                       'Tac::Dir', 'ri' )
      self.intfConfigDir = cdbMg.mount( 'mcs/v1/fromSwitch/intfConfig',
                                       'Tac::Dir', 'ri' )
      self.mcsLocalStatus = Tac.newInstance( "Mcs::AgentStatus",
            "McsHttpAgentLocalStatus" )
      self.cdbIntfStatus = self.mcsLocalStatus.allEthPhyIntfStatusDev
      self.cdbIntfConfig = self.mcsLocalStatus.allEthPhyIntfConfigDev

      self.cdbHw = cdbMg.mount( 'mcs/v1/fromSwitch/hardware',
                                       'Tac::Dir', 'ri' )
      self.cdbCell = cdbMg.mount( 'mcs/v1/fromSwitch/cellStatus',
                                       'Tac::Dir', 'ri' )
      self.cdbReboot = cdbMg.mount( 'mcs/v1/fromSwitch/rebootDir',
                                       'Tac::Dir', 'ri' )
      self.cdbSwitchIntfConfig = cdbMg.mount( 'mcs/v1/fromSwitch/switchIntfConfig',
                                             'Tac::Dir', 'ri' )
      self.cdbMroute = cdbMg.mount( 'mcs/v1/fromSwitch/mcastCommon',
                                 'Tac::Dir', 'ri' )
      self.cdbLldpLocal = cdbMg.mount( 'mcs/v1/fromSwitch/lldpLocal',
                                       'Tac::Dir', 'ri' )
      self.cdbIpInputStatus = cdbMg.mount( 'mcs/v1/fromSwitch/ipInputStatus',
                                       'Tac::Dir', 'ri' )
      self.mcsStatusHA = cdbMg.mount( "mcs/v1/apiCfgRedStatus/switch",
                                      "Tac::Dir", "ri" )

class McsApiBase( object ):
   """All classes that handle a particular URL should be derived from this class"""
   def __init__( self ):


      # The decorator UrlMap.urlHandler won't register the actual object. So, this is
      # a hack to replace the preregistered method for the url with a tuple of the
      # object and the method name. This helps McsApiHandler in calling the method on
      # the real object
      uri = UrlMap.UrlToRegex()( self.api )

      for httpType, reqFunc in UrlMap.urlMap.iteritems():
         if uri in reqFunc:
            f = UrlMap.urlMap[ httpType ][ uri ]
            UrlMap.urlMap[ httpType ][ uri ] = ( self, f.__name__ )

   def get_resp_template( self ):
      return { 'messages': [],
               'success': False
               }

   def get_mcast_resp_template( self ):
      return { 'messages': [],
               'success': False,
               'status': {} }

   def get_mcast_sender_template( self ):
      return { 'messages': [],
               'success': False,
               'trackingID': 0,
               'transactionID': 'AA',
               'receivers': { },
               'senders': { 'action': '',
                           'failedSenders': [],
                           'validSenders': []
                           }
                  }

   def get_mcast_recv_template( self ):
      return { 'messages': [],
               'success': False,
               'trackingID': 0,
               'transactionID': 'AA',
               'receivers': { 'TimeToSetGrpc': '0',
                           'TimeToConfigureMulticastFlow': '0',
                           'TimeToDetermineBwAndRoute': '0',
                           'TimeToGetDbData': '0',
                           'TimeToSetBwUpdate': '0',
                           'messages': [],
                           'failedReceivers': [],
                           'validReceivers': []

                           },
               'senders': { }
                  }
   @property
   def api( self ):
      raise NotImplementedError

class ApiStatus( McsApiBase ):
   api = "/mcs/apiStatus[/]"

   def __init__( self, apiStatus ):
      McsApiBase.__init__( self )
      self.apiStatus = apiStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getApiStatus( self, requestContext ):
      log( "Getting API status " )
      model = McsApiModels.ApiStatusModel()
      model.fromSysdb( self.apiStatus )
      result = simplejson.loads( simplejson.dumps( model, cls=ModelJsonSerializer ) )
      if result:
         return OK, result
      else:
         return NO_CONTENT, None


class PostMcastSender( McsApiBase ):
   api = "/mcs/multicast/senders[/]"

   def __init__( self, apiConfig, apiNotify ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiNotify = apiNotify
      self.MCAST_FLOW_KEYS = [ 'flow-action', 'trackingID', 'data', 'transactionID' ]
      self.SENDER_FIELDS = [ 'destinationIP', 'sourceIP', 'bandwidth', 'inIntfID',
                              'bwType' ]
      self.bwTypeMap = { 'k': 1,
            'm': 1000,
            'g': 1000000 }
      self.senderActionMap = { 'addSenders': 'set',
                            'delSenders': 'del',
                            'modBW': 'modBw' }

   def sendNotify( self, postData, errorCode=None, sender=None, isFailure=True ):
      """Send failure or delete notification messages for a sender requests"""

      action = postData.get( 'flow-action' )
      action = self.senderActionMap.get( action ) if action else 'U/A'
      messages = ( errorCode if isinstance( errorCode, list )
                   else [ errorCode ] if errorCode else [] )
      data = ( sender if isinstance( sender, list )
                         else [ sender ] if sender else [] )
      if isFailure:
         # Adding FailedSenders in post error code
         messages.extend( [ '185' ] )

      # Add the deleted senders, valid and failed, to the message queue
      notifyData = {
         'action': action,
         'trackingID': postData.get( 'trackingID', 0 ),
         'messages': messages,
         'data': data
      }
      notifyType = Tac.Type( "Mcs::ApiNotifyMsgType" )
      notify = Tac.Value( "Mcs::ApiNotifyMsg", notifyType.delSender,
                             simplejson.dumps( notifyData ) )
      self.apiNotify.msgQ.enq( notify )

   def validateSenders( self, postData ):
      # returns storedSenders, senderUpdate, debugMsgs

      storedSenders = self.apiConfig.mcastSender.keys()

      senderUpdate = self.get_mcast_resp_template()
      senderUpdate[ 'status' ] = self.get_mcast_sender_template()
      ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.MCAST_FLOW_KEYS,
                                                         postData )

      # Validate Sender Post Fields
      if ec:
         senderUpdate[ 'status' ][ 'messages' ].append( ec )
         if 'data' in missingKeys:
            failedSender = []
         else:
            failedSender = postData[ 'data' ]
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] = failedSender
         if 'flow-action' in missingKeys:
            senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = 'U/A'

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=failedSender )
         return storedSenders, senderUpdate

      flowAction = postData[ 'flow-action' ]
      if flowAction in self.senderActionMap:

         senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = \
                                    self.senderActionMap[ flowAction ]
      else:
         senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = 'U/A'
         ec = '113'
         error( errorCodes[ ec ].replace( '<postedAction>',
                                             flowAction ) )

         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += \
                                                                  postData[ 'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=postData[ 'data' ] )
         return storedSenders, senderUpdate

      try:
         assert isinstance( postData[ 'data' ], list )
      except AssertionError:
         ec = '121'
         error( errorCodes[ ec ] )
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += \
                                                                  postData[ 'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=postData[ 'data' ] )
         return storedSenders, senderUpdate

      for sender in postData[ 'data' ]:
         if 'messages' not in sender:
            sender[ 'messages' ] = []

         ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.SENDER_FIELDS,
                                                            sender )
         if ec:
            sender[ 'messages' ].append( ec )
            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )

            # Send notification
            self.sendNotify( postData, sender=sender )
            continue
         sourceIP = sender[ 'sourceIP' ]
         destinationIP = sender[ 'destinationIP' ]

         valid, m = McsHttpAgentLib.validateMcastGroup( destinationIP )
         if not valid:
            if m:
               sender[ 'messages' ].append( m )

            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         valid, m = McsHttpAgentLib.validateSourceIP( sourceIP )
         if not valid:
            if m:
               sender[ 'messages' ].append( m )

            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         valid, ec = McsHttpAgentLib.validateDeviceID( sender[ 'inIntfID' ] )
         if not valid:
            if ec != '137':
               error( errorCodes[ ec ] )
            else:
               error( errorCodes[ ec ].replace( '<deviceId>',
                           sender[ 'inIntfID' ].split( '-' )[ 0 ] ) )
            sender[ 'messages' ].append( ec )
            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         device, intfId = sender[ 'inIntfID' ].split( '-' )
         postedBw = sender[ 'bandwidth' ]

         valid = McsHttpAgentLib.validateBandwidth( postedBw )
         if not valid:
            ec = '123'
            error( errorCodes[ ec ] )
            sender[ 'messages' ].append( ec )
            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         valid = McsHttpAgentLib.validateBandwidthType( sender[ 'bwType' ] )
         if not valid:
            ec = '124'
            error( errorCodes[ ec ].replace( '<bwType>', sender[ 'bwType' ] ) )
            sender[ 'messages' ].append( ec )

            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         parsedBw = postedBw * self.bwTypeMap[ sender[ 'bwType' ] ]

         mcastKey = Tac.Value( "Mcs::McastKey", sourceIP, destinationIP )
         storedBw = 0
         if mcastKey in storedSenders:

            storedBw = self.apiConfig.mcastSender[ mcastKey ].bw.value
            currentDevice = self.apiConfig.mcastSender[ mcastKey ].senderId.device
            currentIntfId = self.apiConfig.mcastSender[ mcastKey ].senderId.intfId
            if currentDevice != device or currentIntfId != intfId:
               ec = '122'
               wordMap = { '<deviceId>': device,
                           '<port>': intfId }

               info( parsedBw )
               error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )
               sender[ 'messages' ].append( ec )
               senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                                                                        sender )
               # Send notification
               self.sendNotify( postData, sender=sender )
               continue

            if flowAction != "modBW":
               if storedBw != parsedBw:
                  ec = '120'
                  wordMap = { '<sourceIP>': sourceIP,
                              '<destinationIP>': destinationIP,
                              '<actualBandwidth>': storedBw }
                  info( parsedBw )
                  error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                                                                           sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue

            if flowAction == 'addSenders':
               if storedBw == parsedBw:
                  ec = '119'
                  info( errorCodes[ ec ] )
                  sender[ 'messages' ].append( ec )
                  # Send notification
                  self.sendNotify( postData, sender=sender )

         if '122' in sender[ 'messages' ]:
            continue

         setBwType = "k"
         sender[ 'bandwidth' ] = parsedBw
         sender[ 'bwType' ] = setBwType
         senderUpdate[ 'status' ][ 'senders' ][ 'validSenders' ].append( sender )

      senderUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
      senderUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
      senderUpdate[ 'success' ] = True

      return storedSenders, senderUpdate

   @UrlMap.urlHandler( ( 'POST', ), api )
   def postMcastSenders( self, requestContext ):
      """
      Sample post:
      {
           "flow-action": "addSenders",
            "transactionID": "test",
            "trackingID": 69,
            "data": [{
                       "destinationIP": "224.2.0.11",
                       "sourceIP": "10.1.1.100",
                       "bandwidth": 500, "bwType": "m",
                       "inIntfID": "2899.3a8f.998d-Ethernet6"
                   }]}
      """
      info( "Configuring Multicast Sender" )
      # XXX TODO: Log trackingID to track posts
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      info( 'Received data:', bv( data ) )

      storedSenders, senderUpdate = self.validateSenders( data )
      validSenders = senderUpdate[ 'status' ][ 'senders' ][ 'validSenders' ]
      if validSenders:
         action = senderUpdate[ 'status' ][ 'senders' ][ 'action' ]

         log( data )
         if action == 'set':
            model = McsApiModels.McastSenderModel()
            model.populateModelFromJson( data )
            if model.getModelField( 'data' ).hasBeenSet:
               model.toSysdb( self.apiConfig )

            else:
               senderUpdate[ 'success' ] = False
               return OK, senderUpdate
         else:


            for sender in validSenders:
               source = sender[ 'sourceIP' ]
               group = sender[ 'destinationIP' ]

               mcastKey = Tac.Value( "Mcs::McastKey", source, group )
               if mcastKey in storedSenders:
                  # Get the sender
                  if action == 'del':
                     del self.apiConfig.mcastSender[ mcastKey ]
                     # Add the deleted senders to the message queue
                     self.sendNotify( data, sender=sender, isFailure=False )
                  if action == 'modBw':

                     modSender = self.apiConfig.mcastSender[ mcastKey ]
                     modSender.bw = ( sender[ 'bandwidth' ], 'kilobps' )

               else:
                  if action == 'del':
                     ec = '105'
                     msg = errorCodes[ ec ].replace( '<sender post>', str( sender ) )
                     warn( msg )
                     # FIXME for this error code the fialureSenders is not set!
                  if action == 'modBw':
                     ec = '106'
                     wordMap = { '<sourceIp>': source, '<destinationIp>': group,
                              '<inIntfID>': sender[ 'inIntfID' ] }
                     msg = McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap )
                     error( msg )
                     del validSenders[ sender ]
                     senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders'
                                                                   ].append( sender )
                  senderUpdate[ 'status' ][ 'messages' ].append( ec )
                  self.sendNotify( data, errorCode=ec, sender=sender )

      if not senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ]:
         senderUpdate[ 'status' ][ 'success' ] = True

      else:
         senderUpdate[ 'status' ][ 'messages' ].append( '185' )

      return OK, senderUpdate

class Oui ( McsApiBase ):
   api = "/mcs/oui[/]"

   def __init__( self, apiConfig ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):

      response = { 'success': True,
                   'messages': [],
                   'oui': [] }

      for ouiId, value in self.apiConfig.vendor.items():
         response[ 'oui' ].append( { ouiId: value.vendorName } )

      return OK, response

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext ):
      """ Sample post :
      { "oui": "08:00:20",
        "vendorName": "sample" }
      """
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      log( 'Received data:', bv( data ) )
      validOui, oui = McsHttpAgentLib.validateOui( data )
      response = { 'success': False,
                  'messages': [] }
      if validOui:
         data.update( { "oui": int( '0x' + oui, 16 ) } )

         model = McsApiModels.OuiModel()
         model.populateModelFromJson( data )
         if model.getModelField( 'oui' ).hasBeenSet:
            model.toSysdb( self.apiConfig )
            response[ 'success' ] = True
      else:
         response[ 'messages' ].append( "186" )
         error( errorCodes[ '186' ] )

      return OK, response

class GetMcastSender( McsApiBase ):
   api = "/mcs/senders[/]"

   def __init__( self, apiConfig ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getMcastSenders( self, requestContext ):

      info( "Getting Multicast Senders" )

      model = McsApiModels.McastSenderModel()
      model.fromSysdb( self.apiConfig )
      result = simplejson.loads( simplejson.dumps( model, cls=ModelJsonSerializer ) )
      senders = self.get_resp_template()
      senders[ 'senders' ] = []
      if result:
         senders[ 'success' ] = True

         for sender in result[ 'data' ]:
            senderId, senderPort = sender[ 'inIntfID' ].split( '-' )
            senders[ 'senders' ].append( {
               "bandwidth": sender[ 'bandwidth' ],
               "destinationIP": sender[ 'destinationIP' ],
               "messages": [],
               "senderId": senderId,
               "senderPort": senderPort,
               "sourceIP": sender[ 'sourceIP' ],
               "success": True,
               "trackingID": result[ 'trackingID' ]
         } )
         return OK, senders
      else:
         return NO_CONTENT, None

class PostMcastReceiver( McsApiBase ):
   api = "/mcs/multicast/receivers[/]"

   def __init__( self, apiConfig, apiNotify ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiNotify = apiNotify
      self.MCAST_FLOW_KEYS = [ 'flow-action', 'trackingID', 'data', 'transactionID' ]
      self.RECEIVER_FIELDS = [ 'sourceIP', 'destinationIP', 'outIntfID' ]
      self.receiverActionMap = { 'addReceivers': 'set',
                            'delReceivers': 'del' }

   def sendNotify( self, postData, errorCode=None, receiver=None, isFailure=True ):
      """Send failure or delete notification messages for a receiver requests"""

      action = postData.get( 'flow-action' )
      action = self.receiverActionMap.get( action ) if action else 'U/A'
      messages = ( errorCode if isinstance( errorCode, list )
                   else [ errorCode ] if errorCode else [] )
      data = ( receiver if isinstance( receiver, list )
                           else [ receiver ] if receiver else [] )
      if isFailure:
         messages.extend( [ "114" ] )

      # Add the deleted senders, valid and failed, to the message queue
      notifyData = {
         'action': action,
         'trackingID': postData.get( 'trackingID', 0 ),
         'messages': messages,
         'data': data
      }
      notifyType = Tac.Type( "Mcs::ApiNotifyMsgType" )
      notify = Tac.Value( "Mcs::ApiNotifyMsg", notifyType.delReceiver,
                           simplejson.dumps( notifyData ) )
      self.apiNotify.msgQ.enq( notify )

   def validateReceivers( self, postData ):
      receiverUpdate = self.get_mcast_resp_template()
      receiverUpdate[ 'status' ] = self.get_mcast_recv_template()
      ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.MCAST_FLOW_KEYS,
                                                         postData )
      # Validate Receiver Post Fields
      if ec:
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )
         if 'data' in missingKeys:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] = []
         else:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         if 'flow-action' in missingKeys:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = 'U/A'

         # Send notification
         self.sendNotify( postData, errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      flowAction = postData[ 'flow-action' ]
      if flowAction in self.receiverActionMap:

         receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = \
                                    self.receiverActionMap[ flowAction ]
      else:
         receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = 'U/A'
         ec = '113'
         error( errorCodes[ ec ].replace( '<postedAction>',
                                             flowAction ) )

         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData,
                          errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      try:
         assert isinstance( postData[ 'data' ], list )
      except AssertionError:
         ec = '121'
         error( errorCodes[ '121' ] )
         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData,
                          errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      for receiver in postData[ 'data' ]:
         if 'messages' not in receiver:
            receiver[ 'messages' ] = []

         ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.RECEIVER_FIELDS,
                                                            receiver )
         if ec:
            receiver[ 'messages' ].append( ec )
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue
         sourceIP = receiver[ 'sourceIP' ]
         destinationIP = receiver[ 'destinationIP' ]

         valid, m = McsHttpAgentLib.validateMcastGroup( destinationIP )
         if not valid:
            if m:
               receiver[ 'messages' ].append( m )

            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue

         valid, m = McsHttpAgentLib.validateSourceIP( sourceIP )
         if not valid:
            if m:
               receiver[ 'messages' ].append( m )

            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue
         storedSenders = self.apiConfig.mcastSender.keys()
         currentKey = Tac.Value( "Mcs::McastKey", sourceIP, destinationIP )
         ec = ''

         def getFailedReceiver( receiver, outIntfID, ec ):
            failedReceiver = {}
            failedReceiver[ 'destinationIP' ] = receiver[ 'destinationIP' ]
            failedReceiver[ 'sourceIP' ] = receiver[ 'sourceIP' ]
            failedReceiver[ 'outIntfID' ] = [ outIntfID ]
            failedReceiver[ 'messages' ] = [ ec ]
            return failedReceiver

         if currentKey in storedSenders:
            for outIntfID in receiver[ 'outIntfID' ]:
               valid, ec = McsHttpAgentLib.validateDeviceID( outIntfID )

               if not valid:
                  if ec != '137':
                     error( errorCodes[ ec ] )
                  else:
                     error( errorCodes[ ec ].replace( '<deviceId>',
                                 outIntfID.split( '-' )[ 0 ] ) )

                  failedReceiver = getFailedReceiver( receiver, outIntfID, ec )
                  receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].\
                                                            append( failedReceiver )
                  # Send notification
                  self.sendNotify( postData, receiver=failedReceiver )
                  continue
               outDevice, outIntf = outIntfID.split( '-' )
               senderDev = self.apiConfig.mcastSender[ currentKey ].senderId.device
               senderIntf = self.apiConfig.mcastSender[ currentKey ].senderId.intfId
               if senderDev == outDevice and senderIntf == outIntf:
                  ec = "159"
                  sg = sourceIP + ':' + destinationIP
                  receiverId = outDevice + '-' + outIntf
                  senderId = senderDev + '-' + senderIntf
                  wordMap = { '<sourceIp:destinationIp> ': sg,
                              'sender=<deviceId:port>': 'sender=' + senderId,
                              'receiver=<deviceId:port>': ' receiver=' + receiverId
                              }
                  error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )

                  failedReceiver = getFailedReceiver( receiver, outIntfID, ec )
                  receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].\
                                                            append( failedReceiver )
                  # Send Notification
                  self.sendNotify( postData, receiver=failedReceiver )

         if not ec:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ].append(
                                                                         receiver )
      receiverUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
      receiverUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
      receiverUpdate[ 'success' ] = True
      return receiverUpdate

   @UrlMap.urlHandler( ( 'POST', ), api )
   def postMcastReceivers( self, requestContext ):
      """
      Sample post:
      {
         "flow-action": "addReceivers",
         "transactionID": "test", "trackingID": 76,
         "data": [
               {
                  "destinationIP": "224.2.0.11",
                  "sourceIP": "10.1.1.100",
                  "outIntfID": [
                  "001c.738d.1569-Ethernet8"]
               }]
      }
      """
      info( "Configuring Multicast Receiver" )
      # XXX TODO: Log trackingID to track posts
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      log( 'Received data:', bv( data ) )

      receiverUpdate = self.validateReceivers( data )
      validReceivers = receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ]
      # pylint: disable-msg=R1702
      if validReceivers:
         action = receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ]

         log( data )

         # Also check if this is an existing receiver- In this case, we can avoid
         # update and return null op.

         if action == 'set':
            model = McsApiModels.McastReceiverModel()
            model.populateModelFromJson( data )
            if model.getModelField( 'data' ).hasBeenSet:
               model.toSysdb( self.apiConfig )
            else:

               receiverUpdate[ 'success' ] = False
               return OK, receiverUpdate
         else:

            storedReceivers = self.apiConfig.mcastReceiver.keys()
            for receiver in validReceivers:
               source = receiver[ 'sourceIP' ]
               group = receiver[ 'destinationIP' ]
               for outIntfID in receiver[ 'outIntfID' ]:
                  device, intfId = outIntfID.split( '-' )

                  mcastKey = Tac.Value( "Mcs::McastKey", source, group )
                  if mcastKey in storedReceivers:
                     receivers = self.apiConfig.mcastReceiver[ mcastKey ].receivers
                     if ( device in receivers ) and \
                                       ( intfId in receivers[ device ].recvIntfs ):

                        del receivers[ device ].recvIntfs[ intfId ]
                        self.sendNotify( data, receiver=receiver, isFailure=False )

                        if not receivers[ device ].recvIntfs:
                           del receivers[ device ]
                     if not receivers:
                        del self.apiConfig.mcastReceiver[ mcastKey ]
                        # Send notification for the receiver
                        self.sendNotify( data, receiver=receiver, isFailure=False )
                  else:
                     warn( "Deleteing receiver not configured" )

      if not receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ]:
         receiverUpdate[ 'status' ][ 'success' ] = True

      else:
         receiverUpdate[ 'status' ][ 'messages' ].append( '114' )

      if not receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ]:
         receiverUpdate[ 'status' ][ 'messages' ].append( '117' )

      return OK, receiverUpdate

class GetMcastReceiver( McsApiBase ):
   api = "/mcs/receivers[/]"

   def __init__( self, apiConfig ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getMcastReceivers( self, requestContext ):

      info( "Getting Multicast receivers" )

      storedReceivers = McsHttpAgentLib.getStoredReceivers( self.apiConfig )

      receivers = self.get_resp_template()
      receivers[ 'receivers' ] = []
      receivers[ 'success' ] = True
      for _, srcDetail in storedReceivers.items():
         for _, grpDetail in srcDetail.items():
            for _, tDetail in grpDetail.items():
               for _, devDetail in tDetail.items():
                  receivers[ 'receivers' ].append( devDetail )

      return OK, receivers

class GetFlowsActive( McsApiBase ):
   api = "/mcs/flows-active[/]"

   def __init__( self, agentStatus ):
      McsApiBase.__init__( self )
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getFlowsActive( self, requestContext ):
      info( "Getting active flows" )
      activeFlows = { 'active-flows': [], 'messages': [], 'success': True }
      activeFlows[ 'active-flows' ], _ = McsHttpAgentLib.getFlowBwProgram(
                                                               self.agentStatus )
      return OK, activeFlows

class GetApiErrorCodes( McsApiBase ):
   api = "/mcs/apiErrorCodes[/]"

   def __init__( self ):
      McsApiBase.__init__( self )

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getApiErrorCodes( self, requestContext ):
      info( "Getting MCS Api Error Codes " )
      return OK, errorCodes

linkStatusStringsFormat2 = { 'linkUp': 'connected',
                                'linkDown': 'notconnect',
                                'linkUnknown': 'unknown' }

def getDevices( agentStatus, mcsMounts, apiConfig, deviceId=None,
                network=False, endpoint=False ):

   netDevs = set()
   endEdges = set()
   netDevInfo = []
   endDevInfo = []
   topologyEdges = mcsMounts.cdbTopology.edge
   cdbIntfStatus = mcsMounts.cdbIntfStatus
   cdbIntfConfig = mcsMounts.cdbIntfConfig
   cdbHw = mcsMounts.cdbHw
   cdbCell = mcsMounts.cdbCell
   cdbIpInputStatus = mcsMounts.cdbIpInputStatus
   cdbLldpLocal = mcsMounts.cdbLldpLocal
   cdbReboot = None
   if mcsMounts.cdbReboot:
      cdbReboot = mcsMounts.cdbReboot

   for _, value in topologyEdges.items():
      fromHostMac = Arnet.EthAddr( value.fromPort.host().name ).stringValue
      ouiFrom = fromHostMac[ : 8 ].replace( ':', '' )
      netFrom = Tac.Value( "Mcs::Oui", int( '0x' + ouiFrom, 16 ) ) \
                                                             in apiConfig.vendor

      for toPort in value.toPort.keys():
         toHostMac = Arnet.EthAddr( toPort.host().name ).stringValue
         ouiTo = toHostMac[ : 8 ].replace( ':', '' )

         netTo = Tac.Value( "Mcs::Oui", int( '0x' + ouiTo, 16 ) ) \
                                                            in apiConfig.vendor
         if netTo and netFrom:
            if toHostMac in agentStatus.mcsDevice.members():
               netDevs.add( toHostMac.replace( ':', '-' ) )
            if fromHostMac in agentStatus.mcsDevice.members():
               netDevs.add( fromHostMac.replace( ':', '-' ) )
         else:

            if not netTo:
               endDevId = toHostMac
               netDevId = fromHostMac
               endPort = toPort.name
            if not netFrom:
               endDevId = fromHostMac
               netDevId = toHostMac
               endPort = value.fromPort.name
            if netDevId in agentStatus.mcsDevice.members():
               endEdge = ( netDevId + '-' + endDevId, endPort )
               endEdges.add( endEdge )

   # pylint: disable-msg=R1702
   if network:
      for dev in netDevs:

         if deviceId and deviceId != cdbHw[ dev ].systemMacAddr:
            continue
         netDev = { 'build-time': '',
                  'id': '',
                  'image-file': '',
                  'interfaces': [],
                  'last-reset-reason': '',
                  'mac-address': '',
                  'messages': [],
                  'mgmt-ip': '',
                  'name': '',
                  'product-description': '',
                  'product-id': '',
                  'serial-number': '',
                  'success': True,
                  'uptime': '',
                  'vendor': '',
                  'version': '' }
         netDev[ 'id' ] = cdbHw[ dev ].systemMacAddr
         netDev[ 'name' ] = cdbCell[ dev ].fqdn
         netDev[ 'serial-number' ] = cdbHw[ dev ].fixedSystem.serialNum
         netDev[ 'product-id' ] = cdbHw[ dev ].fixedSystem.modelName
         netDev[ 'vendor' ] = 'Arista Networks'
         netDev[ 'mac-address' ] = cdbHw[ dev ].systemMacAddr
         netDev[ 'version' ] = cdbHw[ dev ].fixedSystem.softwareRev
         # XXX Get from MCS Client
         netDev[ 'image-file' ] = ''
         netDev[ 'uptime' ] = ''
         # Get from /Sysdb/cell/1/sys/reset/history/rebootHistory/0/causeQ/0
         rebootCause = ''
         if cdbReboot[ dev ].rebootHistory.keys():

            for instance in cdbReboot[ dev ].rebootHistory.keys():
               for cId in cdbReboot[ dev ].rebootHistory[ instance ].causeQ.keys():
                  cause = cdbReboot[ dev ].rebootHistory[ instance ].causeQ[ cId ]
                  rebootCause = cause.description
                  if rebootCause:
                     break
               if rebootCause:
                  break
         netDev[ 'last-reset-reason' ] = rebootCause
         # From Lldp local-info
         cdbMan = cdbLldpLocal[ dev ].managementAddress
         prio = cdbMan[ cdbLldpLocal[ dev ].managementAddressPrio.values()[ 0 ] ]
         netDev[ 'mgmt-ip' ] = formatAddr( prio.manAddrSubtype, prio.manAddr )
         netDev[ 'product-description' ] = cdbLldpLocal[ dev ].sysDesc
         apiIntfs = []
         for intf, inSt in cdbIntfStatus[ dev ].items():
            intfInfo = { 'MTU': '',
                     'admin-status': '',
                     'built-in-mac-address': '',
                     'description': '',
                     'down-reason': '',
                     'if-name': '',
                     'ip-address': '',
                     'mac-address': '',
                     'mode': '',
                     'oper-status': '',
                     'operating-speed': '' }
            intfInfo[ 'operating-speed' ] = McsHttpAgentLib.get_bandwidth_num(
                                                                        inSt.speed )
            intfInfo[ 'built-in-mac-address' ] = inSt.addr
            intfInfo[ 'if-name' ] = intf

            desc = ''
            if intf in cdbIntfConfig[ dev ]:
               desc = cdbIntfConfig[ dev ][ intf ].description
            intfInfo[ 'description' ] = desc
            adminStatus = linkStatusStringsFormat2[ inSt.linkStatus ] \
                        if inSt.linkStatus in linkStatusStringsFormat2 else 'N/A'
            intfInfo[ 'admin-status' ] = adminStatus
            intfInfo[ 'oper-status' ] = intfOperStatusToEnum( inSt.operStatus )
            intfInfo[ 'down-reason' ] = ''
            intfInfo[ 'mac-address' ] = inSt.addr
            # ip address from L3/Intf/status
            ipAddr = ''
            ipIntf = cdbIpInputStatus[ dev ].ipIntfStatus
            if intf in ipIntf:
               ipAddr = ipIntf[ intf ].activeAddrWithMask.address
            intfInfo[ 'ip-address' ] = ipAddr
            intfInfo[ 'MTU' ] = inSt.mtu
            intfInfo[ 'mode' ] = _forwardingModelMap[ inSt.forwardingModel ]
            apiIntfs.append( intfInfo.copy() )
         netDev[ 'interfaces' ] = apiIntfs
         netDevInfo.append( netDev.copy() )

   if endpoint:

      for endEdge in endEdges:
         netDevId, endDevId = endEdge[ 0 ].split( '-' )
         if deviceId and deviceId != netDevId:
            continue

         endDev = {
                  "chassis-id": endDevId,
                  "messages": [],
                  "network-device-id": netDevId,
                  "port-id": endPort,
                  "success": True
               }

         endDevInfo.append( endDev )
   return endDevInfo, netDevInfo

def getEdges( agentStatus, mcsMounts, apiConfig, deviceId=None,
                                                network=False, endpoint=False ):

   edges = []
   uniqueEndEdges = set()
   topologyEdges = mcsMounts.cdbTopology.edge
   cdbIntfStatus = mcsMounts.cdbIntfStatus

   for _, value in topologyEdges.items():
      edgeItem = {}
      fromHostMac = Arnet.EthAddr( value.fromPort.host().name ).stringValue

      ouiFrom = fromHostMac[ : 8 ].replace( ':', '' )
      netFrom = Tac.Value( "Mcs::Oui", int( '0x' + ouiFrom, 16 ) ) \
                                                             in apiConfig.vendor

      if deviceId and deviceId != fromHostMac:
         continue
      for toPort in value.toPort.keys():
         toHostMac = Arnet.EthAddr( toPort.host().name ).stringValue
         ouiTo = toPort.host().name[ : 8 ].replace( ':', '' )

         netTo = Tac.Value( "Mcs::Oui", int( '0x' + ouiTo, 16 ) ) \
                                                            in apiConfig.vendor

         if netTo and netFrom:
            if network:
               if fromHostMac in agentStatus.mcsDevice.members():
                  hStatus = cdbIntfStatus[ fromHostMac.replace( ':', '-' ) ]
                  edgeItem[ "network-device-id" ] = fromHostMac
                  edgeItem[ "network-interface" ] = value.fromPort.name
                  edgeItem[ "peer-network-interface" ] = toPort.name
                  edgeItem[ "peer-network-device-id" ] = toHostMac
                  edgeItem[ "speed" ] = hStatus[ value.fromPort.name ].speed
                  edgeItem[ "success" ] = True
                  edgeItem[ "messages" ] = []
                  edgeItem[ "endpoint-link" ] = False
                  edges.append( edgeItem )
         else:
            if endpoint:

               netDev = toHostMac
               netDev = toHostMac if netTo else fromHostMac
               if netDev in agentStatus.mcsDevice.members():
                  netPort = toPort.name if netTo else value.fromPort.name
                  endDev = toHostMac if not netTo else fromHostMac
                  endPort = toPort.name if not netTo else value.fromPort.name
                  # Use unique edges to not append same network-endpoint edge again
                  uEdge = netDev + '-' + netPort + '=' + endDev + '-' + endPort

                  uniqueEndEdges.add( uEdge )

   for edge in uniqueEndEdges:
      edgeItem = {}
      netId, endId = edge.split( '=' )
      netDev, netPort = netId.split( '-' )
      endDev, endPort = endId.split( '-' )
      hStatus = cdbIntfStatus[ netDev.replace( ':', '-' ) ]
      edgeItem[ "network-device-id" ] = netDev
      edgeItem[ "network-interface" ] = netPort
      edgeItem[ "endpoint-id" ] = endDev
      edgeItem[ "endpoint-port-id" ] = endPort
      edgeItem[ "speed" ] = hStatus[ netPort ].speed
      edgeItem[ "messages" ] = []
      edgeItem[ "endpoint-link" ] = True
      edgeItem[ "success" ] = True
      edges.append( edgeItem )

   return edges

class GetNetworkLinks( McsApiBase ):
   api = "/mcs/network-links[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkLinks( self, requestContext ):
      """
      [
         {
            "endpoint-link": false,
            "network-device-id": "2899.3abb.c1ee",
            "network-interface": "Ethernet2",
            "peer-network-device-id": "001c.7318.630e",
            "peer-network-interface": "Ethernet23",
            "speed": "speed10Gbps",
            "messages": [],
            "success": true
         }
      ]
      """
      info( "Getting MCS Network Links" )
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                        network=True )
      return OK, edges

class GetNetworkLink( McsApiBase ):
   api = "/mcs/network-link/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkLink( self, requestContext, deviceId ):
      """
      [
         {
            "endpoint-link": false,
            "network-device-id": "2899.3abb.c1ee",
            "network-interface": "Ethernet2",
            "peer-network-device-id": "001c.7318.630e",
            "peer-network-interface": "Ethernet23",
            "speed": "speed10Gbps",
            "messages": [],
            "success": true
         }
      ]
      """
      info( "Getting MCS Network Links for device %s" % ( deviceId ) )
      edges = []
      arDeviceId = Arnet.EthAddr( deviceId ).stringValue
      if arDeviceId != deviceId or \
                           not McsHttpAgentLib.validateDeviceMac( deviceId ):

         ec = '137'
         error( errorCodes[ ec ] )
         edge = { "endpoint-link": False,
                  "network-device-id": deviceId,
                  "network-interface": "",
                  "peer-network-device-id": "",
                  "peer-network-interface": "",
                  "speed": "",
                  "messages": [ ec ],
                  "success": False }
         edges.append( edge )
         return OK, edges

      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
               deviceId=arDeviceId, network=True )
      return OK, edges

class GetNetworkDevices( McsApiBase ):
   api = "/mcs/network-devices[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkDevices( self, requestContext ):
      """
      [
         {
            "build-time": "",
            "id": "74:83:ef:00:06:c9",
            "image-file": "flash:/openconfig-eos-trunk.swi",
            "interfaces": [
                  {
                     "MTU": 1500,
                     "admin-status": "connected",
                     "built-in-mac-address": "74:83:ef:00:06:c9",
                     "description": "",
                     "down-reason": "",
                     "if-name": "Ethernet8",
                     "ip-address": "10.37.8.1",
                     "mac-address": "74:83:ef:00:06:c9",
                     "mode": "routed",
                     "oper-status": "up",
                     "operating-speed": 1000000000
                  }
            ],
            "last-reset-reason": "Reload requested by the user.",
            "mac-address": "74:83:ef:00:06:c9",
            "messages": [],
            "mgmt-ip": "172.20.0.37",
            "name": "cs-lf37.sjc.aristanetworks.com",
            "product-description": "Arista Networks EOS version",
            "product-id": "DCS-7060SX2-48YC6-F",
            "serial-number": "SSJ17261113",
            "success": True,
            "uptime": " 07:40:23 up 48 days, 12:26, 0 users, load average:
                           0.50, 0.71, 0.78\n",
            "vendor": "Arista Networks",
            "version": "4.21.0F-INT-9623699.eostrunk (engineering build)"
         }
      ]
      """
      info( "Getting MCS Network Devices" )
      _, netDevs = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              network=True )
      return OK, netDevs

class GetNetworkDevice( McsApiBase ):
   api = "/mcs/network-device/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkDevice( self, requestContext, deviceId ):
      """
      {
         "build-time": "",
         "id": "28:99:3a:bb:c1:ee",
         "image-file": "flash:/EOS.swi",
         "interfaces": [
            {
                  "MTU": 1500,
                  "admin-status": "connected",
                  "built-in-mac-address": "28:99:3a:bb:c1:ee",
                  "description": "NTNX-AHV-4",
                  "down-reason": "",
                  "if-name": "Ethernet8",
                  "ip-address": "10.23.8.1",
                  "mac-address": "28:99:3a:bb:c1:ee",
                  "mode": "routed",
                  "oper-status": "up",
                  "operating-speed": 1000000000
            }
         ],
         "last-reset-reason": "Reload requested by the user.",
         "mac-address": "28:99:3a:bb:c1:ee",
         "messages": [],
         "mgmt-ip": "172.20.0.23",
         "name": "cs-lf23.sjc.aristanetworks.com",
         "product-description": "Arista Networks EOS version 4.20.2.1F running on
                  an Arista Networks DCS-7160-48YC6",
         "product-id": "DCS-7160-48YC6-F",
         "serial-number": "SSJ17164906",
         "success": true,
         "uptime": " 07:19:19 up 6 days, 21:39,  2 users,
            load average: 1.56, 1.57, 1.59\n",
         "vendor": "Arista Networks",
         "version": "4.20.2.1F"
      }
      """

      info( "Getting MCS Network Device of id %s" % ( deviceId ) )
      netDev = { "mac-address": deviceId,
               "messages": [],
               "success": True }
      # Normalize deviceId to xx.xx.xx.xx format

      arDeviceId = Arnet.EthAddr( deviceId ).stringValue

      if arDeviceId != deviceId or \
                           not McsHttpAgentLib.validateDeviceMac( deviceId ):

         ec = '137'
         netDev[ 'messages' ].append( ec )
         netDev[ 'success' ] = False
         error( errorCodes[ ec ] )
         return OK, netDev

      _, netDevs = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              deviceId=arDeviceId, network=True )
      if netDevs:
         netDev.update( netDevs[ 0 ] )

      return OK, netDev

class GetEndpointDevices( McsApiBase ):
   api = "/mcs/endpoints[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )

      self.apiConfig = apiConfig
      self.mcsMounts = mcsMounts
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointDevices( self, requestContext ):
      """
      [
      {
         "chassis-id": "0050.56ba.32aa",
         "messages": [],
         "network-device-id": "2899.3abb.c1ee",
         "port-id": "00:50:56:ba:32:aa",
         "success": true
      }
      ]
      """
      info( "Getting MCS Endpoint Devices" )
      endDevs, _ = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              endpoint=True )
      return OK, endDevs

class GetEndpointDevice( McsApiBase ):
   api = "/mcs/endpoint/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointDevice( self, requestContext, deviceId ):
      """
      [
      {
         "chassis-id": "0050.56ba.32aa",
         "messages": [],
         "network-device-id": "2899.3abb.c1ee",
         "port-id": "00:50:56:ba:32:aa",
         "success": true
      }
      ]
      """
      info( "Getting MCS Endpoint Device %s" % ( deviceId ) )

      endDevs = []
      arDeviceId = Arnet.EthAddr( deviceId ).stringValue
      if arDeviceId != deviceId or \
                           not McsHttpAgentLib.validateDeviceMac( deviceId ):

         ec = '137'
         error( errorCodes[ ec ] )
         endDev = { "chassis-id": "",
                     "messages": [ ec ],
                     "network-device-id": arDeviceId,
                     "port-id": "",
                     "success": False }
         endDevs.append( endDev )
         return OK, endDevs
      endDevs, _ = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              deviceId=arDeviceId, endpoint=True )
      return OK, endDevs

class GetEndpointLinks( McsApiBase ):
   api = "/mcs/endpoint-links[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointinks( self, requestContext ):
      """
      [
         {
            "endpoint-id": "0050.56ba.fb78",
            "endpoint-link": true,
            "endpoint-port-id": "00:50:56:ba:fb:78",
            "network-device-id": "2899.3abb.c1ee",
            "network-interface": "Ethernet8",
            "speed": "speed1Gbps",
            "success": true,
            "messages": []
         }
      ]
      """
      info( "Getting MCS Endpoint Links" )
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                        endpoint=True )
      return OK, edges

class GetEndpointLink( McsApiBase ):
   api = "/mcs/endpoint-link/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointink( self, requestContext, deviceId ):
      """
      [
         {
            "endpoint-id": "0050.56ba.fb78",
            "endpoint-link": true,
            "endpoint-port-id": "00:50:56:ba:fb:78",
            "network-device-id": "2899.3abb.c1ee",
            "network-interface": "Ethernet8",
            "speed": "speed1Gbps",
            "success": true,
            "messages": []
         }
      ]
      """
      info( "Getting MCS Endpoint Links" )
      edges = []
      arDeviceId = Arnet.EthAddr( deviceId ).stringValue
      if arDeviceId != deviceId or \
                           not McsHttpAgentLib.validateDeviceMac( deviceId ):

         ec = '137'
         error( errorCodes[ ec ] )
         edge = {
                  "endpoint-id": "",
                  "endpoint-link": True,
                  "endpoint-port-id": "",
                  "network-device-id": arDeviceId,
                  "network-interface": "",
                  "speed": "",
                  "success": False,
                  "messages": [ ec ]
               }
         edges.append( edge )
         return OK, edges
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                                                deviceId=arDeviceId, endpoint=True )
      return OK, edges

class GetInterfaces( McsApiBase ):
   api = "/mcs/interface-bandwidth[/]"

   def __init__( self, mcsMounts, agentStatus, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.agentStatus = agentStatus
      self.apiConfig = apiConfig


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getInterfaces( self, requestContext ):
      """
      [
         {
            "availableRxBw": 25000000,
            "availableTxBw": 25000000,
            "deviceID": "2899.3abb.c1ee",
            "interfaceId": "Ethernet35",
            "messages": [],
            "portSpeed": 25000000,
            "reservation-percentage": "1.0",
            "rxTotal": 0,
            "success": True,
            "totalMcastBw": 0,
            "txTotal": 0,
            "vlanId": "1",
            "vlanMode": "access"
         }
      ]
      """
      interfaces = []
      info( "Getting MCS Interfaces" )
      _, bwProgrammed = McsHttpAgentLib.getFlowBwProgram( self.agentStatus )
      for mac, intfStatus in self.mcsMounts.cdbIntfStatus.items():
         sConfig = self.mcsMounts.cdbSwitchIntfConfig[ mac ]
         for intf, intfInfo in intfStatus.items():
            intfDet = {}
            bw = McsHttpAgentLib.get_bandwidth_num( intfInfo.speed )
            if intf not in sConfig.switchIntfConfig:
               continue
            swIntf = sConfig.switchIntfConfig[ intf ]
            mac = mac.replace( '-', ':' )
            intfDet[ 'deviceID' ] = mac
            intfDet[ 'portSpeed' ] = bw
            intfDet[ 'interfaceId' ] = intf
            intfDet[ 'vlanId' ] = str( swIntf.nativeVlan )
            intfDet[ 'vlanMode' ] = swIntf.switchportMode
            intfDet[ 'availableRxBw' ] = bw
            intfDet[ 'availableTxBw' ] = bw
            intfDet[ 'rxTotal' ] = 0
            intfDet[ 'txTotal' ] = 0
            intfDet[ 'totalMcastBw' ] = 0
            intfDet[ 'linkStatus' ] = intfInfo.linkStatus

            resKey = Tac.Value( "Mcs::DeviceAndIntfId",
                                    Arnet.EthAddr( mac ).stringValue, intf )
            resvPercent = 1.0
            resvPercentEntity = self.apiConfig.reservationPercentage.get( resKey )
            if resvPercentEntity:
               resvPercent = resvPercentEntity.percentValue
            intfDet[ 'reservation-percentage' ] = str( resvPercent )
            if mac in bwProgrammed and intf in bwProgrammed[ mac ]:
               intfDet[ 'rxTotal' ] = bwProgrammed[ mac ][ intf ][ 'rxTotal' ]
               intfDet[ 'txTotal' ] = bwProgrammed[ mac ][ intf ][ 'txTotal' ]
               intfDet[ 'totalMcastBw' ] = intfDet[ 'rxTotal' ] + \
                                          intfDet[ 'txTotal' ]
            availBw = round( float( resvPercent ), 2 ) * bw
            intfDet[ 'availableRxBw' ] = int( availBw ) - intfDet[ 'rxTotal' ]
            intfDet[ 'availableTxBw' ] = int( availBw ) - intfDet[ 'txTotal' ]

            intfDet[ 'success' ] = True
            intfDet[ 'messages' ] = []
            interfaces.append( intfDet )

      return OK, interfaces

class ReservationPercentage( McsApiBase ):
   api = "/mcs/reservation-percentage[/][{deviceId}]"

   def __init__( self, apiConfig ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext, deviceId ):
      """
      [
         {
            "deviceId": "001c.738d.1569",
            "interface-name": "all",
            "messages": [],
            "reservation-percent": "1.0",
            "success": true
         }
         ]
      """
      resvPr = []
      valid = McsHttpAgentLib.validateDeviceMac( deviceId )
      if not valid:
         ec = '137'
         data = { 'success': False, 'messages': [ ec ] }
         error( errorCodes[ ec ] )
         resvPr.append( data )
         return OK, resvPr
      devPresent = False
      for deviceAndIntfId, rpValue in self.apiConfig.reservationPercentage.items():
         if deviceAndIntfId.device == Arnet.EthAddr( deviceId ).stringValue:
            devPresent = True
            resvPr.append( { 'deviceId': deviceId,
                     'interface-name': deviceAndIntfId.intfId,
                     'success': True,
                     'messages': [],
                     'reservation-percent': str( round( rpValue.percentValue, 2 ) )
                     } )

      if not devPresent:
         resvPr.append( { 'deviceId': deviceId,
                     'interface-name': 'all',
                     'success': True,
                     'messages': [],
                     'reservation-percent': '1.0'
                     } )

      return OK, resvPr

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext, deviceId=None ):
      """
      Sample post:
      {"chassis-id":"00:1c:73:8d:15:69",
      "interface-name":"Ethernet1",
      "reservation-percent":"1.0"}

      Response:
      {
         "messages": [],
         "response": [
            {
               "deviceId": "001c.738d.1569",
               "interfaceId": "Ethernet1",
               "reservation-percentage": "1.0"
            }
         ],
         "success": true
         }

      """
      info( "Posting Reservation percentages" )
      data = requestContext.getRequestContent()
      data = simplejson.loads( data )
      info( 'Received data:', bv( data ) )
      response = { 'success': False,
               'messages': [],
               'response': [] }
      valid, ec = McsHttpAgentLib.validateRp( data )
      if not valid:
         response[ 'messages' ].append( ec )
      else:
         model = McsApiModels.RPModel()
         model.populateModelFromJson( data )

         if model.getModelField( 'chassis-id' ).hasBeenSet:
            model.toSysdb( self.apiConfig )
            response[ 'success' ] = True
            response[ 'response' ].append( {
                     "deviceId": model.getModelField( 'chassis-id' ).value,
                     "interfaceId": model.getModelField( 'interface-name' ).value,
                     "reservation-percentage":
                     str( model.getModelField( 'reservation-percent' ).value ) } )
      return OK, response


class AgentStatus( McsApiBase ):
   api = "/mcs/agentStatus[/]"

   def __init__( self, agentStatus ):
      info( "Registering AgentStatus" )
      McsApiBase.__init__( self )
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getAgentStatus( self, requestContext ):
      info( "Getting MCS Agent status " )
      model = McsApiModels.AgentStatusModel()
      model.fromSysdb( self.agentStatus )
      result = simplejson.loads( simplejson.dumps( model, cls=ModelJsonSerializer ) )
      if result:
         return OK, result
      else:
         return NO_CONTENT, None

class ShutdownReactor( object ):
   def __init__( self, mcsConfig, httpConfig,
                 apiStatus ):
      self.mcsConfig = mcsConfig

      self.httpConfig = httpConfig
      self.apiStatus = apiStatus
      self.mcsReactor = GenericReactor( self.apiStatus, [ 'enabled' ],
                                        self.handleEnabled )

      self.httpReactor = GenericReactor( self.httpConfig, [ 'enabled' ],
                                        self.handleEnabled )
      self.handleEnabled()

   def handleEnabled( self, notifiee=None ):
      log( "mcs=%s httpd=%s" %
          ( self.mcsConfig.enabled, self.httpConfig.enabled ) )
      enabled = True
      if not self.mcsConfig.enabled:
         self.apiStatus.ready = False
         enabled = False
      elif not self.httpConfig.enabled:
         # If Httpd is shutdown, we'll shutdown McsHttpAgent but continue to run
         # Mcs service. So, cleanup shouldn't be done
         enabled = False
      self.apiStatus.enabled = enabled

class McsApiHandler( Agent.Agent ):
   # pylint: disable-msg=W0201
   def __init__( self, em, blocking=False ):
      Agent.Agent.__init__( self, em, agentName="McsHttpAgent" )
      self.warm_ = False
      self.aaaManager = None
      self.sysname = em.sysname()
      self.controllerdbMountsComplete = False
      self.blocking = blocking
      self.sysdbMountsComplete = False
      self.apiConfig = None
      self.mcsControllerMounts = None
      self.agentStatus = None
      self.cliConfig = None
      self.apiStatus = None
      self.apiNotify = None
      self.readyReactor = None
      self.mcsNotify = None
      self.mcsHA = None
      qtfile = "%s%s.qt" % ( self.agentName, "-%d" if "QUICKTRACEDIR"
                             not in os.environ else "" )
      BothTrace.initialize( qtfile, "8, 128, 8, 8, 8, 128, 8, 8, 128, 8" )

      cdbSocket = os.environ.get(
         'CONTROLLERDBSOCKNAME',
         Tac.Value( 'Controller::Constants' ).controllerdbDefaultSockname )

      self.cdbEm = Controllerdb(
         em.sysname(), controllerdbSockname_=cdbSocket, mountRoot=False )

   def doMaybeFinishInit( self ):
      if self.sysdbMountsComplete and self.controllerdbMountsComplete:
         self.mountsComplete()
         self.warm_ = True
         if self.mcsControllerMounts:
            if ( self.mcsControllerMounts.intfStatusDir and
                 self.mcsControllerMounts.intfConfigDir and
                 self.apiConfig ):
               self.allIntfStatusSm = Tac.newInstance( "Mcs::AllIntfStatusSm",
                     self.mcsControllerMounts.intfStatusDir,
                     self.mcsControllerMounts.intfConfigDir,
                     self.mcsControllerMounts.mcsLocalStatus,
                     self.apiConfig )
               self.allIntfStatusSm.initialized = True
            else:
               info( 'Unable to start allIntfStatusSm' )
      else:
         log( "Sysdb and controllerdb mounts incomplete" )
         log( self.controllerdbMountsComplete )
         log( self.sysdbMountsComplete )

   # Parameters differ from overridden 'doInit' method
   # pylint: disable-msg=arguments-differ
   def doControllerdbMounts( self ):
      def _onControllerdbMountsComplete():
         info( 'controllerdb mounts complete' )
         self.controllerdbMountsComplete = True
         self.doMaybeFinishInit()
         self.finishAgentInitalization()

      cdbMg = self.cdbEm.mountGroup()
      cdbMg.mount( '', 'Tac::Dir', 'rt' )  # t=toplevel rootMount, must do this first
      self.mcsControllerMounts = McsControllerMounts( cdbMg )

      if self.blocking:
         info( 'doing Controllerdb mounts: blocking' )
         cdbMg.close( blocking=True )
      else:
         info( 'doing Controllerdb mounts: non-blocking' )
         cdbMg.close( callback=_onControllerdbMountsComplete )

   def mountsComplete( self ):

      self.readyReactor = GenericReactor( self.agentStatus, [ 'enabled' ],
                                          self.handleReady, callBackNow=True )
      self.mountReadyReactor = GenericReactor( self.agentStatus, [ 'mountDone' ],
                                               self.handleHAReplicate,
                                               callBackNow=True )
      self.shutdownReactor = ShutdownReactor(
                                    self.cliConfig,
                                    self.httpConfig.service[ 'mcs' ],
                                    self.apiStatus )
      self.notifyAckReactor = GenericReactor( self.agentStatus, [ 'notifyAck' ],
                                               self.handleNotifyAck,
                                               callBackNow=True )

   def doSysdbMounts( self, em ):
      def _onMountsComplete():
         info( 'sysdb mounts complete' )
         self.sysdbMountsComplete = True
         self.doMaybeFinishInit()
         self.finishAgentInitalization()

      mg = em.mountGroup()

      self.apiConfig = mg.mount( "mcs/config/api", "Mcs::ApiConfig", "w" )
      self.cliConfig = mg.mount( "mcs/config/cli", "Mcs::CliConfig", "r" )
      self.agentStatus = mg.mount( "mcs/status/agent", "Mcs::AgentStatus", "r" )
      self.apiStatus = mg.mount( "mcs/status/api", "Mcs::ApiStatus", "w" )
      self.apiNotify = mg.mount( "mcs/status/notify", "Mcs::ApiNotifyQueue", "w" )
      self.httpConfig = mg.mount( "mgmt/capi/config", "HttpService::Config", "r" )
      self.clusterStatus = mg.mount( "controller/cluster/statusDir",
                                     "ControllerCluster::ClusterStatusDir", "r" )

      if self.blocking:
         info( 'doing Sysdb mounts: blocking' )
         mg.close( blocking=True )  # synchronous mounting
         _onMountsComplete()
      else:
         info( 'doing Sysdb mounts: non-blocking' )
         mg.close( callback=_onMountsComplete )


   def mcsStateHAReplicate( self ):
      # Initialize replication object
      self.mcsHA = McsStateHAReplicate.McsStateReplicate(
         self.apiConfig,
         self.clusterStatus,
         self.apiStatus,
         self.agentStatus,
         self.mcsControllerMounts.mcsStatusHA )

   def doInit( self, em ):
      info( "doInit..." )
      self.doControllerdbMounts()
      self.doSysdbMounts( em )
      info( "doInit done" )

   def finishAgentInitalization( self ):
      if self.sysdbMountsComplete and self.controllerdbMountsComplete:
         info( "finishAgentInitalization..." )
         AgentStatus( self.agentStatus )
         ApiStatus( self.apiStatus )
         PostMcastSender( self.apiConfig, self.apiNotify )
         GetMcastSender( self.apiConfig )
         PostMcastReceiver( self.apiConfig, self.apiNotify )
         Oui( self.apiConfig )
         GetMcastReceiver( self.apiConfig )
         GetFlowsActive( self.agentStatus )
         GetNetworkLinks( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetEndpointLinks( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetNetworkLink( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetEndpointLink( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetInterfaces( self.mcsControllerMounts, self.agentStatus, self.apiConfig )
         ReservationPercentage( self.apiConfig )
         GetApiErrorCodes()

         GetNetworkDevices( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )

         GetEndpointDevices( self.agentStatus, self.mcsControllerMounts,
                              self.apiConfig )

         GetNetworkDevice( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )

         GetEndpointDevice( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )

         self.mcsStateHAReplicate()

         info( "finishAgentInitalization done" )

   def handleReady( self, notifiee=None ):
      info( "Start API", self.apiStatus.enabled )

   def handleHAReplicate( self, notifiee=None ):
      if self.mcsHA:
         self.mcsHA.syncApiConfig()

   def handleNotifyAck( self, notifiee=None ):
      if self.apiNotify:
         del self.apiNotify.msgQ[ self.agentStatus.notifyAck ]

   def processRequest( self, request ):
      @self.mcsHA.replicate()
      def doProcessRequest( request ):
         try:
            requestContext = UwsgiRequestContext( request, self.aaaManager )
            requestType = requestContext.getRequestType()
            parsedUrl = requestContext.getParsedUrl()
            log( 'processRequest(): request type:', bv( requestType ),
                 ', parsed url:',
                 bv( parsedUrl ), 'received data:',
                 bv( requestContext.getRequestContent() ) )
            func, urlArgs = UrlMap.getHandler( requestType, parsedUrl )

            log( "Got", func, "and", urlArgs, "for url", parsedUrl )
            if func is None:
               # pylint: disable-msg=no-value-for-parameter
               raise HttpNotFound( "Invalid endpoint requested" )
               # pylint: enable-msg=no-value-for-parameter
            obj, funcName = func
            status, result = getattr( obj, funcName )( requestContext, **urlArgs )
            result = simplejson.dumps( result )
         except HttpNotFound as e:
            log( "processRequest HttpNotFound : %s", bv( e.message ) )
            ec = '187'
            errMsg = simplejson.dumps( { 'messages': ec, 'success': False } )
            error( errorCodes[ ec ] )
            return ( e.code, "application/json",
                     e.additionalHeaders, errMsg )
         else:
            return ( status, 'application/json', None, result )
      return doProcessRequest( request )

   def doCleanup( self ):
      pass



class McsHttpAgent( object ):
   def __init__( self ):
      self.container_ = Agent.AgentContainer( [ McsApiHandler ],
                                              passiveMount=True,
                                              agentTitle="McsHttpAgent" )
      self.container_.startAgents()
      # Run ActivityLoop in a separate thread in order to serve HTTP requests
      # in the main thread
      Tac.activityThread().start( daemon=True )
      Tac.waitFor( self.mcsHttpAgentWarm,
                   maxDelay=1,
                   sleep=True, description="Mcs HTTP Agent warmup" )
      self.agent_ = self.container_.agents_[ 0 ]

   def mcsHttpAgentWarm( self ):
      return self.container_.agents_ and self.container_.agents_[ 0 ].warm()

   def __call__( self, request, start_response ):

      ( responseCode, contentType, headers, body ) = \
            self.agent_.processRequest( request )
      responseStr = '%d %s' % ( responseCode, responses[ responseCode ] )
      headers = headers or []
      headers.append( ( 'Content-type', contentType ) )
      if body:
         headers.append( ( 'Content-length', str( len( body ) ) ) )
      start_response( responseStr, UwsgiConstants.DEFAULT_HEADERS + headers )
      return [ body ]
