# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Arnet
from ApiBaseModels import BaseModel, Str, Bool, Int, List

class ApiSettingsModel( BaseModel ):
   apiVersion = Str( apiName='apiVersion', tacName='apiVersion', versions=[ 1 ],
                     inputOk=False, outputOk=True,
                     description='Current version of MCS API' )
   apiTitle = Str( apiName='apiTitle', tacName='apiTitle', versions=[ 1 ],
                   inputOk=False, outputOk=True, description='Title of MCS API' )
   apiDescription = Str( apiName='apiDescription', tacName='apiDescription',
                        versions=[ 1 ], inputOk=False, outputOk=True,
                        description='Description of MCS API' )

   # pylint: disable-msg=arguments-differ
   def fromSysdb( self, apiSettings ):
      self.getModelField( 'apiVersion' ).value = apiSettings.apiVersion
      self.getModelField( 'apiTite' ).value = apiSettings.apiTitle

class RPModel ( BaseModel ):
   deviceId = Str( apiName='chassis-id', tacName='deviceId', versions=[ 1 ],
               inputOk=True, outputOk=True, description="Device MACId" )
   intfName = Str( apiName='interface-name', tacName='intfName', versions=[ 1 ],
               inputOk=True, outputOk=True, description="Interface Name" )
   rp = Str( apiName='reservation-percent', tacName='rp', versions=[ 1 ],
               inputOk=True, outputOk=True, description="Reservation Percentage" )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, apiConfig ):
      deviceId = self.getModelField( 'chassis-id' ).value
      intfName = self.getModelField( 'interface-name' ).value
      rp = self.getModelField( 'reservation-percent' ).value
      rpKey = Tac.Value( "Mcs::DeviceAndIntfId", deviceId, intfName )
      apiConfig.newReservationPercentage( rpKey )
      apiConfig.reservationPercentage[ rpKey ].percentValue = round( float( rp ), 2 )

class OuiModel ( BaseModel ):
   oui = Int( apiName='oui', tacName='oui', versions=[ 1 ], inputOk=True,
               outputOk=True, description="OUI Id" )
   vendorName = Str( apiName='vendorName', tacName='vendorName', versions=[ 1 ],
               inputOk=True, outputOk=True, description="Vendor Name" )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, apiConfig ):
      oui = apiConfig.vendor.newMember( Tac.Value( "Mcs::Oui",
                                        self.getModelField( 'oui' ).value ) )
      oui.vendorName = self.getModelField( 'vendorName' ).value


class AgentStatusModel( BaseModel ):
   enabled = Bool( apiName='enabled', tacName='enabled', versions=[ 1 ],
                   inputOk=False, outputOk=True, description='Status of MCS Agent' )

   # pylint: disable-msg=arguments-differ

   def fromSysdb( self, status ):
      self.getModelField( 'enabled' ).value = status.enabled

class ApiStatusModel( BaseModel ):
   # Represents Same API state as Agent Status for now.
   # Can be expanded to represent holistic API status including agent status, cvx
   # client status, etc
   enabled = Bool( apiName='enabled', tacName='enabled', versions=[ 1 ],
                   inputOk=False, outputOk=True, description='Status of MCS API' )
   # pylint: disable-msg=arguments-differ

   def fromSysdb( self, status ):
      self.getModelField( 'enabled' ).value = status.enabled

class McastSendKeyModel( BaseModel ):
   source = Str( apiName='sourceIP', tacName='source', versions=[ 1 ],
                  inputOk=True, outputOk=True, description='source' )
   group = Str( apiName='destinationIP', tacName='group', versions=[ 1 ],
               inputOk=True, outputOk=True, description='group' )
   bwValue = Int( apiName='bandwidth', tacName='value', versions=[ 1 ],
                  inputOk=True, outputOk=True, description='bandwidth value' )
   bwType = Str( apiName='bwType', tacName='type', versions=[ 1 ], inputOk=True,
                outputOk=True, description='bwType' )
   inIntfID = Str( apiName='inIntfID', tacName='', versions=[ 1 ], inputOk=True,
                  outputOk=True, description='inIntfID' )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, trackingId, transactionId, config ):
      bwTypeTacMap = { 'k': 'kilobit',
                     'm': 'megabit',
                     'g': 'gigabit' }

      bwTypeMap = { 'kilobit': 1,
                  'megabit': 1000,
                  'gigabit': 1000000 }

      postedBwType = self.getModelField( 'bwType' ).value
      source = self.getModelField( 'sourceIP' ).value
      group = self.getModelField( 'destinationIP' ).value
      device, intfId = self.getModelField( 'inIntfID' ).value.split( '-' )
      mcastKey = Tac.Value( "Mcs::McastKey", source, group )

      sender = config.mcastSender.newMember( mcastKey )
      sender.trackingId = trackingId
      sender.transactionId = transactionId
      bwType = bwTypeTacMap[ postedBwType ]
      bandwidth = bwTypeMap[ bwType ] * self.getModelField( 'bandwidth' ).value
      sender.bw = ( bandwidth, 'kilobps' )
      sender.senderId = Tac.Value( "Mcs::DeviceAndIntfId", device, intfId )

   # pylint: disable-msg=arguments-differ
   def fromSysdb( self, config, mcastKey ):
      self.getModelField( 'sourceIP' ).value = mcastKey.source
      self.getModelField( 'destinationIP' ).value = mcastKey.group
      configBwValue = config.mcastSender[ mcastKey ].bw.value
      configDevice = config.mcastSender[ mcastKey ].senderId.device
      configIntfId = config.mcastSender[ mcastKey ].senderId.intfId
      configInIntfID = configDevice + '-' + configIntfId
      self.getModelField( 'bandwidth' ).value = configBwValue
      self.getModelField( 'bwType' ).value = 'k'
      self.getModelField( 'inIntfID' ).value = configInIntfID

class McastSenderModel( BaseModel ):

   data = List( apiName='data', tacName='', versions=[ 1 ], inputOk=True,
               outputOk=True, description='McastKey', valueType=McastSendKeyModel )
   transactionId = Str( apiName='transactionID', tacName='transactionId',
                        versions=[ 1 ], inputOk=True, outputOk=True,
                        description='transactionID' )
   trackingId = Int( apiName='trackingID', tacName='trackingId', versions=[ 1 ],
                     inputOk=True, outputOk=True, description='trackingID' )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, config ):
      if self.getModelField( 'data' ).value:
         for flow in self.getModelField( 'data' ).value:
            flow.toSysdb( self.getModelField( 'trackingID' ).value,
                           self.getModelField( 'transactionID' ).value, config )

   # pylint: disable-msg=arguments-differ
   def fromSysdb( self, config ):
      data = []
      for mcastKey in config.mcastSender:
         mcastSendKeyModel = McastSendKeyModel()
         mcastSendKeyModel.fromSysdb( config, mcastKey )
         data.append( mcastSendKeyModel )
         configTransactionId = config.mcastSender[ mcastKey ].transactionId
         configTrackingId = config.mcastSender[ mcastKey ].trackingId
         self.getModelField( 'transactionID' ).value = configTransactionId
         self.getModelField( 'trackingID' ).value = configTrackingId
      self.getModelField( 'data' ).value = data

class McastRecvKeyModel( BaseModel ):
   source = Str( apiName='sourceIP', tacName='source', versions=[ 1 ],
                  inputOk=True, outputOk=True, description='source' )
   group = Str( apiName='destinationIP', tacName='group', versions=[ 1 ],
               inputOk=True, outputOk=True, description='group' )
   outIntfID = List( apiName='outIntfID', tacName='', versions=[ 1 ], inputOk=True,
                  outputOk=True, description='outIntfID', valueType=Str )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, trackingId, transactionId, config ):

      source = self.getModelField( 'sourceIP' ).value
      group = self.getModelField( 'destinationIP' ).value
      mcastKey = Tac.Value( "Mcs::McastKey", source, group )

      if mcastKey in config.mcastReceiver.keys():

         recv = config.mcastReceiver.get( mcastKey )
      else:
         recv = config.mcastReceiver.newMember( mcastKey )

      for outID in self.getModelField( 'outIntfID' ).value:

         device, intfId = outID.split( '-' )
         arNetDevice = Arnet.EthAddr( device ).stringValue

         if arNetDevice not in recv.receivers.keys():
            recv.newReceivers( arNetDevice )

         intfKey = Tac.Value( "Arnet::IntfId", intfId )

         # If this is an existing intfKey just update it's trackingId, transactionId
         # to keep track of last transaction to modify the receiver

         if intfKey not in recv.receivers[ arNetDevice ].recvIntfs:
            tacIntfId = recv.receivers[ arNetDevice ].recvIntfs.newMember( intfKey )
         else:
            tacIntfId = recv.receivers[ arNetDevice ].recvIntfs[ intfKey ]

         tacIntfId.trackingId = trackingId
         tacIntfId.transactionId = transactionId

   # pylint: disable-msg=arguments-differ
   def fromSysdb( self, data ):
      self.getModelField( 'sourceIP' ).value = data[ 'sourceIP' ]
      self.getModelField( 'destinationIP' ).value = data[ 'destinationIP' ]
      self.getModelField( 'outIntfID' ).value = data[ 'outIntfID' ]

class McastReceiverModel( BaseModel ):

   data = List( apiName='data', tacName='', versions=[ 1 ], inputOk=True,
               outputOk=True, description='McastKey', valueType=McastRecvKeyModel )
   transactionId = Str( apiName='transactionID', tacName='transactionId',
                        versions=[ 1 ], inputOk=True, outputOk=True,
                        description='transactionID' )
   trackingId = Int( apiName='trackingID', tacName='trackingId', versions=[ 1 ],
                     inputOk=True, outputOk=True, description='trackingID' )

   # pylint: disable-msg=arguments-differ
   def toSysdb( self, config ):

      if self.getModelField( 'data' ).value:

         for flow in self.getModelField( 'data' ).value:
            flow.toSysdb( self.getModelField( 'trackingID' ).value,
                           self.getModelField( 'transactionID' ).value, config )
