# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress, IpGenericPrefix
from CliModel import Model, List, Dict, Bool, Int, Float, Str, Submodel, \
                     DeferredModel
from IntfModels import Interface

class MulticastRibRoute( Model ):
   '''Cli Model for show ip route multicast'''
   __revision__ = 2
   vias = Dict( help="set of next-hop addresses that this route is reachable via",
         keyType=IpGenericAddress, valueType=str )
   routeProtocol = Str( help="Routing protocol through which this route was learnt" )
   routePreference = Int( help="Route preference" )
   routeMetric = Int( help="Route metric" )
   routeCreationTime = Str( help="Route creation time", optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1 and not dictRepr.get( 'routeCreationTime' ):
         dictRepr[ 'routeCreationTime' ] = 'N/A'
      return dictRepr

class MulticastRibRoutes( Model ):
   __revision__ = 2
   routes = Dict( help="Multicast static routes", keyType=IpGenericPrefix,
         valueType=MulticastRibRoute )
   def render( self ):
      print '''Codes: C - connected, S - static, K - kernel,
       O - OSPF, IA - OSPF inter area, E1 - OSPF external type 1,
       E2 - OSPF external type 2, N1 - OSPF NSSA external type 1,
       N2 - OSPF NSSA external type2, B I - iBGP, B E - eBGP,
       R - RIP, I - IS-IS, A B - BGP Aggregate, A O - OSPF Summary,
       NG - Nexthop Group Static Route, M - Martian'''
      print

      for route in sorted( self.routes ):
         routeInfo = self.routes[ route ] # this way or itervalues()?
         outputStr = "%-6s %-17s [%d/%d]" % ( routeInfo.routeProtocol,
               route, routeInfo.routeMetric, routeInfo.routePreference )
         if not routeInfo.vias:
            print outputStr
            continue
         print outputStr,
         indent = ""
         for via in sorted( routeInfo.vias ):
            routeCreationTimeStr = ( ( str( routeInfo.routeCreationTime ) + ", " )
                  if routeInfo.routeCreationTime is not None else "" )
            print indent + "via %s, %s%s" % ( via, routeCreationTimeStr,
                  routeInfo.vias[ via ] )
            indent = " " * ( len( outputStr ) + 1 )

class MulticastRib6Routes( Model ):
   routes = Dict( help="A mapping from IPv6 prefix to MulticastRibRoute",
                  keyType=IpGenericPrefix, valueType=MulticastRibRoute )

   def render( self ):
      print 'Codes: C - connected, S - static, B - BGP'
      print

      for route in sorted( self.routes ):
         routeInfo = self.routes[ route ]
         outputStr = "%-6s %-17s [%d/%d]" % ( routeInfo.routeProtocol,
               route, routeInfo.routeMetric, routeInfo.routePreference )
         print outputStr
         indent = " " * 8
         for via in sorted( routeInfo.vias ):
            routeCreationTimeStr = ( ( str( routeInfo.routeCreationTime ) + ", " )
                  if routeInfo.routeCreationTime is not None else "" )
            print indent + "via %s, %s%s" % ( via, routeCreationTimeStr,
                  routeInfo.vias[ via ] )

class Counter( DeferredModel ):
   active = Bool( help="Counter active in hardware", optional=True )
   value = Int( help="Stored count retrieved from hardware when counter is active",
                optional=True )

class MfibHwRoute( DeferredModel ):
   programmed = Bool( help="Route programmed in hardware" )
   creationTime = Float( help="UTC time when the hardware route was created" )
   routeActivity = Float(
         help="Last polled UTC time when the route was seen as active in hardware",
         optional=True )

class MfibRoute( DeferredModel ):
   hwRoute = Submodel(
         help="Hardware route information",
         valueType=MfibHwRoute, optional=True )
   bytesCounter = Submodel(
         help="Hardware multicast traffic counter, in bytes, if supported",
         valueType=Counter, optional=True )
   packetsCounter = Submodel(
         help="Hardware multicast traffic counter, in packets, if supported",
         valueType=Counter, optional=True )

   incomingInterface = Interface( help="Incoming interface for the route",
                                  optional=True )
   incomingInterfaceFrr = \
         Interface( help="Incoming interface for fast failover route",
                    optional=True )
   notifyInterface = Interface( help="Notify interface for the route",
                                optional=True )
   staticFastdropInterface = Interface(
         help="Interface with static fastdrop installed", optional=True )

   rpaIndex = Int( help="Rendezvous point address index for the route",
                   optional=True )
   outgoingInterfaces = List( help="List of outgoing interfaces for the route",
                   valueType=Interface, optional=True )
   toCpu = Bool( help="Route has local listeners", optional=True )

   fastdropInterfaces = List( help="List of fastdrop interfaces",
                              valueType=Interface, optional=True )

class MfibGroup( DeferredModel ):
   routes = Dict( help="Multicast routes for this group",
                   keyType=IpGenericAddress, valueType=MfibRoute )

class MfibGroups( DeferredModel ):
   hwPollingInterval = Int( help="Hardware polling interval" )
   hwResourceFull = Bool( help="No space remaining in hardware" )
   groups = Dict( help="Map of multicast groups",
                  keyType=IpGenericAddress, valueType=MfibGroup )

class MfibRoutesCount( DeferredModel ):
   sgRoutes = Int( help="Number of (S,G) routes" )
   wcRoutes = Int( help="Number of (*,G) routes" )
   fastdropRoutes = Int( help="Number of Fastdrop routes" )
   moFrrRoutes = Int( help="Number of Multicast-only fast reroute routes" )
   prefixRoutes = Int( help="Number of Prefix routes" )

   unprogrammedSgRoutes = Int( help="Number of unprogrammed (S,G) routes" )
   unprogrammedWcRoutes = Int( help="Number of unprogrammed (*,G) routes" )
   unprogrammedFastdropRoutes = Int( help="Number of unprogrammed Fastdrop routes" )
   unprogrammedMoFrrRoutes = Int(
         help="Number of unprogrammed Multicast-only fast reroute routes" )
   unprogrammedPrefixRoutes = Int( help="Number of unprogrammed Prefix routes" )
