# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import LogMgrSyslogMsgs
import Tac
# pylint: disable-msg=W0611
from LoggingDefs import consoleLogFilename, consoleSyncLogFilename
from LoggingDefs import monitorLogFilename, monitorSyncLogFilename
from LoggingDefs import consoleSyncLogSkippedMsg, consoleSyncLogRotatedMsg
from LoggingDefs import consoleSyncLogRunCmdMsg
# pylint: enable-msg=W0611

# Logging CLI utility functions -- also used by the Logging CLI tests

# Map from syslog facility CLI to value
logFacilityCliToValueMap = { "kern" : 0,
                             "user" : 1,
                             "mail" : 2,
                             "daemon" : 3,
                             "auth" : 4,
                             "syslog" : 5,
                             "lpr" : 6,
                             "news" : 7,
                             "uucp" : 8,
                             "sys9" : 9,
                             "sys10" : 10,
                             "sys11" : 11,
                             "sys12" : 12,
                             "sys13" : 13,
                             "sys14" : 14,
                             "cron" : 15,
                             "local0" : 16,
                             "local1" : 17,
                             "local2" : 18,
                             "local3" : 19,
                             "local4" : 20,
                             "local5" : 21,
                             "local6" : 22,
                             "local7" : 23 }

# Map from syslog facility value to TACC
logFacilityValueToTaccMap = { 0 : "logKern",
                              1 : "logUser", 
                              2 : "logMail",
                              3 : "logDaemon",
                              4 : "logAuth",
                              5 : "logSyslog",
                              6 : "logLpr",
                              7 : "logNews",
                              8 : "logUucp",
                              9 : "logCron",
                              10 : "logAuthpriv",
                              11 : "logFtp",
                              12 : "logSys12",
                              13 : "logSys13",
                              14 : "logSys14",
                              15 : "logSys15",
                              16 : "logLocal0",
                              17 : "logLocal1",
                              18 : "logLocal2",
                              19 : "logLocal3",
                              20 : "logLocal4",
                              21 : "logLocal5",
                              22 : "logLocal6",
                              23 : "logLocal7" }

# Map from syslog facility TACC to CLI
logFacilityTaccToCliMap = { "logKern" : "kern",
                            "logUser" : "user", 
                            "logMail" : "mail",
                            "logDaemon" : "daemon",
                            "logAuth" : "auth",
                            "logSyslog" : "syslog",
                            "logLpr" : "lpr",
                            "logNews" : "news",
                            "logUucp" : "uucp",
                            "logCron" : "sys9",
                            "logAuthpriv" : "sys10",
                            "logFtp" : "sys11",
                            "logSys12" : "sys12",
                            "logSys13" : "sys13",
                            "logSys14" : "sys14",
                            "logSys15" : "cron",
                            "logLocal0" : "local0",
                            "logLocal1" : "local1",
                            "logLocal2" : "local2",
                            "logLocal3" : "local3",
                            "logLocal4" : "local4",
                            "logLocal5" : "local5",
                            "logLocal6" : "local6",
                            "logLocal7" : "local7" }

# Map from severity values to TACC severity names.
logSevValueToTaccMap = { 0 : "logEmergency",
                         1 : "logAlert",
                         2 : "logCritical",
                         3 : "logError",
                         4 : "logWarning",
                         5 : "logNotice",
                         6 : "logInfo",
                         7 : "logDebug" }

# Map from TACC severity names to severity values
logSevTaccToValueMap = { "logEmergency" : 0,
                         "logAlert" : 1,
                         "logCritical" : 2,
                         "logError" : 3,
                         "logWarning" : 4,
                         "logNotice" : 5,
                         "logInfo" : 6,
                         "logDebug" : 7 }

# Map from CLI severity names to severity values.
logSevCliToValueMap = { "emergencies" : 0,
                        "alerts" : 1,
                        "critical" : 2,
                        "errors" : 3,
                        "warnings" : 4,
                        "notifications" : 5,
                        "informational" : 6,
                        "debugging" : 7 }

# Map from TACC severity names to CLI severity names.
logSevTaccToCliMap = { "logEmergency" : "emergencies",
                       "logAlert" : "alerts",
                       "logCritical" : "critical",
                       "logError" : "errors",
                       "logWarning" : "warnings",
                       "logNotice" : "notifications",
                       "logInfo" : "informational",
                       "logDebug" : "debugging" }

logSevTaccToSyslogMap = { "logEmergency" : "emerg",
                          "logAlert" : "alert",
                          "logCritical" : "crit",
                          "logError" : "err",
                          "logWarning" : "warning",
                          "logNotice" : "notice",
                          "logInfo" : "info",
                          "logDebug" : "debug" }

# List of CLI severity level names
cliSeverities = [ ( 'emergencies', 'System is unusable' ),
                  ( 'alerts', 'Immediate action needed' ),
                  ( 'critical', 'Critical conditions' ),
                  ( 'errors', 'Error conditions' ),
                  ( 'warnings', 'Warning conditions' ),
                  ( 'notifications', 'Normal but significant conditions' ),
                  ( 'informational', 'Informational messages' ),
                  ( 'debugging', 'Debugging messages' ) ]

# Map CLI severity values to Logging Id
logSevValueToSendLogIdMap = { 0 : LogMgrSyslogMsgs.SYS_LOGMSG_EMRG,
                            1 : LogMgrSyslogMsgs.SYS_LOGMSG_ALERT,
                            2 : LogMgrSyslogMsgs.SYS_LOGMSG_CRIT,
                            3 : LogMgrSyslogMsgs.SYS_LOGMSG_ERR,
                            4 : LogMgrSyslogMsgs.SYS_LOGMSG_WARN,
                            5 : LogMgrSyslogMsgs.SYS_LOGMSG_NOTICE,
                            6 : LogMgrSyslogMsgs.SYS_LOGMSG_INFO,
                            7 : LogMgrSyslogMsgs.SYS_LOGMSG_DBG }

# Map from TACC timestamp format names to CLI format names.
logTimestampFormatTaccToCliMap = { "tsRFC3164" : "traditional",
                                   "tsRFC3339" : "high-resolution" }

logTimestampFormatCliToTaccMap = { "traditional"     : "tsRFC3164",
                                   "high-resolution" : "tsRFC3339" }

# Methods
def facilityValueToTacc( facility ):
   return logFacilityValueToTaccMap[ facility ]

def facilityCliToValue( facility ):
   return logFacilityCliToValueMap[ facility ]

def facilityTaccToCli( facility ):
   return logFacilityTaccToCliMap[ facility ]

def facilityTaccToValue( facility ):
   facility = facilityTaccToCli( facility )
   return facilityCliToValue( facility )

def severityValueToTacc( sev ):
   return logSevValueToTaccMap[ sev ]

def severityCliToValue( sev ):
   return logSevCliToValueMap[ sev ]

def severityTaccToCli( sev ):
   return logSevTaccToCliMap[ sev ]

def severityTaccToValue( sev ):
   return logSevTaccToValueMap[ sev ]

def severityTaccToSyslog( sev ):
   return logSevTaccToSyslogMap[ sev ]

def lowerTaccSeverity( sev1, sev2 ):
# The severity is in reverse order of the severity value, so
# the max of the two severity values
   if severityTaccToValue( sev1 ) >= severityTaccToValue( sev2 ):
      return sev1
   return sev2

def incTaccSeverity(sev, adj=1 ):
   # Lower or raise the severity by adj
   newSev = severityTaccToValue( sev ) + adj
   if newSev > 7: newSev = 7
   if newSev < 0: newSev = 0
   return severityValueToTacc( newSev )

def timestampFormatToTacc( tsf ):
   return logTimestampFormatCliToTaccMap[ tsf ]

def timestampFormatToCli( tsf ):
   return logTimestampFormatTaccToCliMap[ tsf ]

def logTrapSystemValue( facility=None, severity=None, tag="", contain="" ):
   flag = Tac.Value( 'LogMgr::LogFacSevFlag' )
   if facility is not None:
      flag.facility = True
   else:
      facility = 'logLocal4'
   if severity is not None:
      flag.severity = True
   else:
      severity = 'logInfo'

   return Tac.Value( 'LogMgr::LogTrapSystemConfig',
                     flag, facility, severity, tag, contain )

def doSanityCheck():
   for cli in logFacilityCliToValueMap.keys():
      x = logFacilityValueToTaccMap[ logFacilityCliToValueMap[ cli ] ]
      assert logFacilityTaccToCliMap[ x ] == cli

   for ts in logSevTaccToCliMap.keys():
      x = logSevCliToValueMap[ logSevTaccToCliMap[ ts ] ]
      assert logSevValueToTaccMap[ x ] == ts

   for cli in logTimestampFormatCliToTaccMap.keys():
      tacc = logTimestampFormatCliToTaccMap[ cli ]
      assert logTimestampFormatTaccToCliMap[ tacc ] == cli

# Rule for logging severity
maxSeverityDescLen = max( len( desc ) for ( n, desc ) in cliSeverities )
loggingSeverityMap_ = {
   name : '%-*s (severity=%d)' % ( maxSeverityDescLen,
                                   description,
                                   severityCliToValue( name ) )
   for ( name, description ) in cliSeverities }

class SeverityExpression( CliCommand.CliExpression ):
   expression = "SEVERITY_VALUE | SEVERITY_NAME"
   data = {
      "SEVERITY_VALUE" : CliMatcher.IntegerMatcher( 0, 7,
                                            helpdesc='Severity level value' ),
      "SEVERITY_NAME" : CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                          loggingSeverityMap_ )
      }
   @staticmethod
   def adapter( mode, args, argsList ):
      sev = args.pop( "SEVERITY_NAME", None )
      if sev is not None:
         # We can be in an iteration, but all current users
         # only have one occurance (SetRule).
         if isinstance( sev, list ):
            sev = sev[ 0 ]
         sev = severityCliToValue( sev )
      else:
         sev = args.pop( "SEVERITY_VALUE", None )
         if isinstance( sev, list ):
            sev = sev[ 0 ]

      if sev is not None:
         args[ "SEVERITY" ] = sev

# Simple class that is used to read logging output
# from different sources, terminated by the match
# string
class LogOutput:
   def output( self, match ):
      raise NotImplementedError

# Reads log output from an expect source. 
class LogOutputExpect( LogOutput ):
   def __init__( self, expecter ):
      self.expecter_ = expecter
   def output( self, match ):
      try:
         self.expecter_.expect( match, timeout=600 )
         return self.expecter_.before + self.expecter_.after
      except:
         self.expecter_.close( 1 )
         raise      

if __name__ == "__main__":
   doSanityCheck()
