#!/usr/bin/python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import optparse
import os
import re
import sys

from ExtensionMgr import ( 
      errors,
      logs,
)
import ExtensionMgrLib
import SignatureVerificationMapLib as SigVerifyMapLib
import SslCertKey
import SwiSignLib
import Tac

MgmtSslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

class LoadExtensionsError( errors.Error ):
   pass


class ParseError( errors.Error ):
   pass


def getFormatFromName( name ):
   if name.lower().endswith( '.rpm' ):
      return 'formatRpm'
   if name.lower().endswith( '.swix' ):
      return 'formatSwix'
   return 'formatUnknown'

def parseConfig( lines ):
   """Takes an iterable containing lines from a boot-extensions config file
   and returns a list of ( extension name, force, type, dependencies ) tuples."""
   extensions = []
   specialCharRe = re.compile( ".*[\x00-\x08\x0b-\x1f\x7f-\xff]+.*" )
   lineRe = re.compile( r'(?P<file>\S+)((\s+(?P<force>force|no)))?'
                        r'(\s+(?P<format>\S+)\s*(?P<deps>.*))?' )
   blankRe = re.compile( "^\\s*$" )
   lineNum = 0
   for line in lines:
      lineNum += 1
      if line.startswith( '#' ):
         continue
      elif blankRe.match( line ) is not None:
         continue
      elif specialCharRe.match( line ) is not None:
         msg = "Invalid character found on line %d" % lineNum
         raise ParseError( msg )
      m = lineRe.match( line )
      if m is None:
         msg = "Parse error on line %d: %s" % ( lineNum, line )
         raise ParseError( msg )
      groups = m.groupdict()
      if groups[ 'format' ] is None:
         groups[ 'format' ] = getFormatFromName( groups[ 'file' ] )
      extensions.append( ( groups[ 'file' ], groups[ 'force' ], groups[ 'format' ], 
                         groups[ 'deps' ] ) )
   return extensions


usage = """
LoadExtensions -f <config-file> -d <extensions dir> [options]

Reads the config file, installs the extensions specified in the config,
and generates output indicating the status of each extension, i.e. whether
each extension installed properly or not.

This program must be run as root.
"""
def main( argv, stdout=sys.stdout, stderr=sys.stderr ):
   parser = optparse.OptionParser( usage=usage )
   parser.add_option( "-d", "--dir", action="store",
      help="the extensions directory" )
   parser.add_option( "-f", "--config", action="store",
      help="the config file" )
   parser.add_option( "-o", "--output", action="store", default="",
      help="the output file (default: stdout)" )
   parser.add_option( "-q", "--quiet", action="store_true", default=False,
      help="suppress output (default: %default)" )
   parser.add_option( "-l", "--logging", action="store", default="",
      help="where to log the output, syslog if none specified" )

   options, args = parser.parse_args( args=argv )
   if args:
      parser.error( "Unrecognized argument" )
   if not options.config:
      parser.error( "Must specify -f <config file>" )
   if not options.dir:
      parser.error( "Must specify -d <extensions directory>" )
   if options.logging:
      ExtensionMgrLib.LOGFILE = options.logging

   def err( *args, **kwargs ):
      if not options.quiet:
         stderr.write( *args, **kwargs )

   if not os.path.exists( options.config ):
      msg = "%s not found: no extensions will be installed" % options.config
      raise LoadExtensionsError( msg )

   try:
      f = file( options.config, "r" )
      lines = f.readlines()
      f.close()
      extensionsToLoad = parseConfig( lines )
   except ParseError, e:
      raise LoadExtensionsError( str( e ) )
   except Exception, e:
      msg = "Error reading config file: %s" % e
      raise LoadExtensionsError( msg )

   if options.output:
      try:
         output = file( options.output, "w" )
      except Exception, e:
         msg = "Error opening output file: %s\n" % e
         raise LoadExtensionsError( msg )
   else:
      output = stdout

   # print output header
   output.write( "# --config=%s\n" % options.config )
   output.write( "# --dir=%s\n" % options.dir )

   # Construct a dummy Extension::Status instance to pass to ExtensionMgrLib
   # functions that need one.  This script runs before Sysdb starts, so it's
   # not possible to get the "real" Extension::Status.
   status = Tac.newInstance( "Extension::Status", "status" )
   for extensionName, force, pType, deps in extensionsToLoad:
      depsList = []
      if deps:
         # strip all the excess and get a list of the file names
         depsList = [ x.strip()[ 1:-1 ] for x in deps[ 1:-1 ].split( ',' ) ]
      shouldForce = force == 'force'
      path = extensionName

      if path[0] != '/':
         path = os.path.join( options.dir, path )
      ExtensionMgrLib.readExtensionFile( path, status, pType=pType, deps=depsList )
      info = ExtensionMgrLib.latestExtensionForName( extensionName, status )
      if info is None:
         err( "Extension %s is not present\n" % extensionName )
         # create a fake info
         key = Tac.Value( "Extension::InfoKey", extensionName,
                          pType or 'formatUnknown', 1 )
         info = Tac.newInstance( "Extension::Info", key )
         info.presence = "absent"
         info.status = "notInstalled"
         ExtensionMgrLib.printExtensionInfo( output, info )
         continue

      try:
         sigValid = _verifySignature( info )
         ExtensionMgrLib.installExtension( status, info, force=shouldForce )
      except errors.InstallError as e:
         err( "Error installing %s: %s\n" % ( extensionName, e ) )
      finally:
         ExtensionMgrLib.printExtensionInfo( output, info, sigValid=sigValid )

   output.close()

def _verifySignature( info ):
   if info.format == 'formatSwix':
      return _verifySwixSignature( info.filepath )
   # Only SWIXs have signature verification
   return None

def _verifySwixSignature( filepath ):
   ''' Verify SWIX signatures using info from the signature-verification
   mapping file, generate a syslog for signature status, and 
   return whether SWIX signature is valid (or None, which means the feature
   is not enabled)'''
   try:
      basename = os.path.basename( filepath )
      sigVerifyFileMgr = SigVerifyMapLib.FileMgr( 'extension' )
      sigVerifyAttrs = sigVerifyFileMgr.readAttrs()
      enabled = sigVerifyAttrs.get( SigVerifyMapLib.FILE_ATTR.ENFORCE_SIGNATURE )
      sslProfile = sigVerifyAttrs.get( SigVerifyMapLib.FILE_ATTR.SSL_PROFILE )
      if enabled == 'True':
         if sslProfile:
            trustedCertsPath = MgmtSslConstants.trustedCertsPath( sslProfile )
            trustedCerts = SslCertKey.getAllPem( trustedCertsPath )
            if not trustedCerts:
               ExtensionMgrLib.log( logs.EXTENSION_SIGNATURE_INVALID, basename, 
                     "Unable to verify signature. No trusted certificates defined." )
               return False
            sigValid, reason = SwiSignLib.verifySwixSignature( filepath, 
                                             trustedCerts, rootCAIsFile=False )
            if sigValid:
               ExtensionMgrLib.log( logs.EXTENSION_SIGNATURE_VALID, basename )
               return True
            else:
               ExtensionMgrLib.log( logs.EXTENSION_SIGNATURE_INVALID,
                                    basename, reason )
         else:
            ExtensionMgrLib.log( logs.EXTENSION_SIGNATURE_INVALID, basename, 
                     "Unable to verify signature. SSL profile is unknown." )
      else:
         return None # Feature is not enabled
   except IOError as e:
      ExtensionMgrLib.log( logs.EXTENSION_SIGNATURE_INVALID, basename,
            "Unable to verify signature: %s" % e )
   return False
