#!/usr/bin/python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ExtensionMgrLib, PyClient, Tac, Tracing
import optparse, os, re, sys

t0 = Tracing.trace0

usage = """
LoadExtensionStatus -f <extension-status-file> -d <extensions-dir> [options]

Populates Sysdb with extension status, including information about all
extension files in the extensions directory and the information in the
extension status file, which is expected to be output from the LoadExtensions
program.
"""

def parseLoadedExtensionFile( lines, extensionConfig, extensionStatus, err ):
   # This code is purposely tolerant of badly formatted data.  Lines that are
   # not properly formatted are ignored.  Each line is expected to contain 6
   # tab-separated columns: name, presence, presenceDetail, status,
   # statusDetail, signedStr, and installed rpms.  
   # That last column is a space-separated
   # list of rpm names that were installed by this extension.
   lineRe = re.compile(
      "(\\S+)\t(\\S+)\t(\\S+)\t([^\t]*)\t(\\S+)\t([^\t]*)\t([^\t]*)\t([^\t]*)\n" )
   blankRe = re.compile( "^\\s*$" )
   lineNum = 0
   index = 0

   deps = set() 

   for line in lines:
      lineNum += 1
      if line.startswith( '#' ):
         continue
      if blankRe.match( line ) is not None:
         continue
      m = lineRe.match( line )
      if m is None:
         msg = "Parse error on line %d: %s\n" % ( lineNum, line )
         err.write( msg )
         continue
      ( _name, pType, presence, presenceDetail, status, 
        statusDetail, signedStr, rpm ) = m.groups()
      t0( "Parsed status:", _name, ",", pType, ",", presence, ",", presenceDetail, 
          ",", status, ",", statusDetail, ",", signedStr, ",", rpm )
      info = ExtensionMgrLib.latestExtensionForName( _name, extensionStatus )
      if not info:
         err.write( "No Sysdb state found for extension %s\n" % _name )
         key = Tac.Value( "Extension::InfoKey", _name, 'formatUnknown', 1 )
         info = extensionStatus.info.newMember( key )
         info.presence = 'absent'
         info.presenceDetail = presenceDetail
         info.status = 'notInstalled'
         info.statusDetail = statusDetail
         continue
      # I think it's better to leave presence/presenceDetail alone since the
      # code that populated Sysdb's notion of this state ran later than the
      # code that wrote the status file.
      # info.presence = presence
      # info.presenceDetail = presenceDetail
      info.status = status
      info.statusDetail = statusDetail
      # On bootup, there is no need to restart any agent, so clear those install
      # "side-effects".
      info.agentsToRestart.clear()
      info.finalizationWarning = ""
      info.postError = ""
      if status in ( 'installed', 'forceInstalled' ):
         item = extensionConfig.installation.newMember( _name, pType, index )
         item.force = bool( status == 'forceInstalled' )
         if signedStr == 'notSigned':
            item.signatureIgnored = True
         index += 1
      rpms = rpm.split( " " )
      for r in rpms:
         deps.add( r )
         rpmPtr = info.package.get( r )
         if rpmPtr is None:
            err.write( "Error on line %d: no rpm %s in extension" % ( lineNum, r ) )
         else:
            info.installedPackage.addMember( rpmPtr )

   for info in extensionStatus.info.values():
      if info.filename in deps:
         if info.status == 'notInstalled':
            info.status = 'installed'
         rpmPtr = info.package.get( info.filename )
         info.installedPackage.addMember( rpmPtr )

class LoadExtensionStatusError( Exception ):
   pass

def loadExtensionStatus( path, extensionConfig, extensionStatus, err ):
   try:
      f = file( path, "r" )
      lines = f.readlines()
      parseLoadedExtensionFile( lines, extensionConfig, extensionStatus, err )
   except Exception, e:
      t0( "Failed to load extension status file", path, ":", e )
      raise LoadExtensionStatusError( str( e ) )

def main( argv, stderr=sys.stderr, stdout=sys.stdout ):
   parser = optparse.OptionParser( usage=usage )
   parser.add_option( "-d", "--dir", action="store",
      help="the extensions directory" )
   parser.add_option( "-f", "--config", action="store",
      help="the config file" )   
   parser.add_option( "-s", "--sysname", action="store", default="ar",
      help="system name (default: %default)" )
   options, args = parser.parse_args( args=argv )
   if len( args ) != 0:
      parser.error( "Unrecognized options: " + " ".join( args ) )
   if not options.config:
      parser.error( "Must specify -f <config file>" )
   if not options.dir:
      parser.error( "Must specify -d <extensions directory>" )

   pc = PyClient.PyClient( options.sysname, "Sysdb" )
   root = pc.agentRoot()
   status = root.entity[ ExtensionMgrLib.statusPath() ]
   config = root.entity[ ExtensionMgrLib.configPath() ]

   if status.initializationStatus != 'notInitialized':
      msg = "Error: extension status has already been initialized."
      raise LoadExtensionStatusError( msg )

   def updateStatus( state, details='' ):
      status.initializationStatus = state 
      status.initializationStatusDetail = details

   updateStatus( 'inProgress' )

   try:
      ExtensionMgrLib.readExtensionsDir( options.dir, status,
         continueOnError=True )
   except Exception, e:
      msg = "Error reading extensions directory: %s" % e
      updateStatus( 'failed', msg )
      raise LoadExtensionStatusError( msg )
   
   try:
      if os.path.exists( options.config ):
         loadExtensionStatus( options.config, config, status, stderr )
   except Exception, e:
      msg = "Error reading extensions status file %s: %s" % ( options.config, e )
      updateStatus( 'failed', msg )
      raise LoadExtensionStatusError( msg )

   updateStatus( 'completed' )

def name():
   return 'LoadExtensionStatus'
